/*:
 * @target MZ
 * @plugindesc スクリーンショット付きセーブプラグイン
 * @author GrayOgre
 * @url https://grayogre.info/rmmz/plugin/index.html
 * @help
 * 
 * このプラグインはサブ画面にスクリーンショットを含む詳細情報を表示する
 * セーブ／ロード画面を提供します。
 * 
 * Version 1.1　ピクチャ画像を設定出来るように改変
 * Version 1.2　プラグインコマンドで任意画像をサムネイルにしてセーブ画面へ遷移
 * Version 1.3　サムネイル生成サイズ・画質・現実時間表示を追加
 * Version 1.4　OpenSaveWithPicture でマップ名を上書き設定可能に
 * 
 * 制約事項
 * オートセーブ時のスクリーンショットには対応していません。
 * 
 * 移植元プラグイン
 * sai_Scenefile.js (sairi様作)
 * 
 * Copyright (c) 2021-2023 GrayOgre
 * Released under the MIT license
 * https://opensource.org/licenses/mit-license.php
 * 
 * @param pictureFile
 *   @text ピクチャ画像
 *   @desc 背景として使用するピクチャ画像のファイル名（拡張子無し）
 *   @type file
 *   @dir img/pictures
 *   @default
 * 
 * @param bgPicture
 *   @text 背景画像
 *   @desc 背景画像のファイル名です。（拡張子無し、PNGファイル）
 *   @type file
 *   @default
 * @param fontSize
 *   @text フォントサイズ
 *   @desc 表示用フォントサイス
 *   @type number
 *   @default 20
 * @param totalHeight
 *   @text 高さ（全体）
 *   @desc 画面全体の高さです。
 *   @type number
 *   @default 550
 * @param listWidth
 *   @text 選択ウインドウ幅
 *   @desc 選択ウィンドウの幅です。
 *   @type number
 *   @default 350
 * @param statusWidth
 *   @text ステータスウィンドウ幅
 *   @desc ステータスウィンドウの幅です
 *   @type number
 *   @default 400
 * @param saveListCols
 *   @text セーブリストの列数
 *   @desc セーブリストの横の列数
 *   @type number
 *   @default 2
 * @param saveSlotMax
 *   @text セーブスロットの最大数
 *   @desc セーブできる領域の最大数
 *   @type number
 *   @default 20
 * @param saveListVisibleRows
 *   @text セーブリスト表示行数
 *   @desc セーブリストに表示させる行数
 *   @type number
 *   @default 10
 * @param ssRatio
 *   @text スクリーンショット縮小率
 *   @desc 表示するスクリーンショットの縮小率(%)
 *   @type number
 *   @default 30
 *
 * @param thumbGenWidth
 *   @text サムネイル生成幅
 *   @desc サムネイルを生成する幅。0 の場合は元画像×縮小率を使用
 *   @type number
 *   @default 0
 *
 * @param thumbGenHeight
 *   @text サムネイル生成高さ
 *   @desc サムネイルを生成する高さ。0 の場合は元画像×縮小率を使用
 *   @type number
 *   @default 0
 *
 * @param thumbFormat
 *   @text サムネイル画像形式
 *   @desc サムネイル保存時の画像形式
 *   @type select
 *   @option 自動
 *   @value auto
 *   @option PNG
 *   @value png
 *   @option JPEG
 *   @value jpeg
 *   @default auto
 *
 * @param thumbJpegQuality
 *   @text JPEG画質
 *   @desc JPEG使用時の画質(1〜100)。値が高いほど高画質
 *   @type number
 *   @min 1
 *   @max 100
 *   @default 80
 *
 * @param windowOpacity
 *   @text ウィンドウ透過度
 *   @desc 表示するウィンドウの透過度
 *   @type number
 *   @default 0
 * @param showSaveTitle
 *   @text セーブ名表示
 *   @desc trueの場合、セーブ名を表示する
 *   @type boolean
 *   @default true
 * @param saveTitle_X
 *   @text セーブ名X座標
 *   @dexc セーブ名の表示位置のX座標
 *   @type number
 *   @default 0
 * @param saveTitle_Y
 *   @text セーブ名Y座標
 *   @dexc セーブ名の表示位置のY座標
 *   @type number
 *   @default 0
 * @param showThumbnail
 *   @text サムネイルの表示
 *   @desc trueの場合、サムネイルを表示する
 *   @type boolean
 *   @default true
 * @param thumbnail_X
 *   @text サムネイルX座標
 *   @desc サムネイルの表示位置のX座標
 *   @type number
 *   @default 0
 * @param thumbnail_Y
 *   @text サムネイルY座標
 *   @desc サムネイルの表示位置のY座標
 *   @type number
 *   @default 40
 * @param showSaveTime
 *   @text セーブ日時表示
 *   @desc trueの場合、セーブ日時を表示する
 *   @type boolean
 *   @default true
 * @param saveTime_X
 *   @text セーブ日時X座標
 *   @desc セーブ日時の表示位置のX座標
 *   @type number
 *   @default 0
 * @param saveTime_Y
 *   @text セーブ日時Y座標
 *   @desc セーブ日時の表示位置のY座標
 *   @type number
 *   @default 230
 * @param showLocation
 *   @text マップ名の表示
 *   @desc trueの場合マップ名を表示
 *   @type boolean
 *   @default true
 * @param locationTitle
 *   @text マップ名見出し
 *   @desc マップ名の前に表示する見出し
 *   @type string
 *   @default 記録場所　：
 * @param location_X
 *   @text マップ名X座標
 *   @desc マップ名の表示位置のX座標
 *   @type number
 *   @default 0
 * @param location_Y
 *   @text マップ名Y座標
 *   @desc マップ名の表示位置のY座標
 *   @type number
 *   @default 260
 * @param showPlayTime
 *   @text プレイ時間の表示
 *   @desc trueの場合プレイ時間を表示
 *   @type boolean
 *   @default true
 * @param playTimeTitle
 *   @text プレイ時間見出し
 *   @desc プレイ時間の前に表示する見出し
 *   @type string
 *   @default プレイ時間：
 * @param playTime_X
 *   @text プレイ時間X座標
 *   @desc プレイ時間の表示位置のX座標
 *   @type number
 *   @default 0
 * @param playTime_Y
 *   @text プレイ時間Y座標
 *   @desc プレイ時間の表示位置のY座標
 *   @type number
 *   @default 290
 *
 * @param showRealTime
 *   @text 現実時間の表示
 *   @desc trueの場合、セーブ時の現実の時間を表示する
 *   @type boolean
 *   @default false
 *
 * @param realTimeTitle
 *   @text 現実時間見出し
 *   @desc 現実時間の前に表示する見出し
 *   @type string
 *   @default 現実時間　：
 *
 * @param realTime_X
 *   @text 現実時間X座標
 *   @desc 現実時間の表示位置のX座標
 *   @type number
 *   @default 0
 *
 * @param realTime_Y
 *   @text 現実時間Y座標
 *   @desc 現実時間の表示位置のY座標
 *   @type number
 *   @default 410
 * @param showMoney
 *   @text 所持金表示
 *   @desc trueの場合、所持金を表示する
 *   @type boolean
 *   @default true
 * @param moneyTitle
 *   @text 所持金見出し
 *   @desc 所持金の前に表示する見出し
 *   @type string
 *   @default 所持金　　：
 * @param money_X
 *   @text 所持金X座標
 *   @desc 所持金の表示位置のX座標
 *   @type number
 *   @default 0
 * @param money_Y
 *   @text 所持金Y座標
 *   @desc 所持金の表示位置のY座標
 *   @type number
 *   @default 320
 * @param showLevel
 *   @text レベル表示
 *   @desc trueの場合、レベルを表示する
 *   @type boolean
 *   @default true
 * @param level_X
 *   @text レベルX座標
 *   @desc レベルの表示位置のX座標
 *   @type number
 *   @default 0
 * @param level_Y
 *   @text レベルY座標
 *   @desc レベルの表示位置のY座標
 *   @type number
 *   @default 350
 * @param showPageButton
 *   @text ページ送りボタン表示
 *   @desc trueの場合、セーブリストのページ送りボタンを表示する
 *   @type boolean
 *   @default true
 * 
 
 * 
 * @param enableActionMenu
 *   @text ファイル選択後のメニュー表示
 *   @desc true の場合、ファイル選択後に「セーブ／ロード」を表示します
 *   @type boolean
 *   @default true
 *
 * @param actionItemWidth
 *   @text メニュー項目 幅
 *   @desc 「セーブ／ロード」の当たり判定の幅
 *   @type number
 *   @default 220
 *
 * @param actionItemHeight
 *   @text メニュー項目 高さ
 *   @desc 「セーブ／ロード」の当たり判定の高さ
 *   @type number
 *   @default 48
 *
 * @param actionSaveX
 *   @text セーブ X座標
 *   @desc 「セーブ」の表示位置（画面座標）
 *   @type number
 *   @default 40
 *
 * @param actionSaveY
 *   @text セーブ Y座標
 *   @desc 「セーブ」の表示位置（画面座標）
 *   @type number
 *   @default 560
 *
 * @param actionLoadX
 *   @text ロード X座標
 *   @desc 「ロード」の表示位置（画面座標）
 *   @type number
 *   @default 280
 *
 * @param actionLoadY
 *   @text ロード Y座標
 *   @desc 「ロード」の表示位置（画面座標）
 *   @type number
 *   @default 560
 *
 * @param actionMenuFontSize
 *   @text メニュー文字サイズ
 *   @desc 「セーブ／ロード」の文字サイズ（0 なら既存のフォントサイズ）
 *   @type number
 *   @default 0
 *
 * @param actionSaveText
 *   @text セーブ表示名（アクション）
 *   @desc ファイル選択後メニューの「セーブ」表示名
 *   @type string
 *   @default セーブ
 *
 * @param actionLoadText
 *   @text ロード表示名（アクション）
 *   @desc ファイル選択後メニューの「ロード」表示名
 *   @type string
 *   @default ロード


* @param showSaveString
 *   @text Savestringの表示
 *   @desc trueの場合、Savestringを表示する
 *   @type boolean
 *   @default true
 *
 * @param saveString_X
 *   @text Savestring X座標
 *   @desc Savestringの表示位置のX座標
 *   @type number
 *   @default 0
 *
 * @param saveString_Y
 *   @text Savestring Y座標
 *   @desc Savestringの表示位置のY座標
 *   @type number
 *   @default 380
 * 
 * @param customString
 *   @text カスタム文字列
 *   @desc セーブ画面に表示する任意の文字列
 *   @type string
 *   @default Custom Text
 *
 * @param customString_X
 *   @text カスタム文字列X座標
 *   @desc カスタム文字列の表示位置のX座標
 *   @type number
 *   @default 0
 *
 * @param customString_Y
 *   @text カスタム文字列Y座標
 *   @desc カスタム文字列の表示位置のY座標
 *   @type number
 *   @default 420
 *
 * @command OpenSaveWithPicture
 * @text 画像指定でセーブ画面を開く
 * @desc 指定したピクチャをサムネイル用画像としてセットし、セーブ画面を開きます。
 *
 * @arg pictureFile
 * @text サムネイル用画像
 * @desc サムネイルに使用するピクチャ（img/pictures 内・拡張子なし）
 * @type file
 * @dir img/pictures
 * @default
 *
 * @arg locationName
 * @text マップ名(上書き)
 * @desc このコマンドから開いたセーブで使うマップ名。空欄なら通常のマップ名を使用
 * @type string
 * @default
 */

(() => {
    'use strict';

    const pluginName = document.currentScript.src.replace(/^.*\/(.*).js$/, function() {
        return arguments[1];
    });
    const parameters = PluginManager.parameters(pluginName);
    const pictureFile = parameters['pictureFile'];
    const fontSize = Number(parameters.fontSize);
    const totalHeight = Number(parameters.totalHeight);
    const listWidth = Number(parameters.listWidth);
    const statusWidth = Number(parameters.statusWidth);
    const saveListCols = Number(parameters.saveListCols);
    const saveSlotMax = Number(parameters.saveSlotMax);
    const saveListVisibleRows = Number(parameters.saveListVisibleRows);
    const ssRatio = Number(parameters.ssRatio);
    const thumbGenWidth = Number(parameters.thumbGenWidth || 0);
    const thumbGenHeight = Number(parameters.thumbGenHeight || 0);
    const thumbFormat = String(parameters.thumbFormat || 'auto');
    const thumbJpegQuality = Number(parameters.thumbJpegQuality || 80);
    const windowOpacity = Number(parameters.windowOpacity);
    const showSaveTitle = parameters.showSaveTitle === 'true';
    const saveTitle_X = Number(parameters.saveTitle_X);
    const saveTitle_Y = Number(parameters.saveTitle_Y);
    const showThumbnail = parameters.showThumbnail === 'true';
    const thumbnail_X = Number(parameters.thumbnail_X);
    const thumbnail_Y = Number(parameters.thumbnail_Y);
    const showSaveTime = parameters.showSaveTime === 'true';
    const saveTime_X = Number(parameters.saveTime_X);
    const saveTime_Y = Number(parameters.saveTime_Y);
    const showLocation = parameters.showLocation === 'true';
    const locationTitle = parameters.locationTitle;
    const location_X = Number(parameters.location_X);
    const location_Y = Number(parameters.location_Y);
    const showPlayTime = parameters.showPlayTime === 'true';
    const playTimeTitle = parameters.playTimeTitle;
    const playTime_X = Number(parameters.playTime_X);
    const playTime_Y = Number(parameters.playTime_Y);
    const showRealTime = parameters.showRealTime === 'true';
    const realTimeTitle = parameters.realTimeTitle;
    const realTime_X = Number(parameters.realTime_X);
    const realTime_Y = Number(parameters.realTime_Y);
    const showMoney = parameters.showMoney === 'true';
    const moneyTitle = parameters.moneyTitle;
    const money_X = Number(parameters.money_X);
    const money_Y = Number(parameters.money_Y);
    const showLevel = parameters.showLevel === 'true';
    const level_X = Number(parameters.level_X);
    const level_Y = Number(parameters.level_Y);
    const showPageButton = parameters.showPageButton === 'true';
    const enableActionMenu = parameters.enableActionMenu === 'true';
    const actionItemWidth = Number(parameters.actionItemWidth || 220);
    const actionItemHeight = Number(parameters.actionItemHeight || 48);
    const actionSaveX = Number(parameters.actionSaveX || 40);
    const actionSaveY = Number(parameters.actionSaveY || 560);
    const actionLoadX = Number(parameters.actionLoadX || 280);
    const actionLoadY = Number(parameters.actionLoadY || 560);
    const actionDeleteX = Number(parameters.actionDeleteX || 520);
    const actionDeleteY = Number(parameters.actionDeleteY || 560);
    const actionMenuFontSize = Number(parameters.actionMenuFontSize || 0);

        const actionSaveText = String(parameters.actionSaveText || "セーブ");
    const actionLoadText = String(parameters.actionLoadText || "ロード");
const showSaveString = parameters['showSaveString'] === 'true';
    const saveString_X = Number(parameters.saveString_X);
    const saveString_Y = Number(parameters.saveString_Y);
    const PROGRESS_VARIABLE_ID = 68; // 進捗を保存する変数ID
    const customString = parameters['customString'] || "Custom Text";
    const customString_X = Number(parameters.customString_X);
    const customString_Y = Number(parameters.customString_Y);

    //===============================
    // プラグインコマンド
    //===============================
    PluginManager.registerCommand(pluginName, "OpenSaveWithPicture", args => {
        const fileName = args.pictureFile || "";
        const locName = args.locationName || "";
        $gameTemp.setSavefileLocationName(locName);
        if (fileName) {
            const bitmap = ImageManager.loadPicture(fileName);
            $gameTemp.setSavefileBitmap(bitmap);
        } else {
            const snap = SceneManager.snap();
            $gameTemp.setSavefileBitmap(snap);
        }
        SceneManager.push(Scene_Save);
    });

    // Bitmap
    Bitmap.prototype.toDataURL = function() {
        if (!this._canvas) {
            return "";
        }
        const q = Math.max(0, Math.min(1, thumbJpegQuality / 100));
        if (thumbFormat === 'png') {
            return this._canvas.toDataURL('image/png');
        } else if (thumbFormat === 'jpeg') {
            return this._canvas.toDataURL('image/jpeg', q);
        } else {
            const png = this._canvas.toDataURL('image/png');
            const jpeg = this._canvas.toDataURL('image/jpeg', q);
            return (png.length < jpeg.length) ? png : jpeg;
        }
    };
    
    // DataManager
    
    DataManager._snapUrls = {};

    DataManager.maxSavefiles = function() {
        return saveSlotMax;
    };

    const _dataManager_saveGame = DataManager.saveGame;
    DataManager.saveGame = function(savefileId) {
        const result = _dataManager_saveGame.call(this, savefileId);

        const bitmap = this.makeSavefileBitmap();
        if (bitmap){
            const snapUrl = bitmap.toDataURL();
            this.saveThumbnail(savefileId, snapUrl);
        }

        return result;
    };

    const _dataManager_loadAllSavefileImages = DataManager.loadAllSavefileImages;
    DataManager.loadAllSavefileImages = function() {
        _dataManager_loadAllSavefileImages.call(this);
        for (let id = 0; id < DataManager.maxSavefiles(); id++) {
            this.loadThumbnail(id);
        }
    };

    DataManager.getThumbnail = function(savefileId) {
        return this._snapUrls[savefileId];
    };

    const _dataManager_makeSavefileInfo = DataManager.makeSavefileInfo;
    DataManager.makeSavefileInfo = function(){
        const info = _dataManager_makeSavefileInfo.call(this);

        const defaultLocation =
            $dataMap.displayName !== ""
                ? $dataMap.displayName
                : $dataMapInfos[$gameMap.mapId()].name;

        const overrideName = $gameTemp.getSavefileLocationName();
        info.location = overrideName && overrideName.length > 0 ? overrideName : defaultLocation;

        info.gold = $gameParty.gold();
        info.map_id = $gameMap.mapId();

        const savetime = new Date();
        const dateOptions = {
            year: "numeric",
            month: "2-digit",
            day: "2-digit",
            hour: "2-digit",
            minute: "2-digit"
        };
        info.savetime = savetime.toLocaleDateString("ja-JP", dateOptions);
        info.realTime = savetime.toLocaleTimeString("ja-JP", {
            hour: "2-digit",
            minute: "2-digit"
        });
        info.savestring = $gameVariables.value(PROGRESS_VARIABLE_ID) || "";
        return info;
    };
    
    // セーブファイル用のビットマップを作成
    
    DataManager.makeSavefileBitmap = function(){
        const bitmap = $gameTemp.getSavefileBitmap();
        if (!bitmap){
            return null;
        }
        let targetW, targetH;
        if (thumbGenWidth > 0 && thumbGenHeight > 0) {
            targetW = thumbGenWidth;
            targetH = thumbGenHeight;
        } else {
            const scale = ssRatio / 100;
            targetW = Math.floor(bitmap.width * scale);
            targetH = Math.floor(bitmap.height * scale);
        }
        const newBitmap = new Bitmap(targetW, targetH);
        newBitmap.blt(bitmap, 0, 0, bitmap.width, bitmap.height, 0, 0, targetW, targetH);
        return newBitmap;
    };

    DataManager.makeThumbnailFilename = function(id) {
        return "thumb%1".format(id);
    };

    DataManager.loadThumbnail = function(id) {
        if (!this._snapUrls[id]) {
            const filename = this.makeThumbnailFilename(id);
            if (StorageManager.exists(filename)) {
                this._snapUrls[id] = StorageManager.loadObject(filename).then(url => {
                    this._snapUrls[id] = url;
                    return 0;
                });
            }
        }
        return this._snapUrls[id];
    };

    DataManager.saveThumbnail = function(id, url) {
        if (this._snapUrls[id] !== url) {
            this._snapUrls[id] = url;
            const filename = this.makeThumbnailFilename(id);
            StorageManager.saveObject(filename, url);
        }
    };

    // SceneManager
    SceneManager.snapForBackground = function() {
        this._backgroundBitmap = this.snap();
    };

    // Game_Temp
    const _game_temp_initialize = Game_Temp.prototype.initialize;
    Game_Temp.prototype.initialize = function() {
        _game_temp_initialize.call(this);
        this._savefileBitmap = null;
        this._savefileLocationName = "";
    };

    Game_Temp.prototype.setSavefileBitmap = function(bitmap){
        this._savefileBitmap = bitmap;
    };
    
    Game_Temp.prototype.getSavefileBitmap = function() {
        if (this._savefileBitmap) {
            return this._savefileBitmap;
        } else {
            return SceneManager.backgroundBitmap();
        }
    };

    Game_Temp.prototype.setSavefileLocationName = function(text) {
        this._savefileLocationName = text || "";
    };

    Game_Temp.prototype.getSavefileLocationName = function() {
        return this._savefileLocationName || "";
    };

    // Scene_Map
    const _scene_map_callMenu = Scene_Map.prototype.callMenu;
    Scene_Map.prototype.callMenu = function() {
        $gameTemp.setSavefileBitmap(SceneManager.snap());
        $gameTemp.setSavefileLocationName(""); // メニューからはマップ名上書きなし
        _scene_map_callMenu.call(this);
    };

    // Scene_File
    const _scene_file_create = Scene_File.prototype.create;
    Scene_File.prototype.create = function() {
        _scene_file_create.call(this);
        this.createControlWindow();
        this.createStatusWindow();
        this.createControlButtons();
        this._statusWindow.opacity = windowOpacity;
        this._helpWindow.opacity = windowOpacity;
        this._listWindow.opacity = windowOpacity;
        this._controlWindow.opacity = windowOpacity;
        this._listWindow.statusWindow = this._statusWindow;
        this._statusWindow.listWindow = this._listWindow;
    };

    Scene_File.prototype.createControlWindow = function() {
        const rect = this.controlWindowRect();
        this._controlWindow = new Window_SaveListControl(rect);
        this.addWindow(this._controlWindow);
    };

    Scene_File.prototype.createStatusWindow = function() {
        const rect = this.statusWindowRect();
        this._statusWindow = new Window_SaveFileStatus(rect);
        this._statusWindow.setMode(this.mode());
        this.addWindow(this._statusWindow);
    };

    Scene_File.prototype.listWindowRect = function() {
        const ww = listWidth - 80;
        const wh = (showPageButton) ? totalHeight - this.calcWindowHeight(1, true) - 8 : totalHeight;
        const wx = Math.floor((Graphics.boxWidth - listWidth - statusWidth) / 2) - 20;
        const wy = this.mainAreaTop() + 150;
        return new Rectangle(wx, wy, ww, wh);
    };

    Scene_File.prototype.helpWindowRect = function() {
        const rect = this.listWindowRect();
        const ww = statusWidth;
        const wh = this.calcWindowHeight(1, true);
        const wx = rect.x + rect.width + 160;
        const wy = this.mainAreaTop() + 155;
        return new Rectangle(wx, wy, ww, wh);
    };

    Scene_File.prototype.controlWindowRect = function() {
        const rect = this.listWindowRect();
        const ww = rect.width;
        const wh = totalHeight - rect.height;
        const wx = rect.x;
        const wy = rect.y + rect.height;
        return new Rectangle(wx, wy, ww, wh);
    };

    Scene_File.prototype.statusWindowRect = function() {
        const rect = this.helpWindowRect();
        const ww = statusWidth;
        const wh = totalHeight - rect.height;
        const wx = rect.x;
        const wy = rect.y + rect.height;
        return new Rectangle(wx, wy, ww, wh);
    };

    Scene_File.prototype.createControlButtons = function() {
        this._pageUpButton = new Sprite_Button("pageup");
        this._pageUpButton.x = 4;
        this._pageUpButton.y = 0;
        this._controlWindow.addButton(this._pageUpButton);
        this._pageDownButton = new Sprite_Button("pagedown");
        this._pageDownButton.x = this._controlWindow.width - this._pageDownButton.width * 2 + 16;
        this._pageDownButton.y = 0;
        this._controlWindow.addButton(this._pageDownButton);
    };

    const _scene_file_start = Scene_File.prototype.start;
    Scene_File.prototype.start = function() {
        _scene_file_start.call(this);
        this._listWindow.ensureCursorVisible();
        this._listWindow.callUpdateHelp();
    };

    const _scene_file_createBackground = Scene_File.prototype.createBackground;
    Scene_File.prototype.createBackground = function() {
        _scene_file_createBackground.call(this);
        
        if (pictureFile) {
            this._pictureSprite = new Sprite();
            const bitmap = ImageManager.loadPicture(pictureFile);

            bitmap.addLoadListener(() => {
                this._pictureSprite.bitmap = bitmap;
                this._pictureSprite.opacity = 255;
                this._pictureSprite.z = 10;
                this._pictureSprite.x = 0;
                this._pictureSprite.y = 0;
            });
            this.addChild(this._pictureSprite);
        }
        if (this._pictureSprite) {
            this.setChildIndex(this._pictureSprite, this.children.length - 1);
        }
    };

    // Window_Base
    Window_Base.prototype.setFontSize = function(size) {
        this.contents.fontSize = size;
    };

    // Window_SavefileList
    const _window_savefilelist_initialize = Window_SavefileList.prototype.initialize;
    Window_SavefileList.prototype.initialize = function(rect) {
        _window_savefilelist_initialize.call(this, rect);
        this.setFontSize(fontSize);
    };

    Window_SavefileList.prototype.numVisibleRows = function() {
        return saveListVisibleRows;
    };

    Window_SavefileList.prototype.maxCols = function() {
        return saveListCols;
    };

    const _window_saveFileList_callUpdateHelp = Window_SavefileList.prototype.callUpdateHelp;
    Window_SavefileList.prototype.callUpdateHelp = function() {
        _window_saveFileList_callUpdateHelp.call(this);
        if (this.active && this.statusWindow) {
            this.statusWindow.setId(this.indexToSavefileId(this.index()));
        }
    };

    Window_SavefileList.prototype.drawItem = function(index) {
        const savefileId = this.indexToSavefileId(index);
        const rect = this.itemRectWithPadding(index);
        this.resetTextColor();
               this.changePaintOpacity(this.isEnabled(savefileId));
        this.drawTitle(savefileId, rect.x, rect.y + 4);
    };

    // Window_Help
    const _window_help_resetFontSettings = Window_Help.prototype.resetFontSettings;
    Window_Help.prototype.resetFontSettings = function() {
        _window_help_resetFontSettings.call(this);
        if (SceneManager._scene instanceof Scene_File) {
            this.setFontSize(fontSize);
        }
    };

    // Window_SaveListControl
    function Window_SaveListControl() {
        this.initialize(...arguments);
    }
    
    Window_SaveListControl.prototype = Object.create(Window_Base.prototype);
    Window_SaveListControl.prototype.constructor = Window_SaveListControl;

    Window_SaveListControl.prototype.initialize = function(rect) {
        Window_Base.prototype.initialize.call(this, rect);
    };

    Window_SaveListControl.prototype.addButton = function(button) {
        this.addInnerChild(button);
    };

    // Window_SaveFileStatus
    function Window_SaveFileStatus() {
        this.initialize(...arguments);
    }

    Window_SaveFileStatus.prototype = Object.create(Window_Base.prototype);
    Window_SaveFileStatus.prototype.constructor = Window_SaveFileStatus;

    Window_SaveFileStatus.prototype.initialize = function(rect) {
        Window_Base.prototype.initialize.call(this, rect);
        this.setFontSize(fontSize);
        this._id = 1;
    };

    Window_SaveFileStatus.prototype.setMode = function(mode) {
        this._mode = mode;
    };

    Window_SaveFileStatus.prototype.setId = function(id) {
        this._id = id;
        this.refresh();
    };

    Window_SaveFileStatus.prototype.refresh = function() {
        this.contents.clear();
        const id = this._id;
        const info = DataManager.savefileInfo(id);
        const valid = DataManager.savefileExists(id);
        this.drawSaveTitle(id, saveTitle_X, saveTitle_Y);
        if (info) {
            this.drawSaveTime(info, valid, saveTime_X, saveTime_Y);
            this.drawLocation(info, valid, location_X, location_Y);
            this.drawPlayTime(info, valid, playTime_X, playTime_Y);
            this.drawRealTime(info, valid, realTime_X, realTime_Y);
            this.drawMoney(info, valid, money_X, money_Y);
            this.drawLevel(info, valid, level_X, level_Y);
            this.drawSnappedImage(id, valid, thumbnail_X, thumbnail_Y);
            this.drawSaveString(info, valid, saveString_X, saveString_Y);
        }
        this.drawCustomString(customString, customString_X, customString_Y);
    };

    Window_SaveFileStatus.prototype.drawSaveTitle = function(id, x, y) {
        if (showSaveTitle) {
            if (id === 0) {
                this.drawText(TextManager.autosave, x, y, 180);
            } else {
                this.drawText(TextManager.file + " " + id, x, y, 180);
            }
        }
    };

    Window_SaveFileStatus.prototype.drawSaveTime = function(info, valid, x, y) {
        if (showSaveTime && valid && info.savetime) {
            this.drawText(info.savetime, x, y, 200);
        }
    };

    Window_SaveFileStatus.prototype.drawLocation = function(info, valid, x, y) {
        if (showLocation && valid && info.location) {
            const drawnText = locationTitle + ' ' + (info.location ? info.location : "");
            this.drawTextEx(drawnText, x, y, 200);
        }
    };

    Window_SaveFileStatus.prototype.drawPlayTime = function(info, valid, x, y) {
        if (showPlayTime && valid && info.playtime) {
            const drawnText = playTimeTitle + ' ' + (info.playtime ? info.playtime : "");
            this.drawTextEx(drawnText, x, y, 200);
        }
    };

    Window_SaveFileStatus.prototype.drawRealTime = function(info, valid, x, y) {
        if (showRealTime && valid && info.realTime) {
            const drawnText = realTimeTitle + ' ' + (info.realTime ? info.realTime : "");
            this.drawTextEx(drawnText, x, y, 200);
        }
    };

    Window_SaveFileStatus.prototype.drawMoney = function(info, valid, x, y) {
        if (showMoney && valid && typeof info.gold === 'number') {
            const drawnText = moneyTitle + ' ' + info.gold + ' ' + TextManager.currencyUnit;
            this.drawTextEx(drawnText, x, y, 200);
        }
    };

    Window_SaveFileStatus.prototype.drawLevel = function(info, valid, x, y) {
        if (showLevel && valid && info.level) {
            const drawnText = TextManager.levelA + ' ' + info.level;
            this.drawTextEx(drawnText, x, y, 60);
        }
    };

    Window_SaveFileStatus.prototype.drawSnappedImage = function(savefileId, valid, x, y) {
        const snapUrl = DataManager.getThumbnail(savefileId);
        if (showThumbnail && valid && snapUrl) {
            const bitmap = ImageManager.loadThumbnailFromUrl(snapUrl);
    
            if (bitmap) {
                this.changePaintOpacity(true);
                bitmap.addLoadListener(function() {
                    const thumbnailWidth = bitmap.width * 0.8;
                    const thumbnailHeight = bitmap.height * 0.8;
                    this.contents.blt(
                        bitmap,
                        0,
                        0,
                        bitmap.width,
                        bitmap.height,
                        x,
                        y,
                        thumbnailWidth,
                        thumbnailHeight
                    );
                }.bind(this));
            } else {
                console.error("Bitmapの読み込みに失敗しました。URL:", snapUrl);
            }
        }
    };
    
    Window_SaveFileStatus.prototype.drawSaveString = function(info, valid, x, y) {
        if (showSaveString && valid && info.savestring) {
            this.drawTextEx(info.savestring, x, y, 200);
        }
    };
    
    Window_SaveFileStatus.prototype.drawCustomString = function(text, x, y) {
        if (text) {
            this.drawTextEx(text, x, y, this.contents.width - x);
        }
    };

    const _window_saveFileStatus_resetFontSettings = Window_SaveFileStatus.prototype.resetFontSettings;
    Window_SaveFileStatus.prototype.resetFontSettings = function() {
        _window_saveFileStatus_resetFontSettings.call(this);
        this.setFontSize(fontSize);
    };

    // ImageManager
    ImageManager.loadThumbnailFromUrl = function(url) {
        if (typeof url !== "string") {
            console.error("無効なURL: 文字列が期待されましたが、受け取ったのは", url);
            return null;
        }
        
        const cache = url.includes("/system/") ? this._system : this._cache;
        if (!cache[url]) {
            cache[url] = Bitmap.loadThumbnail(url);
        }
        return cache[url];
    };
    
    Bitmap.loadThumbnail = function(url) {
        const bitmap = Object.create(Bitmap.prototype);
        bitmap.initialize();
        bitmap._url = url;
        bitmap.isThumbnail = true;
        bitmap._startLoading();
        return bitmap;
    };
    
    Bitmap.prototype._startLoading = function() {
        this._image = new Image();
        this._image.onload = this._onLoad.bind(this);
        this._image.onerror = this._onError.bind(this);
        this._destroyCanvas();
        this._loadingState = "loading";
        if (Utils.hasEncryptedImages() && !this.isThumbnail) {
            this._startDecrypting();
        } else {
            this._image.src = this._url;
            if (this._image.width > 0) {
                this._image.onload = null;
                this._onLoad();
            }
        }
    };



    //========================================
    // ファイル選択後「セーブ／ロード」メニュー
    //========================================
    function Window_GOSSFileAction() {
        this.initialize(...arguments);
    }

    Window_GOSSFileAction.prototype = Object.create(Window_Command.prototype);
    Window_GOSSFileAction.prototype.constructor = Window_GOSSFileAction;

    Window_GOSSFileAction.prototype.initialize = function() {
        const rect = new Rectangle(0, 0, Graphics.boxWidth, Graphics.boxHeight);
        Window_Command.prototype.initialize.call(this, rect);
        this.openness = 255;
        this.opacity = 0;
        this.contentsOpacity = 255;
        this._savefileId = 1;
        this._canSave = false;
        this._canLoad = false;
        this._canDelete = false;
        if (actionMenuFontSize > 0) {
            this.contents.fontSize = actionMenuFontSize;
        }
        this.deactivate();
    };

    Window_GOSSFileAction.prototype.updatePadding = function() {
        this.padding = 0;
    };

    Window_GOSSFileAction.prototype.makeCommandList = function() {
        this.addCommand(actionSaveText, "save", this._canSave);
        this.addCommand(actionLoadText, "load", this._canLoad);
        };

    Window_GOSSFileAction.prototype.setSavefileId = function(savefileId) {
        this._savefileId = savefileId;
    };

    Window_GOSSFileAction.prototype.setEnabledFlags = function(canSave, canLoad, canDelete) {
        this._canSave = !!canSave;
        this._canLoad = !!canLoad;
        this._canDelete = !!canDelete;
        this.refresh();
    };

    Window_GOSSFileAction.prototype.itemRect = function(index) {
        const symbol = this.commandSymbol(index);
        const w = actionItemWidth;
        const h = actionItemHeight;
        let x = 0;
        let y = 0;
        if (symbol === "save") {
            x = actionSaveX;
            y = actionSaveY;
        } else if (symbol === "load") {
            x = actionLoadX;
            y = actionLoadY;
        } else if (symbol === "delete") {
            x = actionDeleteX;
            y = actionDeleteY;
        }
        return new Rectangle(x, y, w, h);
    };

    Window_GOSSFileAction.prototype.maxCols = function() {
        return 3;
    };

    Window_GOSSFileAction.prototype.numVisibleRows = function() {
        return 1;
    };

    Window_GOSSFileAction.prototype.drawItem = function(index) {
        const rect = this.itemRect(index);
        const align = "center";
        this.resetTextColor();
        this.changePaintOpacity(this.isCommandEnabled(index));
        this.drawText(this.commandName(index), rect.x, rect.y, rect.width, align);
    };

    Window_GOSSFileAction.prototype.hitTest = function(x, y) {
        if (this.isOpen() && this.visible) {
            const touchPos = new Point(x, y);
            for (let i = 0; i < this.maxItems(); i++) {
                const rect = this.itemRect(i);
                if (rect.contains(touchPos.x, touchPos.y)) return i;
            }
        }
        return -1;
    };

    
    // [Patch] MZ環境で canvasToLocalX/Y が存在しない場合の互換処理
    // TouchInput はキャンバス座標なので、WindowLayer 変換 → ウィンドウ内座標へ落とす
    if (!Window_GOSSFileAction.prototype.canvasToLocalX) {
        Window_GOSSFileAction.prototype.canvasToLocalX = function(x) {
            let lx = x;
            const layer = this._windowLayer || this.parent;
            if (layer && typeof layer.canvasToLocalX === "function") {
                lx = layer.canvasToLocalX(x);
            }
            // ウィンドウの左上基準、内側（padding）基準へ
            return lx - this.x - (this.padding || 0);
        };
    }
    if (!Window_GOSSFileAction.prototype.canvasToLocalY) {
        Window_GOSSFileAction.prototype.canvasToLocalY = function(y) {
            let ly = y;
            const layer = this._windowLayer || this.parent;
            if (layer && typeof layer.canvasToLocalY === "function") {
                ly = layer.canvasToLocalY(y);
            }
            return ly - this.y - (this.padding || 0);
        };
    }

    Window_GOSSFileAction.prototype.onTouchSelect = function(trigger) {
        const lastIndex = this.index();
        const x = this.canvasToLocalX(TouchInput.x);
        const y = this.canvasToLocalY(TouchInput.y);
        const hitIndex = this.hitTest(x, y);
        if (hitIndex >= 0) {
            this.select(hitIndex);
            if (trigger && this.isTouchOkEnabled()) {
                this.processOk();
            }
        } else if (trigger && lastIndex >= 0) {
            this.processCancel();
        }
    };

    // Scene_File に組み込み
    const _goss_scene_file_create = Scene_File.prototype.create;
    Scene_File.prototype.create = function() {
        _goss_scene_file_create.call(this);

        if (enableActionMenu) {
            this.createGOSSActionWindow();
            // ファイル選択時の挙動を差し替え
            if (this._listWindow) {
                this._listWindow.setHandler("ok", this.onGOSSSelectFile.bind(this));
            }
        }
    };

const _goss_scene_file_update = Scene_File.prototype.update;
Scene_File.prototype.update = function() {
    _goss_scene_file_update.call(this);
    if (enableActionMenu) {
        this.gossUpdateActionWindow();
    }
};


    Scene_File.prototype.createGOSSActionWindow = function() {
        this._gossActionWindow = new Window_GOSSFileAction();
        const _saveFn = this.onGOSSActionSave || Scene_File.prototype.onGOSSActionSave;
        const _loadFn = this.onGOSSActionLoad || Scene_File.prototype.onGOSSActionLoad;
        const _cancelFn = this.onGOSSActionCancel || Scene_File.prototype.onGOSSActionCancel;

        this._gossActionWindow.setHandler("save", (_saveFn ? _saveFn.bind(this) : function(){}));
        this._gossActionWindow.setHandler("load", (_loadFn ? _loadFn.bind(this) : function(){}));
        this._gossActionWindow.setHandler("cancel", (_cancelFn ? _cancelFn.bind(this) : function(){}));

        this.addWindow(this._gossActionWindow);
        // 常時表示（ただし操作はファイル決定後にアクティブ化）
        this._gossActionWindow.show();
        this._gossActionWindow.deactivate();
        this.gossUpdateActionWindow();
    };

    
Scene_File.prototype.gossCanSaveInThisScene = function() {
    // タイトル画面付近では $gameSystem は存在しますが、マップ未セットのため mapId() が 0 になります
    // その状態ではセーブを禁止します
    const inGameMap = $gameMap && typeof $gameMap.mapId === "function" && $gameMap.mapId() > 0;
    const saveEnabled = $gameSystem && typeof $gameSystem.isSaveEnabled === "function" ? $gameSystem.isSaveEnabled() : true;
    return !!inGameMap && !!saveEnabled;
};

Scene_File.prototype.gossUpdateActionWindow = function() {
    if (!this._gossActionWindow || !this._listWindow) return;
    const savefileId = this.savefileId();
    if (this._gossLastActionSavefileId === savefileId) return;
    this._gossLastActionSavefileId = savefileId;

    const canSave = this.gossCanSaveInThisScene();
    const exists = DataManager.savefileExists(savefileId);
    const canLoad = exists;

    this._gossActionWindow.setSavefileId(savefileId);
    this._gossActionWindow.setEnabledFlags(canSave, canLoad, false);
};
Scene_File.prototype.onGOSSSelectFile = function() {
        SoundManager.playOk(); // ★ 追加：ファイル決定時のSE
        const savefileId = this.savefileId();
        const canSave = this.gossCanSaveInThisScene();
        const canLoad = DataManager.savefileExists(savefileId);
        const canDelete = DataManager.savefileExists(savefileId);

        this._gossActionWindow.setSavefileId(savefileId);
        this._gossActionWindow.setEnabledFlags(canSave, canLoad, false);

        this._listWindow.deactivate();
        this._gossActionWindow.select(0);
        this._gossActionWindow.activate();
    };

    Scene_File.prototype.onGOSSActionCancel = function() {
        this._gossActionWindow.deactivate();
        this._listWindow.activate();
    };

    
    // -------------------------------------------------------------------------
    // 確認ダイアログ（FesStyleConfirm 互換）
    // -------------------------------------------------------------------------
    Scene_File.prototype.gossHasConfirm = function() {
        return (typeof Window_Confirm !== "undefined") && (typeof this.createConfirmWindow === "function");
    };

    Scene_File.prototype.gossOpenConfirm = function(mode, onOk) {
        // mode: "save" or "load"
        if (!this.gossHasConfirm()) {
            onOk();
            return;
        }

        this.createConfirmWindow();

        // ハンドラを差し替え（GOSSSave側で完結）
        this._confirmWindow.setHandler("ok", () => {
            onOk();
            this._confirmWindow.close();
            this._confirmWindow.deactivate();
        });
        this._confirmWindow.setHandler("cancel", () => {
            this._confirmWindow.close();
            this._confirmWindow.deactivate();
            // アクションウィンドウに戻す
            if (this._gossActionWindow) {
                this._gossActionWindow.activate();
            } else if (this.activateListWindow) {
                this.activateListWindow();
            }
        });

        // 上書き確認（saveのみ）
        if (mode === "save" && typeof this._confirmWindow.setOverwrite === "function") {
            const savefileId = this.savefileId();
            const isOverwrite = !!DataManager.savefileInfo(savefileId);
            this._confirmWindow.setOverwrite(isOverwrite);
        }

        // 表示開始
        this._confirmWindow.startConfirm(mode);
    };
Scene_File.prototype.onGOSSActionSave = function() {
        const savefileId = this.savefileId();
        this._gossActionWindow.deactivate();


        const doSave = () => {
        // 既存の Scene_Save の処理が使えるならそれを優先
        if (typeof this.executeSave === "function") {
            this.executeSave(savefileId);
            return;
        }

        SoundManager.playSave();
        DataManager.saveGame(savefileId).then(() => {
            this._listWindow.refresh();
            this._listWindow.callUpdateHelp();
            this.onGOSSActionCancel();
        }).catch(() => {
            SoundManager.playBuzzer();
            this.onGOSSActionCancel();
        });
    };

    

        // 確認ダイアログを表示してから実行
        this.gossOpenConfirm("save", doSave);
    };

    Scene_File.prototype.onGOSSActionLoad = function() {
        const savefileId = this.savefileId();
        this._gossActionWindow.deactivate();

        const doLoad = () => {
            // ロード前の音を残さない（標準 + MUSH）
            AudioManager.stopAll();
            if (typeof AudioManager.stopAllMushBgm === "function") AudioManager.stopAllMushBgm(0);
            if (typeof AudioManager.stopAllMushBgs === "function") AudioManager.stopAllMushBgs(0);
            if (typeof AudioManager.closeFadingBgms === "function") AudioManager.closeFadingBgms();
            if (typeof AudioManager.closeFadingBgss === "function") AudioManager.closeFadingBgss();

            const afterLoad = () => {
                // ロード後、マップに入ったタイミングで音状態を確定させる
                if ($gameTemp) $gameTemp._gossPostLoadAudioApply = true;

                SoundManager.playLoad();
                SceneManager.goto(Scene_Map);
            };

            // 既存の Scene_Load の処理が使えるならそれを優先
            if (typeof this.executeLoad === "function") {
                // onLoadSuccess をフック（bind 未定義対策：存在確認をしてから）
                const _onLoadSuccess = this.onLoadSuccess;
                this.onLoadSuccess = function() {
                    if (typeof _onLoadSuccess === "function") _onLoadSuccess.call(this);
                    afterLoad();
                };
                this.executeLoad(savefileId);
                return;
            }

            DataManager.loadGame(savefileId).then(() => {
                afterLoad();
            }).catch(() => {
                SoundManager.playBuzzer();
                this.onGOSSActionCancel();
            });
        };

        // 確認ダイアログを表示してから実行
        this.gossOpenConfirm("load", doLoad);
    };



    // オートセーブ時にもサムネイルを保存
    const _dataManager_autoSaveGame = DataManager.autoSaveGame;
    DataManager.autoSaveGame = function() {
        const result = _dataManager_autoSaveGame.call(this);

        const bitmap = this.makeSavefileBitmap();
        if (bitmap) {
            const snapUrl = bitmap.toDataURL();
            this.saveThumbnail(0, snapUrl);
        }

        return result;
    };

    
    

// -------------------------------------------------------------------------
// ロード直後の音（標準 + MUSH）を「1歩待ち」なしで反映する
// -------------------------------------------------------------------------
const _GOSS_Scene_Map_start = Scene_Map.prototype.start;
Scene_Map.prototype.start = function() {
    _GOSS_Scene_Map_start.call(this);

    if ($gameTemp && $gameTemp._gossPostLoadAudioApply) {
        $gameTemp._gossPostLoadAudioApply = false;

        // セーブデータ側の状態を反映（標準BGM/BGS等）
        if ($gameSystem && typeof $gameSystem.onAfterLoad === "function") {
            $gameSystem.onAfterLoad();
        }

        // マップ設定の自動再生（必要なら）
        if ($gameMap && typeof $gameMap.autoplay === "function") {
            $gameMap.autoplay();
        }

        // MUSHの空間音量を即時更新（1歩動かないと反映されない問題対策）
        if ($gamePlayer) {
            if (typeof $gamePlayer.requestSpatialAudioUpdate_full === "function") {
                $gamePlayer.requestSpatialAudioUpdate_full();
            } else if (typeof $gamePlayer.requestSpatialAudioUpdate_general === "function") {
                $gamePlayer.requestSpatialAudioUpdate_general();
            } else if (typeof $gamePlayer.requestSpatialAudioUpdate === "function") {
                $gamePlayer.requestSpatialAudioUpdate();
            }
        }
    }
};


// -------------------------------------------------------------------------
// [Fix] タイトル→コンティニュー直後は「セーブ」を禁止する
// 目的：イベントコマンド「タイトルに戻る」後にタイトルからコンティニューした場合、
//       本来セーブ不可にしたいのにセーブできてしまう問題を確実に防ぐ。
// 方針：タイトルの「コンティニュー」を実行したかどうかをフラグで管理し、
//       マップに入った時点（Scene_Map.start）で解除する。
//       既存コードは全保持し、判定関数だけを上書きで拡張する。
// -------------------------------------------------------------------------
let _gossFromTitleContinue = false;

// タイトルの「コンティニュー」経由を検出
if (typeof Scene_Title !== "undefined" && Scene_Title.prototype && Scene_Title.prototype.commandContinue) {
    const _GOSS_Scene_Title_commandContinue = Scene_Title.prototype.commandContinue;
    Scene_Title.prototype.commandContinue = function() {
        _gossFromTitleContinue = true;
        _GOSS_Scene_Title_commandContinue.call(this);
    };
}

// マップに入ったタイミングで解除（ロード成功後、Scene_Mapに遷移してから start が呼ばれる）
if (typeof Scene_Map !== "undefined" && Scene_Map.prototype && Scene_Map.prototype.start) {
    const _GOSS_Scene_Map_start_ForTitleGuard = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function() {
        _GOSS_Scene_Map_start_ForTitleGuard.call(this);
        _gossFromTitleContinue = false;
    };
}

// 既存の可否判定を「全保持」のまま拡張（上書き）
// 既存：mapId>0 かつ isSaveEnabled()==true のみ
// 追加：タイトル→コンティニュー直後は常に false
if (Scene_File && Scene_File.prototype && typeof Scene_File.prototype.gossCanSaveInThisScene === "function") {
    const _GOSS_gossCanSaveInThisScene_Original = Scene_File.prototype.gossCanSaveInThisScene;
    Scene_File.prototype.gossCanSaveInThisScene = function() {
        if (_gossFromTitleContinue) return false;
        return _GOSS_gossCanSaveInThisScene_Original.call(this);
    };
}

})();