/*:
 * @param BgvFolder
 * @text BGVフォルダ
 * @type string
 * @default bgv
 *
 * @param DefaultVolume
 * @text 既定音量(0〜100)
 * @type number
 * @min 0
 * @max 100
 * @default 100
 *
 * @param DefaultPitch
 * @text 既定ピッチ(50〜150)
 * @type number
 * @min 50
 * @max 150
 * @default 100
 *
 * @param DefaultPan
 * @text 既定パン(-100〜100)
 * @type number
 * @min -100
 * @max 100
 * @default 0
 *
 * @command PlayBGV
 * @text BGV再生(ファイル名)
 * @desc audio/bgv/ から再生。例: A01 / mama/A01
 *
 * @arg name
 * @text ファイル名
 * @type string
 *
 * @arg volume
 * @text 音量(0〜100)
 * @type number
 * @min 0
 * @max 100
 * @default 100
 *
 * @arg pitch
 * @text ピッチ(50〜150)
 * @type number
 * @min 50
 * @max 150
 * @default 100
 *
 * @arg pan
 * @text パン(-100〜100)
 * @type number
 * @min -100
 * @max 100
 * @default 0
 *
 * @arg loop
 * @text ループ再生
 * @type boolean
 * @on する
 * @off しない
 * @default true
 *
 * @arg channel
 * @text チャンネル番号
 * @type number
 * @min 1
 * @max 8
 * @default 1
 * @desc 同時再生したい場合にチャンネルを分けます。未指定時は1。
 *
 * @command PlayBGV_Path
 * @text BGV再生(自由パス)
 * @desc スラッシュ含む自由パス。例: A01 / mama/A01 / audio/bgv/mama/A01
 *
 * @arg path
 * @text パス
 * @type string
 *
 * @arg volume2
 * @text 音量(0〜100)
 * @type number
 * @min 0
 * @max 100
 * @default 100
 *
 * @arg pitch2
 * @text ピッチ(50-150)
 * @type number
 * @min 50
 * @max 150
 * @default 100
 *
 * @arg pan2
 * @text パン(-100〜100)
 * @type number
 * @min -100
 * @max 100
 * @default 0
 *
 * @arg loop2
 * @text ループ再生
 * @type boolean
 * @on する
 * @off しない
 * @default true
 *
 * @arg channel2
 * @text チャンネル番号
 * @type number
 * @min 1
 * @max 8
 * @default 1
 * @desc 同時再生したい場合にチャンネルを分けます。未指定時は1。
 *
 * @command StopBGV
 * @text BGV停止
 * @desc BGVを停止します。チャンネル0または未指定で全チャンネル停止。
 *
 * @arg channel
 * @text チャンネル番号
 * @type number
 * @min 0
 * @max 8
 * @default 0
 * @desc 0で全チャンネル停止。1〜8で該当チャンネルのみ停止。
 *
 * @arg fadeSec
 * @text フェードアウト秒
 * @type number
 * @decimals 2
 * @min 0
 * @default 0
 * @desc 0で即時停止。0より大きいと指定秒数かけて音量を0にしてから停止。
 */

(() => {
"use strict";

const pluginName = "KCH_BackgroundVoice_AudioVoice";
const params = PluginManager.parameters(pluginName);
const bgvFolder = String(params["BgvFolder"] || "bgv");

const KCH_BgvManager = {
  _channels: {},
  _voiceSet: new Set(),
  _pendingResume: false,
  _resumeDelayFrames: 0,

  _getChannel(id) {
    const chId = Number(id || 1);
    if (!this._channels[chId]) {
      this._channels[chId] = {
        id: chId,
        buffer: null,
        folder: "",
        name: "",
        p: null,
        playing: false,
        pausedByVoice: false,
        pausePos: 0,
        fadeFramesRemaining: 0,
        fadeTotalFrames: 0
      };
    }
    return this._channels[chId];
  },

  _clearChannel(id) {
    const ch = this._channels[id];
    if (!ch) return;
    if (ch.buffer) ch.buffer.stop();
    ch.buffer = null;
    ch.folder = "";
    ch.name = "";
    ch.p = null;
    ch.playing = false;
    ch.pausedByVoice = false;
    ch.pausePos = 0;
    ch.fadeFramesRemaining = 0;
    ch.fadeTotalFrames = 0;
  },

  stopAll(fadeSec = 0) {
    const sec = Number(fadeSec || 0);
    if (sec > 0) {
      const frames = Math.max(1, Math.floor(sec * 60));
      for (const id in this._channels) {
        const ch = this._channels[id];
        if (!ch || !ch.buffer || !ch.p) continue;
        ch.fadeTotalFrames = frames;
        ch.fadeFramesRemaining = frames;
      }
    } else {
      for (const id in this._channels) {
        this._clearChannel(Number(id));
      }
    }
    if (sec <= 0) {
      this._voiceSet.clear();
      this._pendingResume = false;
      this._resumeDelayFrames = 0;
    }
  },

  stop() {
    this.stopAll(0);
  },

  _mix(vol) {
    const indiv = Number(vol ?? 100);
    const bgvOpt = typeof ConfigManager !== "undefined" ? Number(ConfigManager.bgvVolume ?? 100) : 100;
    const mast = (typeof AudioManager !== "undefined" && typeof AudioManager.masterVolume !== "undefined")
      ? Number(AudioManager.masterVolume ?? 100)
      : 100;
    const i = Math.max(0, Math.min(100, indiv));
    const b = Math.max(0, Math.min(100, bgvOpt));
    const m = Math.max(0, Math.min(100, mast));
    return (i / 100) * (b / 100) * (m / 100);
  },


_isSameRequest(ch, folder, name, p) {
  if (!ch) return false;
  if (String(ch.folder || "") !== String(folder || "")) return false;
  if (String(ch.name || "") !== String(name || "")) return false;
  const cp = ch.p || {};
  return (
    Number(cp.volume ?? 90) === Number(p.volume ?? 90) &&
    Number(cp.pitch  ?? 100) === Number(p.pitch  ?? 100) &&
    Number(cp.pan    ?? 0) === Number(p.pan    ?? 0) &&
    Boolean(cp.loop) === Boolean(p.loop)
  );
},

_applyOrUpdateIfSame(ch, folder, name, p) {
  const same = this._isSameRequest(ch, folder, name, p);
  if (!same) return false;

  ch.p = { ...p };

  if (ch.buffer && (ch.playing || ch.pausedByVoice)) {
    try {
      ch.buffer.pitch = ch.p.pitch / 100;
      ch.buffer.pan   = ch.p.pan / 100;
    } catch (_) {}
  }
  return true;
},
  playByName(name, volume, pitch, pan, loop, channelId = 1) {
    if (!name) {
      this.stopChannel(channelId, 0);
      return;
    }
    const ch = this._getChannel(channelId);

    const p = {
      volume: Number(volume ?? 90),
      pitch:  Number(pitch ?? 100),
      pan:    Number(pan ?? 0),
      loop:   loop !== false
    };

    if (this._applyOrUpdateIfSame(ch, bgvFolder, String(name), p)) {
      return;
    }

    this._clearChannel(channelId);

    ch.folder = bgvFolder;
    ch.name   = String(name);
    ch.p = p;

    const buf = AudioManager.createBuffer(ch.folder, ch.name);
    ch.buffer = buf;
    buf.volume = this._mix(ch.p.volume);
    buf.pitch  = ch.p.pitch / 100;
    buf.pan    = ch.p.pan / 100;
    buf.play(ch.p.loop, 0);
    ch.playing = true;
    ch.pausedByVoice = false;
    ch.pausePos = 0;
    ch.fadeFramesRemaining = 0;
    ch.fadeTotalFrames = 0;
  },

  playByPath(path, volume, pitch, pan, loop, channelId = 1) {
    if (!path) {
      this.stopChannel(channelId, 0);
      return;
    }
    const ch = this._getChannel(channelId);
const fixed = path.indexOf("audio/") === 0 ? path.replace(/^audio\//, "") : (bgvFolder + "/" + path);
const parts = fixed.split("/");
const name = parts.pop();
const folder = parts.join("/");

const p = {
  volume: Number(volume ?? 90),
  pitch:  Number(pitch ?? 100),
  pan:    Number(pan ?? 0),
  loop:   loop !== false
};

if (this._applyOrUpdateIfSame(ch, folder, name, p)) {
  return;
}

this._clearChannel(channelId);

ch.folder = folder;
ch.name   = name;
ch.p = p;

    const buf = AudioManager.createBuffer(ch.folder, ch.name);
    ch.buffer = buf;
    buf.volume = this._mix(ch.p.volume);
    buf.pitch  = ch.p.pitch / 100;
    buf.pan    = ch.p.pan / 100;
    buf.play(ch.p.loop, 0);
    ch.playing = true;
    ch.pausedByVoice = false;
    ch.pausePos = 0;
    ch.fadeFramesRemaining = 0;
    ch.fadeTotalFrames = 0;
  },

  stopChannel(channelId = 1, fadeSec = 0) {
    const chId = Number(channelId || 1);
    const ch = this._channels[chId];
    const sec = Number(fadeSec || 0);
    if (!ch) return;

    if (sec > 0 && ch.buffer && ch.p) {
      const frames = Math.max(1, Math.floor(sec * 60));
      ch.fadeTotalFrames = frames;
      ch.fadeFramesRemaining = frames;
    } else {
      this._clearChannel(chId);
    }
  },

  updateVolume() {
    const voiceActive = this._voiceSet.size > 0;
    const muteByVoice = voiceActive || this._pendingResume || this._resumeDelayFrames > 0;

    for (const id in this._channels) {
      const ch = this._channels[id];
      if (!ch || !ch.buffer || !ch.p) continue;

      const base = this._mix(ch.p.volume);
      let vol = base;

      if (ch.fadeTotalFrames > 0 && ch.fadeFramesRemaining > 0) {
        const t = ch.fadeFramesRemaining / ch.fadeTotalFrames;
        vol *= t;
        ch.fadeFramesRemaining--;
        if (ch.fadeFramesRemaining <= 0) {
          this._clearChannel(Number(id));
          continue;
        }
      }

      if (muteByVoice) {
        vol = 0;
      }

      ch.buffer.volume = vol;
    }

    this._updateResumeStatus();
  },

  pauseByVoice() {
    for (const id in this._channels) {
      const ch = this._channels[id];
      if (!ch || !ch.buffer || !ch.playing || ch.pausedByVoice) continue;

      try {
        if (ch.buffer.seek) {
          ch.pausePos = ch.buffer.seek();
        } else {
          ch.pausePos = 0;
        }
      } catch (_) {
        ch.pausePos = 0;
      }

      try {
        ch.buffer.volume = 0;
      } catch (_) {}
      ch.buffer.stop();

      ch.playing = false;
      ch.pausedByVoice = true;
    }
  },

  resumeAfterVoice() {
    for (const id in this._channels) {
      const ch = this._channels[id];
      if (!ch || !ch.buffer || !ch.pausedByVoice || !ch.p) continue;

      const p = ch.p;
      const pos = ch.pausePos || 0;
      try {
        ch.buffer.play(p.loop, pos);
        ch.buffer.volume = this._mix(p.volume);
        ch.buffer.pitch  = p.pitch / 100;
        ch.buffer.pan    = p.pan / 100;
        ch.playing = true;
      } catch (_) {
        const b = AudioManager.createBuffer(ch.folder, ch.name);
        ch.buffer = b;
        b.volume = this._mix(p.volume);
        b.pitch  = p.pitch / 100;
        b.pan    = p.pan / 100;
        b.play(p.loop, pos);
        ch.playing = true;
      }
      ch.pausedByVoice = false;
      ch.pausePos = 0;
    }
  },


  _voiceStarted(buffer) {
    if (!buffer) return;
    const wasEmpty = this._voiceSet.size === 0;
    this._voiceSet.add(buffer);
    if (wasEmpty) {
      this._pendingResume = false;
      this._resumeDelayFrames = 0;
      this.pauseByVoice();
    }
  },

  _voiceStopped(buffer) {
    if (!buffer) return;
    this._voiceSet.delete(buffer);
    if (this._voiceSet.size === 0) {
      if ($gameMessage && $gameMessage.isBusy && $gameMessage.isBusy()) {
        this._pendingResume = true;
        this._resumeDelayFrames = 0;
      } else {
        this._pendingResume = true;
        this._resumeDelayFrames = 4;
      }
    }
  },

  onVoiceStart(buffer) {
    if (buffer) {
      this._voiceStarted(buffer);
    } else {
      const dummy = {};
      this._voiceStarted(dummy);
    }
  },

  onVoiceStopOne(buffer) {
    if (buffer) {
      this._voiceStopped(buffer);
    } else {
      this.onAllVoicesStopped();
    }
  },

  onAllVoicesStopped() {
    if (this._voiceSet.size > 0) this._voiceSet.clear();
    if ($gameMessage && $gameMessage.isBusy && $gameMessage.isBusy()) {
      this._pendingResume = true;
      this._resumeDelayFrames = 0;
      return;
    }
    this._pendingResume = true;
    this._resumeDelayFrames = 4;
  },

  _updateResumeStatus() {
    if (this._voiceSet.size > 0) {
      this._resumeDelayFrames = 0;
      return;
    }
    if (!this._pendingResume) return;

    if (this._resumeDelayFrames > 0) {
      this._resumeDelayFrames--;
      return;
    }

    this._pendingResume = false;
    this.resumeAfterVoice();
  },

  _tryResumeAfterMessage() {
    if (this._voiceSet.size > 0) {
      this._pendingResume = false;
      this._resumeDelayFrames = 0;
      return;
    }
    if (this._pendingResume) {
      if (this._resumeDelayFrames <= 0) {
        this._resumeDelayFrames = 4;
      }
    }
  }
};

if (typeof ConfigManager !== "undefined") {
  if (ConfigManager.bgvVolume == null) ConfigManager.bgvVolume = 100;

  const _makeData = ConfigManager.makeData;
  ConfigManager.makeData = function() {
    const c = _makeData.call(this);
    c.bgvVolume = this.bgvVolume;
    return c;
  };

  const _applyData = ConfigManager.applyData;
  ConfigManager.applyData = function(data) {
    _applyData.call(this, data);
    this.bgvVolume = this.readVolume(data, "bgvVolume");
  };
}

PluginManager.registerCommand(pluginName, "PlayBGV", args => {
  const name   = String(args.name || "");
  const volume = Number(args.volume || 90);
  const pitch  = Number(args.pitch || 100);
  const pan    = Number(args.pan || 0);
  const loop   = args.loop !== "false";
  const ch     = Number(args.channel || 1);
  KCH_BgvManager.playByName(name, volume, pitch, pan, loop, ch);
});

PluginManager.registerCommand(pluginName, "PlayBGV_Path", args => {
  const path   = String(args.path || "");
  const volume = Number(args.volume2 || 90);
  const pitch  = Number(args.pitch2 || 100);
  const pan    = Number(args.pan2 || 0);
  const loop   = args.loop2 !== "false";
  const ch     = Number(args.channel2 || 1);
  KCH_BgvManager.playByPath(path, volume, pitch, pan, loop, ch);
});

PluginManager.registerCommand(pluginName, "StopBGV", args => {
  const ch    = Number(args && args.channel != null ? args.channel : 0);
  const fade  = Number(args && args.fadeSec != null ? args.fadeSec : 0);
  if (ch > 0) {
    KCH_BgvManager.stopChannel(ch, fade);
  } else {
    KCH_BgvManager.stopAll(fade);
  }
});

const _Scene_Base_update = Scene_Base.prototype.update;
Scene_Base.prototype.update = function() {
  _Scene_Base_update.call(this);
  KCH_BgvManager.updateVolume();
};

(() => {
  if (typeof WebAudio === "undefined") return;

  let voicePrefix = "audio/voice/";
  try {
    if (window.AudioVoice && typeof AudioVoice._Plugpath === "string" && AudioVoice._Plugpath) {
      voicePrefix = AudioVoice._Plugpath;
      if (!voicePrefix.endsWith("/")) voicePrefix += "/";
    }
  } catch (_) {}

  const _play = WebAudio.prototype.play;
  WebAudio.prototype.play = function(loop, offset) {
    try {
      const url = String(this._url || "");
      if (url.indexOf(voicePrefix) >= 0) {
        if (!this._kchBgvVoiceTracked) {
          this._kchBgvVoiceTracked = true;
          KCH_BgvManager.onVoiceStart(this);
        }
      }
    } catch (_) {}
    return _play.call(this, loop, offset);
  };

  const _stop = WebAudio.prototype.stop;
  WebAudio.prototype.stop = function() {
    const wasVoice = this._kchBgvVoiceTracked;
    _stop.call(this);
    if (wasVoice) {
      this._kchBgvVoiceTracked = false;
      KCH_BgvManager.onVoiceStopOne(this);
    }
  };

  const _onEnd = WebAudio.prototype._onEnd;
  WebAudio.prototype._onEnd = function() {
    const wasVoice = this._kchBgvVoiceTracked;
    if (_onEnd) _onEnd.apply(this, arguments);
    if (wasVoice) {
      this._kchBgvVoiceTracked = false;
      KCH_BgvManager.onVoiceStopOne(this);
    }
  };
})();

if (typeof Scene_Title !== "undefined") {
  const _Scene_Title_start = Scene_Title.prototype.start;
  Scene_Title.prototype.start = function() {
    KCH_BgvManager.stopAll(0);
    _Scene_Title_start.call(this);
  };
}

if (typeof Scene_Options !== "undefined" && Scene_Options.prototype.commandToTitle) {
  const _Scene_Options_commandToTitle = Scene_Options.prototype.commandToTitle;
  Scene_Options.prototype.commandToTitle = function() {
    KCH_BgvManager.stopAll(0);
    _Scene_Options_commandToTitle.call(this);
  };
}

(() => {
  if (!window.AudioVoice) return;

  function hook(fnName) {
    const orig = AudioVoice[fnName];
    if (typeof orig !== "function") return;

    AudioVoice[fnName] = function() {
      const ret = orig.apply(this, arguments);

      try {
        const list = AudioVoice._vcBuffers || this._vcBuffers || [];
        const buffer = list[list.length - 1];
        if (buffer && !buffer._kchBgvVoiceTracked && !buffer._kchBgvAVTracked) {
          buffer._kchBgvAVTracked = true;
          KCH_BgvManager.onVoiceStart(buffer);

          const _stop = buffer.stop;
          buffer.stop = function() {
            const was = this._kchBgvAVTracked;
            const r = _stop.apply(this, arguments);
            if (was) {
              this._kchBgvAVTracked = false;
              KCH_BgvManager.onVoiceStopOne(this);
            }
            return r;
          };

          if (buffer._onEnd) {
            const _onEnd = buffer._onEnd;
            buffer._onEnd = function() {
              const was = this._kchBgvAVTracked;
              _onEnd.apply(this, arguments);
              if (was) {
                this._kchBgvAVTracked = false;
                KCH_BgvManager.onVoiceStopOne(this);
              }
            };
          }
        }
      } catch (_) {}

      return ret;
    };
  }

  hook("playVc");
  hook("playbattleVc");
  hook("explaybattleVc");
})();

(() => {
  if (typeof Window_Message === "undefined") return;
  const _terminateMessage = Window_Message.prototype.terminateMessage;
  Window_Message.prototype.terminateMessage = function() {
    _terminateMessage.call(this);
    try { KCH_BgvManager._tryResumeAfterMessage(); } catch (_) {}
  };
})();

})();
