/*:
 * @target MZ
 * @plugindesc [KCH] Mush BGS pause persists across map transfer (works with Add Spacial BGS).
 * @author KCH
 *
 * @command PauseMushBgs
 * @text BGS一時停止（チャンネル）
 * @desc 指定チャンネルのMUSH BGSを一時停止し、以後「一時停止維持モード」に入ります。
 * @arg Channel
 * @type number
 * @min 1
 * @default 1
 *
 * @command ContinueMushBgs
 * @text BGS再開（チャンネル）
 * @desc 指定チャンネルの一時停止を解除します。全停止維持モードは解除しません（必要ならContinueAll）。
 * @arg Channel
 * @type number
 * @min 1
 * @default 1
 *
 * @command PauseAllMushBgs
 * @text BGS一時停止（全チャンネル）
 * @desc 現在の全MUSH BGSを一時停止し、以後「一時停止維持モード」に入ります（場所移動しても鳴らない）。
 *
 * @command ContinueAllMushBgs
 * @text BGS再開（全チャンネル）
 * @desc 「一時停止維持モード」を解除し、一時停止中の全MUSH BGSを再開します。
 */

(() => {
  const pluginName = "KCH_MushBgsPauseAddOn";

  // -----------------------------
  // 永続状態（場所移動でバッファが作り直されても維持する）
  // -----------------------------
  const KEY_GLOBAL = "_kchMushBgsGlobalPause";
  const KEY_LIST = "_kchMushBgsPausedChannels";

  function sys() {
    return (typeof $gameSystem !== "undefined" && $gameSystem) ? $gameSystem : null;
  }
  function ensureList() {
    const s = sys();
    if (!s) return [];
    if (!Array.isArray(s[KEY_LIST])) s[KEY_LIST] = [];
    // 数値＆重複排除
    s[KEY_LIST] = [...new Set(s[KEY_LIST].map(Number).filter(n => n > 0))];
    return s[KEY_LIST];
  }
  function setGlobalPause(v) {
    const s = sys();
    if (!s) return;
    s[KEY_GLOBAL] = !!v;
  }
  function isGlobalPaused() {
    const s = sys();
    return !!(s && s[KEY_GLOBAL]);
  }
  function markPausedChannel(ch, paused) {
    const list = ensureList();
    const n = Number(ch);
    if (!(n > 0)) return;
    const idx = list.indexOf(n);
    if (paused) {
      if (idx < 0) list.push(n);
    } else {
      if (idx >= 0) list.splice(idx, 1);
    }
  }
  function isChannelPaused(ch) {
    const n = Number(ch);
    if (!(n > 0)) return false;
    return ensureList().includes(n);
  }

  // -----------------------------
  // 便利関数
  // -----------------------------
  function getBgsBuffer(ch) {
    if (!AudioManager.getBgsFromChannel) return null;
    return AudioManager.getBgsFromChannel(ch);
  }
  function allCurrentBgsChannels() {
    const buffers = AudioManager._bgsBuffers || [];
    const chs = [];
    for (const b of buffers) {
      if (b && b.channel != null) chs.push(Number(b.channel));
    }
    return [...new Set(chs.filter(n => n > 0))];
  }
  function pauseChannelNow(ch) {
    const n = Number(ch);
    if (!(n > 0)) return;
    if (AudioManager.pauseMushBgs) AudioManager.pauseMushBgs(n);
    const buf = getBgsBuffer(n);
    if (buf) {
      buf.currentlyPaused = true;
      // playされた直後に止めても「位置」がnullになりうるので、最低0を入れておく
      if (buf.currentPosition == null) buf.currentPosition = 0;
    }
  }
  function applyGlobalPauseToAllExistingBuffers() {
    const chs = allCurrentBgsChannels();
    for (const ch of chs) pauseChannelNow(ch);
  }

  // -----------------------------
  // プラグインコマンド
  // -----------------------------
  PluginManager.registerCommand(pluginName, "PauseMushBgs", args => {
    const ch = Number(args.Channel || 0);
    if (!(ch > 0)) return;
    setGlobalPause(true);
    markPausedChannel(ch, true);
    pauseChannelNow(ch);
  });

  PluginManager.registerCommand(pluginName, "ContinueMushBgs", args => {
    const ch = Number(args.Channel || 0);
    if (!(ch > 0)) return;
    markPausedChannel(ch, false);

    // 全体停止維持中なら「個別再開」しても鳴らさない（意図通り：全体解除は ContinueAll）
    if (isGlobalPaused()) return;

    const buf = getBgsBuffer(ch);
    if (buf && buf.currentlyPaused && AudioManager.continueMushBgs) {
      AudioManager.continueMushBgs(ch);
      buf.currentlyPaused = false;
    }
  });

  PluginManager.registerCommand(pluginName, "PauseAllMushBgs", () => {
    setGlobalPause(true);
    for (const ch of allCurrentBgsChannels()) markPausedChannel(ch, true);
    applyGlobalPauseToAllExistingBuffers();
  });

  PluginManager.registerCommand(pluginName, "ContinueAllMushBgs", () => {
    setGlobalPause(false);
    const pausedList = ensureList().slice();
    // listはこれで空にする（「停止維持モード解除」なので）
    const s = sys();
    if (s) s[KEY_LIST] = [];

    for (const ch of pausedList) {
      const buf = getBgsBuffer(ch);
      if (buf && buf.currentlyPaused && AudioManager.continueMushBgs) {
        AudioManager.continueMushBgs(ch);
        buf.currentlyPaused = false;
      }
    }
  });

  // -----------------------------
  // 핵：どこからBGSが「新規に起動」されても止める
  //  - Add Spacial BGS は playMushBgs を必ず通る :contentReference[oaicite:2]{index=2}
  // -----------------------------
  if (AudioManager.playMushBgs) {
    const _playMushBgs = AudioManager.playMushBgs;

    AudioManager.playMushBgs = function(bgs, channel, autoRemover, interrupt, pos) {
      _playMushBgs.call(this, bgs, channel, autoRemover, interrupt, pos);

      const ch = Number(channel);
      // 1) 全体停止維持中
      // 2) もしくは、そのチャンネルが停止対象として記録されている
      if (isGlobalPaused() || isChannelPaused(ch)) {
        markPausedChannel(ch, true);
        pauseChannelNow(ch);
      }
    };
  }

  if (AudioManager.continueMushBgs) {
    const _continueMushBgs = AudioManager.continueMushBgs;

    AudioManager.continueMushBgs = function(channel) {
      const ch = Number(channel);
      // 停止維持中は再開させない（他プラグインがcontinueを直呼びしても鳴らない）
      if (isGlobalPaused() || isChannelPaused(ch)) return;
      _continueMushBgs.call(this, ch);
    };
  }

  // -----------------------------
  // 場所移動直後は、別イベントが走って Add Spacial BGS 等が実行されがちなので
  // onMapLoaded のタイミングで「今あるバッファ」を必ず再停止しておく（保険）
  // -----------------------------
  if (typeof Scene_Map !== "undefined" && Scene_Map.prototype.onMapLoaded) {
    const _onMapLoaded = Scene_Map.prototype.onMapLoaded;
    Scene_Map.prototype.onMapLoaded = function() {
      _onMapLoaded.call(this);
      if (isGlobalPaused()) applyGlobalPauseToAllExistingBuffers();
    };
  }
})();
