/*:
 *
 * @param Timers
 * @text タイマー設定一覧
 * @type struct<TimerSlot>[]
 * @default []
 *
 * @param DebugLog
 * @text デバッグログ
 * @type boolean
 * @default false
 *
 * @param FlashDefault
 * @text 既定：演出（設定が無いIdのフォールバック）
 * @type struct<FlashSetting>
 * @default {"duration":"24","maxOpacity":"192","blendAdd":"true"}
 *
 * @param FlashPlacement
 * @text 既定：位置（設定が無いIdのフォールバック）
 * @type struct<FlashPlacement>
 * @default {"CoordSpace":"container","anchor":"center","x":"0","y":"0","w":"0","h":"0"}
 *
 * @param FlashMap
 * @text 共通フラッシュ設定（Gauge.Idごと）
 * @type struct<FlashItem>[]
 * @default []
 */

/*~struct~TimerSlot:
 * @param boostSwitchId
 * @text ブーストスイッチID（効果2倍）
 * @type switch
 * @default 0
 *
 * @param enabled
 * @text 有効
 * @type boolean
 * @default true
 *
 * @param label
 * @text メモ/ラベル
 * @type string
 * @default
 *
 * @param switchId
 * @text 対象スイッチ（ONの間だけ進行）
 * @type switch
 * @default 1
 *
 * @param disableSwitchId
 * @text 無効スイッチ（このスロットのみ停止）
 * @type switch
 * @default 0
 *
 * @param seconds
 * @text 加算間隔（秒）
 * @type number
 * @decimals 2
 * @min 0.01
 * @default 1.00
 *
 * @param addValue
 * @text 加算値Z（±）
 * @type number
 * @decimals 2
 * @min -999999
 * @default 1
 *
 * @param affectVariable
 * @text 変数に加算
 * @type boolean
 * @default true
 *
 * @param variableId
 * @text 対象変数
 * @type variable
 * @default 1
 *
 * @param affectActorHP
 * @text アクターHPに加算
 * @type boolean
 * @default false
 *
 * @param affectActorMP
 * @text アクターMPに加算
 * @type boolean
 * @default false
 *
 * @param actorId
 * @text 対象アクター
 * @type actor
 * @default 1
 *
 * @param showPopup
 * @text ポップアップ表示
 * @type boolean
 * @default true
 *
 * @param posX
 * @text 表示X（px／中央合わせ）
 * @type number
 * @default 100
 *
 * @param posY
 * @text 表示Y（px／中央合わせ）
 * @type number
 * @default 100
 *
 * @param fontSize
 * @text フォントサイズ
 * @type number
 * @min 8
 * @default 28
 *
 * @param outlineWidth
 * @text ふち幅
 * @type number
 * @min 0
 * @default 4
 *
 * @param duration
 * @text 表示フレーム数（ポップアップ）
 * @type number
 * @min 1
 * @default 60
 *
 * @param rise
 * @text 上昇量（px）
 * @type number
 * @default 24
 *
 * @param popupText
 * @text ポップアップ文字列
 * @type string
 * @default {Z}
 *
 * @param textColor
 * @text 文字色（HEX/CSS）
 * @type string
 * @default #00FF88
 *
 * @param commonEventId
 * @text コモンイベント（毎回）
 * @type common_event
 * @default 0
 */

/*~struct~FlashSetting:
 * @param duration
 * @text 継続フレーム
 * @type number
 * @min 1
 * @default 24
 *
 * @param maxOpacity
 * @text 最大Opacity
 * @type number
 * @min 1
 * @max 255
 * @default 192
 *
 * @param blendAdd
 * @text 加算合成を使う
 * @type boolean
 * @default true
 */

/*~struct~FlashPlacement:
 * @param CoordSpace
 * @text 座標系
 * @type select
 * @option container
 * @option screen
 * @default container
 *
 * @param anchor
 * @text アンカー
 * @type select
 * @option center
 * @option topleft
 * @default center
 *
 * @param x
 * @text X（px）
 * @type number
 * @default 0
 *
 * @param y
 * @text Y（px）
 * @type number
 * @default 0
 *
 * @param w
 * @text 幅（省略=ゲージサイズ）
 * @type number
 * @min 0
 * @default 0
 *
 * @param h
 * @text 高さ（省略=ゲージサイズ）
 * @type number
 * @min 0
 * @default 0
 */

/*~struct~FlashItem:
 * @param GaugeId
 * @text ID（Gauge.Id）
 * @type string
 * @default
 *
 * @param CoordSpace
 * @text 座標系
 * @type select
 * @option container
 * @option screen
 * @default container
 *
 * @param x
 * @text X（px）
 * @type number
 * @default 0
 *
 * @param y
 * @text Y（px）
 * @type number
 * @default 0
 *
 * @param duration
 * @text 継続フレーム
 * @type number
 * @min 1
 * @default 24
 *
 * @param opacity
 * @text 最大Opacity
 * @type number
 * @min 1
 * @max 255
 * @default 192
 *
 * @param blendAdd
 * @text 加算合成
 * @type boolean
 * @default true
 */

(() => {
  'use strict';
  const PN = 'KZ_ExtraGauge_TimerFlash';
  const P  = PluginManager.parameters(PN);
  const DEBUG = String(P.DebugLog || 'false') === 'true';
  const dlog  = (...a)=>{ if (DEBUG) console.log('[KZ-EGTF]', ...a); };

  const ICON_W = 32, ICON_H = 32;

  const DEF_FLASH = (() => {
    const j = JSON.parse(P.FlashDefault || '{}');
    return {
      duration: Number(j.duration || 24),
      maxOpacity: Number(j.maxOpacity || 192),
      blendAdd: String(j.blendAdd || 'true') === 'true',
    };
  })();

  const DEF_PLACE = (() => {
    const j = JSON.parse(P.FlashPlacement || '{}');
    return {
      space: String(j.CoordSpace || 'container'),
      anchor: String(j.anchor || 'center'), 
      x: Number(j.x || 0),
      y: Number(j.y || 0),
      w: Number(j.w || 0),
      h: Number(j.h || 0),
    };
  })();

  const FLASH_MAP = (() => {
    const arr = JSON.parse(P.FlashMap || '[]');
    const table = new Map();
    for (const it of arr) {
      const o = JSON.parse(it);
      const id = String(o.GaugeId || '').trim().toLowerCase();
      if (!id) continue;
      table.set(id, {
        space: String(o.CoordSpace || 'container'),
        x: Number(o.x || 0),
        y: Number(o.y || 0),
        duration: Math.max(1, Number(o.duration || DEF_FLASH.duration)),
        opacity: Math.max(1, Math.min(255, Number(o.opacity || DEF_FLASH.maxOpacity))),
        blendAdd: String(o.blendAdd || 'true') === 'true',
      });
    }
    return table;
  })();

  const normId = s => String(s || '').trim().toLowerCase();

  const SLOTS = JSON.parse(P.Timers || '[]').map(s => {
    const o = JSON.parse(s);
    return {
      boostSwitchId: Number(o.boostSwitchId || 0), 
      enabled: String(o.enabled) !== 'false',
      label: String(o.label || ''),
      switchId: Number(o.switchId || 1),
      disableSwitchId: Number(o.disableSwitchId || 0),
      seconds: Math.max(0.001, Number(o.seconds || 1)),
      addValue: Number(o.addValue || 1),

      affectVariable: String(o.affectVariable) !== 'false',
      variableId: Number(o.variableId || 1),

      affectActorHP: String(o.affectActorHP) === 'true',
      affectActorMP: String(o.affectActorMP) === 'true',
      actorId: Number(o.actorId || 1),

      showPopup: String(o.showPopup) !== 'false',
      posX: Number(o.posX || 100),
      posY: Number(o.posY || 100),
      fontSize: Number(o.fontSize || 28),
      outlineWidth: Number(o.outlineWidth || 4),
      duration: Math.max(1, Number(o.duration || 60)),
      rise: Number(o.rise || 24),
      popupText: String(o.popupText || '{Z}'),
      textColor: String(o.textColor || '#00FF88'),

      commonEventId: Number(o.commonEventId || 0),
    };
  });

  function tokenizeRich(text) {
    const segs = [];
    const re = /\\I\[(\d+)\]/gi;
    let last = 0, m;
    while ((m = re.exec(text)) !== null) {
      if (m.index > last) segs.push({ type:'t', text:text.slice(last, m.index) });
      segs.push({ type:'i', id:Number(m[1]) });
      last = re.lastIndex;
    }
    if (last < text.length) segs.push({ type:'t', text:text.slice(last) });
    return segs;
  }
  function measureRichWidth(text, fontSize, outlineWidth) {
    const segs = tokenizeRich(text);
    const tmp = new Bitmap(1,1);
    tmp.fontSize = fontSize; tmp.outlineWidth = outlineWidth;
    const iw = ICON_W, ih = ICON_H;
    const iconW = Math.round(iw * (fontSize / ih));
    let w = 0;
    for (const s of segs) w += (s.type === 't') ? Math.ceil(tmp.measureTextWidth(s.text)) : iconW;
    return w;
  }
  function drawRichCentered(bmp, text, fontSize, outlineWidth, color) {
    bmp.fontSize = fontSize;
    bmp.outlineWidth = outlineWidth;
    bmp.outlineColor = 'rgba(0,0,0,0.75)';
    bmp.textColor = (color && color.trim()) ? color : '#ffffff';

    const segs = tokenizeRich(text);
    const totalW = measureRichWidth(text, fontSize, outlineWidth);
    const startX = Math.floor((bmp.width - totalW) / 2);

    const iconBmp = (function(){ try { return ImageManager.loadSystem('IconSet'); } catch(_) { return null; } })();
    const iw = ICON_W, ih = ICON_H;
    const iconH = fontSize, iconW = Math.round(iw * (iconH / ih));
    const dy = Math.floor((bmp.height - iconH) / 2);

    const meas = new Bitmap(1,1);
    meas.fontSize = fontSize; meas.outlineWidth = outlineWidth;

    const drawAll = () => {
      bmp.clear();
      bmp.fontSize = fontSize;
      bmp.outlineWidth = outlineWidth;
      bmp.outlineColor = 'rgba(0,0,0,0.75)';
      bmp.textColor = (color && color.trim()) ? color : '#ffffff';
      let x = startX;
      for (const s of segs) {
        if (s.type === 't') {
          const w = Math.ceil(meas.measureTextWidth(s.text));
          bmp.drawText(s.text, x, 0, w, bmp.height, 'left');
          x += w;
        } else {
          if (iconBmp && typeof iconBmp.isReady === 'function' && iconBmp.isReady()) {
            const index = s.id;
            const sx = (index % 16) * iw;
            const sy = Math.floor(index / 16) * ih;
            bmp.blt(iconBmp, sx, sy, iw, ih, x, dy, iconW, iconH);
          }
          x += iconW;
        }
      }
    };
    const hasIcon = /\\I\[(\d+)\]/i.test(text);
    if (hasIcon && iconBmp && typeof iconBmp.isReady === 'function' && !iconBmp.isReady() && typeof iconBmp.addLoadListener === 'function') {
      iconBmp.addLoadListener(drawAll);
    } else {
      drawAll();
    }
  }

  class KZ_Popup extends Sprite {
    constructor(text, cfg) {
      const marginW = 16, marginH = 8;
      const w = Math.max(1, Math.ceil(measureRichWidth(text, cfg.fontSize, cfg.outlineWidth)) + marginW);
      const h = Math.max(1, cfg.fontSize + marginH);
      super(new Bitmap(w, h));
      this._cfg = cfg;
      this._text = text;
      this._life = cfg.duration;
      this.opacity = 255;
      this._bx = cfg.posX - Math.floor(w/2);
      this._by = cfg.posY - Math.floor(h/2);
      this.x = this._bx;
      this.y = this._by;
      drawRichCentered(this.bitmap, text, cfg.fontSize, cfg.outlineWidth, cfg.textColor);
    }
    update() {
      super.update();
      if (this._life > 0) {
        this._life--;
        const t = 1 - this._life / Math.max(1, this._cfg.duration);
        this.y = this._by - (this._cfg.rise * t);
        this.opacity = 255 * (1 - t);
        if (this._life <= 0) {
          this.parent && this.parent.removeChild(this);
          this.destroy({children:true, texture:false, baseTexture:false});
        }
      }
    }
  }
  class KZ_PopupLayer extends Sprite {
    spawn(text, cfg){ this.addChild(new KZ_Popup(text, cfg)); }
    clear(){ this.removeChildren(); }
  }

  function isExtraGaugeContainer(node) {
    const g = node && node._gauge;
    if (!g) return false;
    const ctor = window.Sprite_ExtraGauge;
    return ctor ? (g instanceof ctor) : (!!g._data && g.bitmap !== undefined);
  }
  function forEachExtraGaugeContainerInScene(fn) {
    const scene = SceneManager._scene;
    if (!scene) return;
    const visit = (node) => {
      if (!node || !node.children) return;
      for (const ch of node.children) {
        if (isExtraGaugeContainer(ch)) fn(ch);
        visit(ch);
      }
    };
    visit(scene);
    if (scene._spriteset) visit(scene._spriteset);
    if (scene._windowLayer) visit(scene._windowLayer);
  }

  function resolveFlashConfigFor(gaugeId) {
    const id = normId(gaugeId);
    const item = FLASH_MAP.get(id);
    const place = {
      space: (item && item.space) || DEF_PLACE.space,
      anchor: DEF_PLACE.anchor,
      x: (item ? item.x : DEF_PLACE.x),
      y: (item ? item.y : DEF_PLACE.y),
      w: DEF_PLACE.w,
      h: DEF_PLACE.h,
    };
    const eff = {
      duration: (item ? item.duration : DEF_FLASH.duration),
      maxOpacity: (item ? item.opacity  : DEF_FLASH.maxOpacity),
      blendAdd: (item ? item.blendAdd  : DEF_FLASH.blendAdd),
    };
    return { place, eff };
  }

  function startGaugeFlash(container) {
    const g = container._gauge;
    const d = g && g._data;
    const gid = d && d.Id;
    const { place, eff } = resolveFlashConfigFor(gid);

    const gw = g ? (g.bitmap ? g.bitmap.width : g.width || 1) : 1;
    const gh = g ? (g.bitmap ? g.bitmap.height: g.height || 1): 1;

    const fw = Math.max(1, place.w || gw);
    const fh = Math.max(1, place.h || gh);

    const over = new Sprite(new Bitmap(fw, fh));
    over.bitmap.fillAll('white');
    over.opacity = eff.maxOpacity;
    if (eff.blendAdd && PIXI && PIXI.BLEND_MODES) over.blendMode = PIXI.BLEND_MODES.ADD;

    if (place.anchor === 'center') over.anchor.set(0.5, 0.5); else over.anchor.set(0, 0);
    const px = Number(place.x || 0), py = Number(place.y || 0);
    if (place.space === 'screen') {
      over.x = px; over.y = py;
      SceneManager._scene.addChild(over);
    } else {
      over.x = px; over.y = py;
      container.addChild(over);
    }

    const dur = Math.max(1, eff.duration);
    over._life = dur;
    const baseUpdate = over.update.bind(over);
    over.update = function(){
      baseUpdate();
      this._life--;
      this.opacity = eff.maxOpacity * (this._life / dur);
      if (this._life <= 0) {
        this.parent && this.parent.removeChild(this);
        this.destroy({children:true, texture:false, baseTexture:false});
      }
    };
  }

  function flashByVariableId(varId) {
    forEachExtraGaugeContainerInScene(container => {
      const d = container._gauge && container._gauge._data;
      const cur = d && d.CurrentMethod;
      if (cur && Number(cur.VariableId || 0) === varId) startGaugeFlash(container);
    });
  }

  function isHpScript(script){ return script && /\bbattler\.(?:hp|hprate\(\))\b/i.test(String(script)); }
  function isMpScript(script){ return script && /\bbattler\.(?:mp|mprate\(\)|sp|mana)\b/i.test(String(script)); }

  function flashByActorParam(actorId, kind) {
    forEachExtraGaugeContainerInScene(container => {
      const g = container._gauge;
      const battler = g && g._battler;
      if (!battler || typeof battler.actorId !== 'function') return;
      if (Number(battler.actorId()) !== Number(actorId)) return;

      const d = g._data, cur = d && d.CurrentMethod;
      if (!cur) return;
      if (Number(cur.VariableId || 0) > 0) return; // 変数は別ルート
      const scr = cur.Script || '';
      if (kind === 'hp' && isHpScript(scr)) startGaugeFlash(container);
      if (kind === 'mp' && isMpScript(scr)) startGaugeFlash(container);
    });
  }
  const KZEGTF = {
    _slots: [],
    popupLayer: null,
    _parallelInterpreters: [],
    _prevFrame: 0,

    setup() {
      this._slots = SLOTS.filter(s => s.enabled !== false).map((cfg, i)=> new TimerSlot(cfg, i));
      this._prevFrame = Graphics.frameCount;
      dlog('setup slots:', this._slots.length);
    },
    ensurePopupLayer() {
      const sc = SceneManager._scene;
      if (!sc) return;
      if (!this.popupLayer) {
        this.popupLayer = new KZ_PopupLayer();
        sc.addChild(this.popupLayer);
      }
    },
    popup(text, cfg) {
      if (!cfg.showPopup) return;
      this.ensurePopupLayer();
      if (this.popupLayer) this.popupLayer.spawn(text, cfg);
    },
    runCommonEventParallel(ceId) {
      const data = $dataCommonEvents[ceId];
      if (!data || !data.list || !data.list.length) return;
      const it = new Game_Interpreter();
      it.setup(data.list, 0);
      this._parallelInterpreters.push(it);
    },
    _updateParallel() {
      if (!this._parallelInterpreters.length) return;
      const alive = [];
      for (const it of this._parallelInterpreters) {
        it.update();
        if (it.isRunning ? it.isRunning() : !!it._list) alive.push(it);
      }
      this._parallelInterpreters = alive;
    },
    updateOncePerFrame() {
      const now = Graphics.frameCount;
      if (this._prevFrame === now) return;
      const frames = Math.max(0, now - this._prevFrame);
      this._prevFrame = now;
      const dt = frames / 60;
      for (const s of this._slots) s.update(dt);
      this._updateParallel();
    }
  };

  class TimerSlot {
    constructor(cfg, idx){
      this.cfg = cfg;
      this.idx = idx;
      this.elapsed = 0;
    }
    update(dtSec) {
      if (!$gameSwitches) return;
      const on  = $gameSwitches.value(this.cfg.switchId);
      const dis = this.cfg.disableSwitchId > 0 && $gameSwitches.value(this.cfg.disableSwitchId);
      if (!on || dis) return;
      this.elapsed += dtSec;
      while (this.elapsed + 1e-9 >= this.cfg.seconds) {
        this.elapsed -= this.cfg.seconds;
        this.applyIncrement();
      }
    }
    applyIncrement() {
        let z = this.cfg.addValue;

  const boostId = this.cfg.boostSwitchId || 0;
  if (boostId > 0 && $gameSwitches.value(boostId)) {
    z *= 2;
  }

      let oldValue = 0, newValue = 0, vId = this.cfg.variableId;
      if (this.cfg.affectVariable) {
        oldValue = Number($gameVariables.value(vId) || 0);
        newValue = oldValue + z;
        $gameVariables.setValue(vId, newValue);
      } else {
        oldValue = Number($gameVariables.value(vId) || 0);
        newValue = oldValue;
      }

      const actor = $gameActors.actor(this.cfg.actorId);
      if (actor) {
        if (this.cfg.affectActorHP) actor.gainHp(z);
        if (this.cfg.affectActorMP) actor.gainMp(z);
      }

      const text = this.formatPopup({z, vId, oldValue, newValue});
      KZEGTF.popup(text, this.cfg);

      if (this.cfg.commonEventId > 0) KZEGTF.runCommonEventParallel(this.cfg.commonEventId);

      if (this.cfg.affectVariable) flashByVariableId(vId);
      if (this.cfg.affectActorHP && actor) flashByActorParam(actor.actorId(), 'hp');
      if (this.cfg.affectActorMP && actor) flashByActorParam(actor.actorId(), 'mp');
    }
    formatPopup(ctx){
      let s = String(this.cfg.popupText || '{Z}');
      s = s.replace(/\{Z\}/g, String(ctx.z));
      s = s.replace(/\{VALUE\}/g, String(ctx.newValue));
      s = s.replace(/\{OLD\}/g, String(ctx.oldValue));
      s = s.replace(/\\V\[(\d+)\]/gi, (_,n)=> String($gameVariables.value(Number(n))||0));
      return s;
    }
  }

  const _Scene_Boot_start = Scene_Boot.prototype.start;
  Scene_Boot.prototype.start = function(){
    _Scene_Boot_start.apply(this, arguments);
    KZEGTF.setup();
  };
  const _SceneManager_updateMain = SceneManager.updateMain;
  SceneManager.updateMain = function(){
    _SceneManager_updateMain.apply(this, arguments);
    KZEGTF.updateOncePerFrame();
  };
})();
