/*:
 * @param ---- Lanes ----
 * @param LaneLeftX @text 左レーンX @type number @min 0 @default 260
 * @param LaneRightX @text 右レーンX @type number @min 0 @default 1020
 * @param LaneTopY @text レーン上端Y @type number @min 0 @default 80
 * @param LaneBottomY @text レーン下端Y @type number @min 0 @default 620
 * @param TargetY @text ターゲットY @type number @min 0 @default 420
 * @param TargetRadius @text ターゲット半径 @type number @min 8 @default 86
 *
 * @param ---- Flow ----
 * @param NoteSpeed @text ノーツ速度(px/秒) @type number @min 10 @default 520
 * @param SpawnInterval @text 生成間隔(秒・基準) @type number @decimals 2 @min 0.10 @default 0.70
 * @param SpawnJitterMin @text 生成間隔ランダム最小(秒) @type number @decimals 2 @min 0 @default 0.50
 * @param SpawnJitterMax @text 生成間隔ランダム最大(秒) @type number @decimals 2 @min 0 @default 1.00
 * @param GameDuration @text 制限時間(秒) @type number @min 3 @default 30
 * @param JudgeWindow @text 判定幅(ピクセル) @type number @min 4 @default 40
 * @param TouchSafeMargin @text タッチ左右判定の幅 @type number @min 16 @default 140
 *
 * @param ---- Note Images ----
 * @param NotePictureLeft  @text 左ノーツ画像 @type file @dir img/pictures @default
 * @param NotePictureRight @text 右ノーツ画像 @type file @dir img/pictures @default
 * @param NoteImageSize @text ノーツ画像サイズ(px) @desc 0で原寸 @type number @min 0 @default 64
 *
 * @param ---- Gauge (images first) ----
 * @param GaugeX @text ゲージX @type number @min 0 @default 820
 * @param GaugeY @text ゲージY @type number @min 0 @default 690
 * @param GaugeBgImage @text 背景画像 @type file @dir img/pictures @default
 * @param GaugeFillImage @text ゲージ画像 @type file @dir img/pictures @default
 * @param GaugeMaskImage @text マスク画像 @type file @dir img/pictures @default
 * @param GaugeFillFromLeft @text 左から伸ばす @type boolean @on はい @off いいえ @default true
 *
 * @param GaugeW @text [矩形描画用]幅 @type number @min 1 @default 420
 * @param GaugeH @text [矩形描画用]高さ @type number @min 1 @default 24
 * @param GaugeBgColor @text [矩形描画用]背景色CSS @type string @default rgba(0,0,0,0.25)
 * @param GaugeFillColor @text [矩形描画用]色CSS @type string @default #7ad21b
 * @param PassThresholdPercent @text 合格しきい値(%) @type number @min 1 @max 100 @default 80
 *
 * @param ---- UI: Help ----
 * @param HelpText @text ヘルプ文字列 @type string @default ヘルプテキスト
 * @param HelpX @text ヘルプX @type number @min 0 @default 28
 * @param HelpY @text ヘルプY @type number @min 0 @default 690
 * @param HelpFontSize @text ヘルプ文字サイズ @type number @min 8 @default 26
 *
 * @param ---- UI: Time Label & Number ----
 * @param TimeLabelText @text 残り時間ラベル @type string @default 残り時間
 * @param TimeLabelX @text ラベルX @type number @min 0 @default 900
 * @param TimeLabelY @text ラベルY @type number @min 0 @default 650
 * @param TimeLabelFontSize @text ラベル文字サイズ @type number @min 8 @default 26
 * @param TimeNumX @text 秒数X @type number @min 0 @default 965
 * @param TimeNumY @text 秒数Y @type number @min 0 @default 700
 * @param TimeNumFontSize @text 秒数文字サイズ @type number @min 8 @default 64
 * @param TimeNumWidth @text 秒数テキスト幅 @type number @min 1 @default 160
 * @param TimeNumHeight @text 秒数テキスト高さ @type number @min 1 @default 72
 *
 * @param ---- UI: Satisfaction Texts ----
 * @param SatLeftText @text 満足度左テキスト @type string @default ママ 満足度
 * @param SatLeftX @text 左テキストX @type number @min 0 @default 900
 * @param SatLeftY @text 左テキストY @type number @min 0 @default 690
 * @param SatLeftFontSize @text 左テキストサイズ @type number @min 8 @default 26
 * @param SatRightText @text 右テキスト @type string @default 満足
 * @param SatRightX @text 右テキストX @type number @min 0 @default 1250
 * @param SatRightY @text 右テキストY @type number @min 0 @default 690
 * @param SatRightFontSize @text 右テキストサイズ @type number @min 8 @default 26
 *
 * @param ---- Guides ----
 * @param DrawGuides @text ガイド表示 @type boolean @on 表示 @off 非表示 @default true
 *
 * @param ---- Common Events ----
 * @param StartCommonEvent @text 開始時CE @type common_event @default 0
 * @param SuccessLeftCommonEvent @text 左成功CE @type common_event @default 0
 * @param SuccessRightCommonEvent @text 右成功CE @type common_event @default 0
 * @param MissCommonEvent @text 失敗CE @type common_event @default 0
 * @param EndCommonEvent @text 終了時CE @type common_event @default 0
 *
 * @param ---- Result Vars/Switch ----
 * @param ScoreVarId @text スコア変数ID @type variable @default 11
 * @param ComboVarId @text 最高コンボ変数ID @type variable @default 12
 * @param ResultSwitchId @text 合否スイッチID @type switch @default 21
 * @param PassScore @text 参考:スコア閾値 @desc 未使用・互換のため残置 @type number @default 10
 *
 * @command StartGame
 * @text ミニゲーム開始
 * @arg duration @text 制限時間(秒) @type number @min 3 @default 0
 * @arg interval @text 生成間隔(秒・基準) @type number @decimals 2 @min 0.10 @default 0
 * @arg speed @text 速度(px/秒) @type number @min 10 @default 0
 * @arg successTarget @text 必要成功数 @type number @min 1 @default 0
 */
(() => {
  "use strict";
  const pluginName = "KZ_ShoulderTapMiniGame";
  const P = PluginManager.parameters(pluginName);
  const N = n => Number(n || 0);
  const S = s => String(s || "");

  const param = {
    lx: N(P.LaneLeftX || 260),
    rx: N(P.LaneRightX || 1020),
    topY: N(P.LaneTopY || 80),
    bottomY: N(P.LaneBottomY || 620),
    ty: N(P.TargetY || 420),
    tr: N(P.TargetRadius || 86),

    speed: N(P.NoteSpeed || 520),
    intervalBase: N(P.SpawnInterval || 0.7),
    jitterMin: N(P.SpawnJitterMin || 0.5),
    jitterMax: N(P.SpawnJitterMax || 1.0),

    duration: N(P.GameDuration || 30),
    judge: N(P.JudgeWindow || 40),
    touchMargin: N(P.TouchSafeMargin || 140),

    noteImgL: S(P.NotePictureLeft),
    noteImgR: S(P.NotePictureRight),
    noteImgSize: N(P.NoteImageSize || 64),

    gX: N(P.GaugeX || 820),
    gY: N(P.GaugeY || 690),

    gBgImg: S(P.GaugeBgImage),
    gFillImg: S(P.GaugeFillImage),
    gMaskImg: S(P.GaugeMaskImage),
    gLeft: P.GaugeFillFromLeft === "true",

    gW: N(P.GaugeW || 420),
    gH: N(P.GaugeH || 24),
    gBgCss: S(P.GaugeBgColor || "rgba(0,0,0,0.25)"),
    gFillCss: S(P.GaugeFillColor || "#7ad21b"),
    passPct: N(P.PassThresholdPercent || 80),

    helpText: S(P.HelpText || "ヘルプテキスト"),
    helpX: N(P.HelpX || 28),
    helpY: N(P.HelpY || 690),
    helpFs: N(P.HelpFontSize || 26),

    tlText: S(P.TimeLabelText || "残り時間"),
    tlX: N(P.TimeLabelX || 900),
    tlY: N(P.TimeLabelY || 650),
    tlFs: N(P.TimeLabelFontSize || 26),

    tnX: N(P.TimeNumX || 965),
    tnY: N(P.TimeNumY || 700),
    tnFs: N(P.TimeNumFontSize || 64),
    tnW:  N(P.TimeNumWidth || 160),
    tnH:  N(P.TimeNumHeight || 72),

    satLText: S(P.SatLeftText || "ママ 満足度"),
    satLX: N(P.SatLeftX || 900),
    satLY: N(P.SatLeftY || 690),
    satLFs: N(P.SatLeftFontSize || 26),
    satRText: S(P.SatRightText || "満足"),
    satRX: N(P.SatRightX || 1250),
    satRY: N(P.SatRightY || 690),
    satRFs: N(P.SatRightFontSize || 26),

    drawGuides: P.DrawGuides === "true",

    ceStart: N(P.StartCommonEvent || 0),
    ceGoodL: N(P.SuccessLeftCommonEvent || 0),
    ceGoodR: N(P.SuccessRightCommonEvent || 0),
    ceMiss: N(P.MissCommonEvent || 0),
    ceEnd:  N(P.EndCommonEvent || 0),

    scoreVar: N(P.ScoreVarId || 11),
    comboVar: N(P.ComboVarId || 12),
    resultSw: N(P.ResultSwitchId || 21),
  };

  PluginManager.registerCommand(pluginName, "StartGame", args => {
    const duration      = N(args.duration || 0) || param.duration;
    const intervalBase  = N(args.interval || 0) || param.intervalBase;
    const speed         = N(args.speed || 0)    || param.speed;
    const successTarget = N(args.successTarget || 0) || 10;
    SceneManager.push(Scene_ShoulderTap);
    SceneManager.prepareNextScene({ duration, intervalBase, speed, successTarget });
  });

  function drawCircleBmp(bmp, cx, cy, r, color, thick) {
    const ctx = bmp.context;
    ctx.save();
    ctx.lineWidth = thick || 8;
    ctx.strokeStyle = color || "rgba(0,200,255,0.9)";
    ctx.beginPath();
    ctx.arc(cx, cy, r, 0, Math.PI * 2);
    ctx.stroke();
    ctx.restore();
    bmp._baseTexture.update();
  }

  class Note {
    constructor(side, x, y, speed, imgName, imgSize) {
      this.side = side;
      this.x = x;
      this.y = y;
      this.speed = speed;
      this.hit = false;
      this.dead = false;

      if (imgName) {
        const sprite = new Sprite(ImageManager.loadPicture(imgName));
        sprite.anchor.set(0.5, 0.5);
        if (imgSize > 0) {
          const applyScale = bm => {
            sprite.scale.x = imgSize / Math.max(1, bm.width);
            sprite.scale.y = imgSize / Math.max(1, bm.height);
          };
          sprite.bitmap?.addLoadListener(applyScale);
        }
        this.sprite = sprite;
      } else {
        const g = new PIXI.Graphics();
        g.beginFill(0x00bfff, 0.85);
        g.drawCircle(0, 0, 26);
        g.endFill();
        g.lineStyle(4, 0xffffff, 0.9);
        g.drawCircle(0, 0, 26);
        const s = new Sprite(); s.addChild(g); this.sprite = s;
      }
      this.sprite.x = this.x; this.sprite.y = this.y; this.sprite.alpha = 0.95;
    }
    update(dt){ if(!this.dead){ this.y += this.speed * dt; this.sprite.y = this.y; } }
    kill(){ this.dead = true; if (this.sprite.parent) this.sprite.parent.removeChild(this.sprite); this.sprite.destroy({children:true}); }
  }

  class Scene_ShoulderTap extends Scene_Base {
    initialize() {
      super.initialize();
      this._settings = { duration:param.duration, intervalBase:param.intervalBase, speed:param.speed, successTarget:10 };
      this._intervalNext = this.nextInterval();
      this._notes = [];
      this._elapsed = 0;
      this._spawnElapsed = 0;
      this._score = 0;
      this._combo = 0;
      this._maxCombo = 0;
      this._succCount = 0;
      this._ended = false;
      this._exitPending = false;
      this._interpreter = new Game_Interpreter();
    }

    clearMiniGameUi() {
      for (const n of this._notes) {
        if (!n.dead) n.kill();
      }
      this._notes = [];

      const removeSprite = s => {
        if (!s) return;
        if (s.parent) {
          s.parent.removeChild(s);
        }
        if (s !== this._spritesetBase) {
          s.destroy({ children: true });
        }
      };

      removeSprite(this._guide);
      removeSprite(this._help);
      removeSprite(this._timeLabel);
      removeSprite(this._timeNum);
      removeSprite(this._satLeft);
      removeSprite(this._satRight);
      removeSprite(this._gCont);
      removeSprite(this._gBg);
      removeSprite(this._gFill);
      removeSprite(this._touchBlocker);

      this._guide = null;
      this._help = null;
      this._timeLabel = null;
      this._timeNum = null;
      this._satLeft = null;
      this._satRight = null;
      this._gCont = null;
      this._gBg = null;
      this._gFill = null;
      this._touchBlocker = null;
    }
        
    prepare(s){ if(s) this._settings = s; }
    nextInterval(){
      const min = Math.min(param.jitterMin, param.jitterMax);
      const max = Math.max(param.jitterMin, param.jitterMax);
      const r = min + Math.random()*(max-min);
      const sign = Math.random()<0.5 ? -1 : 1;
      return Math.max(0.10, this._settings.intervalBase + sign*r);
    }

    create() {
      super.create();
      this.createBackgroundLayer();
      this.createGuideLayer();
      this.createUiLayer();
      this._touchBlocker = new Sprite(new Bitmap(Graphics.width, Graphics.height));
      this._touchBlocker.opacity = 0;
      this.addChild(this._touchBlocker);
      this.setupTouchHandlers();
    }
    start(){
      super.start();
      this._elapsed = 0; this._spawnElapsed = 0;
      if (param.ceStart>0) $gameTemp.reserveCommonEvent(param.ceStart);
    }

    update(){
      super.update();

      if (this._interpreter.isRunning()) {
        this._interpreter.update();
      } else if ($gameTemp.isCommonEventReserved()) {
        const ce = $gameTemp.retrieveCommonEvent();
        this._interpreter.setup(ce.list);
        this._interpreter.update();
      }

      if (this._interpreter.isRunning() || $gameMessage.isBusy()) return;

      if (this._exitPending) { SceneManager.pop(); return; }
      if (this._ended) return;

      const dt = Graphics._deltaTime || 1/60;
      this._elapsed += dt; this._spawnElapsed += dt;

      if (this._spawnElapsed >= this._intervalNext) {
        this._spawnElapsed = 0; this._intervalNext = this.nextInterval();
        this.spawnNote();
      }
      for (const n of this._notes) n.update(dt);
      this.processAutoMiss();
      this.updateKeyInput();
      this.updateUi();

      if (this._elapsed >= this._settings.duration) this.finishAndReturn();
    }

    createBackgroundLayer(){ this._spritesetBase = new Spriteset_Base(); this.addChild(this._spritesetBase); }

    createGuideLayer(){
      this._guide = new Sprite(new Bitmap(Graphics.width, Graphics.height));
      this.addChild(this._guide);
      if (!param.drawGuides) return;
      const b = this._guide.bitmap;
      b.paintOpacity = 160;
      b.fillRect(param.lx-2, param.topY, 4, param.bottomY-param.topY, "#77aaff");
      b.fillRect(param.rx-2, param.topY, 4, param.bottomY-param.topY, "#77aaff");
      b.paintOpacity = 255;
      drawCircleBmp(b, param.lx, param.ty, param.tr, "rgba(0,200,255,0.85)", 8);
      drawCircleBmp(b, param.lx, param.ty, param.tr-12, "rgba(255,255,255,0.9)", 4);
      drawCircleBmp(b, param.rx, param.ty, param.tr, "rgba(0,200,255,0.85)", 8);
      drawCircleBmp(b, param.rx, param.ty, param.tr-12, "rgba(255,255,255,0.9)", 4);
    }

    createUiLayer(){
      this._help = new Sprite(new Bitmap(Graphics.width, 80));
      this._help.x = param.helpX; this._help.y = param.helpY; this.addChild(this._help);

      this._timeLabel = new Sprite(new Bitmap(Graphics.width, 50));
      this._timeLabel.x = param.tlX; this._timeLabel.y = param.tlY; this.addChild(this._timeLabel);

      this._timeNum = new Sprite(new Bitmap(param.tnW, param.tnH));
      this._timeNum.x = param.tnX; this._timeNum.y = param.tnY; this.addChild(this._timeNum);

      this._satLeft = new Sprite(new Bitmap(Graphics.width, 50));
      this._satLeft.x = param.satLX; this._satLeft.y = param.satLY; this.addChild(this._satLeft);

      this._satRight = new Sprite(new Bitmap(Graphics.width, 50));
      this._satRight.x = param.satRX; this._satRight.y = param.satRY; this.addChild(this._satRight);

      if (param.gFillImg || param.gBgImg || param.gMaskImg) this.createImageGauge();
      else this.createRectGauge();

      this.redrawStaticUi();
      this.updateUi();
    }

    createImageGauge() {
      this._gCont = new Sprite();
      this._gCont.x = param.gX; this._gCont.y = param.gY;
      this.addChild(this._gCont);

      if (param.gBgImg) {
        this._gBgSpr = new Sprite(ImageManager.loadPicture(param.gBgImg));
        this._gCont.addChild(this._gBgSpr);
      }

      this._gFillSpr = new Sprite(param.gFillImg ? ImageManager.loadPicture(param.gFillImg) : new Bitmap(1,1));
      if (!param.gFillImg) this._gFillSpr.bitmap.fillAll(param.gFillCss);
      this._gCont.addChild(this._gFillSpr);

      if (param.gMaskImg) {
        this._gMaskSpr = new Sprite(ImageManager.loadPicture(param.gMaskImg));
        const align = () => { this._gMaskSpr.x = 0; this._gMaskSpr.y = 0; this._gFillSpr.mask = this._gMaskSpr; };
        if (this._gMaskSpr.bitmap.isReady()) align(); else this._gMaskSpr.bitmap.addLoadListener(align);
        this._gCont.addChild(this._gMaskSpr);
      }

      this._gFillSpr.anchor.x = param.gLeft ? 0 : 1;
      this._gFillSpr.x = param.gLeft ? 0 : (this._gBgSpr?.bitmap?.width || this._gFillSpr.bitmap.width);

      const initFrame = () => {
        const bw = this._gFillSpr.bitmap.width;
        const bh = this._gFillSpr.bitmap.height;
        if (param.gLeft) this._gFillSpr.setFrame(0, 0, 0, bh);
        else this._gFillSpr.setFrame(bw, 0, 0, bh);
      };
      if (this._gFillSpr.bitmap.isReady()) initFrame(); else this._gFillSpr.bitmap.addLoadListener(initFrame);
    }

    createRectGauge() {
      this._gBg = new Sprite(new Bitmap(param.gW, param.gH));
      this._gFill = new Sprite(new Bitmap(param.gW, param.gH));
      this._gBg.x = param.gX; this._gBg.y = param.gY;
      this._gFill.x = param.gX; this._gFill.y = param.gY;
      this.addChild(this._gBg); this.addChild(this._gFill);
      this.cssFillRect(this._gBg.bitmap, param.gBgCss);
    }

    cssFillRect(bmp, css){
      const ctx = bmp.context;
      ctx.clearRect(0,0,bmp.width,bmp.height);
      ctx.fillStyle = css;
      ctx.fillRect(0,0,bmp.width,bmp.height);
      bmp._baseTexture.update();
    }

    redrawStaticUi(){
      const hb = this._help.bitmap; hb.clear();
      hb.fontFace = $gameSystem.mainFontFace(); hb.fontSize = param.helpFs; hb.textColor = "#000";
      hb.drawText(param.helpText, 0, 0, Graphics.width, 40, "left");

      const lb = this._timeLabel.bitmap; lb.clear();
      lb.fontFace = $gameSystem.mainFontFace(); lb.fontSize = param.tlFs; lb.textColor = "#000";
      lb.drawText(param.tlText, 0, 0, Graphics.width, 40, "left");

      const sl = this._satLeft.bitmap; sl.clear();
      sl.fontFace = $gameSystem.mainFontFace(); sl.fontSize = param.satLFs; sl.textColor = "#000";
      sl.drawText(param.satLText, 0, 0, Graphics.width, 40, "left");

      const sr = this._satRight.bitmap; sr.clear();
      sr.fontFace = $gameSystem.mainFontFace(); sr.fontSize = param.satRFs; sr.textColor = "#000";
      sr.drawText(param.satRText, 0, 0, Graphics.width, 40, "left");
    }

    updateUi(){
      const remain = Math.max(0, Math.ceil(this._settings.duration - this._elapsed));
      const nb = this._timeNum.bitmap; nb.clear();
      nb.fontFace = $gameSystem.mainFontFace();
      nb.fontSize = param.tnFs;
      nb.textColor = "#000";
      nb.drawText(String(remain), 0, 0, param.tnW, param.tnH, "center");

      const pct = this.gaugePercent();
      if (this._gFillSpr) {
        const bw = this._gFillSpr.bitmap.width;
        const bh = this._gFillSpr.bitmap.height;
        const w = Math.round((pct/100) * bw);
        if (param.gLeft) this._gFillSpr.setFrame(0, 0, w, bh);
        else { const sx = bw - w; this._gFillSpr.setFrame(sx, 0, w, bh); }
      } else if (this._gFill) {
        const w = Math.round((pct/100) * param.gW);
        const fb = this._gFill.bitmap; fb.clear();
        if (w > 0){ const ctx = fb.context; ctx.fillStyle = param.gFillCss; ctx.fillRect(0,0,w,param.gH); fb._baseTexture.update(); }
      }
    }

    gaugePercent(){
      const tgt = Math.max(1, this._settings.successTarget);
      const v = Math.min(Math.max(this._succCount, 0), tgt);
      return (v / tgt) * 100;
    }

    spawnNote(){
      const side = Math.random()<0.5 ? "L" : "R";
      const x = side==="L" ? param.lx : param.rx;
      const y = param.topY - 24;
      const img = side==="L" ? param.noteImgL : param.noteImgR;
      const n = new Note(side, x, y, this._settings.speed, img, param.noteImgSize);
      this.addChild(n.sprite);
      this._notes.push(n);
    }

    judgeInput(side){
      let judged = false;
      for (const n of this._notes){
        if (n.dead || n.hit) continue;
        if ((side==="L" && n.side!=="L") || (side==="R" && n.side!=="R")) continue;
        const dy = Math.abs(n.y - param.ty);
        if (dy <= param.judge){
          n.hit = true; judged = true;
          this.flashCircle(side);
          this._score += 1;
          this._combo += 1;
          this._maxCombo = Math.max(this._maxCombo, this._combo);
          this._succCount = Math.min(this._succCount + 1, this._settings.successTarget);
          if (side==="L" && param.ceGoodL>0) $gameTemp.reserveCommonEvent(param.ceGoodL);
          if (side==="R" && param.ceGoodR>0) $gameTemp.reserveCommonEvent(param.ceGoodR);
          n.kill(); break;
        }
      }
      if (!judged) this.onMiss();
    }

    onMiss(){
      this._combo = 0;
      this._succCount = Math.max(0, this._succCount - 1);
      if (param.ceMiss>0) $gameTemp.reserveCommonEvent(param.ceMiss);
    }

    processAutoMiss(){
      for (const n of this._notes){
        if (n.dead) continue;
        if (n.y > param.bottomY + 40){ n.kill(); this.onMiss(); }
      }
      this._notes = this._notes.filter(n=>!n.dead);
    }

    updateKeyInput(){
      if (Input.isTriggered("left")) this.judgeInput("L");
      if (Input.isTriggered("right")) this.judgeInput("R");
    }

    setupTouchHandlers(){
      const self = this;
      this._touchBlocker.update = function(){
        if (TouchInput.isTriggered()){
          const x = TouchInput.x, y = TouchInput.y;
          if (y < param.ty - param.tr - 60 || y > param.ty + param.tr + 60) return;
          const dxL = Math.abs(x - param.lx);
          const dxR = Math.abs(x - param.rx);
          if (dxL <= param.touchMargin && dxL <= dxR) self.judgeInput("L");
          else if (dxR <= param.touchMargin) self.judgeInput("R");
        }
      };
    }

    flashCircle(side){
      const x = side==="L" ? param.lx : param.rx;
      const y = param.ty;
      const g = new PIXI.Graphics();
      g.lineStyle(10, 0x66ffcc, 1.0);
      g.drawCircle(0, 0, param.tr + 8);
      g.x = x; g.y = y; g.alpha = 1.0;
      this.addChild(g);
      const step = () => {
        g.alpha -= 0.08;
        g.scale.set(g.scale.x + 0.06, g.scale.y + 0.06);
        if (g.alpha <= 0){ this.removeChild(g); g.destroy(); return; }
        requestAnimationFrame(step);
      };
      requestAnimationFrame(step);
    }

    finishAndReturn(){
      if (this._ended) return;
      this._ended = true;

      this.clearMiniGameUi();

      $gameVariables.setValue(param.scoreVar, this._score);
      $gameVariables.setValue(param.comboVar, this._maxCombo);

      const pass = this.gaugePercent() >= param.passPct;
      if (param.resultSw>0) $gameSwitches.setValue(param.resultSw, pass);

      if (param.ceEnd>0) $gameTemp.reserveCommonEvent(param.ceEnd);
      this._exitPending = true;
    }

  }
})();
