/*:
 * @param Pictures
 * @text ピクチャ設定
 * @type struct<PictureConfig>[]
 * @default []
 * @desc ピクチャを設定します。
 */

/*~struct~PictureConfig:
 * @param switchId
 * @text 表示制御スイッチID
 * @type switch
 * @desc このスイッチのオン／オフでピクチャの表示／非表示を制御します。
 * 
 * @param alternateImages
 * @text 切替画像設定
 * @type struct<AlternateImage>[]
 * @default []
 * @desc 複数のSwitchや変数に基づき表示する画像を設定します。
 * 
 * @param commonEventId
 * @text コモンイベントID
 * @type common_event
 * @default 0
 * @desc ピクチャをクリックした際に起動するコモンイベントです。
 * 
 * @param actorOpacity
 * @text アクター重複時の透明度
 * @type number
 * @min 0
 * @max 255
 * @default 50
 * @desc ピクチャにアクターが重なった場合の透明度を指定します。
 * 
 * @param hideDuringEvent
 * @text イベント中に非表示
 * @type boolean
 * @default true
 * @desc イベント中に自動的に非表示にするかどうか。
 * 
 * @param x
 * @text ピクチャX座標
 * @type number
 * @default 0
 * @desc ピクチャのX座標です。
 * 
 * @param y
 * @text ピクチャY座標
 * @type number
 * @default 0
 * @desc ピクチャのY座標です。
 */

/*~struct~AlternateImage:
 * @param switchId
 * @text 切替スイッチID
 * @type switch
 * @default 0
 * @desc このスイッチがオンの場合に表示される画像です。（0の場合は無視）
 * 
 * @param variableId
 * @text 変数ID
 * @type variable
 * @default 0
 * @desc この変数の値が指定された値の場合に表示される画像です。（0の場合は無視）
 * 
 * @param variableValue
 * @text 変数値
 * @type text
 * @default
 * @desc variableIdで指定された変数の値が一致した場合にこの画像が表示されます。
 * 
 * @param image
 * @text 画像ファイル
 * @type file
 * @dir img/pictures/
 * @desc 表示する画像ファイルを指定します。
 */

(() => {
    const parameters = PluginManager.parameters('Kamichichi_InteractivePictureMap');
    const pictureConfigs = JSON.parse(parameters['Pictures'] || '[]').map(config => JSON.parse(config));

    const PLUGIN_PICTURE_ID_OFFSET = 100;
    let isEventRunning = false;

    function isActorWithinPicture(config, pictureId) {
        const picture = $gameScreen.picture(pictureId);
        if (picture) {
            const pictureX = Number(config.x);
            const pictureY = Number(config.y);
            const { width: pictureWidth, height: pictureHeight } = getPictureSize(picture._name);

            const actorX = $gamePlayer.screenX();
            const actorY = $gamePlayer.screenY();

            return (
                actorX >= pictureX &&
                actorX <= pictureX + pictureWidth &&
                actorY >= pictureY - 96 &&
                actorY <= pictureY - 96 + pictureHeight
            );
        }
        return false;
    }

    function updatePictureOpacityWithFade(config, pictureId, targetOpacity, fadeSpeed = 10) {
        const picture = $gameScreen.picture(pictureId);
        if (picture) {
            const currentOpacity = picture._opacity || 255;
            const newOpacity = currentOpacity < targetOpacity
                ? Math.min(currentOpacity + fadeSpeed, targetOpacity)
                : Math.max(currentOpacity - fadeSpeed, targetOpacity);

            if (currentOpacity !== newOpacity) {
                $gameScreen.showPicture(
                    pictureId,
                    picture._name,
                    0,
                    Number(config.x),
                    Number(config.y),
                    100,
                    100,
                    newOpacity,
                    0
                );
            }
        }
    }

    function getPictureSize(pictureName) {
        const bitmap = ImageManager.loadPicture(pictureName);
        if (bitmap && bitmap.isReady()) {
            return { width: bitmap.width, height: bitmap.height };
        }
        return { width: 0, height: 0 };
    }

    function getSelectedImage(config) {
        const alternateImages = JSON.parse(config.alternateImages || '[]').map(ai => JSON.parse(ai));
        for (const altImage of alternateImages) {
            const isSwitchValid = Number(altImage.switchId) > 0 && $gameSwitches.value(Number(altImage.switchId));
            const isVariableValid =
                Number(altImage.variableId) > 0 &&
                String($gameVariables.value(Number(altImage.variableId))) === String(altImage.variableValue);

            if (isSwitchValid || isVariableValid) {
                return altImage.image;
            }
        }
        return null;
    }

    function updatePicture(config, pictureIndex) {
        const pictureId = PLUGIN_PICTURE_ID_OFFSET + pictureIndex;
        if (isEventRunning) {
            $gameScreen.erasePicture(pictureId);
            return;
        }

        const switchOn = $gameSwitches.value(Number(config.switchId));
        if (switchOn) {
            const selectedImage = getSelectedImage(config);
            if (selectedImage) {
                const picture = $gameScreen.picture(pictureId);
                if (!picture || picture._name !== selectedImage) {
                    $gameScreen.showPicture(
                        pictureId,
                        selectedImage,
                        0,
                        Number(config.x),
                        Number(config.y),
                        100,
                        100,
                        255,
                        0
                    );
                }
            } else {
                $gameScreen.erasePicture(pictureId);
            }
        } else {
            $gameScreen.erasePicture(pictureId);
        }
    }

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function () {
        _Scene_Map_update.call(this);

        pictureConfigs.forEach((config, index) => {
            updatePicture(config, index);

            const pictureId = PLUGIN_PICTURE_ID_OFFSET + index;
            const isActorInside = isActorWithinPicture(config, pictureId);
            const targetOpacity = isActorInside ? Number(config.actorOpacity) : 255;
            updatePictureOpacityWithFade(config, pictureId, targetOpacity);
        });
    };

    const _Game_Map_setupStartingEvent = Game_Map.prototype.setupStartingEvent;
    Game_Map.prototype.setupStartingEvent = function () {
        const started = _Game_Map_setupStartingEvent.call(this);
        isEventRunning = this.isEventRunning();
        return started;
    };

    const _Game_Map_updateInterpreter = Game_Map.prototype.updateInterpreter;
    Game_Map.prototype.updateInterpreter = function () {
        _Game_Map_updateInterpreter.call(this);
        isEventRunning = this.isEventRunning();
    };
})();
