/*:
 * @param GaugeWidth
 * @text ゲージ横幅
 * @type number
 * @default 400
 *
 * @param GaugeHeight
 * @text ゲージ高さ
 * @type number
 * @default 26
 *
 * @param GaugePadding
 * @text 内側余白
 * @type number
 * @default 4
 *
 * @param GaugeOffsetY
 * @text ゲージYオフセット
 * @type number
 * @default 8
 *
 * @command setNextTimer
 * @text 次の選択肢をタイマーにする
 * @desc 次に開く選択肢だけ制限時間を付けます（次の1回だけ）
 *
 * @arg duration
 * @text 制限時間（フレーム）
 * @type number
 * @default 180
 *
 * @arg action
 * @text タイムアウト時動作
 * @type select
 * @option cancel
 * @option ok
 * @option selectFirst
 * @default cancel
 *
 * @arg textColorId
 * @text 次の選択肢テキスト色ID
 * @desc 次の選択肢ウィンドウの選択肢テキスト色を変更します（0で通常色）。ツクールの文字色IDを指定。
 * @type number
 * @min 0
 * @max 31
 * @default 0
 */

(() => {
  "use strict";

  const pluginName = "LL_GalgeChoiceWindow_TimerGaugeAddOn";
  const p = PluginManager.parameters(pluginName);

  const GW = Number(p.GaugeWidth || 400);
  const GH = Number(p.GaugeHeight || 26);
  const GP = Number(p.GaugePadding || 4);
  const GOY = Number(p.GaugeOffsetY || 0);

  // 次の1回だけ有効
  let nextOverride = null;

  PluginManager.registerCommand(pluginName, "setNextTimer", args => {
    nextOverride = {
      duration: Number(args.duration || 0),
      action: String(args.action || "cancel"),
      textColorId: Number(args.textColorId || 0),
    };
  });

  function drawRoundRectPath(ctx, x, y, width, height, radius) {
    const r = Math.min(radius, height / 2, width / 2);
    ctx.beginPath();
    ctx.moveTo(x + r, y);
    ctx.lineTo(x + width - r, y);
    ctx.quadraticCurveTo(x + width, y, x + width, y + r);
    ctx.lineTo(x + width, y + height - r);
    ctx.quadraticCurveTo(x + width, y + height, x + width - r, y + height);
    ctx.lineTo(x + r, y + height);
    ctx.quadraticCurveTo(x, y + height, x, y + height - r);
    ctx.lineTo(x, y + r);
    ctx.quadraticCurveTo(x, y, x + r, y);
    ctx.closePath();
  }

  function ensureState(win) {
    if (!win._lltg) {
      win._lltg = { enabled: false, max: 0, left: 0, action: "cancel", sprite: null, textColorId: 0 };
    }
  }

  function removeGauge(win) {
    const scene = SceneManager._scene;
    if (!scene || !win || !win._lltg || !win._lltg.sprite) return;
    scene.removeChild(win._lltg.sprite);
    win._lltg.sprite.destroy({ children: true, texture: true, baseTexture: true });
    win._lltg.sprite = null;
  }

  function createGauge(win) {
    const scene = SceneManager._scene;
    if (!scene) return;
    removeGauge(win);
    win._lltg.sprite = new Sprite(new Bitmap(GW, GH));
    scene.addChild(win._lltg.sprite);
  }

  function updateGauge(win) {
    const st = win._lltg;
    if (!st || !st.sprite || !st.sprite.bitmap) return;

    const rate = st.max > 0 ? Math.max(0, Math.min(1, st.left / st.max)) : 0;

    st.sprite.x = Math.round((Graphics.width - GW) / 2);
    st.sprite.y = Math.round(win.y + win.height + GOY);

    const bmp = st.sprite.bitmap;
    const ctx = bmp.context;
    const w = bmp.width;
    const h = bmp.height;

    bmp.clear();

    const outerRadius = h / 2;
    const innerX = GP;
    const innerY = GP;
    const innerW = w - GP * 2;
    const innerH = h - GP * 2;
    const innerRadius = innerH / 2;
    const fillW = innerW * rate;

    ctx.fillStyle = "#5a2e1f";
    drawRoundRectPath(ctx, 0, 0, w, h, outerRadius);
    ctx.fill();

    ctx.fillStyle = "#ffffff";
    drawRoundRectPath(ctx, innerX, innerY, innerW, innerH, innerRadius);
    ctx.fill();

    if (fillW > 0) {
      ctx.save();
      ctx.beginPath();
      ctx.rect(innerX, innerY, fillW, innerH);
      ctx.clip();

      ctx.fillStyle = "#f4b000";
      drawRoundRectPath(ctx, innerX, innerY, innerW, innerH, innerRadius);
      ctx.fill();

      ctx.restore();
    }

    bmp._baseTexture.update();
  }

  function doTimeout(win) {
    const st = win._lltg;
    if (!st) return;

    const canCancel = !!(win.isCancelEnabled && win.isCancelEnabled());
    const action = (st.action === "cancel" && !canCancel) ? "ok" : st.action;

    if (action === "cancel") {
      if (win.callCancelHandler) win.callCancelHandler();
      return;
    }
    if (action === "selectFirst") {
      if (win.select) win.select(0);
    }
    if (win.callOkHandler) win.callOkHandler();
  }

  function consumeNextOverride() {
    const cfg = nextOverride;
    nextOverride = null;
    return cfg;
  }

  // ここが今回のクラッシュ対策：LL側の背景ロードのコールバックを安全化
  function patchLLBackgroundSafety() {
    if (typeof Window_GalgeChoiceList === "undefined") return;
    const proto = Window_GalgeChoiceList.prototype;
    if (proto._lltg_bgSafePatched) return;
    proto._lltg_bgSafePatched = true;

    const _createOriginalBackground = proto.createOriginalBackground;
    if (typeof _createOriginalBackground !== "function") return;

    proto.createOriginalBackground = function() {
      _createOriginalBackground.apply(this, arguments);

      // 既に背景スプライトがある場合だけ、ロード後処理を「安全なもの」に差し替える
      const bg = this._bgSprite;
      const bmp = bg && bg.bitmap;
      if (!bg || !bmp || !bmp.addLoadListener) return;

      // addLoadListener は複数登録できるので「安全版」を追加して、元が落ちてもこちらは落ちない
      bmp.addLoadListener(() => {
        // ウィンドウが閉じられて _bgSprite が null でも落ちないようにする
        if (!this._bgSprite) return;
        if (this._bgSprite !== bg) return; // 途中で差し替わっていたら触らない

        const b = this._bgSprite.bitmap;
        if (!b) return;

        this._bgSprite.x = this.width / 2 - b.width / 2;
        this._bgSprite.y = this.height / 2 - b.height / 2;
      });
    };
  }

  function patchTimerAndAlign(win) {
    if (!win || win._lltg_patched) return;
    win._lltg_patched = true;

    ensureState(win);

    // タイマー有効時だけ左寄せ
    const _itemTextAlign = typeof win.itemTextAlign === "function" ? win.itemTextAlign.bind(win) : null;
    win.itemTextAlign = function() {
      ensureState(this);
      if (this._lltg && this._lltg.enabled) return "left";
      return _itemTextAlign ? _itemTextAlign() : "center";
    };

    // タイマー有効時の「選択肢テキスト色」を反映するため、drawItem を差し替え
    // （元のLL側が drawText で描画している前提で、同等処理に色差し込みを追加）
    if (!win._lltg_drawItemColorPatched) {
      win._lltg_drawItemColorPatched = true;

      win.drawItem = function(index) {
        const rect = this.itemLineRect(index);
        const align = this.itemTextAlign();

        this.resetTextColor();
        this.changePaintOpacity(this.isCommandEnabled(index));

        ensureState(this);
        const st = this._lltg;

        // ★ここが追加：次の選択肢がタイマー対象のときだけ色を変更
        const colorId = Number(st && st.enabled ? (st.textColorId ?? 0) : 0);
        if (colorId > 0 && typeof ColorManager !== "undefined" && ColorManager.textColor) {
          this.changeTextColor(ColorManager.textColor(colorId));
        }

        // LL本体と同じく drawText を使う（制御文字なし運用）
        const name = String(this.commandName(index));
        this.drawText(name, rect.x, rect.y + 8, rect.width, align);

        this.resetTextColor();
      };
    }

    const _start = win.start;
    win.start = function() {
      _start.apply(this, arguments);

      ensureState(this);
      const st = this._lltg;

      // デフォルト無効
      st.enabled = false;
      st.max = 0;
      st.left = 0;
      st.action = "cancel";
      st.textColorId = 0;
      removeGauge(this);

      const cfg = consumeNextOverride();
      if (!cfg) return;

      const dur = Number(cfg.duration || 0);
      if (dur <= 0) return;

      st.enabled = true;
      st.max = dur;
      st.left = dur;
      st.action = String(cfg.action || "cancel");
      st.textColorId = Number(cfg.textColorId || 0);

      createGauge(this);
      updateGauge(this);

      if (this.refresh) this.refresh(); // 左寄せ／色変更を確実に反映
    };

    const _update = win.update;
    win.update = function() {
      _update.apply(this, arguments);

      const st = this._lltg;
      if (!st || !st.enabled) return;
      if (!this.isOpen || !this.isOpen()) return;
      if (!this.active) return;

      if (st.left > 0) {
        st.left--;
        updateGauge(this);
        if (st.left <= 0) {
          removeGauge(this);
          st.enabled = false;
          st.textColorId = 0;
          if (this.refresh) this.refresh();
          doTimeout(this);
        }
      }
    };

    const _close = win.close;
    win.close = function() {
      removeGauge(this);
      if (this._lltg) {
        this._lltg.enabled = false;
        this._lltg.textColorId = 0;
      }
      _close.apply(this, arguments);
    };
  }

  // LLが Scene_Message にウィンドウを作るタイミングで拾う
  const _create = Scene_Message.prototype.createGalgeChoiceListWindow;
  if (typeof _create !== "function") return;

  Scene_Message.prototype.createGalgeChoiceListWindow = function() {
    // 先にLLの背景安全化（Window生成より前にやっておくのが安全）
    patchLLBackgroundSafety();

    _create.apply(this, arguments);

    // ここで生成済みの実体にタイマー/左寄せ/色変更を注入
    patchTimerAndAlign(this._galgeChoiceListWindow);
  };



  
})();
