//=============================================================================
// MessageWindowHidden.js
// ----------------------------------------------------------------------------
// (C)2015 Triacontane
// This software is released under the MIT License.
// http://opensource.org/licenses/mit-license.php
// http://opensource.org/licenses/mit-license.php
// ----------------------------------------------------------------------------
// Version
// 2.9.0 2025/11/26 メッセージ非表示時に自動表示するピクチャとテキストを構造体で指定できる機能を追加
// 2.8.2 2023/03/14 決定動作により復帰するとき、スイッチ同期が設定されていると一瞬復帰したあと元に戻ってしまう問題を修正
// 1.0.0 2015/12/30 初版
// ----------------------------------------------------------------------------
// [Blog]   : https://triacontane.blogspot.jp/
// [Twitter]: https://twitter.com/triacontane/
// [GitHub] : https://github.com/triacontane/
//=============================================================================

/*:
 * @plugindesc メッセージウィンドウ一時消去プラグイン
 * @target MZ
 * @url https://github.com/triacontane/RPGMakerMV/tree/mz_master/MessageWindowHidden.js
 * @author トリアコンタン
 *
 * @param triggerButton
 * @text ボタン名称
 * @desc ウィンドウを消去するボタンです。(複数登録可能) プラグイン等で入力可能なボタンを追加した場合は直接入力
 * @default ["右クリック"]
 * @type combo[]
 * @option 右クリック
 * @option shift
 * @option control
 * @option tab
 * @option pageup
 * @option pagedown
 * @option debug
 *
 * @param triggerSwitch
 * @text トリガースイッチ
 * @desc 指定したスイッチに連動させてウィンドウを消去します。並列処理などを使ってON/OFFを適切に管理してください。
 * @default 0
 * @type switch
 * 
 * @param syncSwitch
 * @text トリガースイッチ同期
 * @desc 有効にすると、手動で消去した場合も含めてウィンドウの表示状態とトリガースイッチが同期します。
 * @default false
 * @type boolean
 *
 * @param linkPictureNumbers
 * @text 連動ピクチャ番号
 * @desc ウィンドウ消去時に連動して不透明度を[0]にするピクチャの番号です。
 * @default []
 * @type number[]
 *
 * @param linkShowPictureNumbers
 * @text 連動表示ピクチャ番号
 * @desc ウィンドウ消去時に連動して不透明度を[255]にするピクチャの番号です。
 * @default []
 * @type number[]
 *
 * @param disableLinkSwitchId
 * @text 連動ピクチャ無効スイッチ
 * @desc 指定した番号のスイッチがONのとき、ピクチャの連動が無効になります。
 * @default 0
 * @type switch
 *
 * @param disableSwitchId
 * @text 無効スイッチ
 * @desc 指定した番号のスイッチがONのとき、プラグイン全体の機能が無効になります。
 * @default 0
 * @type switch
 *
 * @param disableInBattle
 * @text 戦闘中無効化
 * @desc trueのとき、戦闘中にプラグインの機能を無効にします。
 * @default false
 * @type boolean
 *
 * @param disableInChoice
 * @text 選択肢表示中は無効
 * @desc 選択肢の表示中はウィンドウを非表示にできなくなります。
 * @default true
 * @type boolean
 *
 * @param restoreByDecision
 * @text 決定動作で復帰
 * @desc メッセージ消去時、決定動作により非表示になっていたメッセージウィンドウを再表示できます。
 * @default false
 * @type boolean
 *
 * @param hiddenPictures
 * @text 非表示時ピクチャ一覧
 * @desc メッセージウィンドウを非表示にしている間だけ自動表示するピクチャとテキストです。
 * @default []
 * @type struct<HiddenPicture>[]
 *
 * @help メッセージウィンドウを表示中に指定したボタンを押下することで
 * メッセージウィンドウを消去します。もう一度押すと戻ります。
 *
 * ウィンドウ消去時に連動して不透明度を[0]にするピクチャを指定することができます。
 * 背景に特定のピクチャを使用している場合などに指定してください。
 * 再表示すると不透明度は、ウィンドウ非表示まえの不透明度で復元されます。
 *
 * さらに、パラメーター[非表示時ピクチャ一覧]に登録した項目は、
 * メッセージウィンドウを非表示にしている間だけ自動表示されます。
 * ピクチャについてはイベント側で「ピクチャの表示」を事前に書く必要はありません。
 * テキストも同様に、自動で重ねて表示されます。
 *
 * メッセージウィンドウを再表示すると、これらのピクチャとテキストは自動で消えます。
 *
 * このプラグインにはプラグインコマンドはありません。
 *
 * 利用規約：
 *  作者に無断で改変、再配布が可能で、利用形態（商用、18禁利用等）
 *  についても制限はありません。
 *  このプラグインはもうあなたのものです。
 */

/*~struct~HiddenPicture:
 * @param id
 * @text ピクチャ番号
 * @type number
 * @min 1
 * @default 1
 * @desc 表示に使用するピクチャ番号です。他イベントと被らない番号を指定してください。
 *
 * @param picture
 * @text ピクチャ
 * @type file
 * @dir img/pictures
 * @default
 * @desc 表示するピクチャのファイル名です。（拡張子不要）
 *
 * @param x
 * @text X座標
 * @type number
 * @default 0
 * @desc ピクチャの表示X座標です。
 *
 * @param y
 * @text Y座標
 * @type number
 * @default 0
 * @desc ピクチャの表示Y座標です。
 *
 * @param text
 * @text テキスト
 * @type string
 * @default
 * @desc ピクチャの上に表示するテキストです。空欄ならテキストは表示されません。
 *
 * @param align
 * @text テキスト揃え
 * @type select
 * @option 左
 * @value left
 * @option 中央
 * @value center
 * @option 右
 * @value right
 * @default left
 * @desc テキストの揃え方向です。
 *
 * @param textX
 * @text テキストX座標
 * @type number
 * @default 0
 * @desc テキストの表示X座標です。
 *
 * @param textY
 * @text テキストY座標
 * @type number
 * @default 0
 * @desc テキストの表示Y座標です。
 */

(function() {
    'use strict';

    const createPluginParameter = function(pluginName) {
        const paramReplacer = function(key, value) {
            if (value === 'null') {
                return value;
            }
            if (value && value[0] === '"' && value[value.length - 1] === '"') {
                return value;
            }
            try {
                return JSON.parse(value);
            } catch (e) {
                return value;
            }
        };
        const parameter = JSON.parse(JSON.stringify(PluginManager.parameters(pluginName), paramReplacer));
        PluginManager.setParameters(pluginName, parameter);
        return parameter;
    };
    const param = createPluginParameter('MessageWindowHidden');

    // 非表示時ピクチャ一覧（構造体）を扱いやすい形に変換
    const hiddenPictures = (param.hiddenPictures || []).map(function(item) {
        if (!item) return null;
        const id    = Number(item.id || 0);
        const name  = item.picture || '';
        const x     = Number(item.x || 0);
        const y     = Number(item.y || 0);
        const text  = item.text || '';
        const align = item.align || 'left';
        const textX = Number(item.textX || 0);
        const textY = Number(item.textY || 0);
        if (!id || !name) return null;
        return { id, name, x, y, text, align, textX, textY };
    }).filter(function(item) {
        return !!item;
    });

    //=============================================================================
    // Game_Picture
    //  メッセージウィンドウの表示可否と連動します。
    //=============================================================================
    Game_Picture.prototype.linkWithMessageWindow = function(opacity) {
        this._opacity       = opacity;
        this._targetOpacity = opacity;
    };

    //=============================================================================
    // Window_MessageHiddenOverlay
    //  非表示時テキストを描画するための透明な全画面ウィンドウ
    //=============================================================================
    function Window_MessageHiddenOverlay() {
        this.initialize(...arguments);
    }

    Window_MessageHiddenOverlay.prototype = Object.create(Window_Base.prototype);
    Window_MessageHiddenOverlay.prototype.constructor = Window_MessageHiddenOverlay;

    Window_MessageHiddenOverlay.prototype.initialize = function(rect) {
        Window_Base.prototype.initialize.call(this, rect);
        this.opacity = 0;
        this.contentsOpacity = 255;
        this.refresh();
    };

    Window_MessageHiddenOverlay.prototype.refresh = function() {
        this.contents.clear();
        if (!hiddenPictures.length) return;
        hiddenPictures.forEach(function(entry) {
            if (!entry.text) return;
            const x = entry.textX;
            const y = entry.textY;
            const maxWidth = this.contentsWidth() - x;
            const align = entry.align || 'left';
            this.drawText(entry.text, x, y, maxWidth, align);
        }, this);
    };

    Window_MessageHiddenOverlay.prototype.update = function() {
        Window_Base.prototype.update.call(this);
    };

    Window_MessageHiddenOverlay.prototype.windowWidth = function() {
        return Graphics.boxWidth;
    };

    Window_MessageHiddenOverlay.prototype.windowHeight = function() {
        return Graphics.boxHeight;
    };

    Window_MessageHiddenOverlay.prototype.windowRect = function() {
        const wx = 0;
        const wy = 0;
        const ww = this.windowWidth();
        const wh = this.windowHeight();
        return new Rectangle(wx, wy, ww, wh);
    };

    //=============================================================================
    // Scene_Map / Scene_Battle
    //  非表示時テキスト用ウィンドウを生成して制御
    //=============================================================================
    const _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
    Scene_Map.prototype.createAllWindows = function() {
        _Scene_Map_createAllWindows.call(this);
        this.createMessageHiddenOverlay();
    };

    Scene_Map.prototype.createMessageHiddenOverlay = function() {
        const rect = new Rectangle(0, 0, Graphics.boxWidth, Graphics.boxHeight);
        this._messageHiddenOverlayWindow = new Window_MessageHiddenOverlay(rect);
        this._messageHiddenOverlayWindow.hide();
        this.addWindow(this._messageHiddenOverlayWindow);
    };

    Scene_Map.prototype.showMessageHiddenOverlay = function() {
        if (!this._messageHiddenOverlayWindow) return;
        if ($gameSwitches.value(param.disableLinkSwitchId)) return;
        this._messageHiddenOverlayWindow.refresh();
        this._messageHiddenOverlayWindow.show();
    };

    Scene_Map.prototype.hideMessageHiddenOverlay = function() {
        if (!this._messageHiddenOverlayWindow) return;
        this._messageHiddenOverlayWindow.hide();
    };

    const _Scene_Battle_createAllWindows = Scene_Battle.prototype.createAllWindows;
    Scene_Battle.prototype.createAllWindows = function() {
        _Scene_Battle_createAllWindows.call(this);
        this.createMessageHiddenOverlay();
    };

    Scene_Battle.prototype.createMessageHiddenOverlay = function() {
        const rect = new Rectangle(0, 0, Graphics.boxWidth, Graphics.boxHeight);
        this._messageHiddenOverlayWindow = new Window_MessageHiddenOverlay(rect);
        this._messageHiddenOverlayWindow.hide();
        this.addWindow(this._messageHiddenOverlayWindow);
    };

    Scene_Battle.prototype.showMessageHiddenOverlay = function() {
        if (!this._messageHiddenOverlayWindow) return;
        if ($gameSwitches.value(param.disableLinkSwitchId)) return;
        this._messageHiddenOverlayWindow.refresh();
        this._messageHiddenOverlayWindow.show();
    };

    Scene_Battle.prototype.hideMessageHiddenOverlay = function() {
        if (!this._messageHiddenOverlayWindow) return;
        this._messageHiddenOverlayWindow.hide();
    };

    //=============================================================================
    // Window_Message
    //  指定されたボタン押下時にウィンドウとサブウィンドウを非表示にします。
    //=============================================================================
    const _Window_Message_updateWait = Window_Message.prototype.updateWait;
    Window_Message.prototype.updateWait = function() {
        if (!this.isClosed() && this.isTriggeredHidden() && this.isEnableInChoice()) {
            if (!this.isHidden()) {
                this.hideAllWindow();
            } else {
                this.showAllWindow();
            }
        } else if (this.isHidden() && this.isTriggered() && param.restoreByDecision) {
            this.showAllWindow();
            if (param.syncSwitch) {
                $gameSwitches.setValue(param.triggerSwitch, false);
            }
            Input.update();
            TouchInput.update();
        }
        const wait = _Window_Message_updateWait.apply(this, arguments);
        if (this.isHidden() && this.visible) {
            this.hideAllWindow();
        }
        return wait;
    };

    Window_Message.prototype.isEnableInChoice = function() {
        return !(param.disableInChoice && $gameMessage.isChoice());
    };

    Window_Message.prototype.hideAllWindow = function() {
        this.hide();
        this.subWindows().forEach(function(subWindow) {
            this.hideSubWindow(subWindow);
        }.bind(this));
        this._originalPictureOpacities = {};
        this.linkPictures(0, param.linkPictureNumbers);
        this.linkPictures(255, param.linkShowPictureNumbers);
        this._hideByMessageWindowHidden = true;
        this.showHiddenPictures();
        this.showHiddenTexts();
    };

    Window_Message.prototype.subWindows = function() {
        return [this._goldWindow, this._nameBoxWindow,
            this._choiceListWindow, this._numberInputWindow,
            this._eventItemWindow
        ];
    };

    Window_Message.prototype.showAllWindow = function() {
        this.show();
        this.subWindows().forEach(function(subWindow) {
            this.showSubWindow(subWindow);
        }.bind(this));
        this.linkPictures(null, param.linkShowPictureNumbers);
        this.linkPictures(null, param.linkPictureNumbers);
        this._hideByMessageWindowHidden = false;
        this.eraseHiddenPictures();
        this.eraseHiddenTexts();
    };

    Window_Message.prototype.isHidden = function() {
        return this._hideByMessageWindowHidden;
    };

    Window_Message.prototype.linkPictures = function(opacity, pictureNumbers) {
        if (!pictureNumbers || $gameSwitches.value(param.disableLinkSwitchId)) {
            return;
        }
        pictureNumbers.forEach(function(pictureId) {
            this.linkPicture(opacity, pictureId);
        }, this);
    };

    Window_Message.prototype.linkPicture = function(opacity, pictureId) {
        const picture = $gameScreen.picture(pictureId);
        if (!picture) {
            return;
        }
        if (opacity === null) {
            if (!this._originalPictureOpacities.hasOwnProperty(pictureId)) {
                return;
            }
            opacity = this._originalPictureOpacities[pictureId];
        } else {
            this._originalPictureOpacities[pictureId] = picture.opacity();
        }
        picture.linkWithMessageWindow(opacity);
    };

    Window_Message.prototype.hideSubWindow = function(subWindow) {
        subWindow.prevVisible = subWindow.visible;
        subWindow.hide();
    };

    Window_Message.prototype.showSubWindow = function(subWindow) {
        if (subWindow.prevVisible) subWindow.show();
        subWindow.prevVisible = undefined;
    };

    Window_Message.prototype.hasNameWindow = function() {
        return this._nameWindow && typeof Window_NameBox !== 'undefined';
    };

    Window_Message.prototype.nameWindowIsSubWindow = function() {
        return this.subWindows().filter(function(subWindow) {
            return subWindow === this._nameWindow;
        }, this).length > 0;
    };

    Window_Message.prototype.disableWindowHidden = function () {
        return (param.disableSwitchId > 0 && $gameSwitches.value(param.disableSwitchId)) ||
            (param.disableInBattle && $gameParty.inBattle());
    };

    Window_Message.prototype.isTriggeredHidden = function() {
        if (this.disableWindowHidden()) {
            return false;
        }
        if (param.triggerSwitch > 0 && this.isTriggeredHiddenSwitch()) {
            return true;
        }
        const result = param.triggerButton.some(function(button) {
            switch (button) {
                case '':
                case '右クリック':
                case 'light_click':
                    return TouchInput.isCancelled();
                case 'ok':
                    return false;
                default:
                    return Input.isTriggered(button);
            }
        });
        if (param.syncSwitch && param.triggerSwitch > 0 && result) {
            $gameSwitches.setValue(param.triggerSwitch, !this.isHidden());
        }
        return result;
    };

    Window_Message.prototype.isTriggeredHiddenSwitch = function() {
        const triggerSwitch = $gameSwitches.value(param.triggerSwitch);
        if (triggerSwitch && !this.isHidden()) {
            return true;
        }
        if (!triggerSwitch && this.isHidden()) {
            return true;
        }
        return false;
    };

    const _Window_Message_updateInput = Window_Message.prototype.updateInput;
    Window_Message.prototype.updateInput = function() {
        if (this.isHidden()) return true;
        return _Window_Message_updateInput.apply(this, arguments);
    };

    // 非表示時専用ピクチャ表示
    Window_Message.prototype.showHiddenPictures = function() {
        if (!hiddenPictures.length) return;
        if ($gameSwitches.value(param.disableLinkSwitchId)) return;
        hiddenPictures.forEach(function(data) {
            const pictureId = data.id;
            const name      = data.name;
            const x         = data.x;
            const y         = data.y;
            const origin    = 0;
            const scaleX    = 100;
            const scaleY    = 100;
            const opacity   = 255;
            const blendMode = 0;
            $gameScreen.showPicture(pictureId, name, origin, x, y, scaleX, scaleY, opacity, blendMode);
        });
    };

    Window_Message.prototype.eraseHiddenPictures = function() {
        if (!hiddenPictures.length) return;
        if ($gameSwitches.value(param.disableLinkSwitchId)) return;
        hiddenPictures.forEach(function(data) {
            const pictureId = data.id;
            $gameScreen.erasePicture(pictureId);
        });
    };

    // 非表示時テキスト表示
    Window_Message.prototype.showHiddenTexts = function() {
        const scene = SceneManager._scene;
        if (!scene) return;
        if (typeof scene.showMessageHiddenOverlay === 'function') {
            scene.showMessageHiddenOverlay();
        }
    };

    Window_Message.prototype.eraseHiddenTexts = function() {
        const scene = SceneManager._scene;
        if (!scene) return;
        if (typeof scene.hideMessageHiddenOverlay === 'function') {
            scene.hideMessageHiddenOverlay();
        }
    };

    //=============================================================================
    // Window_ChoiceList、Window_NumberInput、Window_EventItem
    //  非表示の間は更新を停止します。
    //=============================================================================
    const _Window_ChoiceList_update = Window_ChoiceList.prototype.update;
    Window_ChoiceList.prototype.update = function() {
        if (!this.visible) return;
        _Window_ChoiceList_update.apply(this, arguments);
    };

    const _Window_NumberInput_update = Window_NumberInput.prototype.update;
    Window_NumberInput.prototype.update = function() {
        if (!this.visible) return;
        _Window_NumberInput_update.apply(this, arguments);
    };

    const _Window_EventItem_update = Window_EventItem.prototype.update;
    Window_EventItem.prototype.update = function() {
        if (!this.visible) return;
        _Window_EventItem_update.apply(this, arguments);
    };
})();
