//=============================================================================
// MonakaEasing.js
//=============================================================================
// Copyright (c) 2024- 牛乳もなか
// This software is released under the MIT License.
// http://opensource.org/licenses/mit-license.php
//
// Twitter
// https://twitter.com/milkmonaka_1
//----------------------------------------------------------------------------
// Version
// 1.0.2 2024/03/21 ピクチャの透明度が0などからアニメーション開始した場合、透明度が正常に更新されない不具合を修正。
// 1.0.1 2024/03/20 一部の動作が行われないウインドウがあった不具合を修正。
// 1.0.0 2024/03/19 テスト公開
//----------------------------------------------------------------------------
/*:ja
* @target MZ
* @url https://twitter.com/milkmonaka_1
* @plugindesc ピクチャやウインドウに簡単に多彩なイージングを設定します。
* @author milkmonaka
* @help 無償有償問わず自由な使用と改変が可能、使用報告クレジット不要。
* 一部自分用のよくわからない動きをする関数がありますがスルーしてください。
*
* ※簡単な使い方：ピクチャ編
* ピクチャの移動の前にプラグインコマンドでイージング関数を指定するだけです、
* この機能は明示的にResetEasingFunctionを実行しない限り以降の全てのピクチャの移動に適応されます。
* 終了時にはイージングを解除してください。
* 又、解除することでエディタ内の通常のイージングを使用できます。
* このプラグインはSetEasingFunctionを実行後のみ機能が適応され、使用しない時に既存の機能に影響を及ぼしません。
*
* ※簡単な使い方：ウインドウ編
* イージングを実装したいウインドウに方向やイージング関数を設定するだけです。
* 使用しないウインドウはスルーされますので、既存の機能に影響を及ぼしません。
* ただし既に他のイージングプラグインを入れている環境下での使用は恐らく無理です。
* Window_Baseさえ継承していれば独自に作成したウインドウも
* Window_XXXXXと入力すれば使用できる可能性があります(XXXXXはウインドウクラス名)
* 保証はできませんが。
*
* コード内でスプライトに対してこのイージングを使用する場合はjavascript
* 及びコアスクリプトの知見が必要です。
* そちらに対する個別の説明は控えさえて頂きます。
*
* @command SetEasingFunction
* @text イージング関数の設定
* @desc ピクチャの移動に使用するイージング関数を設定します。
*
* @arg easingFunction
* @text イージング関数
* @desc 使用するイージング関数を選択します。
* @type select
* @option linear
* @option easeInSine
* @option easeOutSine
* @option easeInOutSine
* @option easeInQuad
* @option easeOutQuad
* @option easeInOutQuad
* @option easeInCubic
* @option easeOutCubic
* @option easeInOutCubic
* @option easeInQuart
* @option easeOutQuart
* @option easeInOutQuart
* @option easeInQuint
* @option easeOutQuint
* @option easeInOutQuint
* @option easeInBack
* @option easeOutBack
* @option easeOutBack2
* @option easeInOutBack
* @option easeOutBackWithInertia
* @option easeInElastic
* @option easeOutElastic
* @option easeInOutElastic
* @option bounceIn
* @option bounceOut
* @option bounceInOut
* @default linear
*
* @command ResetEasingFunction
* @text イージングを解除
* @desc ピクチャの移動に使用するイージング関数を解除します。
*
* @param Window Settings
* @text ウインドウ設定
* @desc 各ウインドウの動きの設定
* @type struct<WindowAnimation>[]
* @default []
*
*/

/*~struct~WindowAnimation:
* @param WindowClass
* @desc スライドを実行するウィンドウ。一覧にない場合は直接入力。
* @type select
* @default
* @option ヘルプウィンドウ
* @value Window_Help
* @option 所持金ウィンドウ
* @value Window_Gold
* @option メニューコマンドウィンドウ
* @value Window_MenuCommand
* @option メニューステータスウィンドウ
* @value Window_MenuStatus
* @option アイテムカテゴリウィンドウ
* @value Window_ItemCategory
* @option アイテムリストウィンドウ
* @value Window_ItemList
* @option アクター選択ウィンドウ
* @value Window_MenuActor
* @option スキルタイプウィンドウ
* @value Window_SkillType
* @option スキルステータスウィンドウ
* @value Window_SkillStatus
* @option スキルリストウィンドウ
* @value Window_SkillList
* @option 装備ステータスウィンドウ
* @value Window_EquipStatus
* @option 装備コマンドウィンドウ
* @value Window_EquipCommand
* @option 装備スロットウィンドウ
* @value Window_EquipSlot
* @option 装備リストウィンドウ
* @value Window_EquipItem
* @option ステータスウィンドウ
* @value Window_Status
* @option オプションウィンドウ
* @value Window_Options
* @option セーブファイルウィンドウ
* @value Window_SavefileList
* @option ショップコマンドウィンドウ
* @value Window_ShopCommand
* @option 購入アイテムウィンドウ
* @value Window_ShopBuy
* @option 売却アイテムウィンドウ
* @value Window_ShopSell
* @option 購入数入力ウィンドウ
* @value Window_ShopNumber
* @option ショップステータスウィンドウ
* @value Window_ShopStatus
* @option 名前編集ウィンドウ
* @value Window_NameEdit
* @option 名前入力ウィンドウ
* @value Window_NameInput
* @option 選択肢ウィンドウ
* @value Window_ChoiceList
* @option 数値入力ウィンドウ
* @value Window_NumberInput
* @option アイテム選択ウィンドウ
* @value Window_EventItem
* @option メッセージウィンドウ
* @value Window_Message
* @option ネームボックスウィンドウ
* @value Window_NameBox
* @option マップ名ウィンドウ
* @value Window_MapName
* @option タイトルウィンドウ
* @value Window_TitleCommand
* @option ゲーム終了ウィンドウ
* @value Window_GameEnd
*
* @param Move Speed
* @text 移動速度
* @desc ウインドウが移動する速度(ミリ秒:デフォルト値の場合0.5秒)
* @type number
* @default 500
*
* @param Direction
* @text 移動方向
* @desc ウインドウが開く時の移動方向
* @type select
* @option 上
* @value up
* @option 下
* @value down
* @option 左
* @value left
* @option 右
* @value right
* @default up
*
* @param WindowEasing
* @text イージング関数
* @desc 使用するイージング関数を選択します。
* @type select
* @option linear
* @option easeInSine
* @option easeOutSine
* @option easeInOutSine
* @option easeInQuad
* @option easeOutQuad
* @option easeInOutQuad
* @option easeInCubic
* @option easeOutCubic
* @option easeInOutCubic
* @option easeInQuart
* @option easeOutQuart
* @option easeInOutQuart
* @option easeInQuint
* @option easeOutQuint
* @option easeInOutQuint
* @option easeInBack
* @option easeOutBack
* @option easeOutBack2
* @option easeInOutBack
* @option easeOutBackWithInertia
* @option easeInElastic
* @option easeOutElastic
* @option easeInOutElastic
* @option bounceIn
* @option bounceOut
* @option bounceInOut
* @default linear
*/

(() => {
	'use strict';

	// プラグインコマンドの定義
	const pluginName = decodeURIComponent(document.currentScript.src).match(/^.*\/js\/plugins\/(.+)\.js$/)[1];
	const parameters = PluginManager.parameters(pluginName);
	const windowSettings = JSON.parse(parameters['Window Settings'] || '[]').map(e => JSON.parse(e));

	PluginManager.registerCommand(pluginName, "SetEasingFunction", function (args) {
		$gameTemp.selectedEasingFunction = args.easingFunction;
	});

	PluginManager.registerCommand(pluginName, "ResetEasingFunction", args => {
	   $gameTemp.selectedEasingFunction = null;
	});

	// イージング関数の設定
	window.monakaEasing = {
		linear: function(t) {
			return t;
		},
		easeInSine: function(t) {
			return 1 - Math.cos((t * Math.PI) / 2);
		},
		easeOutSine: function(t) {
			return Math.sin((t * Math.PI) / 2);
		},
		easeInOutSine: function(t) {
			return -(Math.cos(Math.PI * t) - 1) / 2;
		},
		easeInQuad: function(t) {
			return t * t;
		},
		easeOutQuad: function(t) {
			return t * (2 - t);
		},
		easeInOutQuad: function(t) {
			return t < 0.5 ? 2 * t * t : -1 + (4 - 2 * t) * t;
		},
		easeInCubic: function(t) {
			return t * t * t;
		},
		easeOutCubic: function(t) {
			return (--t) * t * t + 1;
		},
		easeInOutCubic: function(t) {
			return t < 0.5 ? 4 * t * t * t : 1 - Math.pow(-2 * t + 2, 3) / 2;
		},
		easeInQuart: function(t) {
			return t * t * t * t;
		},
		easeOutQuart: function(t) {
			return 1 - (--t) * t * t * t;
		},
		easeInOutQuart: function(t) {
			if (t < 0.5) {
				return 8 * t * t * t * t;
			} else {
				return 1 - 8 * (--t) * t * t * t;
			}
		},
		easeInQuint: function(t) {
			return t * t * t * t * t;
		},
		easeOutQuint: function(t) {
			return 1 + (--t) * t * t * t * t;
		},
		easeInOutQuint: function(t) {
			return t < 0.5 ? 16 * t * t * t * t * t : 1 - Math.pow(-2 * t + 2, 5) / 2;
		},
		easeInBack: function(t) {
			const c1 = 1.70158;
			const c3 = c1 + 1;
			return c3 * t * t * t - c1 * t * t;
		},
		easeOutBack: function(t) {
			const c1 = 1.90158;
			const c3 = c1 + 1;
			return 1 + c3 * Math.pow(t - 1, 3) + c1 * Math.pow(t - 1, 2);
		},
		easeOutBack2: function(t) {
			const c1 = 1.00158;
			const c3 = c1 + 1;
			return 1 + c3 * Math.pow(t - 1, 3) + c1 * Math.pow(t - 1, 2);
		},
		easeInOutBack: function(t) {
			const c1 = 1.70158;
			const c2 = c1 * 1.525;
			return t < 0.5
			? 0.5 * ((2 * t) ** 2 * ((c2 + 1) * 2 * t - c2))
			: 0.5 * ((2 * t - 2) ** 2 * ((c2 + 1) * (t * 2 - 2) + c2) + 2);
		},
		easeOutBackWithInertia: function(t) {
			const c1 = 1.90158;
			const c3 = c1 + 1;
			const inertiaFactor = 3; // 慣性の強さを調整する係数

			// オーバーシュートの頂点を計算
			const overshoot = 1 + c3 * Math.pow(1 - 1, 3) + c1 * Math.pow(1 - 1, 2);
			const overshootThreshold = 0.9; // オーバーシュートを止めるしきい値（0～1の範囲で調整）

			if (t <= overshootThreshold) {
				// オーバーシュートの範囲内では通常のイージング関数を適用
				return 1 + c3 * Math.pow(t - 1, 3) + c1 * Math.pow(t - 1, 2);
			} else {
				// オーバーシュートの範囲外では加速度を調整して慣性を付ける
				const adjustedT = (t - overshootThreshold) / (1 - overshootThreshold);
				const inertia = Math.pow(adjustedT, inertiaFactor);
				return overshoot + inertia * (1 - overshoot);
			}
		},
		easeInElastic: function(t) {
			const c4 = (2 * Math.PI) / 3;

			return t === 0 ? 0 : t === 1 ? 1 : -Math.pow(2, 10 * t - 10) * Math.sin((t * 10 - 10.75) * c4);
		},
		easeOutElastic: function(t) {
			const c4 = (2 * Math.PI) / 3;
			return t === 0 ? 0 : t === 1 ? 1 : Math.pow(2, -10 * t) * Math.sin((t * 10 - 0.75) * c4) + 1;
		},
		easeInOutElastic: function(t) {
			const c5 = (2 * Math.PI) / 4.5;
			return t === 0 ? 0 : t === 1 ? 1 : t < 0.5 ? -(Math.pow(2, 20 * t - 10) * Math.sin((20 * t - 11.125) * c5)) / 2 : (Math.pow(2, -20 * t + 10) * Math.sin((20 * t - 11.125) * c5)) / 2 + 1;
		},
		bounceIn: function(t) {
			return 1 - window.monakaEasing.bounceOut(1 - t);
		},
		bounceOut: function(t) {
			const n1 = 7.5625;
			const d1 = 2.75;

			if (t < 1 / d1) {
				return n1 * t * t;
			} else if (t < 2 / d1) {
				return n1 * (t -= 1.5 / d1) * t + 0.75;
			} else if (t < 2.5 / d1) {
				return n1 * (t -= 2.25 / d1) * t + 0.9375;
			} else {
				return n1 * (t -= 2.625 / d1) * t + 0.984375;
			}
		},
		bounceInOut: function(t) {
			if (t < 0.5) {
				return (1 - window.monakaEasing.bounceOut(1 - 2 * t)) / 2;
			} else {
				return (1 + window.monakaEasing.bounceOut(2 * t - 1)) / 2;
			}
		}
	};

	// prettier-ignore
	const _Game_Picture_move = Game_Picture.prototype.move;
	Game_Picture.prototype.move = function(
		origin, x, y, scaleX, scaleY, opacity, blendMode, duration, easingType
	) {
		if (!$gameTemp.selectedEasingFunction) {
			_Game_Picture_move.apply(this, arguments);
		} else {
			if (this._duration === 0) {
				this._moveDuration = duration;
				this._currentFrame = 0;
				const selectedEasing = $gameTemp.selectedEasingFunction;
				this._easingFunction = window.monakaEasing[selectedEasing] || window.monakaEasing.linear;
				this._startX = this._x;
				this._startY = this._y;
				this._startScaleX = this._scaleX;
				this._startScaleY = this._scaleY;
				this._startOpacity = this._opacity;
				this._targetX = x;
				this._targetY = y;
				this._targetScaleX = scaleX;
				this._targetScaleY = scaleY;
				this._targetOpacity = opacity;
				this._distanceX = this._targetX - this._startX;
				this._distanceY = this._targetY - this._startY;
				this._distanceScaleX = this._targetScaleX - this._startScaleX;
				this._distanceScaleY = this._targetScaleY - this._startScaleY;
				this._distanceOpacity = this._targetOpacity - this._startOpacity;
			}
		}
	};

	const _Game_Picture_updateMove = Game_Picture.prototype.updateMove;
	Game_Picture.prototype.updateMove = function() {
		if (!$gameTemp.selectedEasingFunction) {
			_Game_Picture_updateMove.call(this);
		} else {
			if (this._currentFrame > this._moveDuration) {
				return;
			}
			this._currentFrame++;
			if (this._currentFrame <= this._moveDuration) {
				const easing = this._easingFunction ? this._easingFunction(this._currentFrame / this._moveDuration) : 1;
				this._x = this._startX + this._distanceX * easing;
				this._y = this._startY + this._distanceY * easing;
				this._scaleX = this._startScaleX + this._distanceScaleX * easing;
				this._scaleY = this._startScaleY + this._distanceScaleY * easing;
				this._opacity = this._startOpacity + this._distanceOpacity * easing;
			}
		}
	};

	const _Window_Base_initialize = Window_Base.prototype.initialize;
	Window_Base.prototype.initialize = function(rect) {
	  _Window_Base_initialize.call(this, rect);
	  // このウインドウの設定を見つける
	  const setting = windowSettings.find(e => e['WindowClass'] === this.constructor.name);
	  if (setting) {
	    const moveSpeed = Number(setting['Move Speed'] || 10);
	    const direction = setting['Direction'] || 'up';
	    // 初期位置と目標位置の設定、方向に応じて
			this.easingFunction = window.monakaEasing[setting['WindowEasing']];
	    this.setupInitialPosition(moveSpeed, direction, rect);
	    this.moveDirection = direction;
	    this.openingAnimation = true; // アニメーション開始フラグ
	  }
	};

	const _Window_ChoiceList_start = Window_ChoiceList.prototype.start;
	Window_ChoiceList.prototype.start = function() {
	    _Window_ChoiceList_start.call(this);
	    const setting = windowSettings.find(e => e['WindowClass'] === this.constructor.name);
	    if (setting) {
	        const moveSpeed = Number(setting['Move Speed'] || 10);
	        const direction = setting['Direction'] || 'up';
					this.easingFunction = window.monakaEasing[setting['WindowEasing']];
					const rect = new Rectangle(this.x, this.y, this.width, this.height);
	        this.setupInitialPosition(moveSpeed, direction, rect);
	        this.moveDirection = direction;
	        this.openingAnimation = true;
	    }
	};

	const _Window_NumberInput_start = Window_NumberInput.prototype.start;
	Window_NumberInput.prototype.start = function() {
			_Window_NumberInput_start.call(this);
			const setting = windowSettings.find(e => e['WindowClass'] === this.constructor.name);
			if (setting) {
					const moveSpeed = Number(setting['Move Speed'] || 10);
					const direction = setting['Direction'] || 'up';
					this.easingFunction = window.monakaEasing[setting['WindowEasing']];
					const rect = new Rectangle(this.x, this.y, this.width, this.height);
					this.setupInitialPosition(moveSpeed, direction, rect);
					this.moveDirection = direction;
					this.openingAnimation = true;
			}
	};

	const _Window_EventItem_start = Window_EventItem.prototype.start;
	Window_EventItem.prototype.start = function() {
			_Window_EventItem_start.call(this);
			const setting = windowSettings.find(e => e['WindowClass'] === this.constructor.name);
			if (setting) {
					const moveSpeed = Number(setting['Move Speed'] || 10);
					const direction = setting['Direction'] || 'up';
					this.easingFunction = window.monakaEasing[setting['WindowEasing']];
					const rect = new Rectangle(this.x, this.y, this.width, this.height);
					this.setupInitialPosition(moveSpeed, direction, rect);
					this.moveDirection = direction;
					this.openingAnimation = true;
			}
	};

	const _Window_Message_startMessage = Window_Message.prototype.startMessage;
	Window_Message.prototype.startMessage = function() {
			_Window_Message_startMessage.call(this);
			if (!this.isOpen()) {
			const setting = windowSettings.find(e => e['WindowClass'] === this.constructor.name);
			if (setting) {
					const moveSpeed = Number(setting['Move Speed'] || 10);
					const direction = setting['Direction'] || 'up';
					this.easingFunction = window.monakaEasing[setting['WindowEasing']];
					const rect = new Rectangle(this.x, this.y, this.width, this.height);
					this.setupInitialPosition(moveSpeed, direction, rect);
					this.moveDirection = direction;
					this.openingAnimation = true;
			}
		}
	};

	const _Window_NameBox_start = Window_NameBox.prototype.start;
	Window_NameBox.prototype.start = function() {
			_Window_NameBox_start.call(this);
			if (!this.isOpen()) {
			const setting = windowSettings.find(e => e['WindowClass'] === this.constructor.name);
			if (setting) {
					const moveSpeed = Number(setting['Move Speed'] || 10);
					const direction = setting['Direction'] || 'up';
					this.easingFunction = window.monakaEasing[setting['WindowEasing']];
					const rect = new Rectangle(this.x, this.y, this.width, this.height);
					this.setupInitialPosition(moveSpeed, direction, rect);
					this.moveDirection = direction;
					this.openingAnimation = true;
			}
		}
	};

	Window_Base.prototype.setupInitialPosition = function(moveSpeed, direction, rect) {
	  this.animationStartTime = Date.now();
	  this.animationEndTime = this.animationStartTime + moveSpeed;

	  switch (direction) {
	    case 'up':
	      this.startY = Graphics.height;
	      this.targetY = rect.y;
	      break;
	    case 'down':
	      this.startY = -this.height;
	      this.targetY = rect.y;
	      break;
	    case 'left':
	      this.startX = Graphics.width;
	      this.targetX = rect.x;
	      break;
	    case 'right':
	      this.startX = -this.width;
	      this.targetX = rect.x;
	      break;
	  }
	};

  const _Window_Base_update = Window_Base.prototype.update;
	Window_Base.prototype.update = function() {
	  _Window_Base_update.call(this);
	  if (this.openingAnimation) {
	    let currentTime = Date.now();
	    let progress = (currentTime - this.animationStartTime) / (this.animationEndTime - this.animationStartTime);
	    progress = this.easingFunction(Math.min(progress, 1)); // イージング関数を適用

	    switch (this.moveDirection) {
	      case 'up':
	      case 'down':
	        this.y = this.startY + (this.targetY - this.startY) * progress;
	        break;
	      case 'left':
	      case 'right':
	        this.x = this.startX + (this.targetX - this.startX) * progress;
	        break;
	    }

	    // アニメーションが終了したかどうかを確認
	    if (currentTime >= this.animationEndTime) {
	      this.openingAnimation = false; // アニメーション終了
	      // 最終位置を確実に設定
	      switch (this.moveDirection) {
	        case 'up':
	        case 'down':
	          this.y = this.targetY;
	          break;
	        case 'left':
	        case 'right':
	          this.x = this.targetX;
	          break;
	      }
	    }
	  }
	};

})();
