/*:
 * @param FocusTimingVariable
 * @text ピント成功時の変数
 * @type variable
 * @desc ピント調整が成功した場合に加算する変数
 * @default 1
 *
 * @param CameraFrameImage
 * @text カメラフレーム画像
 * @type file
 * @dir img/system/
 * @desc カメラのフレーム画像
 * @default CameraFrame
 *
 * @param FocusBarImage
 * @text ピント調整バー画像
 * @type file
 * @dir img/system/
 * @desc ピント調整バーの画像
 * @default FocusBar
 *
 * @param FocusBarBackgroundImage
 * @text ピント調整バー背景画像
 * @type file
 * @dir img/system/
 * @desc ピント調整バーの背景画像
 * @default FocusBarBackground
 *
 * @param ShutterSound
 * @text シャッター音
 * @type file
 * @dir audio/se/
 * @desc 撮影時のシャッター音ファイル
 * @default Camera
 *
 * @param ShutterWaitTime
 * @text シャッター後のウェイト時間
 * @type number
 * @min 0
 * @desc シャッター後にシーンを終わらせるまでのフレーム数
 * @default 60
 *
 * @command StartPhotoMiniGame
 * @text 写真撮影ミニゲーム開始
 * @desc ピント調整ミニゲームを開始する。テキスト設定はプラグインコマンドから指定します。
 *
 * @arg TargetPicture
 * @text 撮影対象のピクチャID
 * @type number
 * @desc 撮影するピクチャのID
 * @default 1
 *
 * @arg TargetPictureImage
 * @text 撮影対象のピクチャ画像
 * @type file
 * @dir img/pictures/
 * @desc 撮影するピクチャの画像ファイル
 *
 * @arg FocusSpeed
 * @text ピント移動速度
 * @type number
 * @desc ピント調整バーの移動速度
 * @default 2
 *
 * @arg DisplayText
 * @text 表示テキスト
 * @desc 画面に表示するテキスト（制御文字が利用可能）
 * @default こんにちは、\V[1]さん！
 *
 * @arg TextX
 * @text テキストのX座標
 * @type number
 * @desc テキスト表示のX座標
 * @default 100
 *
 * @arg TextY
 * @text テキストのY座標
 * @type number
 * @desc テキスト表示のY座標
 * @default 50
 *
 * @arg FontSize
 * @text フォントサイズ
 * @type number
 * @desc テキストのフォントサイズ
 * @default 28
 *
 * @arg TextWindowSkin
 * @text テキストウィンドウ画像
 * @type file
 * @dir img/system/
 * @desc テキストウィンドウの背景に使用する画像。未設定の場合はデフォルトのウィンドウスキンが使用されます。
 * @default
 */

(() => {
    const pluginName = "PhotoMiniGame";
    
    PluginManager.registerCommand(pluginName, "StartPhotoMiniGame", args => {
        const pictureId      = Number(args.TargetPicture);
        const pictureImage   = String(args.TargetPictureImage);
        const focusSpeed     = Number(args.FocusSpeed) || 2;
        const textDisplay    = String(args.DisplayText);
        const textX          = Number(args.TextX);
        const textY          = Number(args.TextY);
        const fontSize       = Number(args.FontSize);
        const textWindowSkin = String(args.TextWindowSkin);
        
        SceneManager._photoMiniGameParams = { 
            pictureId, 
            pictureImage, 
            focusSpeed, 
            textDisplay, 
            textX, 
            textY, 
            fontSize, 
            textWindowSkin
        };
        SceneManager.push(Scene_PhotoMiniGame);
    });

    class Scene_PhotoMiniGame extends Scene_Base {
        initialize() {
            super.initialize();
            const cmdParams = SceneManager._photoMiniGameParams || { pictureId: 1, pictureImage: "", focusSpeed: 2 };
            this._targetPictureId    = cmdParams.pictureId;
            this._targetPictureImage = cmdParams.pictureImage;
            this._focusSpeed         = cmdParams.focusSpeed;
            this._textDisplay        = cmdParams.textDisplay;
            this._textX              = cmdParams.textX;
            this._textY              = cmdParams.textY;
            this._fontSize           = cmdParams.fontSize;
            this._textWindowSkin     = cmdParams.textWindowSkin;
            
            this._focusVariable = Number(PluginManager.parameters(pluginName)["FocusTimingVariable"]) || 1;
            this._shutterSound = PluginManager.parameters(pluginName)["ShutterSound"] || "Camera";
            this._shutterWaitTime = Number(PluginManager.parameters(pluginName)["ShutterWaitTime"]) || 60;
            this._cameraFrameImage = PluginManager.parameters(pluginName)["CameraFrameImage"] || "CameraFrame";
            this._focusBarImage = PluginManager.parameters(pluginName)["FocusBarImage"] || "FocusBar";
            this._focusBarBackgroundImage = PluginManager.parameters(pluginName)["FocusBarBackgroundImage"] || "FocusBarBackground";
            this._focusValue = 100;
            this._focusLocked = false;
            this._shutterWaitCounter = 0;
        }

        create() {
            super.create();
            this.createTargetPicture();
            this.createCameraFrame();
            this.createFocusBar();
            this.createTextWindow(); 
        }

        start() {
            super.start();
            this._focusActive = true;
        }

        createTargetPicture() {
            if (this._targetPictureImage) {
                this._targetPicture = new Sprite(ImageManager.loadPicture(this._targetPictureImage));
                this._targetPicture.anchor.set(0.5);
                this._targetPicture.x = Graphics.width / 2;
                this._targetPicture.y = Graphics.height / 2;
                this.addChild(this._targetPicture);
            }
        }

        createCameraFrame() {
            this._cameraFrame = new Sprite(ImageManager.loadSystem(this._cameraFrameImage));
            this.addChild(this._cameraFrame);
        }

        createFocusBar() {
            this._focusBarBackground = new Sprite(ImageManager.loadSystem(this._focusBarBackgroundImage));
            this._focusBarBackground.anchor.set(0.5);
            this._focusBarBackground.x = Graphics.width / 2;
            this._focusBarBackground.y = Graphics.height - 100;
            this.addChild(this._focusBarBackground);
            
            this._focusBar = new Sprite(ImageManager.loadSystem(this._focusBarImage));
            this._focusBar.anchor.set(0.5);
            this._focusBar.x = Graphics.width / 2;
            this._focusBar.y = Graphics.height - 100;
            this.addChild(this._focusBar);
        }

        createTextWindow() {
            const displayText = this._textDisplay;
            const textX = this._textX;
            const textY = this._textY;
            const fontSize = this._fontSize;
            const windowWidth = Graphics.width - textX * 2;
            const windowHeight = 100;
            const rect = new Rectangle(textX, textY, windowWidth, windowHeight);
            this._textWindow = new Window_Base(rect);
            if (this._textWindowSkin) {
                this._textWindow.windowskin = ImageManager.loadSystem(this._textWindowSkin);
            }
            this._textWindow.contents.fontSize = fontSize;
            this._textWindow.drawTextEx(displayText, 0, 0);
            this.addChild(this._textWindow);
        }

        update() {
            super.update();
            if (this._focusActive && !this._focusLocked) {
                this.updateFocusBar();
                if (Input.isTriggered("ok")) {
                    this.lockFocus();
                }
            } else if (this._shutterWaitCounter > 0) {
                this._shutterWaitCounter--;
                if (this._shutterWaitCounter === 0) {
                    this.endMiniGame();
                }
            }
        }

        updateFocusBar() {
            const barWidth = this._focusBarBackground.width;
            this._focusValue += this._focusSpeed;
            if (this._focusValue >= barWidth || this._focusValue <= 0) {
                this._focusSpeed *= -1;
            }
            this._focusBar.x = this._focusBarBackground.x - barWidth / 2 + this._focusValue;
            this.applyBlurEffect();
        }

        applyBlurEffect() {
            const focusCenter = this._focusBarBackground.width / 2;
            const distanceFromCenter = Math.abs(this._focusValue - focusCenter);
            const maxDistance = this._focusBarBackground.width / 2;
            const percentage = (distanceFromCenter / maxDistance) * 100;

            let blurStrength;
            if (percentage <= 20) {
                blurStrength = 0;
            } else if (percentage <= 40) {
                blurStrength = 3;
            } else if (percentage <= 70) {
                blurStrength = 7;
            } else {
                blurStrength = 15;
            }

            this._targetPicture.filters = [new PIXI.filters.BlurFilter(blurStrength)];
        }

        lockFocus() {
            this._focusLocked = true;
            this._cameraFrame.visible = false; 
            this._focusBar.visible = false;
            this._focusBarBackground.visible = false;
            if (this._textWindow) {
                this._textWindow.visible = false;
            }

            const focusCenter = this._focusBarBackground.width / 2;
            const distanceFromCenter = Math.abs(this._focusValue - focusCenter);
            const maxDistance = this._focusBarBackground.width / 2;
            const percentage = (distanceFromCenter / maxDistance) * 100;

            let focusScore;
            if (percentage <= 20) {
                focusScore = 100;
            } else if (percentage <= 40) {
                focusScore = 80;
            } else if (percentage <= 70) {
                focusScore = 40;
            } else {
                focusScore = 0;
            }

            $gameVariables.setValue(this._focusVariable, focusScore);
            console.log(`ピントスコア: ${focusScore} 点を変数 ${this._focusVariable} に保存`);

            AudioManager.playSe({ 
                name: this._shutterSound, 
                volume: 90, 
                pitch: 100, 
                pan: 0 
            });

            this._shutterWaitCounter = this._shutterWaitTime;
        }

        endMiniGame() {
            console.log("撮影完了。シーン終了。");
            SceneManager.pop();
        }
    }
})();
