//==============================================================================
// PictureSpineExtension_CharacterAction.js
//==============================================================================

/*:
 * @plugindesc キャラクターアクションプラグイン（PictureSpine拡張プラグイン）
 * @author 奏ねこま（おとぶきねこま）
 * @url http://makonet.sakura.ne.jp/rpg_tkool
 * @target MZ
 * 
 * @help
 * 
 * 歩行グラの画像変更を動的に行う方法（移動ルート→スクリプト）
 * 隊列1人目：$gamePlayer.spine()
 * 隊列2人目：$gamePlayer.followers().follower(0).spine()
 * 隊列3人目：$gamePlayer.followers().follower(1).spine()
 * 隊列4人目：$gamePlayer.followers().follower(2).spine()
 * spine()の後ろは.setAnimationなどコマンドを普通に指定してください。
 * 
 * ------------------------------------------------------------------------------
 *   本プラグインの利用はRPGツクール/RPG Makerの正規ユーザーに限られます。
 *   商用、非商用、有償、無償、一般向け、成人向けを問わず利用可能です。
 *   ご利用の際に連絡や報告は必要ありません。また、製作者名の記載等も不要です。
 *   プラグインを導入した作品に同梱する形以外での再配布、転載はご遠慮ください。
 *   本プラグインにより生じたいかなる問題についても一切の責任を負いかねます。
 * ------------------------------------------------------------------------------
 *                                              Copylight (c) 2021 Nekoma Otobuki
 *                                                           Side note KAMICHICHI
 *                                         http://makonet.sakura.ne.jp/rpg_tkool/
 *                                                  https://twitter.com/koma_neko
 */

(() => {
    'use strict';

    if (!PluginManager._scripts.includes('PictureSpine')) {
        return;
    }

    function convars(obj) {
        if (typeof obj == 'string') {
            let _obj = obj.replace(/\\v\[(\d+)\]/gi, (match, p1) => {
                return $gameVariables.value(Number(p1));
            });
            obj = _obj != obj ? convars(_obj) : _obj;
        } else if (obj instanceof Array) {
            obj = obj.map(value => convars(value));
        } else if (typeof obj == 'object') {
            obj = {...obj};
            for (let key in obj) {
                obj[key] = convars(obj[key]);
            }
        }
        return obj;
    }

    function meta(note, tag) {
        let regexp = /<([^<>:]+)(:?)([^>]*)>/gi;
        let meta = [];
        for (let match; match = regexp.exec(note);) {
            if (match[1] == tag) {
                meta.push(match[2] == ':' ? match[3].replace('&gt;', '>') : true);
            }
        }
        return meta;
    }

    //==============================================================================
    // Game_CharacterSpine
    //==============================================================================
    
    class Game_CharacterSpine extends Game_Spine {
        constructor(script = {}) {
            super();
            this._script = script;
            this._action = {};
            this._handler = {};
            this.setupHandler(script);
        }

        setupHandler(script) {
            for (let symbol in script) {
                if (symbol == 'setup') {
                    eval(`this.${script[symbol]}`);
                } else {
                    while (script[symbol][0] == '#') {
                        script[symbol] = script[script[symbol].substr(1)];
                    }
                    this.setHandler(symbol, Function(`this.${script[symbol]}`).bind(this));
                }
            }
        }

        setHandler(symbol, method) {
            this._handler[symbol] = method;
        }

        callHandler(symbol) {
            if (this._handler[symbol]) {
                this._handler[symbol]();
            } else if (this._script[symbol]) {
                this.setHandler(symbol, Function(`this.${this._script[symbol]}`).bind(this));
                this._handler[symbol]();
            }
        }

        update(character) {
            let action = {};
            action.jump = character.isJumping();
            action.dash = character.isDashMoving() && !action.jump;
            action.walk = character.isNextMoving() && !action.jump && !action.dash;
            action.idle = !action.walk && !action.dash && !action.jump;
            action.dir2 = character.direction() == 2;
            action.dir4 = character.direction() == 4;
            action.dir6 = character.direction() == 6;
            action.dir8 = character.direction() == 8;

            action.idle2 = !action.walk && !action.dash && !action.jump && action.dir2;
            action.idle4 = !action.walk && !action.dash && !action.jump && action.dir4;
            action.idle6 = !action.walk && !action.dash && !action.jump && action.dir6;
            action.idle8 = !action.walk && !action.dash && !action.jump && action.dir8;

            action.dash2 = action.dash && action.dir2;
            action.dash4 = action.dash && action.dir4;
            action.dash6 = action.dash && action.dir6;
            action.dash8 = action.dash && action.dir8;

            action.walk2 = action.walk && action.dir2;
            action.walk4 = action.walk && action.dir4;
            action.walk6 = action.walk && action.dir6;
            action.walk8 = action.walk && action.dir8;

            action.jump2 = character.direction() == 2 &&  action.jump && !action.walk && !action.dash;
            action.jump4 = character.direction() == 4 &&  action.jump && !action.walk && !action.dash;
            action.jump6 = character.direction() == 6 &&  action.jump && !action.walk && !action.dash;
            action.jump8 = character.direction() == 8 &&  action.jump && !action.walk && !action.dash;

            action.oshikko8 = action.oshikko8;
            action.mapchip001 = action.mapchip001;
            
/*
            action.oshikko2 = character.direction() == 2 &&  !action.jump && !action.walk && !action.dash;
            action.oshikko4 = character.direction() == 4 &&  !action.jump && !action.walk && !action.dash;
            action.oshikko6 = character.direction() == 6 &&  !action.jump && !action.walk && !action.dash;
            action.oshikko8 = character.direction() == 8 &&  !action.jump && !action.walk && !action.dash;
*/
            for (let symbol in action) {
                if (action[symbol] && action[symbol] != this._action[symbol]) {
                    this.callHandler(symbol);
                }
            }

            this._action = action;
        }
    }

    window.Game_CharacterSpine = Game_CharacterSpine;

    //==============================================================================
    // Sprite_CharacterSpine
    //==============================================================================
    
    class Sprite_CharacterSpine extends Sprite_Spine {
        constructor(...args) {
            super(...args);
            this._character = args[0] instanceof Game_Character ? args[0] : null;
        }

        spine() {
            if (this._spine) {
                return this._spine;
            }
            return this._character ? this._character.spine() : null;
        }
    }

    /* rpg_objects.js */

    //==============================================================================
    // Game_Character
    //==============================================================================
    
    (__initMembers => {
        Game_Character.prototype.initMembers = function() {
            __initMembers.apply(this, arguments);

            this._spine = null;
            this._spineId = -1;
        };
    })(Game_Character.prototype.initMembers);
    
    (__update => {
        Game_Character.prototype.update = function() {
            this._isNextMoving = this.isMoving();

            __update.apply(this, arguments);

            if (this._spine) {
                this._spine.update(this);
            }
        };
    })(Game_Character.prototype.update);
    
    Game_Character.prototype.spine = function() {
        return this._spine;
    };

    Game_Character.prototype.setupSpine = function(id, note) {
        if (id == this._spineId) {
            return;
        }

        let data = meta(note, 'spine_character_action');
        if (data.length == 0) {
            this._spine = null;
            this._spineId = -1;
            return;
        }

        let script = {};
        for (let value of data) {
            script = Object.assign(script, eval(`({${value}})`));
        }

        this._spine = new Game_CharacterSpine(script);
        this._spineId = id;
    };

    Game_Character.prototype.isNextMoving = function() {
        return !!this._isNextMoving && this.hasWalkAnime() || this.hasStepAnime();
    };

    Game_Character.prototype.isDashMoving = function() {
        return false;
    };

    //==============================================================================
    // Game_Player
    //==============================================================================
    
    (__refresh => {
        Game_Player.prototype.refresh = function() {
            __refresh.apply(this, arguments);

            let player = $gameParty.leader();
            if (!player) {
                this._spine = null;
                return;
            }

            let note = player.actor().note;
            this.setupSpine(player.actorId(), note);
        };
    })(Game_Player.prototype.refresh);
    
    Game_Player.prototype.isDashMoving = function() {
        return this.isDashing() && this.isNextMoving();
    };

    //==============================================================================
    // Game_Follower
    //==============================================================================
    
    (__refresh => {
        Game_Follower.prototype.refresh = function() {
            __refresh.apply(this, arguments);

            if (!this.isVisible()) {
                this._spine = null;
                return;
            }

            let actor = this.actor();
            let note = actor.actor().note;
            this.setupSpine(actor.actorId(), note);
        };
    })(Game_Follower.prototype.refresh);
    
    Game_Follower.prototype.isDashMoving = function() {
        return $gamePlayer.isDashing() && this.isNextMoving();
    };

    //==============================================================================
    // Game_Event
    //==============================================================================
    
    (__refresh => {
        Game_Event.prototype.refresh = function() {
            __refresh.apply(this, arguments);

            if (this._pageIndex < 0) {
                this._spine = null;
                return;
            }

            let note = this.event().note;
            this.page().list.forEach(command => {
                if ([108, 408].includes(command.code)) {
                    note += command.parameters[0];
                }
            })
            this.setupSpine(this._pageIndex, note);
        };
    })(Game_Event.prototype.refresh);
    
    /* rpg_sprites.js */

    //==============================================================================
    // Sprite_Character
    //==============================================================================
    
    (__initialize => {
        Sprite_Character.prototype.initialize = function(character) {
            __initialize.apply(this, arguments);

            this.addChild(new Sprite_CharacterSpine(character));
        };
    })(Sprite_Character.prototype.initialize);
    
})();


