//=============================================================================
// RapidAccelerator.js
// ----------------------------------------------------------------------------
// Copyright (c) 2017-2018 Tsumio
// 当ファイルの複製・改変・再配布を固く禁じます。
// このプラグインは神乳様向けに制作されたプラグインです。
// いかなる理由があろうとも、許可されていない場合の複製・改変・再配布を認めません。
// No reproduction or republication without written permission.
// ----------------------------------------------------------------------------
// Version
// 1.0.5 2025/12/06 プラグインコマンドによる機能ON/OFF制御を追加(MV/MZ両対応)。
// 1.0.4 2018/11/27 音楽の高速化処理を削除。それに伴い、不要になったコードも全て削除。
// 1.0.3 2018/11/27 不要になった処理を削除した。
// 1.0.2 2018/11/27 ゲームウィンドウを切り替え時に高速化が維持されている問題を修正。
// 1.0.1 2018/11/27 高速化状態が逆になる現象を暫定的に修正。
// 1.0.0 2018/11/27 公開。
// ----------------------------------------------------------------------------
// [GitHub] : https://github.com/Tsumio/rmmv-plugins
// [Blog]   : http://ntgame.wpblog.jp/
// [Twitter]: https://twitter.com/TsumioNtGame
//=============================================================================

/*:ja
 * @target MV MZ
 * @plugindesc 特定のキーを押している間のみ高速化を実行する機能を追加するプラグイン
 * @author ツミオ
 *
 * @param ----基本的な設定----
 * @desc 
 * @default 
 * 
 * @param 高速スピード
 * @desc 高速化を実行した際のスピード倍率です。(-60...16)
 * 負の値を設定するとスローモーションになります。
 * @default 2
 * @type number
 * @min -60
 * @max 16
 *
 * @command Enable
 * @text 高速化ON
 * @desc このプラグインの高速化機能を有効にします。
 *
 * @command Disable
 * @text 高速化OFF
 * @desc このプラグインの高速化機能を無効にします。
 *
 * @command Toggle
 * @text 高速化切替
 * @desc このプラグインの高速化機能の有効/無効を切り替えます。
 * 
 * @help 特定のキーを押している間のみ高速化を実行する機能を追加するプラグイン
 * 
 * 
 * 【特徴】
 * ・特定のキーを押している間、ゲームを高速化します
 * ・プラグインコマンドで「高速化機能そのもの」をON/OFFできます
 * 
 * 【使用方法】
 * プラグインの導入後、プラグインパラメーターを設定してください。
 * 指定のキーを押すとゲームが高速化します。
 * さらに、イベントコマンド「プラグインコマンド」から
 * 高速化機能のON/OFFを切り替えることができます。
 * 
 * 【MVでのプラグインコマンド】
 *   RapidAccelerator Enable   # 高速化機能を有効
 *   RapidAccelerator Disable  # 高速化機能を無効
 *   RapidAccelerator Toggle   # 有効/無効を切り替え
 * 
 * 【MZでのプラグインコマンド】
 *   プラグイン一覧から本プラグインを選択し、以下のコマンドを使用します。
 *   ・高速化ON（Enable）
 *   ・高速化OFF（Disable）
 *   ・高速化切替（Toggle）
 * 
 * 【備考】
 * 当プラグインを利用したことによるいかなる損害に対しても、制作者は一切の責任を負わないこととします。
 * 
 * 【利用規約】
 * いかなる理由があろうとも、許可されていない場合の複製・改変・再配布を認めません。
 * 
 * 【謝辞】
 * 本プラグインはトリアコンタン様の「DevToolsManage.js」を参考に作成しています。
 * 素敵なプラグインを公開してくださりありがとうございます。
 * https://github.com/triacontane/RPGMakerMV/blob/master/DevToolsManage.js
 * 
 * 
 */

(function() {
    'use strict';
    var pluginName = 'RapidAccelerator';

    var enableSwitchNumber = 9;  //この番号のSwitchがONになっている間のみ有効

////=============================================================================
//// Local function
////  These functions checks & formats plugin's command parameters.
////  I borrowed these functions from Triacontane.Thanks!
////=============================================================================
    var getParamString = function(paramNames) {
        if (!Array.isArray(paramNames)) paramNames = [paramNames];
        for (var i = 0; i < paramNames.length; i++) {
            var name = PluginManager.parameters(pluginName)[paramNames[i]];
            if (name) return name;
        }
        return '';
    };

    var getParamNumber = function(paramNames, min, max) {
        var value = getParamString(paramNames);
        if (arguments.length < 2) min = -Infinity;
        if (arguments.length < 3) max = Infinity;
        return (parseInt(value) || 0).clamp(min, max);
    };

    //This function is not written by Triacontane.Tsumio wrote this function !
    var convertParam = function(param) {
        if(param !== undefined){
            try {
                return JSON.parse(param);
            }catch(e){
                console.group();
                console.error('%cParameter is invalid ! You should check the following parameter !','background-color: #5174FF');
                console.error('Parameter:' + eval(param));
                console.error('Error message :' + e);
                console.groupEnd();
            }
        }
    };

    //This function is not written by Triacontane.Tsumio wrote this function !
    var convertArrayParam = function(param) {
        if(param !== undefined){
            try {
                const array = JSON.parse(param);
                for(let i = 0; i < array.length; i++) {
                    array[i] = JSON.parse(array[i]);
                }
                return array;
            }catch(e){
                console.group();
                console.error('%cParameter is invalid ! You should check the following parameter !','background-color: #5174FF');
                console.error('Parameter:' + eval(param));
                console.error('Error message :' + e);
                console.groupEnd();
            }
        }
    };

    /**
     * Convert to number.Receive converted JSON object.
     * @param {Object} obj
     * 
     */
    //This function is not written by Triacontane.Tsumio wrote this function !
    var convertToNumber = function(obj) {
        for(var prop in obj) {
            obj[prop] = Number(obj[prop]);
        }
        return obj;
    };

////=============================================================================
//// Get and set plugin parameters.
////=============================================================================
    var param                          = {};
    //Basic Settings
    param.rapidSpeed          = getParamNumber(['高速スピード', '高速スピード']);

////=============================================================================
//// RapidAccelerator
////  高速化のためのフラグを管理。静的クラス
////=============================================================================

    class RapidAccelerator {
        static get RAPID_KEY() {
            return 'control';
        }

        // プラグインコマンドによる有効フラグ（未設定時はtrue扱い）
        static get isEnabled() {
            if (this._enabled === undefined) {
                this._enabled = true; // 互換性のためデフォルトは有効
            }
            return this._enabled;
        }

        static setEnabled(value) {
            this._enabled = !!value;
        }

        static get isRapid() {
            // キーを押している + プラグインコマンドで有効 + スイッチがON
            return this._isRapidMode && this.isEnabled && $gameSwitches.value(enableSwitchNumber);
        }

        static get rapidSpeed() {
            return param.rapidSpeed;
        }

        static toggleRapid(value) {
            if(this._isRapidMode !== value) {
                this._isRapidMode = value;
            }
        }
    }
window.RapidAccelerator = window.RapidAccelerator || RapidAccelerator;

////=============================================================================
//// Plugin Commands (MV/MZ両対応)
////=============================================================================
    // MZ用プラグインコマンド
    if (PluginManager.registerCommand) {
        PluginManager.registerCommand(pluginName, 'Enable', function() {
            RapidAccelerator.setEnabled(true);
        });

        PluginManager.registerCommand(pluginName, 'Disable', function() {
            RapidAccelerator.setEnabled(false);
        });

        PluginManager.registerCommand(pluginName, 'Toggle', function() {
            RapidAccelerator.setEnabled(!RapidAccelerator.isEnabled);
        });
    }

    // MV用プラグインコマンド
    if (typeof Game_Interpreter !== 'undefined') {
        var _RA_Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
        Game_Interpreter.prototype.pluginCommand = function(command, args) {
            _RA_Game_Interpreter_pluginCommand.apply(this, arguments);
            if (command === pluginName) {
                var subCommand = (args[0] || '').toUpperCase();
                switch (subCommand) {
                    case 'ENABLE':
                    case 'ON':
                        RapidAccelerator.setEnabled(true);
                        break;
                    case 'DISABLE':
                    case 'OFF':
                        RapidAccelerator.setEnabled(false);
                        break;
                    case 'TOGGLE':
                        RapidAccelerator.setEnabled(!RapidAccelerator.isEnabled);
                        break;
                }
            }
        };
    }

////=============================================================================
//// Scene_Base
////  フェード関係の高速化を提供します
////=============================================================================
    const _Scene_Base_fadeSpeed    = Scene_Base.prototype.fadeSpeed;
    Scene_Base.prototype.fadeSpeed = function() {
        return RapidAccelerator.isRapid ? 1 : _Scene_Base_fadeSpeed.apply(this, arguments);
    };

    const _Scene_Base_startFadeIn    = Scene_Base.prototype.startFadeIn;
    Scene_Base.prototype.startFadeIn = function(duration, white) {
        if (RapidAccelerator.isRapid) arguments[0] = 1;
        _Scene_Base_startFadeIn.apply(this, arguments);
    };

    const _Scene_Base_startFadeOut    = Scene_Base.prototype.startFadeOut;
    Scene_Base.prototype.startFadeOut = function(duration, white) {
        if (RapidAccelerator.isRapid) arguments[0] = 1;
        _Scene_Base_startFadeOut.apply(this, arguments);
    };

    const _Scene_Base_update    = Scene_Base.prototype.update;
    Scene_Base.prototype.update = function(duration, white) {
        _Scene_Base_update.call(this);
        if (Input.isTriggered(RapidAccelerator.RAPID_KEY)) {
            RapidAccelerator.toggleRapid(true);
        } else if(!Input.isPressed(RapidAccelerator.RAPID_KEY)) {
            RapidAccelerator.toggleRapid(false);
        }
    };

////=============================================================================
//// Scene_Map
////  高速モードを実装します
////=============================================================================
    const _Scene_Map_isFastForward    = Scene_Map.prototype.isFastForward;
    Scene_Map.prototype.isFastForward = function() {
        return _Scene_Map_isFastForward.apply(this, arguments) || RapidAccelerator.isRapid;
    };

    const _Scene_Map_updateMainMultiply    = Scene_Map.prototype.updateMainMultiply;
    Scene_Map.prototype.updateMainMultiply = function() {
        _Scene_Map_updateMainMultiply.apply(this, arguments);
        if (this.isFastForward() && RapidAccelerator.isRapid) {
            for (let i = 2; i <= param.rapidSpeed; i++) {
                this.updateMain();
            }
        }
    };

////=============================================================================
//// Window_Base
////  ウィンドウの高速開閉を提供します
////=============================================================================
    const _Window_Base_updateOpen    = Window_Base.prototype.updateOpen;
    Window_Base.prototype.updateOpen = function() {
        if (RapidAccelerator.isRapid && this._opening) this.openness = 255;
        _Window_Base_updateOpen.apply(this, arguments);
    };

    const _Window_Base_updateClose    = Window_Base.prototype.updateClose;
    Window_Base.prototype.updateClose = function() {
        if (RapidAccelerator.isRapid && this._closing) this.openness = 0;
        _Window_Base_updateClose.apply(this, arguments);
    };

////=============================================================================
//// Window_Message
////  メッセージの高速化を提供します
////=============================================================================
    const _Window_Message_isTriggered    = Window_Message.prototype.isTriggered;
    Window_Message.prototype.isTriggered = function() {
        return _Window_Message_isTriggered.apply(this, arguments) || RapidAccelerator.isRapid;
    };

    const _Window_Message_startPause    = Window_Message.prototype.startPause;
    Window_Message.prototype.startPause = function() {
        _Window_Message_startPause.apply(this, arguments);
        if (RapidAccelerator.isRapid) this.startWait(1);
    };


    

})();
