//=============================================================================
// StartUpFullScreen.js
// ----------------------------------------------------------------------------
// (C)2015 Triacontane
// This software is released under the MIT License.
// http://opensource.org/licenses/mit-license.php
// ----------------------------------------------------------------------------
// Version
// 1.6.1 2023/08/05 ElectronForMz.jsとの順序を定義するアノテーションを追加
// 1.6.0 2023/07/23 オプション変更時にフルスクリーン状態を即時反映させる機能を追加
// 1.5.0 2023/06/01 ElectronForMz.jsに対応
// 1.4.0 2023/05/01 デフォルトでフルスクリーン起動できるパラメータを追加
// 1.3.0 2022/09/09 ゲーム終了画面にもシャットダウン項目を追加できる機能を追加
// 1.2.0 2021/12/30 イベントテスト実行時は全画面化を無効にするよう仕様変更
// 1.1.0 2021/11/04 MZで動作するよう修正
// 1.0.3 2019/01/14 1.0.3でコアスクリプトv1.6.1以前で逆に動作しなくなっていた問題を修正
// 1.0.2 2019/01/14 コアスクリプトv1.6.1以降で正常に動作していなかった問題を修正
// 1.0.1 2018/06/30 タイトルコマンドウィンドウのY座標整数になっていなかった問題を修正
// 1.0.0 2016/03/06 初版
// ----------------------------------------------------------------------------
// [Blog]   : https://triacontane.blogspot.jp/
// [Twitter]: https://twitter.com/triacontane/
// [GitHub] : https://github.com/triacontane/
//=============================================================================

/*:
 * @plugindesc Full Screen Startup
 * @target MZ
 * @url https://github.com/triacontane/RPGMakerMV/tree/mz_master/StartUpFullScreen.js
 * @base PluginCommonBase
 * @orderAfter PluginCommonBase
 * @orderAfter ElectronForMz
 * @author triacontane
 *
 * @param Shutdown
 * @text Shutdown
 * @desc The name of the shutdown item to be added to the title screen.
 * It is displayed only when running in the local environment.
 * @default Shutdown
 *
 * @param DefaultFullScreen
 * @text Starts up in full screen by default
 * @desc If enabled, it will start full screen by default.
 * @default false
 * @type boolean
 *
 * @param Immediate
 * @text immediate reflection
 * @desc If enabled, the full screen state will be changed on the fly when the startup options are changed in the options.
 * @default false
 * @type boolean
 *
 * @param StartUpFullScreen
 * @text full screen startup
 * @desc The name of the item to be activated in all screens to be added to the options screen.
 * It is displayed only when running in the local environment.
 * @default Full Screen Startup
 *
 * @param UseGameEnd
 * @text Added to Game End command
 * @desc Add a shutdown item to the "Game End" command.
 * @default true
 * @type boolean
 *
 * @help StartUpFullScreen.js
 *
 * Add "Full Screen Startup" to the options screen.
 * If enabled, the game will launch in full screen.
 * Also add "Shutdown" to the title screen.
 *
 * This plugin is only valid when run in a local environment.
 * When the event test is executed, tempo is prioritized and full-screening is disabled.
 *
 * You need the base plugin "PluginCommonBase.js" to use this plugin.
 * The "PluginCommonBase.js" is stored in the following folder under the installation folder of RPG Maker MZ.
 * dlc/BasicResources/plugins/official
 *
 * User Agreement:
 *  You may alter or redistribute the plugin without permission. There are no restrictions on usage format
 *  (such as adult- or commercial-use only).
 *  This plugin is now all yours.
 */

/*:ja
 * @plugindesc フルスクリーンで起動プラグイン
 * @target MZ
 * @url https://github.com/triacontane/RPGMakerMV/tree/mz_master/StartUpFullScreen.js
 * @base PluginCommonBase
 * @orderAfter PluginCommonBase
 * @orderAfter ElectronForMz
 * @author トリアコンタン
 *
 * @param Shutdown
 * @text シャットダウン
 * @desc タイトル画面に追加するシャットダウンの項目名です。
 * ローカル環境での実行時のみ表示されます。
 * @default シャットダウン
 *
 * @param DefaultFullScreen
 * @text デフォルトでフルスクリーン
 * @desc 有効にするとデフォルトでフルスクリーン起動します。
 * @default false
 * @type boolean
 *
 * @param Immediate
 * @text 即時反映
 * @desc 有効にするとオプションで起動オプションを変更したときに、その場でフルスクリーン状態が変更されます。
 * @default false
 * @type boolean
 *
 * @param StartUpFullScreen
 * @text フルスクリーンで起動
 * @desc オプション画面に追加する全画面で起動の項目名です。
 * ローカル環境での実行時のみ表示されます。
 * @default フルスクリーンで起動
 *
 * @param UseGameEnd
 * @text ゲーム終了画面に追加
 * @desc ゲーム終了画面にシャットダウンの項目を追加します。
 * @default true
 * @type boolean
 *
 * @help StartUpFullScreen.js
 *
 * オプション画面に「フルスクリーンで起動」を追加します。
 * 有効な場合、ゲームをフルスクリーンで起動します。
 * またタイトル画面にシャットダウンを追加します。
 *
 * このプラグインはローカル環境で実行した場合のみ有効です。
 * イベントテスト実行時はテンポを優先し全画面化は無効となります。
 *
 * このプラグインの利用にはベースプラグイン『PluginCommonBase.js』が必要です。
 * 『PluginCommonBase.js』は、RPGツクールMZのインストールフォルダ配下の
 * 以下のフォルダに格納されています。
 * dlc/BasicResources/plugins/official
 *
 * 利用規約：
 *  作者に無断で改変、再配布が可能で、利用形態（商用、18禁利用等）
 *  についても制限はありません。
 *  このプラグインはもうあなたのものです。
 */

(function() {
  'use strict';

  // --------------------------------
  // Platform detection
  // --------------------------------
  const isNw = () => {
    try { return typeof Utils.isNwjs === 'function' ? Utils.isNwjs() : !!window.nw; } catch(_) { return false; }
  };
  const isElectron = () => {
    try { return !!(window && window.electronAPI); } catch(_) { return false; }
  };
  const isDesktop = () => isNw() || isElectron();

  if (!isDesktop()) return;

  // --------------------------------
  // Internal fullscreen flag
  // --------------------------------
  let _fullScreenFlag = false;

  // --------------------------------
  // Graphics hooks
  // --------------------------------
  const _gReq = Graphics._requestFullScreen;
  Graphics._requestFullScreen = function() {
    try {
      if (isElectron()) {
        if (window.electronAPI && typeof window.electronAPI.fullScreen === 'function') {
          window.electronAPI.fullScreen(true);
          _fullScreenFlag = true;
          return;
        }
      } else if (isNw()) {
        const win = nw.Window.get();
        win.enterFullscreen();
        _fullScreenFlag = true;
        return;
      }
    } catch(_) {}
    _gReq.apply(this, arguments);
    _fullScreenFlag = true;
  };

  const _gCan = Graphics._cancelFullScreen;
  Graphics._cancelFullScreen = function() {
    try {
      if (isElectron()) {
        if (window.electronAPI && typeof window.electronAPI.fullScreen === 'function') {
          window.electronAPI.fullScreen(false);
          _fullScreenFlag = false;
          return;
        }
      } else if (isNw()) {
        const win = nw.Window.get();
        win.leaveFullscreen();
        _fullScreenFlag = false;
        return;
      }
    } catch(_) {}
    _gCan.apply(this, arguments);
    _fullScreenFlag = false;
  };

  const _gIs = Graphics._isFullScreen;
  Graphics._isFullScreen = function() {
    try {
      if (isElectron()) {
        if (window.electronAPI && typeof window.electronAPI.isFullScreen === 'function') {
          return !!window.electronAPI.isFullScreen();
        }
        return _fullScreenFlag;
      } else if (isNw()) {
        const win = nw.Window.get();
        return !!(win && win.isFullscreen);
      }
    } catch(_) {}
    return _gIs.apply(this, arguments);
  };

  // デスクトップではストレッチ許可（黒帯を避ける）
  const _gDefStretch = Graphics._defaultStretchMode;
  Graphics._defaultStretchMode = function() {
    return true;
  };

  // --------------------------------
  // Title/GameEnd にシャットダウン追加（従来互換）
  // --------------------------------
  const param = {
    Shutdown: 'Shutdown',
    StartUpFullScreen: 'Full Screen Startup',
    UseGameEnd: true
  };

  const _stCreateCmd = Scene_Title.prototype.createCommandWindow;
  Scene_Title.prototype.createCommandWindow = function() {
    _stCreateCmd.apply(this, arguments);
    this._commandWindow.setHandler('shutdown', this.commandShutdown && this.commandShutdown.bind(this));
  };
  Scene_Title.prototype.commandShutdown = function() {
    this._commandWindow.close();
    this.fadeOutAll();
    SceneManager.goto(Scene_Terminate);
  };

  const _geCreateCmd = Scene_GameEnd.prototype.createCommandWindow;
  Scene_GameEnd.prototype.createCommandWindow = function() {
    _geCreateCmd.apply(this, arguments);
    if (param.UseGameEnd) this._commandWindow.setHandler('shutdown', this.commandShutdown.bind(this));
  };
  Scene_GameEnd.prototype.commandShutdown = function() {
    this._commandWindow.close();
    this.fadeOutAll();
    SceneManager.goto(Scene_Terminate);
  };
  const _geRect = Scene_GameEnd.prototype.commandWindowRect;
  Scene_GameEnd.prototype.commandWindowRect = function() {
    const rect = _geRect.apply(this, arguments);
    if (param.UseGameEnd) rect.height = this.calcWindowHeight(3, true);
    return rect;
  };
  const _geMake = Window_GameEnd.prototype.makeCommandList;
  Window_GameEnd.prototype.makeCommandList = function() {
    _geMake.apply(this, arguments);
    if (param.UseGameEnd) this.addCommand(param.Shutdown, 'shutdown');
  };

  const _toMake = Window_TitleCommand.prototype.makeCommandList;
  Window_TitleCommand.prototype.makeCommandList = function() {
    _toMake.apply(this, arguments);
    this.addCommand(param.Shutdown, 'shutdown');
  };

  // --------------------------------
  // Options: フルスクリーン項目を追加
  // --------------------------------
  const _optAdd = Window_Options.prototype.addGeneralOptions;
  Window_Options.prototype.addGeneralOptions = function() {
    _optAdd.apply(this, arguments);
    this.addCommand(param.StartUpFullScreen, 'startUpFullScreen');
  };

  // --------------------------------
  // ConfigManager: 永続化 + 変更時即時反映
  // --------------------------------
  if (typeof ConfigManager._startUpFullScreen !== 'boolean') {
    ConfigManager._startUpFullScreen = false;
  }

  function applyStartUpFullScreenNow(v) {
    if (v) Graphics._requestFullScreen(); else Graphics._cancelFullScreen();
  }

  Object.defineProperty(ConfigManager, 'startUpFullScreen', {
    get() { return this._startUpFullScreen; },
    set(value) {
      const v = (value === true || value === 1 || value === '1' || value === 'true');
      if (this._startUpFullScreen === v) return;
      this._startUpFullScreen = v;
      applyStartUpFullScreenNow(v);
      try { ConfigManager.save(); } catch(_) {}
    }
  });

  const _cmApply = ConfigManager.applyData;
  ConfigManager.applyData = function(config) {
    _cmApply.apply(this, arguments);
    const raw = config ? config.startUpFullScreen : undefined;
    const v = (raw === true || raw === 1 || raw === '1' || raw === 'true');
    this._startUpFullScreen = v;
  };

  const _cmMake = ConfigManager.makeData;
  ConfigManager.makeData = function() {
    const cfg = _cmMake.apply(this, arguments);
    cfg.startUpFullScreen = !!this._startUpFullScreen;
    return cfg;
  };

  // --------------------------------
  // Boot: 起動時に設定がONなら即全画面
  // --------------------------------
  const _sbStart = Scene_Boot.prototype.start;
  Scene_Boot.prototype.start = function() {
    _sbStart.apply(this, arguments);
    if (ConfigManager.startUpFullScreen && !DataManager.isEventTest()) {
      Graphics._requestFullScreen();
    }
  };

  // --------------------------------
  // Scene_Terminate
  // --------------------------------
  function Scene_Terminate() { this.initialize.apply(this, arguments); }
  Scene_Terminate.prototype = Object.create(Scene_Base.prototype);
  Scene_Terminate.prototype.constructor = Scene_Terminate;
  Scene_Terminate.prototype.start = function() { SceneManager.terminate(); };

  // --------------------------------
  // 外部ユーティリティ
  // --------------------------------
  window.KCH_setFullScreen = function(on) {
    ConfigManager.startUpFullScreen = !!on;
  };
})();

// --------------------------------------------------------------------------
// 画面サイズプリセット（ウィンドウ時のみ）
// フルスクリーン中は一切適用しない（解除されるまで待つ）
// 内部解像度は 1920x1080 に固定、ストレッチ有効
// 切替のたびに必ず「ディスプレイ作業領域の中心」に再配置
// --------------------------------------------------------------------------
(function() {
  'use strict';

  const isNw = () => { try { return typeof Utils.isNwjs === 'function' ? Utils.isNwjs() : !!window.nw; } catch(_) { return false; } };
  const isElectron = () => { try { return !!(window && window.electronAPI); } catch(_) { return false; } };
  const isDesktop = () => isNw() || isElectron();

  if (!isDesktop()) return;

  const LOGICAL_W = 1920;
  const LOGICAL_H = 1080;
  const LSKEY_PRESET = 'KCH_screenPreset'; // "0":1280x720 / "1":1920x1080

  function isFullscreenNow() {
    try { if (typeof Graphics._isFullScreen === 'function' && Graphics._isFullScreen()) return true; } catch(_) {}
    try { if (document.fullscreenElement) return true; } catch(_) {}
    try { if (isNw()) { const w = nw.Window.get(); if (w && w.isFullscreen) return true; } } catch(_) {}
    try { if (isElectron() && window.electronAPI && typeof window.electronAPI.isFullScreen === 'function') { return !!window.electronAPI.isFullScreen(); } } catch(_) {}
    return false;
  }

  // 現在ウィンドウが存在するディスプレイの「作業領域」を取得
  function getWorkAreaForCurrentWindow() {
    // Electron: 優先的にメイン or 現在のディスプレイの workArea を取得
    if (isElectron() && window.electronAPI) {
      try {
        if (typeof window.electronAPI.getCurrentDisplay === 'function') {
          const d = window.electronAPI.getCurrentDisplay(); // {workArea:{x,y,width,height}}
          if (d && d.workArea) return d.workArea;
        }
        if (typeof window.electronAPI.getPrimaryDisplay === 'function') {
          const d = window.electronAPI.getPrimaryDisplay(); // {workArea:{x,y,width,height}}
          if (d && d.workArea) return d.workArea;
        }
      } catch(_) {}
    }

    // NW.js: 当該ウィンドウが属するスクリーンの work_area
    if (isNw()) {
      try {
        nw.Screen && nw.Screen.Init && nw.Screen.Init();
        const win = nw.Window.get();
        const wx = (typeof window.screenX === 'number') ? window.screenX : 0;
        const wy = (typeof window.screenY === 'number') ? window.screenY : 0;
        const screens = (nw.Screen && nw.Screen.screens) || [];
        // 簡易マッチ：ウィンドウ左上が含まれるスクリーン
        for (const s of screens) {
          const a = s.work_area || s.bounds || { x:0, y:0, width: screen.availWidth, height: screen.availHeight };
          if (wx >= a.x && wx < a.x + a.width && wy >= a.y && wy < a.y + a.height) {
            return { x: a.x, y: a.y, width: a.width, height: a.height };
          }
        }
        // 見つからなければプライマリ
        if (screens[0]) {
          const a = screens[0].work_area || screens[0].bounds;
          if (a) return { x: a.x, y: a.y, width: a.width, height: a.height };
        }
      } catch(_) {}
    }

    // フォールバック：ブラウザの利用可能領域
    try {
      const aw = Math.max(1, screen.availWidth  || window.innerWidth  || 1280);
      const ah = Math.max(1, screen.availHeight || window.innerHeight || 720);
      return { x: 0, y: 0, width: aw, height: ah };
    } catch(_) {
      return { x: 0, y: 0, width: 1280, height: 720 };
    }
  }
/*
  // 与えられた外枠サイズで、常に作業領域の中心に配置
  function setOuterSizeAndCenter(outerW, outerH) {
    const area = getWorkAreaForCurrentWindow();
    const newX = Math.round(area.x + (area.width  - outerW) / 2);
    const newY = Math.round(area.y + (area.height - outerH) / 2);

    try {
      if (isNw()) {
        const win = nw.Window.get();
        // サイズ変更→位置移動の順でジッタ防止
        win.resizeTo(outerW, outerH);
        win.moveTo(newX, newY);
        return;
      }
      if (isElectron() && window.electronAPI && typeof window.electronAPI.setBounds === 'function') {
        window.electronAPI.setBounds({ x: newX, y: newY, width: outerW, height: outerH });
        return;
      }
    } catch(_) {}

    // 最終フォールバック
    try { window.resizeTo(outerW, outerH); } catch(_) {}
    try { window.moveTo(newX, newY); } catch(_) {}
  }
*/
// 与えられた外枠サイズで、常に作業領域の中心に配置（領域内に必ず収める）
function setOuterSizeAndCenter(outerW, outerH) {
  const area = getWorkAreaForCurrentWindow();

  // 1) 作業領域に収まるように外枠サイズを丸める（ここが最重要）
  const w = Math.max(1, Math.min(outerW, area.width));
  const h = Math.max(1, Math.min(outerH, area.height));

  // 2) 中央座標
  let x = Math.round(area.x + (area.width  - w) / 2);
  let y = Math.round(area.y + (area.height - h) / 2);

  // 3) 念のため、完全に領域内にクランプ（負数/はみ出し対策）
  const minX = area.x;
  const minY = area.y;
  const maxX = area.x + area.width  - w;
  const maxY = area.y + area.height - h;
  x = Math.max(minX, Math.min(x, maxX));
  y = Math.max(minY, Math.min(y, maxY));

  try {
    if (isNw()) {
      const win = nw.Window.get();
      win.resizeTo(w, h);
      win.moveTo(x, y);
      return;
    }
    if (isElectron() && window.electronAPI && typeof window.electronAPI.setBounds === 'function') {
      window.electronAPI.setBounds({ x, y, width: w, height: h });
      return;
    }
  } catch(_) {}

  try { window.resizeTo(w, h); } catch(_) {}
  try { window.moveTo(x, y); } catch(_) {}
}

  function fixCanvasStyle() {
    try {
      const html = document.documentElement, body = document.body;
      if (html) { html.style.margin = '0'; html.style.padding = '0'; html.style.overflow = 'hidden'; }
      if (body) { body.style.margin = '0'; body.style.padding = '0'; body.style.overflow = 'hidden'; }
      if (Graphics && Graphics._renderer && Graphics._renderer.view) {
        const v = Graphics._renderer.view;
        v.style.display = 'block';
        v.style.margin  = '0 auto';
        v.style.overflow = 'hidden';
      }
    } catch(_) {}
  }

  // 連続適用の揺れを避けるため、同一フレームの重複呼び出しを束ねる
  let _pendingCenter = false;
  function applyResolutionByPreset(preset) {
    if (isFullscreenNow()) return; // フルスクリーン中は触らない
    fixCanvasStyle();

    const outer = (preset === 0) ? { w:1280, h:720 } : { w:1920, h:1080 };

    SceneManager._screenWidth  = LOGICAL_W;
    SceneManager._screenHeight = LOGICAL_H;
    Graphics.boxWidth  = LOGICAL_W;
    Graphics.boxHeight = LOGICAL_H;
    Graphics._stretchEnabled = true;

    try { if (Graphics && Graphics._renderer) Graphics.resize(LOGICAL_W, LOGICAL_H); } catch(_) {}

    // 中央配置はフレーム末尾で一度だけ
    if (!_pendingCenter) {
      _pendingCenter = true;
      requestAnimationFrame(() => {
        setOuterSizeAndCenter(outer.w, outer.h);
        const flush = () => {
          try { if (SceneManager.onResize) SceneManager.onResize(); } catch(_) {}
          try { window.dispatchEvent(new Event('resize')); } catch(_) {}
          try { if (typeof Graphics._centerElement === 'function') Graphics._centerElement(); } catch(_) {}
        };
        flush();
        setTimeout(flush, 0);
        _pendingCenter = false;
      });
    }
  }

  if (typeof ConfigManager._screenPreset !== 'number') ConfigManager._screenPreset = 1;

  Object.defineProperty(ConfigManager, 'screenPreset', {
    get() { return this._screenPreset; },
    set(v) {
      v = (v|0); if (v !== 0 && v !== 1) v = 1;
      if (this._screenPreset === v) return;
      this._screenPreset = v;
      try { localStorage.setItem(LSKEY_PRESET, String(v)); } catch(_) {}
      applyResolutionByPreset(v);
      try { ConfigManager.save(); } catch(_) {}
    }
  });

  const _cmMake2 = ConfigManager.makeData;
  ConfigManager.makeData = function() {
    const cfg = _cmMake2.apply(this, arguments);
    cfg.screenPreset = this._screenPreset;
    return cfg;
  };

  const _cmApply2 = ConfigManager.applyData;
  ConfigManager.applyData = function(config) {
    _cmApply2.apply(this, arguments);
    if (config && typeof config.screenPreset === 'number') {
      this._screenPreset = (config.screenPreset|0) === 0 ? 0 : 1;
    }
  };

  const _smInit = SceneManager.initGraphics;
  SceneManager.initGraphics = function() {
    _smInit.apply(this, arguments);
    try {
      const sp = localStorage.getItem(LSKEY_PRESET);
      if (sp !== null) ConfigManager._screenPreset = (sp === '0') ? 0 : 1;
    } catch(_) {}
    applyResolutionByPreset(ConfigManager._screenPreset);
  };

  const _sbStart2 = Scene_Boot.prototype.start;
  Scene_Boot.prototype.start = function() {
    _sbStart2.apply(this, arguments);
    setTimeout(() => applyResolutionByPreset(ConfigManager._screenPreset), 0);
  };

  window.addEventListener('resize', fixCanvasStyle);
  document.addEventListener('fullscreenchange', fixCanvasStyle);
  document.addEventListener('DOMContentLoaded', fixCanvasStyle);
})();