/*:
 * @param masterSwitchId
 * @text 包括スイッチID
 * @type switch
 * @default 0
 *
 * @param interruptCommonEventId
 * @text 割り込みコモンイベントID
 * @type common_event
 * @default 1
 *
 * @param watchMode
 * @text 監視対象モード
 * @type select
 * @option パーティ全員
 * @value party
 * @option 指定アクターのみ
 * @value actors
 * @default party
 *
 * @param actorIds
 * @text 監視アクターID（指定時）
 * @type actor[]
 * @default []
 *
 * @param checkEveryFrames
 * @text 監視間隔（フレーム）
 * @type number
 * @min 1
 * @default 1
 *
 * @param stopMessageAndChoices
 * @text 選択肢とメッセージを閉じる
 * @type boolean
 * @default true
 * @desc 「今表示中」の選択肢を閉じるだけ。visible/opennessは触らないので次の選択肢は出ます。
 *
 * @param lockUntilMasterOff
 * @text 割り込み後は包括OFFまで再割り込み禁止
 * @type boolean
 * @default true
 *
 * @command resetState
 * @text ロック・発火状態リセット
 */

(() => {
  "use strict";

  const PLUGIN_NAME = (() => {
    const src = document.currentScript?.src;
    return src ? src.split("/").pop().replace(/\.js$/i, "") : "HPZeroInterruptCommonEvent_Full";
  })();

  const params = PluginManager.parameters(PLUGIN_NAME);

  const masterSwitchId = Number(params.masterSwitchId || 0);
  const interruptCommonEventId = Number(params.interruptCommonEventId || 0);
  const watchMode = String(params.watchMode || "party");
  const actorIds = JSON.parse(params.actorIds || "[]").map(n => Number(n));
  const checkEveryFrames = Math.max(1, Number(params.checkEveryFrames || 1));
  const stopMessageAndChoices = String(params.stopMessageAndChoices || "true") === "true";
  const lockUntilMasterOff = String(params.lockUntilMasterOff || "true") === "true";

  const firedByActorId = new Map();
  let locked = false;

  function masterOn() {
    if (masterSwitchId <= 0) return true;
    return !!$gameSwitches.value(masterSwitchId);
  }

  function enabled() {
    if (interruptCommonEventId <= 0) return false;
    return masterOn();
  }

  function getTargets() {
    if (watchMode === "actors") {
      return actorIds.map(id => $gameActors.actor(id)).filter(a => a);
    }
    return $gameParty ? $gameParty.members() : [];
  }

  function unlockIfMasterOff() {
    if (!lockUntilMasterOff) return;
    if (masterSwitchId <= 0) return;
    if (!masterOn()) {
      locked = false;
      firedByActorId.clear();
    }
  }

  // ★ここが今回の修正の本体
  function forceCloseMessageAndChoices() {
    if (!stopMessageAndChoices) return;

    // LL_GalgeChoiceWindow: busy解除のために“状態だけ”消す（ウィンドウ自体は殺さない）
    if ($gameMessage && typeof $gameMessage.setGalgeChoices === "function") {
      $gameMessage.setGalgeChoices([]);
    }

    const scene = SceneManager._scene;
    if (!scene) return;

    // メッセージウィンドウ（進行待ちを外す目的）
    const mw = scene._messageWindow;
    if (mw) {
      if (typeof mw.terminateMessage === "function") mw.terminateMessage();
      if (typeof mw.deactivate === "function") mw.deactivate();
      if (typeof mw.close === "function") mw.close();
      // visible/openness は触らない
    }

    // 標準選択肢
    const cw = scene._choiceListWindow || scene._choiceWindow;
    if (cw) {
      if (typeof cw.deactivate === "function") cw.deactivate();
      if (typeof cw.close === "function") cw.close();
      // visible/openness は触らない（次の選択肢表示を殺さないため）
    }

    // LL ガルゲ選択肢
    const gw = scene._galgeChoiceListWindow;
    if (gw) {
      if (typeof gw.deactivate === "function") gw.deactivate();
      if (typeof gw.close === "function") gw.close();
      // visible/openness は触らない
    }

    // LLのヘルプスプライトが残る場合だけ非表示（これは表示物なのでOK）
    if (scene._gcwHelpSprite) {
      scene._gcwHelpSprite.visible = false;
      if (scene._gcwHelpSprite.bitmap && typeof scene._gcwHelpSprite.bitmap.clear === "function") {
        scene._gcwHelpSprite.bitmap.clear();
      }
    }
  }

  function interruptToCommonEvent() {
    const itp = $gameMap?._interpreter;
    if (!itp) return;

    // 割り込み直前に「今出ている選択肢だけ」閉じる
    forceCloseMessageAndChoices();

    itp.clear();

    $gameTemp.reserveCommonEvent(interruptCommonEventId);
    itp.setupReservedCommonEvent();

    // 割り込み開始直後にも念のためもう一度
    forceCloseMessageAndChoices();
  }

  function watchAndInterruptIfNeeded() {
    if (!$gameActors || !$gameParty || !$gameSwitches || !$gameMap || !$gameTemp) return;

    unlockIfMasterOff();

    if (locked) return;
    if (!enabled()) return;

    const targets = getTargets();
    for (const actor of targets) {
      const id = actor.actorId();
      const hp = actor.hp;
      const fired = !!firedByActorId.get(id);

      if (hp <= 0) {
        if (!fired) {
          firedByActorId.set(id, true);
          if (lockUntilMasterOff) locked = true;
          interruptToCommonEvent();
        }
        return;
      } else {
        if (fired) firedByActorId.set(id, false);
      }
    }
  }

  let frameCounter = 0;

  const _SceneManager_updateMain = SceneManager.updateMain;
  SceneManager.updateMain = function() {
    _SceneManager_updateMain.call(this);
    if (!(SceneManager._scene instanceof Scene_Map)) return;

    frameCounter++;
    if (frameCounter >= checkEveryFrames) {
      frameCounter = 0;
      watchAndInterruptIfNeeded();
    }
  };

  PluginManager.registerCommand(PLUGIN_NAME, "resetState", () => {
    locked = false;
    firedByActorId.clear();
  });
})();
