/*:
 * @param SceneSettings
 * @text シーンごとの設定
 * @type struct<SceneSetting>[]
 * @default []
 * @desc シーンごとに表示エントリーを設定します。前段条件でEntries集合を切り替えできます。
 *
 * @param UpdateInterval
 * @text 判定間隔（フレーム）
 * @type number
 * @min 1
 * @default 1
 * @desc 1で毎フレーム判定。重いと感じる場合は値を増やしてください。
 *
 * @param AlwaysOnTop
 * @text 最前面に表示
 * @type boolean
 * @on はい
 * @off いいえ
 * @default true
 * @desc ウィンドウレイヤーの一番上に来るようにします。
 *
 * @param OverMessage
 * @text メッセージの上に重ねる
 * @type boolean
 * @on はい
 * @off いいえ
 * @default true
 * @desc Window_Message より上に再配置します。
 *
 */

/*~struct~SceneSetting:
 * @param SceneName
 * @text シーン名
 * @type string
 * @default Scene_Map
 * @desc 例：Scene_Map, Scene_Menu, Scene_MyCustom など
 *
 * @param ConditionVariableId
 * @text 前段条件 変数ID（0で無条件）
 * @type variable
 * @default 0
 *
 * @param ConditionMatchType
 * @text 前段条件 条件タイプ
 * @type select
 * @option 文字列＝ @value eqs
 * @option 文字列≠ @value neqs
 * @option 部分一致（含む） @value contains
 * @option 数値＝ @value eqn
 * @option 数値≧ @value gte
 * @option 数値≦ @value lte
 * @option 正規表現 @value regex
 * @default eqs
 *
 * @param ConditionValue
 * @text 前段条件 比較値
 * @type string
 * @default
 *
 * @param ConditionCaseSensitive
 * @text 前段条件 大小文字区別
 * @type boolean
 * @default false
 *
 * @param Entries
 * @text 表示エントリー
 * @type struct<HelpEntry>[]
 * @default []
 */

/*~struct~HelpEntry:
 * @param VariableId
 * @text 変数ID
 * @type variable
 * @default 1
 *
 * @param Equals
 * @text 一致値
 * @type string
 * @default 1
 * @desc $gameVariables.value(変数ID) を文字列化した結果と比較します。
 *
 * @param Text
 * @text 表示テキスト
 * @type note
 * @default ""
 * @desc 制御文字OK（\I[n], \C[n], \V[n] など）。複数行可。
 *
 * @param X
 * @text X座標
 * @type number
 * @min 0
 * @default 24
 *
 * @param Y
 * @text Y座標
 * @type number
 * @min 0
 * @default 24
 *
 * @param Width
 * @text 幅
 * @type number
 * @min 1
 * @default 480
 *
 * @param FontSize
 * @text フォントサイズ
 * @type number
 * @min 1
 * @default 28
 *
 * @param Opacity
 * @text 不透明度（0〜255）
 * @type number
 * @min 0
 * @max 255
 * @default 255
 *
 * @param Padding
 * @text 余白
 * @type number
 * @min 0
 * @default 12
 *
 * @command ShowHelp
 * @text ヘルプ表示ON
 * @desc ヘルプオーバーレイを表示します。
 *
 * @command HideHelp
 * @text ヘルプ表示OFF
 * @desc ヘルプオーバーレイを非表示にします。
 *
 */

(() => {
  const PLUGIN_NAME = "kamichichi_HelpOverlayVar_AllScenes";
  const params = PluginManager.parameters(PLUGIN_NAME);

  // ---- Plugin Commands: show/hide help overlay (saved in Game_System) ----
  const _Game_System_initialize = Game_System.prototype.initialize;
  Game_System.prototype.initialize = function() {
    _Game_System_initialize.call(this);
    if (this._kchHelpOverlayVisible === undefined) {
      this._kchHelpOverlayVisible = true;
    }
  };

  PluginManager.registerCommand(PLUGIN_NAME, "ShowHelp", () => {
    $gameSystem._kchHelpOverlayVisible = true;
  });

  PluginManager.registerCommand(PLUGIN_NAME, "HideHelp", () => {
    $gameSystem._kchHelpOverlayVisible = false;
  });

  function parseNote(n) {
    if (n == null) return "";
    const s = String(n);
    return s.replace(/^"|"$/g, "").replace(/\\n/g, "\n");
  }

  function parseHelpEntry(raw) {
    const o = JSON.parse(raw);
    return {
      VariableId: Number(o.VariableId || 1),
      Equals: String(o.Equals ?? ""),
      Text: String(o.Text || ""),
      X: Number(o.X || 0),
      Y: Number(o.Y || 0),
      Width: Number(o.Width || 480),
      FontSize: Number(o.FontSize || 28),
      Opacity: Number(o.Opacity ?? 255),
      Windowskin: o.Windowskin ? String(o.Windowskin) : "",
      Padding: Number(o.Padding || 12),
    };
  }

  function normalizeBool(v, def = false) {
    if (v === true || v === "true") return true;
    if (v === false || v === "false") return false;
    return def;
  }

  function parseSceneSettings() {
    const arr = JSON.parse(params.SceneSettings || "[]");
    return arr.map(s => {
      const o = JSON.parse(s);
      const entriesRaw = JSON.parse(o.Entries || "[]");
      return {
        SceneName: String(o.SceneName || ""),
        ConditionVariableId: Number(o.ConditionVariableId || 0),
        ConditionMatchType: String(o.ConditionMatchType || "eqs"),
        ConditionValue: String(o.ConditionValue ?? ""),
        ConditionCaseSensitive: normalizeBool(o.ConditionCaseSensitive, false),
        Entries: entriesRaw.map(parseHelpEntry),
      };
    });
  }

  const SCENE_SETTINGS = parseSceneSettings();
  const UPDATE_INTERVAL = Number(params.UpdateInterval || 1);
  const ALWAYS_ON_TOP   = params.AlwaysOnTop === "true";
  const OVER_MESSAGE    = params.OverMessage === "true";

  function sceneNameOf(scene) {
    if (!scene) return "";
    return scene.constructor?.name || "";
  }

  function evalCondition(variableId, matchType, arg, caseSensitive) {
    if (variableId <= 0) return true;
    const v = $gameVariables.value(variableId);
    try {
      switch (matchType) {
        case "eqs": {
          const a = String(v ?? "");
          const b = String(arg ?? "");
          return caseSensitive ? a === b : a.toLowerCase() === b.toLowerCase();
        }
        case "neqs": {
          const a = String(v ?? "");
          const b = String(arg ?? "");
          return caseSensitive ? a !== b : a.toLowerCase() !== b.toLowerCase();
        }
        case "contains": {
          const a = String(v ?? "");
          const b = String(arg ?? "");
          return caseSensitive ? a.includes(b) : a.toLowerCase().includes(b.toLowerCase());
        }
        case "eqn":
          return Number(v) === Number(arg);
        case "gte":
          return Number(v) >= Number(arg);
        case "lte":
          return Number(v) <= Number(arg);
        case "regex": {
          const flags = caseSensitive ? "" : "i";
          const re = new RegExp(String(arg || ""), flags);
          return re.test(String(v ?? ""));
        }
        default:
          return false;
      }
    } catch (e) {
      console.error(PLUGIN_NAME, "evalCondition error", e);
      return false;
    }
  }


  const _Game_Variables_setValue = Game_Variables.prototype.setValue;
  Game_Variables.prototype.setValue = function(variableId, value) {
    _Game_Variables_setValue.call(this, variableId, value);
    if ($gameTemp) {
      if (!$gameTemp._kchHelpOverlayDirtyVars) {
        $gameTemp._kchHelpOverlayDirtyVars = new Set();
      }
      $gameTemp._kchHelpOverlayDirtyVars.add(variableId);
    }
  };

  function markAllWatchedVarsClean(watchIds) {
    if (!$gameTemp || !$gameTemp._kchHelpOverlayDirtyVars) return;
    const set = $gameTemp._kchHelpOverlayDirtyVars;
    for (const id of watchIds) {
      set.delete(id);
    }
  }

  function isAnyWatchedVarDirty(watchIds) {
    if (!$gameTemp || !$gameTemp._kchHelpOverlayDirtyVars) return false;
    const set = $gameTemp._kchHelpOverlayDirtyVars;
    for (const id of watchIds) {
      if (set.has(id)) return true;
    }
    return false;
  }

  function selectSceneSetting(scene, list) {
    if (!list || !list.length) return null;

    let fallback = null;
    for (const s of list) {
      if (s.ConditionVariableId <= 0) {
        if (!fallback) fallback = s;
        continue;
      }
      if (evalCondition(
        s.ConditionVariableId,
        s.ConditionMatchType,
        s.ConditionValue,
        s.ConditionCaseSensitive
      )) {
        return s;
      }
    }
    return fallback;
  }

  class Window_KCH_Help extends Window_Base {
    initialize() {
      const rect = new Rectangle(0, 0, 240, 72);
      super.initialize(rect);
      this.openness = 0;
      this._currentKey = null;
      this._lastSrcText = "";
      this._text = "";
      this._fontSize = 28;
      this.padding = 12;
      this._windowskinName = "";
    }

    setEntry(entry) {
      const key = entry ? `${entry.VariableId}:${entry.Equals}` : null;
      if (this._currentKey === key && this._text === this._lastSrcText) return;

      this._currentKey = key;
      if (!entry) {
        this.close();
        return;
      }

      this._fontSize = entry.FontSize;
      this.padding = entry.Padding;
      this._lastSrcText = parseNote(entry.Text);
      this._text = this.convertEscapeCharacters(this._lastSrcText);
      this._updateWindowskin(entry.Windowskin);

      const width  = entry.Width;
      const height = this.fittingHeight(this._calcTextLines(this._text, width));

      this.move(entry.X, entry.Y, width, height);
      this.opacity = entry.Opacity;

      this.createContents();
      this.contents.clear();

      const textState = this.createTextState(this._text, 0, 0, width);
      this.resetFontSettings();
      this.contents.fontSize = this._fontSize;
      this.processAllText(textState);

      this.open();
    }

    _updateWindowskin(name) {
      const next = name || "";
      if (this._windowskinName === next) return;
      this._windowskinName = next;
      this.windowskin = ImageManager.loadSystem(next || "Window");
    }

    _calcTextLines(text, width) {
      const st = this.createTextState(text, 0, 0, width);
      st.drawing = false;
      this.resetFontSettings();
      this.contents.fontSize = this._fontSize;
      while (st.index < st.text.length) {
        this.processCharacter(st);
      }
      const h = st.outputHeight || this.lineHeight();
      return Math.max(1, Math.ceil(h / this.lineHeight()));
    }
  }

  const _Scene_Base_create = Scene_Base.prototype.create;
  Scene_Base.prototype.create = function() {
    _Scene_Base_create.call(this);

    const name = sceneNameOf(this);
    this._kchSceneSettingsList = SCENE_SETTINGS.filter(s => s.SceneName === name);
    this._kchCurrentSceneSetting = null;
    this._kchHelpEntries = null;
    this._kchHelpTicker = 0;
    this._kchNeedInitialUpdate = false;
    this._kchWatchVarIds = [];

    if (this._kchSceneSettingsList && this._kchSceneSettingsList.length > 0) {
      const ids = [];
      for (const s of this._kchSceneSettingsList) {
        if (s.ConditionVariableId > 0 && !ids.includes(s.ConditionVariableId)) {
          ids.push(s.ConditionVariableId);
        }
        for (const e of s.Entries) {
          if (e.VariableId > 0 && !ids.includes(e.VariableId)) {
            ids.push(e.VariableId);
          }
        }
      }
      this._kchWatchVarIds = ids;
      this._kchNeedInitialUpdate = true;

      this._kchHelpWindow = new Window_KCH_Help();

      if (ALWAYS_ON_TOP) {
        this.addChild(this._kchHelpWindow);
      } else {
        if (!this._windowLayer && this.createWindowLayer) {
          this.createWindowLayer();
        }
        this.addWindow(this._kchHelpWindow);
      }
    } else {
      this._kchHelpWindow = null;
    }
  };

  const _Scene_Base_update = Scene_Base.prototype.update;
  Scene_Base.prototype.update = function() {
    _Scene_Base_update.call(this);

    if (!this._kchHelpWindow || !this._kchSceneSettingsList) return;

    if ($gameSystem && $gameSystem._kchHelpOverlayVisible === false) {
      this._kchHelpWindow.close();
      return;
    }

    let needEvaluate = false;

    if (this._kchNeedInitialUpdate) {
      this._kchNeedInitialUpdate = false;
      needEvaluate = true;
    } else {
      this._kchHelpTicker = (this._kchHelpTicker + 1) % UPDATE_INTERVAL;
      if (this._kchHelpTicker === 0) {
        if (this._kchWatchVarIds && this._kchWatchVarIds.length > 0) {
          if (isAnyWatchedVarDirty(this._kchWatchVarIds)) {
            needEvaluate = true;
          } else {
            needEvaluate = false;
          }
        } else {
          needEvaluate = true;
        }
      }
    }

    if (needEvaluate) {
      const setting = selectSceneSetting(this, this._kchSceneSettingsList);

      if (setting !== this._kchCurrentSceneSetting) {
        this._kchCurrentSceneSetting = setting || null;
        this._kchHelpEntries = setting ? setting.Entries : null;
        this._kchHelpWindow.setEntry(null); 
      }

      if (!this._kchHelpEntries || this._kchHelpEntries.length === 0) {
        this._kchHelpWindow.setEntry(null);
      } else {
        const entry = this._kchHelpEntries.find(e => {
          const v = $gameVariables.value(e.VariableId);
          return String(v) === String(e.Equals);
        });
        this._kchHelpWindow.setEntry(entry || null);
      }

      if (this._kchWatchVarIds && this._kchWatchVarIds.length > 0) {
        markAllWatchedVarsClean(this._kchWatchVarIds);
      }
    }

    if (this._kchHelpWindow && this._kchHelpWindow.parent) {
      const parent = this._kchHelpWindow.parent;
      if (ALWAYS_ON_TOP) {
        if (parent.children[parent.children.length - 1] !== this._kchHelpWindow) {
          parent.removeChild(this._kchHelpWindow);
          parent.addChild(this._kchHelpWindow);
        }
      } else {
        if (OVER_MESSAGE) {
          let hasMessage = false;
          for (let i = parent.children.length - 1; i >= 0; i--) {
            const c = parent.children[i];
            if (c instanceof Window_Message) {
              hasMessage = true;
              break;
            }
          }
          if (hasMessage && parent.children[parent.children.length - 1] !== this._kchHelpWindow) {
            parent.removeChild(this._kchHelpWindow);
            parent.addChild(this._kchHelpWindow);
          }
        } else {
          if (parent.children[parent.children.length - 1] !== this._kchHelpWindow) {
            parent.removeChild(this._kchHelpWindow);
            parent.addChild(this._kchHelpWindow);
          }
        }
      }
    }
  };

  const _Scene_Base_createMessageWindow = Scene_Base.prototype.createMessageWindow;
  Scene_Base.prototype.createMessageWindow = function() {
    _Scene_Base_createMessageWindow.call(this);
    if (!this._kchHelpWindow) return;

    if (ALWAYS_ON_TOP) return;

    if (!OVER_MESSAGE) return;
    const w = this._kchHelpWindow;
    if (w.parent) w.parent.removeChild(w);
    this.addWindow(w);
  };

   // 背景（ウィンドウスキン）を一切描画しない安全化
  const _KCH_WindowHelp_initialize = Window_KCH_Help.prototype.initialize;
  Window_KCH_Help.prototype.initialize = function(rect) {
    _KCH_WindowHelp_initialize.call(this, rect);
    this._kchDisableWindowSkinRender();
  };

  // windowskin を差し替えても背景描画はさせない（安全装置）
  const _KCH_WindowHelp_updateWindowskin = Window_KCH_Help.prototype._updateWindowskin;
  Window_KCH_Help.prototype._updateWindowskin = function(entry) {
    _KCH_WindowHelp_updateWindowskin.call(this, entry);
    this._kchDisableWindowSkinRender();
  };

  Window_KCH_Help.prototype._kchDisableWindowSkinRender = function() {
    // Window_Base が内部で持つ枠/背景スプライトを描画させない
    if (this._backSprite) {
      this._backSprite.visible = false;
      this._backSprite.renderable = false;
    }
    if (this._frameSprite) {
      this._frameSprite.visible = false;
      this._frameSprite.renderable = false;
    }
    // ついでにウィンドウ本体の不透明度もゼロに寄せる（内容だけ表示）
    this.opacity = 0;
    this.backOpacity = 0;
  };

})();
