/*:
 * @param MasterSwitchId
 * @text 統括スイッチID（ONで表示有効）
 * @type switch
 * @default 0
 *
 * @param ShowOnlyWhenMessage
 * @text メッセージ表示中のみ有効
 * @type boolean
 * @default true
 *
 * @param HideWhenTransparent
 * @text 透明メッセ時は表示しない
 * @type boolean
 * @default false
 *
 * @param HideKey
 * @text 非表示切替キー
 * @type select
 * @option なし @value none
 * @option OK(決定) @value ok
 * @option キャンセル @value cancel
 * @option Shift @value shift
 * @option Control @value control
 * @option Tab @value tab
 * @option PageUp @value pageup
 * @option PageDown @value pagedown
 * @default shift
 * @desc 押すたびに表示/非表示をトグル（全項目一括）。メッセージウィンドウが開いている時のみ有効。
 *       右クリック（TouchInput.cancel）は常にトグルとして有効。
 *
 * @param GlobalPositionMode
 * @text 座標基準（全体）
 * @type select
 * @option 画面基準 @value screen
 * @option メッセージウィンドウ基準 @value window
 * @default screen
 * @desc screen: 画面左上を原点に絶対XY。window: 各項目の「座標基準」に従い、メッセージウィンドウ相対も使用可。
 *
 * @param Items
 * @text 表示項目リスト
 * @type struct<CondItem>[]
 * @default []
 *
 */

/*~struct~CondItem:
 * @param variableId
 * @text 変数ID
 * @type variable
 * @default 0
 *
 * @param matchType
 * @text 条件タイプ
 * @type select
 * @option 文字列＝ @value eqs
 * @option 文字列≠ @value neqs
 * @option 部分一致（含む） @value contains
 * @option 数値＝ @value eqn
 * @option 数値≧ @value gte
 * @option 数値≦ @value lte
 * @option 正規表現 @value regex
 * @default eqs
 *
 * @param value
 * @text 比較値（文字列または数値）
 * @type string
 * @default Xbox
 *
 * @param caseSensitive
 * @text 大文字小文字を区別（文字列系）
 * @type boolean
 * @default false
 *
 * @param text
 * @text 表示テキスト（制御文字OK）
 * @type note
 * @default "\\I[1]ウィンドウ消去  /  \\I[2]オート"
 *
 * @param positionMode
 * @text 座標基準（各項目）
 * @type select
 * @option 画面基準 @value screen
 * @option メッセージウィンドウ基準 @value window
 * @default window
 * @desc 「座標基準（全体）」が window のときのみ参照されます。screen のときは無視されます。
 *
 * @param x
 * @text X座標
 * @type number
 * @default 0
 *
 * @param y
 * @text Y座標
 * @type number
 * @default -36
 *
 * @param width
 * @text 幅（0で自動）
 * @type number
 * @min 0
 * @default 0
 *
 * @param zTop
 * @text 最前面に表示
 * @type boolean
 * @default true
 */

(() => {
"use strict";

const PN = "kamichichi_HelpOverlayVar_MessageWindow";
const P = PluginManager.parameters(PN);

const MasterSwitchId      = Number(P["MasterSwitchId"] || 0);
const ShowOnlyWhenMessage = P["ShowOnlyWhenMessage"] === "true";
const HideWhenTransparent = P["HideWhenTransparent"] === "true";
const HideKey             = String(P["HideKey"] || "none");
const GlobalPositionMode  = String(P["GlobalPositionMode"] || "screen");

window.KZ_MSGOVERLAY_HIDDEN = window.KZ_MSGOVERLAY_HIDDEN || false;

function parseItems() {
    const raw = P["Items"] || "[]";
    let arr = [];
    try { arr = JSON.parse(raw); } catch(_) { arr = []; }
    const out = [];
    for (const it of arr) {
        let o = {};
        try { o = (typeof it === "string") ? JSON.parse(it) : it; } catch(_) {}
        const variableId    = Number(o.variableId || 0);
        const matchType     = String(o.matchType || "eqs");
        const value         = String(o.value ?? "");
        const caseSensitive = String(o.caseSensitive || "false") === "true";

        let textRaw = o.text ?? "";
        let textParsed;
        try { textParsed = JSON.parse(textRaw); } catch (e) { textParsed = String(textRaw); }
        textParsed = textParsed.replace(/\\\\n/g, "\\n");
        textParsed = textParsed.replace(/\\\\/g, "\\"); // \\\\ → \\ → \

        const positionMode = String(o.positionMode || "window");
        const x            = Number(o.x || 0);
        const y            = Number(o.y || 0);
        const width        = Number(o.width || 0);
        const zTop         = String(o.zTop || "true") === "true";

        out.push({ variableId, matchType, value, caseSensitive, text: textParsed, positionMode, x, y, width, zTop });
    }
    return out;
}

const KZ_ITEMS = parseItems();

class Window_KZ_CondText extends Window_Base {
    initialize(rect, item) {
        super.initialize(rect);
        this._item = item;
        this.opacity = 0;
        this.contentsOpacity = 255;
        this.backOpacity = 0;
        this._visibleByCond = false;
        this._lastW = 0;
        this._lastH = 0;
        this.refresh();
    }

    shouldShow() {
        const it = this._item;
        if (!it || it.variableId <= 0) return false;
        const v = $gameVariables.value(it.variableId);
        const t = it.matchType;
        const arg = it.value;
        try {
            switch (t) {
                case "eqs": {
                    const a = String(v ?? "");
                    const b = String(arg ?? "");
                    return it.caseSensitive ? a === b : a.toLowerCase() === b.toLowerCase();
                }
                case "neqs": {
                    const a = String(v ?? "");
                    const b = String(arg ?? "");
                    return it.caseSensitive ? a !== b : a.toLowerCase() !== b.toLowerCase();
                }
                case "contains": {
                    const a = String(v ?? "");
                    const b = String(arg ?? "");
                    return it.caseSensitive ? a.includes(b) : a.toLowerCase().includes(b.toLowerCase());
                }
                case "eqn": return Number(v) === Number(arg);
                case "gte": return Number(v) >= Number(arg);
                case "lte": return Number(v) <= Number(arg);
                case "regex": {
                    const flags = it.caseSensitive ? "" : "i";
                    const re = new RegExp(String(arg || ""), flags);
                    return re.test(String(v ?? ""));
                }
                default: return false;
            }
        } catch(_) { return false; }
    }

    updatePosition() {
        const it = this._item;
        if (GlobalPositionMode === "screen") {
            this.x = it.x;
            this.y = it.y;
        } else {
            const scene = SceneManager._scene;
            const w = scene && scene._messageWindow;
            if (it.positionMode === "window" && w) {
                this.x = (w.x || 0) + it.x;
                this.y = (w.y || 0) + it.y;
            } else {
                this.x = it.x;
                this.y = it.y;
            }
        }
        if (it.zTop) this.z = 9999;
    }

    autoWidth() {
        const it = this._item;
        if (it.width > 0) return it.width;
        this.resetFontSettings();
        const size = this.textSizeEx(it.text);
        const cw = size && size.width ? size.width : this.contentsWidth();
        return Math.ceil(cw) + this.padding * 2;
    }

    refresh() {
        this.updatePosition();
        const iconSet = ImageManager.loadSystem("IconSet");
        if (!iconSet.isReady()) {
            iconSet.addLoadListener(() => this.refresh());
        }
        const size = this.textSizeEx(this._item.text);
        const w = Math.max(4, this.autoWidth());
        const hContent = size && size.height ? Math.ceil(size.height) : this.lineHeight();
        const h = hContent + this.padding * 2;

        const sizeChanged = (w !== this._lastW) || (h !== this._lastH);
        this.move(this.x, this.y, w, h);
        if (sizeChanged || !this.contents) {
            this.createContents();
            this._lastW = w;
            this._lastH = h;
        }
        this.resetFontSettings();
        this.contents.clear();
        this.drawTextEx(this._item.text, 0, 0, w - this.padding * 2);
    }

    update() {
        super.update();
        const scene  = SceneManager._scene;
        const msgWin = scene && scene._messageWindow;

        const visibleByWindow = !ShowOnlyWhenMessage || (msgWin && msgWin.isOpen && msgWin.isOpen() && !msgWin.isClosing());
        const transparentOk   = !HideWhenTransparent || (msgWin && msgWin._background !== 2);
        const masterOk        = (MasterSwitchId === 0) || ($gameSwitches && $gameSwitches.value(MasterSwitchId));

        const vis = masterOk && !window.KZ_MSGOVERLAY_HIDDEN && visibleByWindow && transparentOk && this.shouldShow();

        if (vis !== this._visibleByCond) {
            this._visibleByCond = vis;
            this.visible = vis;
            if (vis) this.refresh();
        }
        if (vis) this.updatePosition();
    }
}


const _Scene_Base_createMessageWindow = Scene_Base.prototype.createMessageWindow;
Scene_Base.prototype.createMessageWindow = function() {
    _Scene_Base_createMessageWindow.call(this);

    const msgWin = this._messageWindow;
    if (!msgWin) return;
    if (!KZ_ITEMS.length) return;
    if (typeof $gameSystem === "undefined" || !$gameSystem || typeof $gameMessage === "undefined" || !$gameMessage) {
        return;
    }

    this._kzCondWindows = [];
    for (const it of KZ_ITEMS) {
        const rect = new Rectangle(0, 0, 100, 36);
        const win  = new Window_KZ_CondText(rect, it);
        win.visible = false;
        this._kzCondWindows.push(win);

        const parent = msgWin.parent || this._windowLayer || this;
        parent.addChild(win);
        if (it.zTop && parent.children[parent.children.length - 1] !== win) {
            parent.removeChild(win);
            parent.addChild(win);
        }
    }
};

const _Scene_Base_update = Scene_Base.prototype.update;
Scene_Base.prototype.update = function() {
    _Scene_Base_update.call(this);

    const msgWin = this._messageWindow;
    if (!this._kzCondWindows || !this._kzCondWindows.length || !msgWin) return;

    for (const w of this._kzCondWindows) {
        if (w && w.update) w.update();
    }

    const isMsgOpen = msgWin.isOpen && msgWin.isOpen() && !msgWin.isClosing();
    if (isMsgOpen && (triggerHideKey() || TouchInput.isCancelled())) {
        window.KZ_MSGOVERLAY_HIDDEN = !window.KZ_MSGOVERLAY_HIDDEN;
        for (const w of this._kzCondWindows) {
            if (w && w.update) w.update();
        }
    }
};

function triggerHideKey() {
    switch (HideKey) {
        case "ok":       return Input.isTriggered("ok");
        case "cancel":   return Input.isTriggered("cancel");
        case "shift":    return Input.isTriggered("shift");
        case "control":  return Input.isTriggered("control");
        case "tab":      return Input.isTriggered("tab");
        case "pageup":   return Input.isTriggered("pageup");
        case "pagedown": return Input.isTriggered("pagedown");
        default:         return false;
    }
}

})();
