/*:
 * @command startQTE
 * @text QTE開始
 * @desc 現在のシーンで即座にQTEを開始します。
 *
 * @arg keys
 * @text 判定キー
 * @desc 判定するキーをカンマ区切りで（例：ok,escape,w,a,s,d）
 * @default ok,escape
 *
 * @arg duration
 * @text 制限時間（フレーム）
 * @desc 制限時間（60フレーム=1秒）
 * @type number
 * @default 120
 *
 * @arg successVar
 * @text 成功時変数ID
 * @type variable
 * @default 1
 *
 * @arg successValue
 * @text 成功時格納値
 * @type number
 * @default 1
 *
 * @arg failVar
 * @text 失敗時変数ID
 * @type variable
 * @default 2
 *
 * @arg failValue
 * @text 失敗時格納値
 * @type number
 * @default 0
 *
 * @arg successSE
 * @text 成功時SE
 * @type file
 * @dir audio/se
 * @default Decision1
 *
 * @arg successVolume
 * @text 成功時SE音量
 * @type number
 * @default 90
 *
 * @arg successPitch
 * @text 成功時SEピッチ
 * @type number
 * @default 100
 *
 * @arg failSE
 * @text 失敗時SE
 * @type file
 * @dir audio/se
 * @default Buzzer1
 *
 * @arg failVolume
 * @text 失敗時SE音量
 * @type number
 * @default 90
 *
 * @arg failPitch
 * @text 失敗時SEピッチ
 * @type number
 * @default 100
 *
 * @param GaugeWidth
 * @text ゲージの横幅
 * @type number
 * @default 400
 * @desc カウントダウンゲージ全体の横幅（ピクセル）
 *
 * @param GaugeHeight
 * @text ゲージの高さ（太さ）
 * @type number
 * @default 26
 * @desc カウントダウンゲージ全体の高さ（ピクセル）
 *
 * @param GaugePadding
 * @text 内側余白
 * @type number
 * @default 4
 * @desc 外枠から内側の白バックまでの余白（ピクセル）
 * 
 * @param GaugeY
 * @text ゲージ表示Y座標
 * @type number
 * @default 120
 * @desc ゲージの縦位置。画面中央からのオフセット値。
 * 
 * @param ShowCircleGauge
 * @text 円ゲージ表示
 * @type boolean
 * @on 表示する
 * @off 表示しない
 * @default true
 * @desc ゲージ上にある黒い円の表示・非表示を切り替えます。
 *
 * @param QtePictureEnabled
 * @text QTEピクチャを表示する
 * @type boolean
 * @on 表示する
 * @off 表示しない
 * @default false
 * @desc QTE開始時にピクチャを表示するかどうか
 *
 * @param QtePictureName
 * @text QTEピクチャ名
 * @type file
 * @dir img/pictures
 * @default 
 * @desc 表示したいピクチャ（img/pictures）のファイル名（拡張子なし）
 *
 * @param QtePictureX
 * @text QTEピクチャX座標
 * @type number
 * @default 0
 * @desc QTEピクチャの表示X座標
 *
 * @param QtePictureY
 * @text QTEピクチャY座標
 * @type number
 * @default 0
 * @desc QTEピクチャの表示Y座標
 */

(() => {
    const pluginName = "kamichichi_QTE";

    const parameters   = PluginManager.parameters(pluginName);
    const gaugeWidth   = Number(parameters["GaugeWidth"]   || 400);
    const gaugeHeight  = Number(parameters["GaugeHeight"]  || 26);
    const gaugePadding = Number(parameters["GaugePadding"] || 4);
    const gaugeY       = Number(parameters["GaugeY"]       || 120);
    const showCircleGauge = parameters["ShowCircleGauge"] === "true";

    const qtePictureEnabled = parameters["QtePictureEnabled"] === "true";
    const qtePictureName    = String(parameters["QtePictureName"] || "");
    const qtePictureX       = Number(parameters["QtePictureX"] || 0);
    const qtePictureY       = Number(parameters["QtePictureY"] || 0);

    const KEY_CODES = {
        ok:13, escape:27, shift:16, pageup:33, pagedown:34, left:37, up:38, right:39, down:40,
        a:65,b:66,c:67,x:88,y:89,z:90,control:17,tab:9,w:87,s:83,d:68,space:32,enter:13,q:81,
        e:69,r:82,f:70,v:86,'0':48,'1':49,'2':50,'3':51,'4':52,'5':53,'6':54,'7':55,'8':56,'9':57
    };

    function drawRoundRectPath(ctx, x, y, width, height, radius) {
        const r = Math.min(radius, height / 2, width / 2);
        ctx.beginPath();
        ctx.moveTo(x + r, y);
        ctx.lineTo(x + width - r, y);
        ctx.quadraticCurveTo(x + width, y, x + width, y + r);
        ctx.lineTo(x + width, y + height - r);
        ctx.quadraticCurveTo(x + width, y + height, x + width - r, y + height);
        ctx.lineTo(x + r, y + height);
        ctx.quadraticCurveTo(x, y + height, x, y + height - r);
        ctx.lineTo(x, y + r);
        ctx.quadraticCurveTo(x, y, x + r, y);
        ctx.closePath();
    }

    let QTEManager = {
        duration:0, maxDuration:0, keys:[],
        successVar:0, successValue:0, failVar:0, failValue:0,
        successSE:{}, failSE:{}, _result:null,
        frameSprite:null,
        circleSprite:null,
        pictureSprite:null,

        setup(args){
            this.keys = args.keys.split(",").map(k => k.trim().toLowerCase());
            this.duration = this.maxDuration = Number(args.duration);
            this.successVar   = Number(args.successVar);
            this.successValue = Number(args.successValue);
            this.failVar      = Number(args.failVar);
            this.failValue    = Number(args.failValue);
            this.successSE = {
                name:   args.successSE,
                volume: Number(args.successVolume),
                pitch:  Number(args.successPitch)
            };
            this.failSE = {
                name:   args.failSE,
                volume: Number(args.failVolume),
                pitch:  Number(args.failPitch)
            };
            this._result = null;
            this.createGauge();
        },

        update(){
            if (this.duration > 0) {
                this.duration--;
                this.updateGauge();
                if (this.checkCorrectInput()) this.onSuccess();
                else if (this.checkIncorrectInput()) this.onFail();
            } else if (this._result === null) {
                this.onFail();
            }
        },

        isFinished(){ 
            return this._result !== null; 
        },

        checkCorrectInput(){
            return this.keys.some(key =>
                Input._pressedKeys[KEY_CODES[key]] || Input.isTriggered(key)
            );
        },

        checkIncorrectInput(){
            const allGamepadKeys = [
                "ok","cancel","menu","escape","shift","pageup","pagedown",
                "left","right","up","down","a","b","c","x","y","z",
                "control","tab","w","s","d","space","enter","q","e","r","f","v",
                "0","1","2","3","4","5","6","7","8","9"
            ];
            return allGamepadKeys
                .filter(k => !this.keys.includes(k))
                .some(k => Input._pressedKeys[KEY_CODES[k]] || Input.isTriggered(k));
        },

        onSuccess(){
            $gameVariables.setValue(this.successVar, this.successValue);
            AudioManager.playSe(this.successSE);
            this._result = "success";
            this.removeGauge();
        },

        onFail(){
            $gameVariables.setValue(this.failVar, this.failValue);
            AudioManager.playSe(this.failSE);
            this._result = "fail";
            this.removeGauge();
        },

        createGauge(){
            const scene = SceneManager._scene;

            if (qtePictureEnabled && qtePictureName) {
                const bitmap = ImageManager.loadPicture(qtePictureName);
                this.pictureSprite = new Sprite(bitmap);
                this.pictureSprite.x = qtePictureX;
                this.pictureSprite.y = qtePictureY;
                scene.addChild(this.pictureSprite);
            } else {
                this.pictureSprite = null;
            }

            this.frameSprite = new Sprite(new Bitmap(gaugeWidth, gaugeHeight));
            this.frameSprite.x = (Graphics.width  - gaugeWidth)  / 2;
            this.frameSprite.y = (Graphics.height - gaugeHeight) / 2 + gaugeY;
            scene.addChild(this.frameSprite);

            if (showCircleGauge) {
                this.circleSprite = new Sprite(new Bitmap(200,200));
                this.circleSprite.x = Graphics.width  / 2 - 100;
                this.circleSprite.y = this.frameSprite.y - 200;
                scene.addChild(this.circleSprite);
            } else {
                this.circleSprite = null;
            }
        },

        updateGauge(){
            const rate = this.duration / this.maxDuration;

            if (showCircleGauge && this.circleSprite && this.circleSprite.bitmap) {
                this.circleSprite.bitmap.clear();
                this.circleSprite.bitmap.drawCircle(
                    100, 100, 90, "rgba(255,255,255,0.5)"
                );
                this.circleSprite.bitmap.drawCircle(
                    100, 100, 90 * rate, "rgba(0,0,0,0.9)"
                );
            }

            const bmp = this.frameSprite.bitmap;
            const ctx = bmp.context;
            const w   = bmp.width;
            const h   = bmp.height;

            bmp.clear();

            const outerRadius = h / 2;
            const innerX      = gaugePadding;
            const innerY      = gaugePadding;
            const innerW      = w - gaugePadding * 2;
            const innerH      = h - gaugePadding * 2;
            const innerRadius = innerH / 2;
            const fillW       = innerW * rate;

            ctx.fillStyle = "#5a2e1f";
            drawRoundRectPath(ctx, 0, 0, w, h, outerRadius);
            ctx.fill();

            ctx.fillStyle = "#ffffff";
            drawRoundRectPath(ctx, innerX, innerY, innerW, innerH, innerRadius);
            ctx.fill();

            if (fillW > 0) {
                ctx.save();
                ctx.beginPath();
                ctx.rect(innerX, innerY, fillW, innerH);
                ctx.clip();

                ctx.fillStyle = "#f4b000"; 
                drawRoundRectPath(ctx, innerX, innerY, innerW, innerH, innerRadius);
                ctx.fill();

                ctx.restore();
            }

            bmp._baseTexture.update();
        },

        removeGauge(){
            const scene = SceneManager._scene;

            if (showCircleGauge && this.circleSprite) {
                scene.removeChild(this.circleSprite);
                this.circleSprite = null;
            }

            if (this.frameSprite) {
                scene.removeChild(this.frameSprite);
                this.frameSprite = null;
            }

            if (this.pictureSprite) {
                scene.removeChild(this.pictureSprite);
                this.pictureSprite = null;
            }
        }
    };

    PluginManager.registerCommand(pluginName, "startQTE", function(args){
        QTEManager.setup(args);
        this.setWaitMode("qte");
    });

    const _Interpreter_wait = Game_Interpreter.prototype.updateWaitMode;
    Game_Interpreter.prototype.updateWaitMode = function(){
        if (this._waitMode === "qte") {
            QTEManager.update();
            if (QTEManager.isFinished()) {
                this._waitMode = "";
                return false;
            }
            return true;
        }
        return _Interpreter_wait.call(this);
    };

    Input._pressedKeys = Input._pressedKeys || {};
    document.addEventListener("keydown", e => { Input._pressedKeys[e.keyCode] = true; });
    document.addEventListener("keyup",   e => { Input._pressedKeys[e.keyCode] = false; });
})();
