/*:
 * @param titleText
 * @text タイトル文字列
 * @type string
 * @default テキストログ
 *
 * @param titleFontSize
 * @text タイトル文字サイズ
 * @type number
 * @min 6
 * @default 28
 *
 * @param titleFontName
 * @text タイトル用フォント名（FontLoadのname）
 * @type string
 * @default
 *
 * @param titleX
 * @text タイトルX
 * @type number
 * @min 0
 * @default 18
 *
 * @param titleY
 * @text タイトルY
 * @type number
 * @min 0
 * @default 12
 *
 * @param helpText
 * @text ヘルプ文字列（制御文字可）
 * @type string
 * @default \\I[87] ←/→/決定で操作　キャンセルで閉じる
 *
 * @param helpFontSize
 * @text ヘルプ文字サイズ
 * @type number
 * @min 6
 * @default 20
 *
 * @param helpFontName
 * @text ヘルプ用フォント名（FontLoadのname）
 * @type string
 * @default
 *
 * @param helpX
 * @text ヘルプX
 * @type number
 * @min 0
 * @default 18
 *
 * @param helpY
 * @text ヘルプY（0で下端自動）
 * @type number
 * @min 0
 * @default 0
 *
 */
(() => {
  'use strict';

  const PN = document.currentScript?.src?.split('/').pop().replace(/\.js$/,'') || 'kamichichi_TextLogLabels';
  const P  = PluginManager.parameters(PN);

  const TITLE_TEXT = String(P.titleText ?? 'テキストログ');
  const TITLE_FS   = Number(P.titleFontSize ?? 28);
  const TITLE_FN   = String(P.titleFontName ?? '');
  const TITLE_X    = Number(P.titleX ?? 18);
  const TITLE_Y    = Number(P.titleY ?? 12);

  const HELP_TEXT  = String(P.helpText ?? '\\I[87] ←/→/決定で操作　キャンセルで閉じる');
  const HELP_FS    = Number(P.helpFontSize ?? 20);
  const HELP_FN    = String(P.helpFontName ?? '');
  const HELP_X     = Number(P.helpX ?? 18);
  const HELP_Y_P   = Number(P.helpY ?? 0);

  const SAFETY = 6;
  const MARGIN_Y = 12;

  function isTextLogScene(scene) {
    return scene && scene.constructor && scene.constructor.name === 'Scene_TextLog';
  }

  class Window_TextLabelBase extends Window_Base {
    initialize(rect) {
      super.initialize(rect);
      this.openness = 255;
      this.opacity = 0;
      this.contentsBack.opacity = 0;
      this.backOpacity = 0;
      this.padding = 0;
      this._text = '';
      this._baseFontSize = this.contents.fontSize;
      this._fixedFontName = '';
    }
    setBase(text, baseFontSize, fixedFontName) {
      const t  = String(text || '');
      const fs = Number(baseFontSize || 20);
      const fn = String(fixedFontName || '');
      let dirty = false;
      if (this._text !== t) { this._text = t; dirty = true; }
      if (this._baseFontSize !== fs) { this._baseFontSize = fs; dirty = true; }
      if (this._fixedFontName !== fn) { this._fixedFontName = fn; dirty = true; }
      if (dirty) this.refresh();
    }
    applyFixedFont() {
      this.resetFontSettings();
      this.contents.fontSize = this._baseFontSize;
      this.contents.fontFace = this._fixedFontName ? this._fixedFontName : $gameSystem.mainFontFace();
    }
    ensureContents(w, h) {
      const W = Math.max(1, Math.floor(w));
      const H = Math.max(1, Math.floor(h));
      if (this.width !== W || this.height !== H) {
        this.move(this.x, this.y, W, H);
        this.createContents();
      } else {
        this.contents.clear();
      }
    }
    measuredBlockHeightByTextEx(raw) {
      const size = this.textSizeEx(String(raw || ''));
      const ow = Math.ceil(this.contents.outlineWidth || 0);
      return Math.ceil(size.height + ow + SAFETY);
    }
  }

  class Window_TitleLabel extends Window_TextLabelBase {
    refresh() {
      this.applyFixedFont();
      const needH = Math.max(this._baseFontSize + 10 + SAFETY, 24);
      this.ensureContents(this.width || 1, needH);
      this.contents.clear();
      this.drawText(String(this._text || ''), 0, 0, this.innerWidth, 'left');
    }
  }

  class Window_HelpLabel extends Window_TextLabelBase {
    refresh() {
      this.applyFixedFont();
      const needH = this.measuredBlockHeightByTextEx(this._text);
      this.ensureContents(this.width || 1, needH);

      if (HELP_Y_P === 0) {
        const y = Math.max(0, Graphics.boxHeight - this.height - MARGIN_Y);
        if (this.y !== y) this.y = y;
      }
      this.contents.clear();

      const _resetFontSettings = this.resetFontSettings.bind(this);
      this.resetFontSettings = () => {
        _resetFontSettings();
        this.contents.fontSize = this._baseFontSize;
        this.contents.fontFace = this._fixedFontName ? this._fixedFontName : $gameSystem.mainFontFace();
      };

      this.drawTextEx(String(this._text || ''), 0, 0);

      this.resetFontSettings = _resetFontSettings;
    }
    processEscapeCharacter(code, textState) {
      if (code === 'FN') {
        const font = this.obtainEscapeParam(textState);
        this.contents.fontFace = font ? font : ($gameSystem.mainFontFace());
        return;
      }
      super.processEscapeCharacter(code, textState);
    }
  }

  const _Scene_Base_start = Scene_Base.prototype.start;
  Scene_Base.prototype.start = function() {
    _Scene_Base_start.call(this);
    if (!isTextLogScene(this)) return;

    if (this._kchTitleWin) { this.removeChild(this._kchTitleWin); this._kchTitleWin.destroy({children:true}); }
    if (this._kchHelpWin)  { this.removeChild(this._kchHelpWin);  this._kchHelpWin.destroy({children:true}); }

    const titleRect = new Rectangle(TITLE_X, Math.max(0, TITLE_Y),
      Math.max(1, Graphics.boxWidth - TITLE_X), 1);
    const helpRect  = new Rectangle(HELP_X, 0,
      Math.max(1, Graphics.boxWidth - HELP_X), 1);

    const titleWin = new Window_TitleLabel(titleRect);
    const helpWin  = new Window_HelpLabel(helpRect);

    titleWin.setBase(TITLE_TEXT, TITLE_FS, TITLE_FN);
    helpWin.setBase(HELP_TEXT, HELP_FS, HELP_FN);
    if (HELP_Y_P > 0) helpWin.y = Math.max(0, HELP_Y_P);

    this.addChild(titleWin);
    this.addChild(helpWin);
    this._kchTitleWin = titleWin;
    this._kchHelpWin  = helpWin;

    titleWin.refresh();
    helpWin.refresh();
  };

  const _Scene_Base_onResize = Scene_Base.prototype.onResize;
  Scene_Base.prototype.onResize = function() {
    _Scene_Base_onResize.call(this);
    if (!isTextLogScene(this)) return;

    if (this._kchTitleWin) {
      this._kchTitleWin.move(
        TITLE_X, Math.max(0, TITLE_Y),
        Math.max(1, Graphics.boxWidth - TITLE_X),
        this._kchTitleWin.height
      );
      this._kchTitleWin.refresh();
    }
    if (this._kchHelpWin) {
      this._kchHelpWin.move(
        HELP_X, HELP_Y_P > 0 ? Math.max(0, HELP_Y_P) : this._kchHelpWin.y,
        Math.max(1, Graphics.boxWidth - HELP_X),
        this._kchHelpWin.height
      );
      this._kchHelpWin.refresh();
    }
  };

  const _Scene_Base_update = Scene_Base.prototype.update;
  Scene_Base.prototype.update = function() {
    _Scene_Base_update.call(this);
    if (!isTextLogScene(this)) return;
    if (this._kchTitleWin) { this.removeChild(this._kchTitleWin); this.addChild(this._kchTitleWin); }
    if (this._kchHelpWin)  { this.removeChild(this._kchHelpWin);  this.addChild(this._kchHelpWin); }
  };
})();
