/*~struct~KCHAudio:
 * @param path
 * @text パス（audio直下）
 * @type file
 * @dir audio/
 * @default
 *
 * @param volume
 * @text 音量
 * @type number
 * @min 0
 * @max 100
 * @default 90
 *
 * @param pitch
 * @text ピッチ
 * @type number
 * @min 50
 * @max 150
 * @default 100
 *
 * @param pan
 * @text パン
 * @type number
 * @min -100
 * @max 100
 * @default 0
 *
 * @param muteSwitchId
 * @text 再生しないスイッチ
 * @type switch
 * @default 0
 */

/*~struct~KCHVoicePack:
 * @param label
 * @text ラベル（声優名など）
 * @type string
 * @default
 *
 * @param packMuteSwitchId
 * @text パック停止スイッチ
 * @type switch
 * @default 0
 *
 * @param splashVoices
 * @text スプラッシュ用ボイス
 * @type struct<KCHAudio>[]
 * @default []
 *
 * @param titleVoices
 * @text タイトル用ボイス
 * @type struct<KCHAudio>[]
 * @default []
 */

/*:
 * @target MZ
 * @plugindesc SplashとTitleのボイスを同一パックから再生（構造体統合・競合耐性・voiceVolume対応） 
 * @author KAMICHICHI
 * 
 * @param splashForceWait
 * @text スプラッシュ強制ウェイト
 * @type boolean
 * @default true
 *
 * @param enable
 * @text 有効化
 * @type boolean
 * @default true
 *
 * @param logoCount
 * @text ロゴ枚数
 * @type number
 * @min 0
 * @default 2
 *
 * @param splashDuration
 * @text ロゴ表示時間（フレーム）
 * @type number
 * @min 1
 * @default 60
 *
 * @param splashFade
 * @text フェード速度
 * @type number
 * @min 1
 * @default 2
 *
 * @param fitScreen
 * @text 画面にフィット
 * @type boolean
 * @default true
 *
 * @param voicePacks
 * @text 声優パック一覧
 * @type struct<KCHVoicePack>[]
 * @default []
 *
 * @param titlePlayTiming
 * @text タイトル再生タイミング
 * @type select
 * @option start
 * @option commandShow
 * @default start
 *
 * @param stopTitleBgm
 * @text タイトルBGMを一時停止
 * @type boolean
 * @default false
 *
 * @param pickerMode
 * @text パック選択方式
 * @type select
 * @option random
 * @option first
 * @default random
 *
 * @help
 * 1つの声優パックを起動時に選択し、SplashとTitleで統一再生します。
 * 画像は img/titles2/Splash_0.png, Splash_1.png … を使用します。
 * 旧プラグイン（MOG_TitleSplashScreen / kamichichi_TitleVoice）は無効化してください。
 *
 * 注意：
 * Web配信では自動再生制限により無音になる場合があります。
 * その場合は titlePlayTiming=commandShow を推奨します。
 */

(() => {
  "use strict";

  function currentPluginName(defaultName) {
    try {
      const src = document.currentScript && document.currentScript.src || "";
      const m = src.match(/([^\/]+)\.js$/i);
      return (m && m[1]) || defaultName;
    } catch(_) { return defaultName; }
  }
  function readParams(candidates) {
    for (const name of candidates) {
      const p = PluginManager.parameters(name);
      for (const k in p) return p;
    }
    return {};
  }
  const DEFAULT_NAME = "KCH_TitleIntroUnified";
  const ACTUAL_NAME  = currentPluginName(DEFAULT_NAME);
  const P = readParams([ACTUAL_NAME, DEFAULT_NAME, "kamichichi_TitleIntroUnified"]);

  const N = (v,d=0)=>{ const t=Number(v); return Number.isFinite(t)?t:d; };
  const S = (v,d="")=> String(v ?? d);
  const B = (v,d=false)=> String(v ?? d) === "true";

  const SPLASH_FORCE_WAIT = B(P.splashForceWait, true);
  const ENABLE         = B(P.enable, true);
  const LOGO_N         = Math.max(N(P.logoCount, 2), 0);
  const SPLASH_DUR     = Math.max(N(P.splashDuration, 60), 1);
  const SPLASH_FADE    = Math.max(N(P.splashFade, 2), 1);
  const FIT            = B(P.fitScreen, true);
  const TITLE_TIMING   = S(P.titlePlayTiming, "start");
  const STOP_TITLE_BGM = B(P.stopTitleBgm, false);
  const PICKER_MODE    = S(P.pickerMode, "random");

  function parseStructArray(raw) {
    if (!raw) return [];
    let a; try { a = JSON.parse(raw); } catch { return []; }
    if (!Array.isArray(a)) return [];
    return a.map(e => { try { return JSON.parse(e); } catch { return {}; } });
  }
  function sanitizePath(s){
    return String(s||"").trim().replace(/^\/+|\/+$/g,"").replace(/\.(ogg|m4a|mp3|wav)$/i,"");
  }
  function isOn(swId){
    const id = N(swId,0);
    if (id<=0) return false;
    try { return !!($gameSwitches && $gameSwitches.value(id)); } catch { return false; }
  }
  function pickOne(arr){ return arr && arr.length ? arr[(Math.random()*arr.length)|0] : null; }

  const VOICE_PACKS = parseStructArray(P.voicePacks).map(v => {
    const splashVoices = parseStructArray(v.splashVoices).map(s => ({
      path: sanitizePath(s.path),
      volume: N(s.volume,90),
      pitch:  N(s.pitch,100),
      pan:    N(s.pan,0),
      muteSwitchId: N(s.muteSwitchId,0),
    }));
    const titleVoices = parseStructArray(v.titleVoices).map(s => ({
      path: sanitizePath(s.path),
      volume: N(s.volume,90),
      pitch:  N(s.pitch,100),
      pan:    N(s.pan,0),
      muteSwitchId: N(s.muteSwitchId,0),
    }));
    return {
      label: S(v.label,""),
      packMuteSwitchId: N(v.packMuteSwitchId,0),
      splashVoices, titleVoices
    };
  });

  // voiceVolume 取得（未定義なら100%）
  function voiceRate() {
    try {
      const vv = Number(ConfigManager && ConfigManager.voiceVolume);
      if (Number.isFinite(vv)) return Math.max(0, Math.min(100, vv)) / 100;
    } catch(_) {}
    return 1.0;
  }

  function masterRate() {
    try {
      const mv = Number(ConfigManager && ConfigManager.masterVolume);
      if (Number.isFinite(mv)) return Math.max(0, Math.min(100, mv)) / 100;
    } catch(_) {}
    return 1.0;
  }

  window.KCH_TitleIntro = window.KCH_TitleIntro || {};
  const Shared = window.KCH_TitleIntro;
  Shared.selectedPackIndex = -1;
  Shared._rawSplashWA = null;
  Shared._rawTitleWA  = null;

  function stopRaw(key){
    const wa = Shared[key];
    if (wa) { try{wa.stop();}catch{} Shared[key]=null; }
  }

  function playAudioEntry(entry, key){
    if (!entry || !entry.path) return;
    const path = sanitizePath(entry.path);
    if (!path) return;

    const vol = N(entry.volume,90);
    const pit = N(entry.pitch,100);
    const pan = N(entry.pan,0);
    const vr  = voiceRate();   
    const mr  = masterRate(); 

    if (/^(bgm|bgs|me|se)\//i.test(path)) {
      const [folder, ...rest] = path.split("/");
      const name = rest.join("/");
      if (!name) return;
      const v100 = Math.max(0, Math.min(100, Math.round(vol * vr * mr)));
      const snd = { name, volume: v100, pitch: pit, pan: pan };
      try {
        switch ((folder||"se").toLowerCase()) {
          case "bgm": AudioManager.playBgm(snd); break;
          case "bgs": AudioManager.playBgs(snd); break;
          case "me":  AudioManager.playMe(snd);  break;
          default:    AudioManager.playSe(snd);  break;
        }
      } catch(_) {}
      return;
    }

    try {
      const ext = AudioManager.audioFileExt();
      const url = "audio/" + path + ext;
      const wa  = new WebAudio(url);
      wa.volume = (vol/100) * vr * mr;
      wa.pitch  = pit/100;
      wa.pan    = pan/100;
      wa.play(false);
      Shared[key] = wa;
    } catch(_) {}
  }

  function valid(list){ return (list||[]).filter(a => !isOn(a.muteSwitchId)); }
  function choosePackIndex(){
    const c = VOICE_PACKS.filter(p => !isOn(p.packMuteSwitchId));
    if (!c.length) return -1;
    if (PICKER_MODE === "first") return VOICE_PACKS.indexOf(c[0]);
    const pick = pickOne(c); return VOICE_PACKS.indexOf(pick);
  }

  const _kch_boot_startNormalGame = Scene_Boot.prototype.startNormalGame;
  Scene_Boot.prototype.startNormalGame = function() {
    _kch_boot_startNormalGame.call(this);
    SceneManager.goto(Scene_KCH_Splash);
  };

  function Scene_KCH_Splash(){ this.initialize.apply(this, arguments); }
  Scene_KCH_Splash.prototype = Object.create(Scene_Base.prototype);
  Scene_KCH_Splash.prototype.constructor = Scene_KCH_Splash;

  Scene_KCH_Splash.prototype.initialize = function(){
    Scene_Base.prototype.initialize.call(this);
    this._imgs = [];
    this._sprite = null;
    this._state = { index:0, hold:SPLASH_DUR, fade:SPLASH_FADE, inited:false, played:false };
  };
  Scene_KCH_Splash.prototype.create = function(){
    Scene_Base.prototype.create.call(this);
    if (!ENABLE || LOGO_N <= 0) { this.gotoTitle(); return; }

    this._sprite = new Sprite();
    this._sprite.anchor.set(0.5, 0.5);
    this._sprite.x = Graphics.width / 2;
    this._sprite.y = Graphics.height / 2;
    this.addChild(this._sprite);
    // タイトル背景を先読み（黒表示の揺れ対策）
    this._preloadTitle1 = null;
    this._preloadTitle2 = null;
    try {
      const t1 = $dataSystem?.title1Name || "";
      const t2 = $dataSystem?.title2Name || "";
      if (t1) this._preloadTitle1 = ImageManager.loadTitle1(t1);
      if (t2) this._preloadTitle2 = ImageManager.loadTitle2(t2);
    } catch (_) {}

    for (let i=0;i<LOGO_N;i++) this._imgs.push(ImageManager.loadTitle2("Splash_"+i));
  };
  Scene_KCH_Splash.prototype.start = function(){
    Scene_Base.prototype.start.call(this);
    if (!ENABLE || LOGO_N <= 0) return;
    this.startFadeIn(this.fadeSpeed(), false);
  };
  Scene_KCH_Splash.prototype.update = function(){
    Scene_Base.prototype.update.call(this);
    if (!ENABLE || LOGO_N <= 0) return;

    if (!this._state.inited) {
      if (this._imgs.length===0 || (this._imgs[0] && this._imgs[0].isReady())) {
        this._state.inited = true;
        this._state.index = 0;
        this._state.hold = SPLASH_DUR;
        this._sprite.opacity = 0;
        this._sprite.bitmap = this._imgs[0] || null;
        if (FIT && this._sprite.bitmap) this.fitScreen();

        if (Shared.selectedPackIndex < 0) Shared.selectedPackIndex = choosePackIndex();
        this.playSplashOnce();
      }
      return;
    }

    if (this._state.index >= this._imgs.length) {
      // Splashが終わっても、タイトル背景の準備ができるまで待つ
      const t1Ready = !this._preloadTitle1 || this._preloadTitle1.isReady();
      const t2Ready = !this._preloadTitle2 || this._preloadTitle2.isReady();

      if (t1Ready && t2Ready) {
        stopRaw("_rawSplashWA");
        AudioManager.stopMe();
        this.gotoTitle();
      }
      return;
    }


    if (this._state.hold > 0) {
      this._sprite.opacity += this._state.fade;
      if (this._sprite.opacity >= 255) this._state.hold--;
      if (!SPLASH_FORCE_WAIT && (Input.isTriggered("ok") || TouchInput.isTriggered()) && this._sprite.opacity > 60) this._state.hold = 0;
    } else {
      this._sprite.opacity -= this._state.fade;
      if (!SPLASH_FORCE_WAIT && (Input.isTriggered("ok") || TouchInput.isTriggered())) this._state.index = this._imgs.length;
      if (this._sprite.opacity <= 0) this.nextLogo();
    }
  };

  Scene_KCH_Splash.prototype.fitScreen = function(){
    this._sprite.scale.set(1,1);
    const b = this._sprite.bitmap; if (!b) return;
    if (b.width  < Graphics.width)  this._sprite.scale.x = Graphics.width  / b.width;
    if (b.height < Graphics.height) this._sprite.scale.y = Graphics.height / b.height;
  };
  Scene_KCH_Splash.prototype.playSplashOnce = function(){
    if (this._state.played) return;
    const pidx = Shared.selectedPackIndex;
    const pack = pidx>=0 ? VOICE_PACKS[pidx] : null;
    const pick = pickOne(valid(pack ? pack.splashVoices : []));
    if (pick) { playAudioEntry(pick, "_rawSplashWA"); this._state.played = true; }
  };
  Scene_KCH_Splash.prototype.nextLogo = function(){
    this._state.index++;
    if (this._state.index >= this._imgs.length) return;
    this._sprite.bitmap = this._imgs[this._state.index];
    this._sprite.opacity = 0;
    this._state.hold = SPLASH_DUR;
    if (FIT && this._sprite.bitmap) this.fitScreen();
  };
  Scene_KCH_Splash.prototype.gotoTitle = function(){
    if ($dataSystem.startMapId === 0) throw new Error("Player's starting position is not set");
    this.checkPlayerLocation();
    DataManager.setupNewGame();
    SceneManager.goto(Scene_Title);
    Window_TitleCommand.initCommandPosition();
  };
  Scene_KCH_Splash.prototype.checkPlayerLocation = function(){
    if ($dataSystem.startMapId === 0) throw new Error("Player's starting position is not set");
  };

  const _Scene_Title_start = Scene_Title.prototype.start;
  Scene_Title.prototype.start = function(){
    _Scene_Title_start.call(this);
    if (!ENABLE) return;
    if (TITLE_TIMING === "start") this._kch_playTitleVoiceUnified();
  };
  const _Scene_Title_createCommandWindow = Scene_Title.prototype.createCommandWindow;
  Scene_Title.prototype.createCommandWindow = function(){
    _Scene_Title_createCommandWindow.call(this);
    if (!ENABLE) return;
    if (TITLE_TIMING === "commandShow") this._kch_playTitleVoiceUnified();
  };
  const _Scene_Title_terminate = Scene_Title.prototype.terminate;
  Scene_Title.prototype.terminate = function(){
    _Scene_Title_terminate.call(this);
    stopRaw("_rawTitleWA");
  };
  Scene_Title.prototype._kch_playTitleVoiceUnified = function(){
    const pidx = Shared.selectedPackIndex;
    const pack = pidx>=0 ? VOICE_PACKS[pidx] : null;
    if (!pack) return;
    const pick = pickOne(valid(pack.titleVoices));
    if (!pick) return;
    if (STOP_TITLE_BGM) AudioManager.stopBgm();
    playAudioEntry(pick, "_rawTitleWA");
  };

})();
