/*:
 * @param videos
 * @text 動画リスト
 * @type struct<Video>[]
 * @desc 再生可能な動画のリストを構造体で設定します。
 * @default []
 *
 * @command playVideo
 * @text 動画再生
 * @desc 指定された動画を再生します。
 *
 * @arg videoId
 * @text 動画ID
 * @type number
 * @desc 再生する動画のIDを指定します。
 * @default 1
 *
 * @command removeVideo
 * @text 動画削除
 * @desc 表示中の動画を削除します。
 */

/*~struct~Video:
 * @param id
 * @text 動画ID
 * @type number
 * @desc 動画のID。プラグインコマンドでこのIDを指定して再生します。
 *
 * @param fileName
 * @text ファイル名
 * @desc 再生する動画ファイル名（拡張子なしで指定します）。moviesフォルダにある動画を指定してください。
 *
 * @param loop
 * @text ループ再生
 * @type boolean
 * @desc ループ再生するかどうかを指定します。
 * @default false
 */

(() => {
    const parameters = PluginManager.parameters("kamichichi_VideoOverlayPlugin");
    const videos = JSON.parse(parameters["videos"] || "[]").map(video => JSON.parse(video));

    let videoElement;

function createVideoOverlay(videoData) {
    const el = document.createElement("video");
    videoElement = el;

    el.src = `movies/${videoData.fileName}.webm`;
    el.loop = videoData.loop === "true";
    el.autoplay = true;
    el.style.position = "fixed";
    el.style.top = "0";
    el.style.left = "0";
    el.style.width = "100%";
    el.style.height = "100%";
    el.style.zIndex = "100";
    el.muted = true;

    el.onloadedmetadata = () => {
        if (!el || el !== videoElement) return; // ← null / 別動画防止

        if (!el.loop) {
            const fps = 30;
            const framesBeforeEnd = 5;
            const timeBeforeEnd = framesBeforeEnd / fps;

            const duration = el.duration;
            const timeToRemove = duration - timeBeforeEnd;

            setTimeout(() => {
                if (el === videoElement) removeVideo();
            }, timeToRemove * 1000);
        }
    };

    document.body.appendChild(el);
    el.play().catch(() => {});
}

    function playVideo(videoId) {
        const videoData = videos.find(video => Number(video.id) === videoId);
        if (!videoData) return;

        if (videoElement) {
            removeVideo(); 
        }

        createVideoOverlay(videoData); 
        videoElement.play().catch(() => {});
syncPlaybackRateToRapid(); // 追加：生成直後に反映

    }

    function removeVideo() {
        if (videoElement) {
            videoElement.pause();
            document.body.removeChild(videoElement);
            videoElement = null;
        }
    }

    // 追加：RapidAccelerator の倍速状態に動画再生速度を同期
function syncPlaybackRateToRapid() {
    if (!videoElement) return;

    const ra = window.RapidAccelerator;
    if (ra && ra.isRapid) {
        // video の playbackRate は 0 以下や負数は不可。最低 0.1 に丸める
        const s = Number(ra.rapidSpeed) || 1;
        videoElement.playbackRate = Math.max(0.1, s);
    } else {
        videoElement.playbackRate = 1.0;
    }
}
// 追加：毎フレーム同期（既存の update は保持して呼ぶ）
if (!Scene_Base.prototype.__kchVideoOverlaySyncRatePatched) {
    Scene_Base.prototype.__kchVideoOverlaySyncRatePatched = true;

    const _Scene_Base_update = Scene_Base.prototype.update;
    Scene_Base.prototype.update = function() {
        _Scene_Base_update.call(this);
        syncPlaybackRateToRapid();
    };
}


    PluginManager.registerCommand("kamichichi_VideoOverlayPlugin", "playVideo", args => {
        playVideo(Number(args.videoId));
    });

    PluginManager.registerCommand("kamichichi_VideoOverlayPlugin", "removeVideo", () => {
        removeVideo();
    });
})();
