/*:
 * @param applyToPlayer
 * @text プレイヤーも対象
 * @type boolean
 * @default true
 *
 * @param applyToFollowers
 * @text 隊列メンバーも対象
 * @type boolean
 * @default false
 *
 * @param applyToEvents
 * @text イベントも対象
 * @type boolean
 * @default true
 *
 * @param regionBrushes
 * @text リージョンごとの設定
 * @type struct<RegionBrush>[]
 * @default []
 */
/*~struct~RegionBrush:
 * @param regionId
 * @text リージョンID
 * @type number
 * @min 1
 * @default 1
 *
 * @param brushPixels
 * @text 半透明にする高さ（px）
 * @type number
 * @min 1
 * @default 12
 *
 * @param targetOpacity
 * @text 目標不透明度（0〜255）
 * @type number
 * @min 0
 * @max 255
 * @default 128
 *
 * @param gradient
 * @text グラデーション
 * @type boolean
 * @default true
 */
(() => {
  'use strict';
  const PN = 'kamichichi_brush';
  const P = PluginManager.parameters(PN);
  const APPLY_PLAYER = P.applyToPlayer === 'true';
  const APPLY_FOLLOWERS = P.applyToFollowers === 'true';
  const APPLY_EVENTS = P.applyToEvents !== 'false';

  const parse = (s, d) => { try { return JSON.parse(s); } catch { return d; } };
  const REGION_SET = parse(P.regionBrushes || '[]', []).map(e => {
    const o = parse(e, {});
    return {
      regionId: Number(o.regionId || 0),
      brushPixels: Math.max(1, Number(o.brushPixels || 12)),
      targetOpacity: Math.max(0, Math.min(255, Number(o.targetOpacity ?? 128))),
      gradient: String(o.gradient) === 'true'
    };
  }).filter(e => e.regionId > 0);

  const frag = `
  varying vec2 vTextureCoord;
  uniform sampler2D uSampler;
  uniform float brushRatio;
  uniform float targetAlpha;
  uniform float gradFlag;
  void main(){
    vec4 c = texture2D(uSampler, vTextureCoord);
    float edge = 1.0 - brushRatio;
    float aMul = 1.0;
    if (brushRatio > 0.0){
      if (gradFlag >= 0.5){
        float t = clamp((vTextureCoord.y - edge) / max(brushRatio, 1e-5), 0.0, 1.0);
        aMul = mix(1.0, targetAlpha, t);
      } else {
        aMul = (vTextureCoord.y >= edge) ? targetAlpha : 1.0;
      }
    }
    c.rgb *= aMul;
    c.a   *= aMul;
    gl_FragColor = c;
  }`;

  function findRegionSetting(regionId){
    return REGION_SET.find(s => s.regionId === regionId);
  }

  function isFollower(ch){
    if (!$gamePlayer) return false;
    return $gamePlayer.followers().some(f => f === ch);
  }

  function isTarget(sprite){
    const ch = sprite._character;
    if (!ch) return false;
    if (APPLY_EVENTS && ch instanceof Game_Event) return true;
    if (APPLY_PLAYER && ch === $gamePlayer) return true;
    if (APPLY_FOLLOWERS && isFollower(ch)) return true;
    return false;
  }

  function currentRegion(sprite){
    const ch = sprite._character;
    return $gameMap.regionId(ch.x, ch.y);
  }

  function frameH(sprite){
    try { return sprite.patternHeight ? sprite.patternHeight() : sprite.height || 0; }
    catch { return sprite.height || 0; }
  }

  function ensureFilter(sprite){
    if (!sprite._kchBrush) sprite._kchBrush = {};
    if (!sprite._kchBrush.filter){
      sprite._kchBrush.filter = new PIXI.Filter(undefined, frag, {
        brushRatio: 0.0, targetAlpha: 1.0, gradFlag: 1.0
      });
    }
    return sprite._kchBrush.filter;
  }

  function hasFilter(sprite, f){
    if (!sprite.filters) return false;
    return sprite.filters.includes(f);
  }

  function addFilter(sprite, f){
    const list = (sprite.filters || []).slice();
    list.push(f);
    sprite.filters = list;
  }

  function removeFilter(sprite, f){
    if (!sprite.filters) return;
    sprite.filters = sprite.filters.filter(x => x !== f);
  }

  const _Sprite_Character_initialize = Sprite_Character.prototype.initialize;
  Sprite_Character.prototype.initialize = function(character){
    _Sprite_Character_initialize.apply(this, arguments);
    this._kchBrush = { filter: null, appliedRegion: 0 };
  };

  const _Sprite_Character_updateBush = Sprite_Character.prototype._updateBush;
  Sprite_Character.prototype._updateBush = function(){
    _Sprite_Character_updateBush.apply(this, arguments);
    const ch = this._character;
    if (!ch) return;
    if (!isTarget(this)) return;
    const setting = findRegionSetting(currentRegion(this));
    if (setting){
      this._bushDepth = 0;
    }
  };

  function applyBrush(sprite){
    if (!isTarget(sprite)) {
      if (sprite._kchBrush?.filter) removeFilter(sprite, sprite._kchBrush.filter);
      return;
    }
    const rid = currentRegion(sprite);
    const setting = findRegionSetting(rid);
    if (!setting){
      if (sprite._kchBrush?.filter) removeFilter(sprite, sprite._kchBrush.filter);
      return;
    }
    const h = frameH(sprite);
    if (h <= 0) return;

    const f = ensureFilter(sprite);
    f.uniforms.brushRatio = Math.min(1.0, Math.max(0.0, setting.brushPixels / h));
    f.uniforms.targetAlpha = Math.max(0.0, Math.min(1.0, setting.targetOpacity / 255));
    f.uniforms.gradFlag = setting.gradient ? 1.0 : 0.0;

    if (!hasFilter(sprite, f)) addFilter(sprite, f);
  }

  const _Sprite_Character_update = Sprite_Character.prototype.update;
  Sprite_Character.prototype.update = function(){
    _Sprite_Character_update.apply(this, arguments);
    applyBrush(this); 
  };

  

})();
