/*:
 * @param DigitCount
 * @text 桁数
 * @type number
 * @min 1
 * @max 10
 * @default 3
 *
 * @param PicturePrefix
 * @text 数字画像の接頭辞
 * @type string
 * @default num_
 *
 * @param PlusImageName
 * @text +画像名
 * @type string
 * @default num_plus
 *
 * @param DigitPitch
 * @text デフォルト桁間隔(px)
 * @type number
 * @default 36
 *
 * @param Origin
 * @text 原点
 * @type select
 * @option 左上
 * @value 0
 * @option 中央
 * @value 1
 * @default 0
 *
 * @param DefaultScale
 * @text デフォルト拡大率(%)
 * @type number
 * @min 1
 * @default 100
 *
 * @param Opacity
 * @text 不透明度
 * @type number
 * @default 255
 *
 * @param BlendMode
 * @text 合成方法
 * @type number
 * @default 0
 *
 * @param ZeroPad
 * @text ゼロ埋め
 * @type boolean
 * @default false
 *
 * @param AutoErase
 * @text 終了後に自動消去
 * @type boolean
 * @default true
 *
 * @param HoldFrames
 * @text 終了後保持フレーム
 * @type number
 * @default 30
 *
 * @command StartCountUp
 * @text カウントアップ開始
 *
 * @arg StartPictureId
 * @text 開始ピクチャID
 * @type number
 * @default 50
 *
 * @arg X
 * @text 表示X
 * @type number
 * @default 0
 *
 * @arg Y
 * @text 表示Y
 * @type number
 * @default 0
 *
 * @arg DurationFrames
 * @text 所要フレーム
 * @type number
 * @default 60
 *
 * @arg ScalePercent
 * @text 拡大率(%)
 * @type number
 * @default 0
 *
 * @arg DigitPitch
 * @text 桁の間隔(px / 0でデフォルト)
 * @type number
 * @default 0
 *
 * @arg ShowPlus
 * @text +を表示
 * @type boolean
 * @default true
 *
 * @arg TargetVarId
 * @text 目標値 変数ID
 * @type variable
 * @default 0
 *
 * @arg TargetText
 * @text 目標値 文字列
 * @type string
 * @default \V[1]
 */

(() => {
"use strict";

const PLUGIN_NAME = "kamichichi_countup_picture";
const params = PluginManager.parameters(PLUGIN_NAME);

const P_DIGIT     = Number(params.DigitCount || 3);
const P_PREFIX    = String(params.PicturePrefix || "num_");
const P_PLUS      = String(params.PlusImageName || "num_plus");
const P_DEF_PITCH = Number(params.DigitPitch || 36);
const P_ORIGIN    = Number(params.Origin || 0);
const P_DEF_SCALE = Number(params.DefaultScale || 100);
const P_OPACITY   = Number(params.Opacity || 255);
const P_BLEND     = Number(params.BlendMode || 0);
const P_ZEROPAD   = params.ZeroPad === "true";
const P_AUTOERASE = params.AutoErase === "true";
const P_HOLD      = Number(params.HoldFrames || 0);

function clamp(v,min,max){v=Number(v)||0;return Math.min(max,Math.max(min,v));}

function resolveToNumber(str){
    const rep = String(str).replace(/\\[Vv]\[(\d+)\]/g,(_,n)=>String($gameVariables.value(Number(n))));
    const num = Number(rep);
    return Number.isFinite(num)?num:0;
}

function makeDigits(v){
    const max = Math.pow(10,P_DIGIT)-1;
    v = clamp(v,0,max);
    const s = String(v);
    if(P_ZEROPAD) return s.padStart(P_DIGIT,"0").split("");

    const arr = new Array(P_DIGIT).fill("");
    const cs = s.split("");
    for(let i=0;i<cs.length;i++){
        arr[P_DIGIT-cs.length+i]=cs[i];
    }
    return arr;
}

function showPicture(id,name,x,y,scale){
    if(!$gameScreen.picture(id)){
        $gameScreen.showPicture(id,name,P_ORIGIN,x,y,scale,scale,P_OPACITY,P_BLEND);
    }else{
        const p=$gameScreen.picture(id);
        p._name=name;
        p._x=x; p._y=y;
        p._origin=P_ORIGIN;
        p._scaleX=scale; p._scaleY=scale;
        p._opacity=P_OPACITY; p._blendMode=P_BLEND;
    }
}

function preparePictures(e){
    const base = e.startId;
    let offset = 0;

    if(e.showPlus){
        showPicture(base,P_PLUS,e.x,e.y,e.scale);
        offset = 1;
    }

    for(let i=0;i<P_DIGIT;i++){
        const id = base+offset+i;
        const px = e.x + (offset+i)*e.pitch;
        showPicture(id,`${P_PREFIX}0`,px,e.y,e.scale);
    }
}

function applyDigits(e){
    const base=e.startId;
    const digits = makeDigits(e.value);
    let offset = e.showPlus?1:0;

    for(let i=0;i<P_DIGIT;i++){
        const id = base+offset+i;
        const pic = $gameScreen.picture(id);
        if(!pic) continue;

        const d = digits[i];
        if(d===""){
            pic._opacity = 0;
            continue;
        }
        pic._name = `${P_PREFIX}${d}`;
        pic._opacity = P_OPACITY;
    }
}

function createEntry(args){
    const x = Number(args.X)||0;
    const y = Number(args.Y)||0;

    const duration = clamp(args.DurationFrames,1,999999);
    const startId  = clamp(args.StartPictureId,1,9999);

    const scaleArg = Number(args.ScalePercent)||0;
    const scale = scaleArg>0?scaleArg:P_DEF_SCALE;

    const pitchArg = Number(args.DigitPitch)||0;
    const pitch = pitchArg>0 ? pitchArg : P_DEF_PITCH;

    const showPlus = args.ShowPlus === "true";

    const targetVar = clamp(args.TargetVarId,0,999999);
    const raw = targetVar>0 ? $gameVariables.value(targetVar) : String(args.TargetText||"");
    const target = resolveToNumber(raw);
    const max = Math.pow(10,P_DIGIT)-1;

    const clamped = clamp(target,0,max);
    const perFrame = clamped / duration;

    return {
        startId,
        x,y,
        duration,
        frame:0,

        current:0,
        value:0,
        target:clamped,
        perFrame,

        pitch,
        scale,
        showPlus,
        hold:0,
        done:false
    };
}

PluginManager.registerCommand(PLUGIN_NAME,"StartCountUp", args=>{
    const e = createEntry(args);

    if(!$gameScreen._kchCountUpPicList){
        $gameScreen._kchCountUpPicList = [];
    }
    preparePictures(e);
    applyDigits(e);
    $gameScreen._kchCountUpPicList.push(e);
});

const _Game_Screen_update = Game_Screen.prototype.update;
Game_Screen.prototype.update = function(){
    _Game_Screen_update.call(this);

    const list=this._kchCountUpPicList;
    if(!list) return;

    for(let i=list.length-1;i>=0;i--){
        const e=list[i];
        if(e.done){ list.splice(i,1); continue; }

        e.frame++;

        if(e.frame < e.duration){
            e.current += e.perFrame;
        } else {
            e.current = e.target;
        }

        e.value = Math.floor(e.current);

        applyDigits(e);

        if(e.value >= e.target){
            if(P_AUTOERASE){
                if(e.hold < P_HOLD){
                    e.hold++;
                } else {
                    const total = P_DIGIT + (e.showPlus?1:0);
                    for(let k=0;k<total;k++){
                        $gameScreen.erasePicture(e.startId+k);
                    }
                    e.done=true;
                }
            } else {
                e.done=true;
            }
        }
    }
};

const _Sprite_Picture_loadBitmap = Sprite_Picture.prototype.loadBitmap;
Sprite_Picture.prototype.loadBitmap = function(){
    _Sprite_Picture_loadBitmap.call(this);

    const pic = this.picture();
    if(!pic) return;

    const name = pic.name();
    if(!name) return;

    if(name.startsWith(P_PREFIX) || name===P_PLUS){
        if(this.bitmap){
            this.bitmap.smooth = true;
            if(this.bitmap.baseTexture){
                this.bitmap.baseTexture.scaleMode = PIXI.SCALE_MODES.LINEAR;
            }
        }
    }
};
})();
