/*:
 * @param Tracks
 * @text 計測設定リスト
 * @type struct<TrackSetting>[]
 * @default []
 * @desc 複数の計測設定をまとめて登録します。
 *
 * @command Start
 * @text 計測開始
 * @desc 登録されている全ての計測設定のスイッチをONにし、合計値を初期化します。
 *
 * @command Stop
 * @text 計測停止＆結果確定
 * @desc 登録されている全ての計測設定のスイッチをOFFにし、結果変数へ値を確定させます。
 */

/*~struct~TrackSetting:
 * @param Memo
 * @text メモ
 * @type string
 * @default
 * @desc この設定の用途メモ。わかりやすい名前などを入れてください。
 *
 * @param TrackSwitchId
 * @text 計測スイッチID
 * @type switch
 * @default 0
 * @desc このスイッチがONの間だけ差分を計測します。OFF→ONで開始、ON→OFFで終了とみなします。
 *
 * @param TargetVarIds
 * @text 対象変数IDリスト
 * @type variable[]
 * @default []
 * @desc 増減を監視する変数IDを複数指定できます。ここで指定した全ての変数の差分をこの設定の合計に加算します。
 *
 * @param SumVarId
 * @text 合計変数ID
 * @type variable
 * @default 0
 * @desc 差分を足し込んでいく合計用の変数IDです。
 *
 * @param ResultVarId
 * @text 結果変数ID
 * @type variable
 * @default 0
 * @desc 終了時に確定した結果を入れる変数IDです。
 *
 * @param IgnoreDecrease
 * @text 減少は無視
 * @type boolean
 * @on はい
 * @off いいえ
 * @default false
 * @desc trueなら差分がマイナスのときは0として扱います（増加分だけ合計します）。
 *
 * @param DivideValue
 * @text 割る数（0で割らない）
 * @type number
 * @min 0
 * @default 10
 * @desc 0なら割らない。1以上ならこの数で割って結果変数へ入れます。
 */

(() => {
  "use strict";
  const pluginName = document.currentScript.src.match(/([^/]+)\.js$/)[1];
  const p = PluginManager.parameters(pluginName);

  /**
   * @typedef {Object} TrackConfig
   * @property {string} memo
   * @property {number} trackSwitchId
   * @property {number[]} targetVarIds
   * @property {number} sumVarId
   * @property {number} resultVarId
   * @property {boolean} ignoreDecrease
   * @property {number} divideValue
   */

  /** @type {TrackConfig[]} */
  const tracks = JSON.parse(p.Tracks || "[]").map(str => {
    const obj = str ? JSON.parse(str) : {};
    const targetList = obj.TargetVarIds ? JSON.parse(obj.TargetVarIds) : [];
    return /** @type {TrackConfig} */({
      memo: String(obj.Memo || ""),
      trackSwitchId: Number(obj.TrackSwitchId || 0),
      targetVarIds: targetList.map(v => Number(v || 0)).filter(id => id > 0),
      sumVarId: Number(obj.SumVarId || 0),
      resultVarId: Number(obj.ResultVarId || 0),
      ignoreDecrease: String(obj.IgnoreDecrease || "false") === "true",
      divideValue: Number(obj.DivideValue || 0),
    });
  });

  function isTrackingConfig(cfg) {
    return cfg.trackSwitchId > 0 && $gameSwitches.value(cfg.trackSwitchId);
  }

  function ensureTrackerStore() {
    if (!$gameSystem._kchVarDeltaTrackPrev) {
      $gameSystem._kchVarDeltaTrackPrev = {};
    }
    return $gameSystem._kchVarDeltaTrackPrev;
  }

  function setPrevValue(varId, value) {
    const store = ensureTrackerStore();
    store[varId] = Number(value || 0);
  }

  function getPrevValue(varId, fallback) {
    const store = ensureTrackerStore();
    if (store[varId] === undefined) return Number(fallback || 0);
    return Number(store[varId] || 0);
  }

  PluginManager.registerCommand(pluginName, "Start", () => {
    for (const cfg of tracks) {
      if (cfg.trackSwitchId > 0) {
        $gameSwitches.setValue(cfg.trackSwitchId, true);
        if (cfg.sumVarId > 0) {
          $gameVariables.setValue(cfg.sumVarId, 0);
        }
        if (cfg.resultVarId > 0) {
          $gameVariables.setValue(cfg.resultVarId, 0);
        }
        for (const vid of cfg.targetVarIds) {
          if (vid > 0) {
            setPrevValue(vid, $gameVariables.value(vid));
          }
        }
      }
    }
  });

  PluginManager.registerCommand(pluginName, "Stop", () => {
    for (const cfg of tracks) {
      if (cfg.trackSwitchId > 0) {
        const sum = cfg.sumVarId > 0 ? $gameVariables.value(cfg.sumVarId) : 0;
        const dv = cfg.divideValue;
        const result = (dv && dv > 0) ? Math.floor(sum / dv) : sum;
        if (cfg.resultVarId > 0) {
          $gameVariables.setValue(cfg.resultVarId, result);
        }
        $gameSwitches.setValue(cfg.trackSwitchId, false);
      }
    }
  });

  const _Game_Variables_setValue = Game_Variables.prototype.setValue;
  let _inHook = false;

  Game_Variables.prototype.setValue = function(variableId, value) {
    if (_inHook) {
      _Game_Variables_setValue.call(this, variableId, value);
      return;
    }

    const hasTarget = tracks.some(cfg => cfg.targetVarIds.includes(variableId));

    const before = hasTarget ? this.value(variableId) : null;

    _Game_Variables_setValue.call(this, variableId, value);

    if (!hasTarget) return;

    const after = this.value(variableId);
    const prev = getPrevValue(variableId, before);
    const baseDelta = Number(after) - Number(prev);

    setPrevValue(variableId, after);

    if (baseDelta === 0) return;

    _inHook = true;
    try {
      for (const cfg of tracks) {
        if (!cfg.targetVarIds.includes(variableId)) continue;
        if (!isTrackingConfig(cfg)) continue;

        let delta = baseDelta;
        if (cfg.ignoreDecrease && delta < 0) {
          delta = 0;
        }

        if (cfg.sumVarId > 0) {
          const sum = this.value(cfg.sumVarId);
          this.setValue(cfg.sumVarId, sum + delta);
        }
      }
    } finally {
      _inHook = false;
    }
  };

  const _Game_Switches_setValue = Game_Switches.prototype.setValue;
  Game_Switches.prototype.setValue = function(switchId, value) {
    const before = this.value(switchId);
    _Game_Switches_setValue.call(this, switchId, value);

    if (before === value) return;

    const related = tracks.filter(cfg => cfg.trackSwitchId === switchId);
    if (!related.length) return;

    ensureTrackerStore();

    if (before === false && value === true) {
      for (const cfg of related) {
        if (cfg.sumVarId > 0) {
          $gameVariables.setValue(cfg.sumVarId, 0);
        }
        if (cfg.resultVarId > 0) {
          $gameVariables.setValue(cfg.resultVarId, 0);
        }
        for (const vid of cfg.targetVarIds) {
          if (vid > 0) {
            setPrevValue(vid, $gameVariables.value(vid));
          }
        }
      }
      return;
    }

    if (before === true && value === false) {
      for (const cfg of related) {
        const sum = cfg.sumVarId > 0 ? $gameVariables.value(cfg.sumVarId) : 0;
        const dv = cfg.divideValue;
        const result = (dv && dv > 0) ? Math.floor(sum / dv) : sum;
        if (cfg.resultVarId > 0) {
          $gameVariables.setValue(cfg.resultVarId, result);
        }
      }
    }
  };

})();
