/*:
 * @param EnabledSwitch
 * @text 有効スイッチ
 * @type switch
 * @default 0
 * @desc 0なら常に有効。1以上なら、そのスイッチがONのときだけ動作します。
 *
 * @param Settings
 * @text リージョン設定
 * @type struct<KCHRegionCE>[]
 * @default []
 * @desc 監視したいリージョンと、Enter/Exitで起動するコモンイベントを設定します。
 */
/*~struct~KCHRegionCE:
 * @param Memo
 * @text メモ
 * @type string
 * @default
 *
 * @param RegionId
 * @text リージョンID
 * @type number
 * @min 0
 * @default 0
 *
 * @param EnterCommonEvent
 * @text 乗った瞬間CE
 * @type common_event
 * @default 0
 *
 * @param ExitCommonEvent
 * @text 離れた瞬間CE
 * @type common_event
 * @default 0
 */
(() => {
  "use strict";

  const PLUGIN_NAME = "kamichichi_region_CE";
  const params = PluginManager.parameters(PLUGIN_NAME);

  const enabledSwitch = Number(params.EnabledSwitch || 0);
  const settingsRaw = JSON.parse(params.Settings || "[]").map(s => JSON.parse(s));

  const settings = settingsRaw.map(s => ({
    memo: String(s.Memo || ""),
    regionId: Number(s.RegionId || 0),
    enterCe: Number(s.EnterCommonEvent || 0),
    exitCe: Number(s.ExitCommonEvent || 0),
  })).filter(s => Number.isFinite(s.regionId));

  function isEnabled() {
    if (!enabledSwitch) return true;
    return $gameSwitches.value(enabledSwitch);
  }

  function clampMapX(x) {
    if ($gameMap.isLoopHorizontal()) {
      const w = $gameMap.width();
      return (x % w + w) % w;
    }
    return x.clamp(0, $gameMap.width() - 1);
  }

  function clampMapY(y) {
    if ($gameMap.isLoopVertical()) {
      const h = $gameMap.height();
      return (y % h + h) % h;
    }
    return y.clamp(0, $gameMap.height() - 1);
  }

  function getPlayerTilePosForRegion() {
    try {
      if ($gameMap.isPixelMovement && $gameMap.isPixelMovement()) { 
        if ($gamePlayer && typeof $gamePlayer.getColliderAt === "function") {
          const tw = $gameMap.tileWidth();
          const th = $gameMap.tileHeight();
          const collider = $gamePlayer.getColliderAt({ x: 0, y: 0 });
          if (collider && collider.center) {
            const cx = collider.center.x / tw;
            const cy = collider.center.y / th;
            const tx = clampMapX(Math.floor(cx));
            const ty = clampMapY(Math.floor(cy));
            return { x: tx, y: ty };
          }
        }
      }
    } catch (e) {
    }

    return { x: $gamePlayer.x, y: $gamePlayer.y };
  }

  function regionIdAtTilePos(pos) {
    return $gameMap.regionId(pos.x, pos.y);
  }

  function fireCommonEvent(ceId) {
    if (ceId > 0) $gameTemp.reserveCommonEvent(ceId);
  }

  const _Game_System_initialize = Game_System.prototype.initialize;
  Game_System.prototype.initialize = function() {
    _Game_System_initialize.call(this);
    this._kch_regionCE_lastRegionId = null;
    this._kch_regionCE_lastTileX = null;
    this._kch_regionCE_lastTileY = null;
    this._kch_regionCE_inited = false;
  };

  function ensureInitState() {
    if (!$gameSystem._kch_regionCE_inited) {
      const pos = getPlayerTilePosForRegion();
      $gameSystem._kch_regionCE_lastTileX = pos.x;
      $gameSystem._kch_regionCE_lastTileY = pos.y;
      $gameSystem._kch_regionCE_lastRegionId = regionIdAtTilePos(pos);
      $gameSystem._kch_regionCE_inited = true;
    }
  }

  function processRegionChange(prevRegion, newRegion) {
    for (const s of settings) {
      if (s.regionId === prevRegion) fireCommonEvent(s.exitCe);
    }
    for (const s of settings) {
      if (s.regionId === newRegion) fireCommonEvent(s.enterCe);
    }
  }

  const _Game_Player_update = Game_Player.prototype.update;
  Game_Player.prototype.update = function(sceneActive) {
    _Game_Player_update.call(this, sceneActive);

    if (!settings.length) return;
    if (!isEnabled()) return;
    if (!$gameMap || !$gamePlayer) return;
    if (!$gameMap.isMapLoaded || !$gameMap.isMapLoaded()) return;

    ensureInitState();

    const pos = getPlayerTilePosForRegion();
    const lastX = $gameSystem._kch_regionCE_lastTileX;
    const lastY = $gameSystem._kch_regionCE_lastTileY;

    if (pos.x === lastX && pos.y === lastY) return;

    const prevRegion = $gameSystem._kch_regionCE_lastRegionId;
    const newRegion = regionIdAtTilePos(pos);

    $gameSystem._kch_regionCE_lastTileX = pos.x;
    $gameSystem._kch_regionCE_lastTileY = pos.y;
    $gameSystem._kch_regionCE_lastRegionId = newRegion;

    if (prevRegion === newRegion) return;
    processRegionChange(prevRegion, newRegion);
  };

  const _Game_Player_performTransfer = Game_Player.prototype.performTransfer;
  Game_Player.prototype.performTransfer = function() {
    _Game_Player_performTransfer.call(this);
    if ($gameSystem) {
      $gameSystem._kch_regionCE_inited = false;
    }
  };

})();