/*:
 * @param DefaultFadeOutFrames
 * @text フラッシュフェードアウト(フレーム)
 * @type number
 * @min 0
 * @default 15
 *
 * @param DefaultWhiteHoldFrames
 * @text 白画面保持(フレーム)
 * @type number
 * @min 0
 * @default 6
 *
 * @param DefaultFadeInFrames
 * @text フラッシュフェードイン(フレーム)
 * @type number
 * @min 0
 * @default 15
 *
 * @param DefaultFlashColor
 * @text フラッシュカラー(R,G,B,A)
 * @type string
 * @default 255,255,255,255
 * @desc 例: 255,0,0,200（Aは0〜255）
 *
 * @command StartTransition
 * @text スーパー遷移開始
 * @desc フラッシュアウト→白画面保持→フラッシュイン を実行します。
 *
 * @arg FadeOutFrames
 * @text フェードアウト(フレーム)
 * @type number
 * @min 0
 * @default -1
 * @desc -1ならプラグインパラメータの既定値を使います。
 *
 * @arg WhiteHoldFrames
 * @text 白画面保持(フレーム)
 * @type number
 * @min 0
 * @default -1
 * @desc -1ならプラグインパラメータの既定値を使います。
 *
 * @arg FadeInFrames
 * @text フェードイン(フレーム)
 * @type number
 * @min 0
 * @default -1
 * @desc -1ならプラグインパラメータの既定値を使います。
 *
 * @arg FlashColor
 * @text フラッシュカラー(R,G,B,A)
 * @type string
 * @default
 * @desc 空ならプラグインパラメータの既定値を使います。例: 255,255,255,255
 *
 * @arg WaitFadeIn
 * @text フェードインもウェイト
 * @type boolean
 * @default true
 * @desc true: フェードイン完了まで待つ / false: フェードイン開始時点で待ち解除
 */

(() => {
  "use strict";

  const PLUGIN_NAME = "kamichichi_super_transition";
  const params = PluginManager.parameters(PLUGIN_NAME);

  const DEFAULT_OUT = Number(params.DefaultFadeOutFrames ?? 15);
  const DEFAULT_HOLD = Number(params.DefaultWhiteHoldFrames ?? 6);
  const DEFAULT_IN = Number(params.DefaultFadeInFrames ?? 15);
  const DEFAULT_COLOR_STR = String(params.DefaultFlashColor ?? "255,255,255,255");

  function clampInt(n, min, max) {
    n = Number.isFinite(n) ? Math.trunc(n) : min;
    return Math.max(min, Math.min(max, n));
  }

  function parseColorString(str, fallback) {
    const fb = Array.isArray(fallback) ? fallback : [255, 255, 255, 255];
    if (!str) return fb.slice();
    const parts = String(str).split(",").map(s => s.trim()).filter(s => s.length > 0);
    if (parts.length < 3) return fb.slice();

    const r = clampInt(Number(parts[0]), 0, 255);
    const g = clampInt(Number(parts[1]), 0, 255);
    const b = clampInt(Number(parts[2]), 0, 255);
    const a = clampInt(Number(parts[3] ?? fb[3]), 0, 255);
    return [r, g, b, a];
  }

  const _Game_Screen_initialize = Game_Screen.prototype.initialize;
  Game_Screen.prototype.initialize = function () {
    _Game_Screen_initialize.call(this);
    this._kchSuperTransition = null;
  };

  Game_Screen.prototype.isKchSuperTransitionRunning = function () {
    return !!this._kchSuperTransition;
  };

  Game_Screen.prototype.isKchSuperTransitionWaiting = function () {
    const st = this._kchSuperTransition;
    if (!st) return false;
    if (st.phase === "out") return true;
    if (st.phase === "hold") return true;
    if (st.phase === "in") return !!st.waitFadeIn;
    return false;
  };

  Game_Screen.prototype.startKchSuperTransition = function (fadeOutFrames, whiteHoldFrames, fadeInFrames, flashColor, waitFadeIn) {
    const outF = clampInt(fadeOutFrames, 0, 9999);
    const holdF = clampInt(whiteHoldFrames, 0, 9999);
    const inF = clampInt(fadeInFrames, 0, 9999);

    const color = Array.isArray(flashColor) ? flashColor.slice() : parseColorString(DEFAULT_COLOR_STR, [255, 255, 255, 255]);
    const maxA = clampInt(color[3], 0, 255);

    if (outF === 0 && holdF === 0 && inF === 0) {
      this.clearFlash();
      this._kchSuperTransition = null;
      return;
    }

    this._kchSuperTransition = {
      phase: outF > 0 ? "out" : (holdF > 0 ? "hold" : "in"),
      t: 0,
      outF,
      holdF,
      inF,
      color, 
      maxA,
      waitFadeIn: !!waitFadeIn
    };
  };

  function applyFlashEachFrame(screen, colorArray) {
    screen.startFlash(colorArray, 1);
  }

  function lerpInt(a, b, t01) {
    return Math.round(a + (b - a) * t01);
  }

  Game_Screen.prototype.updateKchSuperTransition = function () {
    const st = this._kchSuperTransition;
    if (!st) return;

    if (st.phase === "out") {
      st.t++;
      const denom = Math.max(1, st.outF);
      const t01 = Math.min(1, st.t / denom);
      const a = lerpInt(0, st.maxA, t01);
      applyFlashEachFrame(this, [st.color[0], st.color[1], st.color[2], a]);

      if (st.t >= st.outF) {
        st.phase = st.holdF > 0 ? "hold" : "in";
        st.t = 0;
      }
      return;
    }

    if (st.phase === "hold") {
      st.t++;
      applyFlashEachFrame(this, [255, 255, 255, 255]);

      if (st.t >= st.holdF) {
        st.phase = "in";
        st.t = 0;
      }
      return;
    }

    if (st.phase === "in") {
      st.t++;
      const denom = Math.max(1, st.inF);
      const t01 = Math.min(1, st.t / denom);
      const a = lerpInt(st.maxA, 0, t01);
      applyFlashEachFrame(this, [st.color[0], st.color[1], st.color[2], a]);

      if (st.t >= st.inF) {
        this.clearFlash();
        this._kchSuperTransition = null;
      }
      return;
    }
  };

  const _Game_Screen_update = Game_Screen.prototype.update;
  Game_Screen.prototype.update = function () {
    _Game_Screen_update.call(this);
    this.updateKchSuperTransition();
  };

  const _Game_Interpreter_updateWaitMode = Game_Interpreter.prototype.updateWaitMode;
  Game_Interpreter.prototype.updateWaitMode = function () {
    if (this._waitMode === "kchSuperTransition") {
      return $gameScreen.isKchSuperTransitionWaiting();
    }
    return _Game_Interpreter_updateWaitMode.call(this);
  };

  PluginManager.registerCommand(PLUGIN_NAME, "StartTransition", function (args) {
    const outArg = Number(args.FadeOutFrames ?? -1);
    const holdArg = Number(args.WhiteHoldFrames ?? -1);
    const inArg = Number(args.FadeInFrames ?? -1);

    const fadeOut = outArg >= 0 ? outArg : DEFAULT_OUT;
    const hold = holdArg >= 0 ? holdArg : DEFAULT_HOLD;
    const fadeIn = inArg >= 0 ? inArg : DEFAULT_IN;

    const defaultColor = parseColorString(DEFAULT_COLOR_STR, [255, 255, 255, 255]);
    const colorStr = String(args.FlashColor ?? "").trim();
    const color = colorStr ? parseColorString(colorStr, defaultColor) : defaultColor;

    const waitFadeIn = String(args.WaitFadeIn ?? "true") === "true";

    $gameScreen.startKchSuperTransition(fadeOut, hold, fadeIn, color, waitFadeIn);

    this.setWaitMode("kchSuperTransition");
  });

})();
