/*:
 * @target MZ
 * @plugindesc [Addon] OptionCustomize：文字色・アウトライン幅（カテゴリ/項目/中身）制御
 * @author kamichichi
 *
 * @param textColor
 * @text 文字色（タイトル/中身 共通）
 * @type string
 * @default #000000
 *
 * @param outlineWidth
 * @text アウトライン幅（タイトル/中身 共通）
 * @type number
 * @min 0
 * @default 0
 *
 * @param outlineColor
 * @text アウトライン色
 * @type string
 * @default rgba(0,0,0,0)
 *
 * @param applyToTitles
 * @text タイトル（カテゴリ/項目名）に適用
 * @type boolean
 * @default true
 *
 * @param applyToValues
 * @text 中身（右側の値）に適用
 * @type boolean
 * @default true
 *
 * @help
 * kzmz_OptionCustomize.js の後に配置。
 * フォントサイズは変更しません。
 */
(() => {
  "use strict";

  const p = PluginManager.parameters("kzmz_OptionCustomize_TextStyleAddon");

  const TEXT_COLOR = String(p.textColor || "#000000");
  const OUTLINE_W  = Number(p.outlineWidth ?? 0);
  const OUTLINE_C  = String(p.outlineColor || "rgba(0,0,0,0)");
  const APPLY_TITLES = String(p.applyToTitles || "true") === "true";
  const APPLY_VALUES = String(p.applyToValues || "true") === "true";

  function isOptionsScene() {
    const s = SceneManager && SceneManager._scene;
    return s && (s instanceof Scene_Options);
  }

  function isTitleTargetWindow(win) {
    if (!win || !win.constructor) return false;
    const n = win.constructor.name;
    // ここがポイント：ローカルクラスでも instance から name は取れる
    return n === "Window_OptionCategory" || n === "Window_Options";
  }

  // --- タイトル（カテゴリ/項目名）側：Window描画に差し込む ---
  // カテゴリは drawText で描いている :contentReference[oaicite:1]{index=1}
  // 項目名は drawTextEx 経由 :contentReference[oaicite:2]{index=2}
  const _resetFontSettings = Window_Base.prototype.resetFontSettings;
  Window_Base.prototype.resetFontSettings = function() {
    _resetFontSettings.call(this);

    if (!APPLY_TITLES) return;
    if (!isOptionsScene()) return;
    if (!isTitleTargetWindow(this)) return;

    if (this.contents) {
      this.contents.textColor = TEXT_COLOR;
      this.contents.outlineWidth = OUTLINE_W;
      this.contents.outlineColor = OUTLINE_C;
    }
  };

  const _resetTextColor = Window_Base.prototype.resetTextColor;
  Window_Base.prototype.resetTextColor = function() {
    _resetTextColor.call(this);

    if (!APPLY_TITLES) return;
    if (!isOptionsScene()) return;
    if (!isTitleTargetWindow(this)) return;

    if (this.contents) {
      this.contents.textColor = TEXT_COLOR;
      this.contents.outlineWidth = OUTLINE_W;
      this.contents.outlineColor = OUTLINE_C;
    }
  };

  // --- 中身（右側の値）側：描画“直前”だけ Bitmap.drawText を差し替え ---
  function withPatchedDrawText(fn) {
    if (!APPLY_VALUES) return fn();

    const orig = Bitmap.prototype.drawText;
    Bitmap.prototype.drawText = function(text, x, y, maxWidth, lineHeight, align) {
      this.textColor = TEXT_COLOR;
      this.outlineWidth = OUTLINE_W;
      this.outlineColor = OUTLINE_C;
      return orig.call(this, text, x, y, maxWidth, lineHeight, align);
    };

    try { return fn(); }
    finally { Bitmap.prototype.drawText = orig; }
  }

  if (typeof Window_Options === "function") {
    const _drawDragBar = Window_Options.prototype.drawDragBar;
    Window_Options.prototype.drawDragBar = function(index, rect) {
      return withPatchedDrawText(() => _drawDragBar.call(this, index, rect));
    };

    const _drawOnOffButton = Window_Options.prototype.drawOnOffButton;
    Window_Options.prototype.drawOnOffButton = function(index, rect) {
      return withPatchedDrawText(() => _drawOnOffButton.call(this, index, rect));
    };
  }
})();
