/*:
 * @target MZ
 * @plugindesc (v1.1) A plugin to cache the most recent Gamepad or Keyboard inputs.
 * @author Zael
 *
 *
 * @help
 * 
 * zActive Remote - Input Checker and Cache
 * Version 1.1
 * by Zael
 *
 * ---------------------
 *     How to use
 * ---------------------
 * To use this plugin, place it in your plugins list and setup the Switches and Variables.
 *
 * The goal of this plugin is to provide a framework of Switches and Variables to allow the
 * user to better track the player's inputs and where they are coming from. As it stands, it
 * is just caching the information into the Switches and Variables, how you use the data
 * gathered by them is up to you. The easiest example is to have different button prompts
 * based on the player's preferred method of input.
 *
 * The plugin will cache the ID of the Controller into a Variable. This Variable will not be
 * a numerical ID, but rather a text string. You can use the variable in a text code and it
 * will display the full ID of the Controller.
 *
 * For example, if you are using an Xbox 360 Controller, and store the ID to Variable 1, when
 * you use '\v[1]' in a Message, it will show up as 'Xbox 360 Controller (XInput STANDARD GAMEPAD)'.
 *
 * This is useful if you want to edit the section below marked Controller Detection to detect
 * specific gamepads that aren't currently supported. The Manufacturer Variable is
 * determined by the Gamepad ID and will also be a text string, however it will be simplified
 * down to a one word name. Currently, only the big 3 Manufacturers are supported, however if
 * you check line 119 below, you will see a section that can be expanded upon to include more
 * manufacturers.
 *
 * Time Stamp is really just here as a bonus and it might be useful to someone so I included
 * the option to store it in a Variable if you want.
 *
 * Lastly, we have our Gamepad Switch which will be flipped on if you are using a Gamepad.
 *
 * ---------------------
 *     Terms of use
 * ---------------------
 * - Free for use in both non-commercial and commercial games.
 * - You may repost and edit this plugin.
 * - Credit to me (Zael) is appreciated, but not required.
 * - Special Thanks to Caethyril and ATT_Turan for their ideas and support.
 * 
 * ---------------------
 *      Changelog
 * ---------------------
 * Version 1.1
 * - Cleaned up code, added Parameters, removed console logging, added Help.
 * Version 1.0
 * - Added Switches and Variables.
 * Version 0.9
 * - Finished the basics of the plugin.
 *
 *
 * @param gamepadIDvariable
 * @text Gamepad ID Variable
 * @desc The ID of the Variable to store the Gamepad ID in. Set to 0 to not use.
 * @type number
 * @min 0
 * @default 0
 *
 * @param timestampVariable
 * @text Time Stamp Variable
 * @desc The ID of the Variable to store the Input Timestamp in. Set to 0 to not use.
 * @type number
 * @min 0
 * @default 0
 *
 * @param gamepadSwitch
 * @text Gamepad Switch
 * @desc The ID of the Switch to turn On if a Gamepad is in use. Set to 0 to not use.
 * @type number
 * @min 0
 * @default 0
 *
 * @param gamepadManufacturerVariable
 * @text Gamepad Manufacturer Variable
 * @desc The ID of the Variable to store the Manufacturer of the Controller in. Set to 0 to not use.
 * @type number
 * @min 0
 * @default 0
 *
 */

(() => {
    const PLUGIN_NAME = "zActiveRemote";

    // Establish Parameters
    const parameters = PluginManager.parameters(PLUGIN_NAME);
    parameters.gamepadIDvariable = Number(parameters.gamepadIDvariable) || 0;
    parameters.timestampVariable = Number(parameters.timestampVariable) || 0;
    parameters.gamepadSwitch = Number(parameters.gamepadSwitch) || 0;
    parameters.gamepadManufacturerVariable = Number(parameters.gamepadManufacturerVariable) || 0;

    // Object to store timestamps of last input for each device
    let lastButtonPressTimestamps = {};
    let lastKeyboardPressTimestamp = 0;
    let lastMousePressTimestamp = 0;

    // --------------------------------------------------------------------
    // 追加：$gameVariables が未生成（タイトル等）でも「最後の入力」を保持するキャッシュ
    // --------------------------------------------------------------------
    let _kchLastInputType = "";          // "gamepad" / "keyboard" / "mouse"
    let _kchLastTimestamp = 0;
    let _kchLastGamepadId = "";
    let _kchLastManufacturer = "";

    function applyCacheToGameVariables() {
        // タイトル等で $gameVariables が無い場合は何もしない（キャッシュは保持される）
        if (!$gameVariables) return;

        if (parameters.timestampVariable != 0) {
            $gameVariables.setValue(parameters.timestampVariable, _kchLastTimestamp);
        }
        if (parameters.gamepadIDvariable != 0) {
            $gameVariables.setValue(parameters.gamepadIDvariable, _kchLastGamepadId);
        }
        if (parameters.gamepadManufacturerVariable != 0) {
            $gameVariables.setValue(parameters.gamepadManufacturerVariable, _kchLastManufacturer);
        }
        if (parameters.gamepadSwitch != 0 && $gameSwitches) {
            $gameSwitches.setValue(parameters.gamepadSwitch, _kchLastInputType === "gamepad");
        }
    }

    // Function to handle gamepad button press events
    function handleGamepadButtonPress(event) {
        let gamepad = event.gamepad;

        // Update the timestamp for the last button press of this gamepad
        lastButtonPressTimestamps[gamepad.index] = Date.now();

        // 追加：キャッシュ更新（$gameVariables が無い場面でも保持）
        _kchLastInputType = "gamepad";
        _kchLastTimestamp = lastButtonPressTimestamps[gamepad.index];

        // Set Current Timestamp to Time Stamp Variable if it is in use
        if (parameters.timestampVariable != 0 && $gameVariables) {
            $gameVariables.setValue(parameters.timestampVariable, lastButtonPressTimestamps[gamepad.index]);
        }

        let gamepadID = navigator.getGamepads()[gamepad.index].id;

        // 追加：キャッシュ更新
        _kchLastGamepadId = gamepadID;

        // Set ID to Gamepad ID Variable if it is in use
        if (parameters.gamepadIDvariable != 0 && $gameVariables) {
            $gameVariables.setValue(parameters.gamepadIDvariable, gamepadID);
        }

        // Set Manufacturer to Variable if it is in use
        if (parameters.gamepadManufacturerVariable != 0 && $gameVariables) {
            if (gamepadID.includes('Xbox')) {
                $gameVariables.setValue(parameters.gamepadManufacturerVariable, "Xbox");
            } else if (gamepadID.includes('054c')) {
                $gameVariables.setValue(parameters.gamepadManufacturerVariable, "Sony");
            } else if (gamepadID.includes('057e')) {
                $gameVariables.setValue(parameters.gamepadManufacturerVariable, "Nintendo");
            } else {
                $gameVariables.setValue(parameters.gamepadManufacturerVariable, "Generic");
            }
        }

        // 追加：キャッシュの manufacturer をここで確定
        if (gamepadID.includes('Xbox')) {
            _kchLastManufacturer = "Xbox";
        } else if (gamepadID.includes('054c')) {
            _kchLastManufacturer = "Sony";
        } else if (gamepadID.includes('057e')) {
            _kchLastManufacturer = "Nintendo";
        } else {
            _kchLastManufacturer = "Generic";
        }

        // 追加：$gameVariables が生成済みなら即反映（未生成ならキャッシュ保持）
        applyCacheToGameVariables();

        compareTimestamps();
    }

    // Function to handle keyboard key press events
    function handleKeyboardKeyPress(event) {
        lastKeyboardPressTimestamp = Date.now();

        // 追加：キャッシュ更新
        _kchLastInputType = "keyboard";
        _kchLastTimestamp = lastKeyboardPressTimestamp;
        _kchLastGamepadId = "Keyboard";
        _kchLastManufacturer = "Standard";

        // Set Current Timestamp to Time Stamp Variable if it is in use
        if (parameters.timestampVariable != 0 && $gameVariables) {
            $gameVariables.setValue(parameters.timestampVariable, lastKeyboardPressTimestamp);
        }

        // Set fake Keyboard ID to Gamepad ID Variable if it is in use
        if (parameters.gamepadIDvariable != 0 && $gameVariables) {
            $gameVariables.setValue(parameters.gamepadIDvariable, "Keyboard");
        }

        // Set fake Standard Manufacturer for Keyboards if the Variable is in use
        if (parameters.gamepadManufacturerVariable != 0 && $gameVariables) {
            $gameVariables.setValue(parameters.gamepadManufacturerVariable, "Standard");
        }

        // 追加：$gameVariables が生成済みなら即反映
        applyCacheToGameVariables();

        compareTimestamps();
    }

    // Function to handle mouse click events
    function handleMouseClick(event) {
        lastMousePressTimestamp = Date.now();

        // 追加：キャッシュ更新
        _kchLastInputType = "mouse";
        _kchLastTimestamp = lastMousePressTimestamp;
        _kchLastGamepadId = "Mouse";
        _kchLastManufacturer = "Standard";

        // Set Current Timestamp to Time Stamp Variable if it is in use
        if (parameters.timestampVariable != 0 && $gameVariables) {
            $gameVariables.setValue(parameters.timestampVariable, lastMousePressTimestamp);
        }

        // Set fake Mouse ID to Gamepad ID Variable if it is in use
        if (parameters.gamepadIDvariable != 0 && $gameVariables) {
            $gameVariables.setValue(parameters.gamepadIDvariable, "Mouse");
        }

        // Set fake Standard Manufacturer for Mouse if the Variable is in use
        if (parameters.gamepadManufacturerVariable != 0 && $gameVariables) {
            $gameVariables.setValue(parameters.gamepadManufacturerVariable, "Standard");
        }

        // 追加：$gameVariables が生成済みなら即反映
        applyCacheToGameVariables();

        compareTimestamps();
    }

    // Function to compare timestamps of gamepad, keyboard, and mouse inputs
    function compareTimestamps() {
        let mostRecentGamepadTimestamp = Math.max(Object.values(lastButtonPressTimestamps));
        let mostRecentInputType = "keyboard";

        if (mostRecentGamepadTimestamp > lastKeyboardPressTimestamp && mostRecentGamepadTimestamp > lastMousePressTimestamp) {
            mostRecentInputType = "gamepad";
        } else if (lastMousePressTimestamp > lastKeyboardPressTimestamp) {
            mostRecentInputType = "mouse";
        }

        // Set Gamepad Switch to On if a Gamepad is in use and the Switch is in use
        if (parameters.gamepadSwitch != 0 && $gameSwitches) {
            $gameSwitches.setValue(parameters.gamepadSwitch, mostRecentInputType === "gamepad");
        }
    }

    // --------------------------------------------------------------------
    // 追加：compareTimestamps の不具合（Math.max(.Object.values...)）を直した版に差し替え
    // ※既存コードは保持し、実際に呼ばれる関数だけ上書き
    // --------------------------------------------------------------------
    const _compareTimestamps_original = compareTimestamps;
    function compareTimestampsFixed() {
        const values = Object.values(lastButtonPressTimestamps);
        const mostRecentGamepadTimestamp = values.length ? Math.max(...values) : 0;

        let mostRecentInputType = "keyboard";
        if (mostRecentGamepadTimestamp > lastKeyboardPressTimestamp && mostRecentGamepadTimestamp > lastMousePressTimestamp) {
            mostRecentInputType = "gamepad";
        } else if (lastMousePressTimestamp > lastKeyboardPressTimestamp) {
            mostRecentInputType = "mouse";
        }

        // Switchはキャッシュにも反映させる（タイトル等で $gameSwitches が無い場合を考慮）
        _kchLastInputType = mostRecentInputType;

        if (parameters.gamepadSwitch != 0 && $gameSwitches) {
            $gameSwitches.setValue(parameters.gamepadSwitch, mostRecentInputType === "gamepad");
        }
    }
    // 実際に呼ばれる関数を差し替え
    compareTimestamps = compareTimestampsFixed;

    // --------------------------------------------------------------------
    // 追加：ニューゲーム/ロード直後に「タイトルで押した入力」を変数へ反映
    // --------------------------------------------------------------------
    if (typeof DataManager !== "undefined") {
        const _DataManager_setupNewGame = DataManager.setupNewGame;
        DataManager.setupNewGame = function() {
            _DataManager_setupNewGame.call(this);
            applyCacheToGameVariables();
        };

        const _DataManager_extractSaveContents = DataManager.extractSaveContents;
        DataManager.extractSaveContents = function(contents) {
            _DataManager_extractSaveContents.call(this, contents);
            applyCacheToGameVariables();
        };
    }

    // Listen for gamepad button press events
    window.addEventListener("gamepadbuttondown", handleGamepadButtonPress);

    // Listen for keyboard key press events
    window.addEventListener("keydown", handleKeyboardKeyPress);

    // Listen for mouse click events
    window.addEventListener("mousedown", handleMouseClick);

    // Check for gamepad input in each animation frame
    function checkGamepadInput() {
        let gamepads = navigator.getGamepads();

        for (let gamepad of gamepads) {
            if (gamepad) {
                for (let i = 0; i < gamepad.buttons.length; i++) {
                    if (gamepad.buttons[i].pressed) {
                        let event = new GamepadEvent("gamepadbuttondown", {
                            gamepad: gamepad
                        });
                        handleGamepadButtonPress(event);
                    }
                }
            }
        }
    }

    // Start checking for gamepad input
    setInterval(checkGamepadInput, 1000 / 60); // Check every frame (assuming 60 FPS)
 })();
