/*:
 * @plugindesc マップ上で左からスライドインするバトルログ風テキスト表示プラグイン（背景色指定可・制御文字対応） 
 * @author 魚音
 *
 * @param Position X
 * @type number
 * @default 100
 * @desc 表示のX座標
 *
 * @param Position Y
 * @type number
 * @default 300
 * @desc 表示のY座標（最下段の行の位置）
 *
 * @param Font Size
 * @type number
 * @default 20
 * @desc フォントサイズ
 *
 * @param Line Spacing
 * @type number
 * @default 4
 * @desc 行間隔
 *
 * @param Max Lines
 * @type number
 * @default 5
 * @desc 最大行数
 *
 * @param Slide Speed
 * @type number
 * @default 6
 * @desc 左からのスライド速度（px/frame）
 *
 * @param Background Width
 * @type number
 * @default 300
 * @desc 背景の横幅
 *
 * @param Background Height
 * @type number
 * @default 28
 * @desc 背景の縦幅（1行分）
 *
 * @param Background Color
 * @type string
 * @default rgba(0,0,0,0.8)
 * @desc 背景グラデーションの開始色（RGBA形式可）
 *
 * @help
 * プラグインコマンド:
 *   BattleLogSlideAdd テキスト
 *     → 新しいテキストを追加表示（\V[1] などの制御文字可）
 *
 *   BattleLogSlideClear
 *     → 表示中のテキストを全て消去
 */
(function(){
    var params = PluginManager.parameters('BattleLogSlide');
    var posX = Number(params['Position X'] || 100);
    var posY = Number(params['Position Y'] || 300);
    var fontSize = Number(params['Font Size'] || 20);
    var lineSpacing = Number(params['Line Spacing'] || 4);
    var maxLines = Number(params['Max Lines'] || 5);
    var slideSpeed = Number(params['Slide Speed'] || 6);
    var bgW = Number(params['Background Width'] || 300);
    var bgH = Number(params['Background Height'] || 28);
    var bgColor = params['Background Color'] || 'rgba(0,0,0,0.8)';

    // 行スプライトクラス
    function BattleLogLine() {
        this.initialize.apply(this, arguments);
    }
    BattleLogLine.prototype = Object.create(PIXI.Container.prototype);
    BattleLogLine.prototype.constructor = BattleLogLine;

    BattleLogLine.prototype.initialize = function(text) {
        PIXI.Container.call(this);

        // 背景グラデーション作成
        var tex = this.createGradientTexture(bgW, bgH);
        var sprite = new PIXI.Sprite(tex);
        this.addChild(sprite);

        // 制御文字対応テキスト描画
        var bitmap = new Bitmap(bgW, bgH);
        bitmap.fontSize = fontSize;
        var tempWindow = new Window_Base(0, 0, bgW, bgH);
        tempWindow.contents = bitmap;
        tempWindow.drawTextEx(text, 8, 0);

        var textSprite = new Sprite(bitmap);
        this.addChild(textSprite);

        this.x = -bgW; // 左外から開始
        this._targetX = 0;
        this._slideSpeed = slideSpeed;
    };

    BattleLogLine.prototype.createGradientTexture = function(width, height) {
        var canvas = document.createElement('canvas');
        canvas.width = width;
        canvas.height = height;
        var ctx = canvas.getContext('2d');
        var grad = ctx.createLinearGradient(0, 0, width, 0);
        grad.addColorStop(0, bgColor);
        grad.addColorStop(1, 'rgba(0,0,0,0)');
        ctx.fillStyle = grad;
        ctx.fillRect(0, 0, width, height);
        return PIXI.Texture.from(canvas);
    };

    BattleLogLine.prototype.update = function() {
        if (this.x < this._targetX) {
            this.x += this._slideSpeed;
            if (this.x > this._targetX) this.x = this._targetX;
        }
    };

    // マップ用コンテナ
    function BattleLogSlideLayer() {
        this.initialize.apply(this, arguments);
    }
    BattleLogSlideLayer.prototype = Object.create(PIXI.Container.prototype);
    BattleLogSlideLayer.prototype.constructor = BattleLogSlideLayer;

    BattleLogSlideLayer.prototype.initialize = function() {
        PIXI.Container.call(this);
        this._lines = [];
    };

    BattleLogSlideLayer.prototype.addLog = function(text) {
        var lineHeight = fontSize + lineSpacing;
        for (var i = 0; i < this._lines.length; i++) {
            var l = this._lines[i];
            l.y -= lineHeight;
        }
        var newLine = new BattleLogLine(text);
        newLine.y = 0;
        this.addChild(newLine);
        this._lines.unshift(newLine);
        while (this._lines.length > maxLines) {
            var old = this._lines.pop();
            this.removeChild(old);
        }
    };

    BattleLogSlideLayer.prototype.clearLogs = function() {
        for (var i = 0; i < this._lines.length; i++) {
            this.removeChild(this._lines[i]);
        }
        this._lines = [];
    };

    BattleLogSlideLayer.prototype.update = function() {
        for (var i = 0; i < this._lines.length; i++) {
            this._lines[i].update();
        }
    };

    // Scene_Mapに追加
    var _Scene_Map_createDisplayObjects = Scene_Map.prototype.createDisplayObjects;
    Scene_Map.prototype.createDisplayObjects = function() {
        _Scene_Map_createDisplayObjects.call(this);
        this._battleLogSlideLayer = new BattleLogSlideLayer();
        this._battleLogSlideLayer.x = posX;
        this._battleLogSlideLayer.y = posY;
        this.addChild(this._battleLogSlideLayer);
    };

    var _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.call(this);
        if (this._battleLogSlideLayer) {
            this._battleLogSlideLayer.update();
        }
    };

    // プラグインコマンド
    var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function(command, args) {
        _Game_Interpreter_pluginCommand.call(this, command, args);
        if (SceneManager._scene instanceof Scene_Map) {
            if (command === 'BattleLogSlideAdd') {
                var text = args.join(' ');
                // Window_Base経由で制御文字展開
                var win = new Window_Base(0, 0, 0, 0);
                text = win.convertEscapeCharacters(text);
                SceneManager._scene._battleLogSlideLayer.addLog(text);
            }
            if (command === 'BattleLogSlideClear') {
                SceneManager._scene._battleLogSlideLayer.clearLogs();
            }
        }
    };

})();
