/*:
 * @plugindesc (v1.0) CustomGaugeMV - 任意のゲージを画面に複数表示するMV用プラグイン
 * @author 
 *
 * @param Gauges
 * @text Gauges
 * @desc プラグ構成のJSON配列。下の説明を参照。
 * @default []
 *
 * @help
 * ■ 概要
 * このプラグインはRPGツクールMV用の簡易的な複数ゲージ描画プラグインです。
 * 設定した複数のゲージをScene_Map/Scene_Battle/Scene_Menuで描画します。
 * ゲージの現在値はゲーム変数を参照し、最大値・最小値は固定値または変数で指定可能。
 * スイッチで表示のON/OFF制御、位置・サイズ・色・ラベル・値表示の設定が可能です。
 *
 * ■ Gauges パラメータ（JSON配列の例）
 * 以下の配列をそのままプラグインパラメータに貼り付けてください。
 *
 * [
 *  {
 *    "id": "hp1",
 *    "scene": "Scene_Map",
 *    "switchId": 0,
 *    "variableId": 41,
 *    "min": 0,
 *    "max": 100,
 *    "x": 400,
 *    "y": 200,
 *    "width": 400,
 *    "height": 24,
 *    "color": "255,0,0",
 *    "backColor": "0,0,0",
 *    "showValue": true,
 *    "label": "HP",
 *    "fontSize": 20,
 *    "fontColor": "255,255,255"
 *  }
 * ]
 *
 * ■ フィールド説明（各ゲージオブジェクト）
 * id         : ゲージ識別文字列（任意）
 * scene      : 描画するシーン名。"Scene_Map" / "Scene_Battle" / "Scene_Menu" / "All" のいずれか
 * switchId   : 0の場合スイッチ無効。1以上でそのスイッチがONの時のみ表示
 * variableId : ゲージの現在値を取得するゲーム変数ID（必須）
 * min        : 最小値（数値か変数参照の文字列例: "v5" は変数5番を参照）
 * max        : 最大値（数値か変数参照の文字列）
 * x,y,width,height : 表示座標・サイズ（ピクセル）
 * color      : ゲージ前景色（"r,g,b"）
 * backColor  : 背景色（"r,g,b"）
 * showValue  : 値テキストを表示するか（true/false）
 * label      : ラベル文字列（空欄可）
 * fontSize   : テキストサイズ
 * fontColor  : テキスト色（"r,g,b"）
 *
 * ■ 動作
 * - variableId の値を元に (value - min) / (max - min) の割合でゲージ幅が決まります。
 * - min/max に "vN" と書くと変数Nの値を参照します（例: "v10"）。
 * - switchId が 0 の場合はスイッチ判定なしで常に表示されます。
 * - Scene の指定を "All" にするとすべての対応シーンで描画されます。
 *
 * ■ インストール
 * 1. このファイルを "js/plugins/CustomGaugeMV.js" として保存
 * 2. ツクールMVのプラグイン管理で有効化
 * 3. プラグインパラメータ "Gauges" に上記JSON配列を設定
 *
 * ■ 注意
 * - 本プラグインはシンプルに実装してあるため、高度なエフェクトや画像差し替え等は実装していません。
 * - 大量のゲージを常時更新するとフレームに負荷がかかる可能性があります。
 *
 */

(function() {
  'use strict';

  var pluginName = 'CustomGaugeMV';
  var parameters = PluginManager.parameters(pluginName) || {};
  var gaugesParam = parameters['Gauges'] || '[]';
  var _gauges = [];
  try {
    _gauges = JSON.parse(gaugesParam);
  } catch (e) {
    // 文字列が空や不正なら空配列
    _gauges = [];
  }

  // ヘルパー
  function toNum(v, def) {
    if (v === undefined || v === null || v === '') return def || 0;
    var n = Number(v);
    return isNaN(n) ? def || 0 : n;
  }

  function parseRGB(str, def) {
    if (!str) return def || [255,255,255];
    var parts = (''+str).split(',').map(function(s){return parseInt(s,10);});
    if (parts.length < 3) return def || [255,255,255];
    return [parts[0]||0, parts[1]||0, parts[2]||0];
  }

  function getVarOrNumber(token) {
    // token like "v5" => $gameVariables.value(5)
    if (typeof token === 'string' && token.match(/^v(\d+)$/i)) {
      return $gameVariables.value(Number(RegExp.$1)) || 0;
    }
    return Number(token) || 0;
  }

  // Gauge Sprite
  function Sprite_CustomGauge(cfg) {
    this.initialize.apply(this, arguments);
  }
  Sprite_CustomGauge.prototype = Object.create(Sprite.prototype);
  Sprite_CustomGauge.prototype.constructor = Sprite_CustomGauge;

  Sprite_CustomGauge.prototype.initialize = function(cfg) {
    Sprite.prototype.initialize.call(this);
    this.cfg = cfg || {};
    this.bitmap = new Bitmap(toNum(this.cfg.width, 200), toNum(this.cfg.height, 20));
    this.anchor.x = 0.5; this.anchor.y = 0.5;
    this.x = toNum(this.cfg.x, Graphics.width/2);
    this.y = toNum(this.cfg.y, Graphics.height/2);
    this._lastValue = null;
    this.refresh();
  };

  Sprite_CustomGauge.prototype.getCurrent = function() {
    var vid = toNum(this.cfg.variableId, 0);
    return $gameVariables.value(vid) || 0;
  };

  Sprite_CustomGauge.prototype.getMin = function() {
    return getVarOrNumber(this.cfg.min === undefined ? 0 : this.cfg.min);
  };

  Sprite_CustomGauge.prototype.getMax = function() {
    return getVarOrNumber(this.cfg.max === undefined ? 100 : this.cfg.max);
  };

  Sprite_CustomGauge.prototype.isVisibleBySwitch = function() {
    var sid = toNum(this.cfg.switchId||0, 0);
    if (sid === 0) return true;
    return $gameSwitches.value(sid);
  };

  Sprite_CustomGauge.prototype.refresh = function() {
    var bm = this.bitmap;
    bm.clear();
    // draw background
    var back = parseRGB(this.cfg.backColor, [0,0,0]);
    bm.fillRect(0,0,bm.width,bm.height, 'rgb('+back.join(',')+')');
    // compute ratio
    var min = this.getMin();
    var max = this.getMax();
    if (max <= min) max = min + 1;
    var cur = this.getCurrent();
    var ratio = (cur - min) / (max - min);
    ratio = Math.max(0, Math.min(1, ratio));
    // draw gauge
    var col = parseRGB(this.cfg.color, [255,0,0]);
    var gw = Math.round(bm.width * ratio);
    bm.fillRect(0,0,gw,bm.height, 'rgb('+col.join(',')+')');
    // draw border
    bm.fillRect(0,0,bm.width,1,'rgba(0,0,0,0.5)');
    bm.fillRect(0,bm.height-1,bm.width,1,'rgba(0,0,0,0.5)');
    // draw label & value
    var fs = toNum(this.cfg.fontSize, 18);
    var fcol = parseRGB(this.cfg.fontColor, [255,255,255]);
    bm.fontSize = fs;
    if (this.cfg.label) {
      bm.drawText(String(this.cfg.label), 6, 0, bm.width - 12, bm.height, 'left');
    }
    if (this.cfg.showValue) {
      var maxv = this.getMax();
      var minv = this.getMin();
      var txt = Math.round(cur) + '/' + Math.round(maxv);
      bm.drawText(txt, 6, 0, bm.width - 12, bm.height, 'right');
    }
  };

  Sprite_CustomGauge.prototype.update = function() {
    Sprite.prototype.update.call(this);
    // visibility by switch
    this.visible = this.isVisibleBySwitch();
    // update position in case Graphics changed
    this.x = toNum(this.cfg.x, Graphics.width/2);
    this.y = toNum(this.cfg.y, Graphics.height/2);
    var cur = this.getCurrent();
    if (cur !== this._lastValue) {
      this._lastValue = cur;
      this.refresh();
    }
  };

  // Container creation helper
  function createGaugeSpritesForScene(scene) {
    if (!scene) return;
    if (!scene._customGaugeContainer) {
      scene._customGaugeContainer = new Sprite();
      scene.addChild(scene._customGaugeContainer);
      // create for each config
      scene._customGaugeSprites = [];
      _gauges.forEach(function(cfg) {
        var sceneName = (cfg.scene || 'All');
        if (sceneName === 'All' || sceneName === scene.constructor.name) {
          var sp = new Sprite_CustomGauge(cfg);
          scene._customGaugeContainer.addChild(sp);
          scene._customGaugeSprites.push(sp);
        }
      });
    }
  }

  // Hook into Scene_Map
  var _Scene_Map_createDisplayObjects = Scene_Map.prototype.createDisplayObjects;
  Scene_Map.prototype.createDisplayObjects = function() {
    _Scene_Map_createDisplayObjects.call(this);
    createGaugeSpritesForScene(this);
  };

  // Hook into Scene_Battle
  var _Scene_Battle_createDisplayObjects = Scene_Battle.prototype.createDisplayObjects;
  Scene_Battle.prototype.createDisplayObjects = function() {
    _Scene_Battle_createDisplayObjects.call(this);
    createGaugeSpritesForScene(this);
  };

  // Hook into Scene_Menu (menu scene uses create())
  var _Scene_Menu_create = Scene_Menu.prototype.create;
  Scene_Menu.prototype.create = function() {
    _Scene_Menu_create.call(this);
    createGaugeSpritesForScene(this);
  };

})();
