/*:
 * @target MV
 * @plugindesc スイッチONでマップ別ピクチャキャッシュ生成・OFFで破棄するプラグイン（ピクチャ同階層描画）
 *
 * @param MapPictureSets
 * @text マップ別ピクチャ設定リスト
 * @type struct<MapPictureSet>[]
 * @default []
 *
 * @param LoadSwitchId
 * @text キャッシュ管理スイッチID
 * @type switch
 * @desc このスイッチがONでキャッシュ生成、OFFで破棄されます。
 * @default 1
 *
 * @help
 * 【概要】
 * - スイッチがONになると、そのマップ専用のピクチャキャッシュを生成。
 * - OFFになるとキャッシュ破棄＆表示中ピクチャ削除。
 * - 「イベントコマンド：ピクチャの表示」と同じ仕組み・順序で描画。
 *
 * 【使い方】
 * 1. プラグインパラメータで各マップごとにピクチャセットを定義
 * 2. キャッシュ管理スイッチを指定（例：ID=5）
 * 3. イベントでスイッチONにすると、そのマップの画像がキャッシュされる
 * 4. 以下のスクリプトで制御：
 *    UIPicture.show(id);
 *    UIPicture.hide(id);
 *
 * 【注意】
 * - マップ移動時、再度スイッチONでそのマップのキャッシュが再生成されます。
 * - OFF時にキャッシュ破棄＆表示ピクチャ削除。
 */

/*~struct~MapPictureSet:
 * @param mapId
 * @text 対象マップID
 * @type number
 * @default 1
 *
 * @param pictures
 * @text ピクチャ設定リスト
 * @type struct<PictureData>[]
 * @default []
 */

/*~struct~PictureData:
 * @param id
 * @text ピクチャID
 * @type number
 * @default 1
 *
 * @param name
 * @text ファイル名
 * @type file
 * @dir img/pictures
 *
 * @param origin
 * @text 原点
 * @type select
 * @option 左上
 * @value 0
 * @option 中心
 * @value 1
 * @default 0
 *
 * @param x
 * @text X座標
 * @type number
 * @default 0
 *
 * @param y
 * @text Y座標
 * @type number
 * @default 0
 *
 * @param scaleX
 * @text 拡大率X(%)
 * @type number
 * @default 100
 *
 * @param scaleY
 * @text 拡大率Y(%)
 * @type number
 * @default 100
 *
 * @param opacity
 * @text 不透明度
 * @type number
 * @max 255
 * @default 255
 *
 * @param blendMode
 * @text 合成方法
 * @type select
 * @option 通常
 * @value 0
 * @option 加算
 * @value 1
 * @option 乗算
 * @value 2
 * @option スクリーン
 * @value 3
 * @default 0
 */

(() => {
  const pluginName = document.currentScript.src.match(/([^/]+)\.js$/)[1];
  const params = PluginManager.parameters(pluginName);
  const mapPictureSets = JSON.parse(params["MapPictureSets"] || "[]").map(d => JSON.parse(d));
  const loadSwitchId = Number(params["LoadSwitchId"] || 1);

  // 現在マップのキャッシュ
  let _currentCache = null;
  let _cacheInitialized = false;

  // ============================================================
  // 内部関数
  // ============================================================
  function getMapPictures(mapId) {
    const entry = mapPictureSets.find(e => Number(e.mapId) === mapId);
    if (!entry) return [];
    return JSON.parse(entry.pictures || "[]").map(d => JSON.parse(d));
  }

  function createCacheForMap(mapId) {
    if (_cacheInitialized) return;
    console.log(`UIPicture: マップ${mapId} キャッシュ生成開始`);

    const dataList = getMapPictures(mapId);
    _currentCache = {};
    for (const data of dataList) {
      const id = Number(data.id);
      const name = data.name;
      _currentCache[id] = {
        name: name,
        origin: Number(data.origin || 0),
        x: Number(data.x || 0),
        y: Number(data.y || 0),
        scaleX: Number(data.scaleX || 100),
        scaleY: Number(data.scaleY || 100),
        opacity: Number(data.opacity || 255),
        blendMode: Number(data.blendMode || 0),
        visible: false
      };
      ImageManager.loadPicture(name);
    }
    _cacheInitialized = true;
    console.log("UIPicture: キャッシュ生成完了");
  }

  function destroyCache() {
    if (!_cacheInitialized) return;
    console.log("UIPicture: キャッシュ破棄");
    // 表示中ピクチャ削除
    if (_currentCache) {
      for (const id in _currentCache) {
        $gameScreen.erasePicture(Number(id));
      }
    }
    _currentCache = null;
    _cacheInitialized = false;
  }

  // ============================================================
  // スイッチON/OFF検知
  // ============================================================
  const _Game_Switches_setValue = Game_Switches.prototype.setValue;
  Game_Switches.prototype.setValue = function(switchId, value) {
    const old = this.value(switchId);
    _Game_Switches_setValue.call(this, switchId, value);
    if (switchId === loadSwitchId && value !== old) {
      if (value) {
        createCacheForMap($gameMap.mapId());
      } else {
        destroyCache();
      }
    }
  };

  // ============================================================
  // マップ移動時のキャッシュ更新
  // ============================================================
  const _Scene_Map_start = Scene_Map.prototype.start;
  Scene_Map.prototype.start = function() {
    _Scene_Map_start.call(this);
    if ($gameSwitches.value(loadSwitchId)) {
      createCacheForMap($gameMap.mapId());
      if (_currentCache) {
        for (const id in _currentCache) {
          const pic = _currentCache[id];
          if (pic.visible) UIPicture._applyPicture(Number(id), pic);
        }
      }
    }
  };

  // ============================================================
  // グローバル制御オブジェクト
  // ============================================================
  window.UIPicture = {
    show(id) {
      if (!_cacheInitialized) return console.warn("UIPicture: キャッシュ未生成。スイッチONを確認してください。");
      const pic = _currentCache[id];
      if (!pic) return console.warn(`UIPicture: ID ${id} は未登録。`);
      pic.visible = true;
      this._applyPicture(id, pic);
    },
    hide(id) {
      if (!_cacheInitialized) return console.warn("UIPicture: キャッシュ未生成。スイッチONを確認してください。");
      const pic = _currentCache[id];
      if (!pic) return console.warn(`UIPicture: ID ${id} は未登録。`);
      pic.visible = false;
      $gameScreen.erasePicture(id);
    },
    _applyPicture(id, pic) {
      $gameScreen.showPicture(
        id,
        pic.name,
        pic.origin,
        pic.x,
        pic.y,
        pic.scaleX,
        pic.scaleY,
        pic.opacity,
        pic.blendMode
      );
    }
  };
})();