/*:
 * @target MV
 * @plugindesc スイッチ単位でピクチャキャッシュ生成・OFFで破棄するプラグイン（ピクチャ同階層描画）
 *
 * @param SwitchPictureSets
 * @text スイッチ別ピクチャ設定リスト
 * @type struct<SwitchPictureSet>[]
 * @default []
 *
 * @help
 * 【概要】
 * - スイッチONでそのスイッチ用のピクチャキャッシュを生成
 * - OFFで破棄＆表示中ピクチャ削除
 * - UIPicture.show(id, switchId) / hide(id, switchId) で表示/非表示切替
 */

/*~struct~SwitchPictureSet:
 * @param switchId
 * @text 対象スイッチID
 * @type switch
 *
 * @param pictures
 * @text ピクチャ設定リスト
 * @type struct<PictureData>[]
 * @default []
 */

/*~struct~PictureData:
 * @param id
 * @text ピクチャID
 * @type number
 * @default 1
 *
 * @param name
 * @text ファイル名
 * @type file
 * @dir img/pictures
 *
 * @param origin
 * @text 原点
 * @type select
 * @option 左上
 * @value 0
 * @option 中心
 * @value 1
 * @default 0
 *
 * @param x
 * @text X座標
 * @type number
 * @default 0
 *
 * @param y
 * @text Y座標
 * @type number
 * @default 0
 *
 * @param scaleX
 * @text 拡大率X(%)
 * @type number
 * @default 100
 *
 * @param scaleY
 * @text 拡大率Y(%)
 * @type number
 * @default 100
 *
 * @param opacity
 * @text 不透明度
 * @type number
 * @max 255
 * @default 255
 *
 * @param blendMode
 * @text 合成方法
 * @type select
 * @option 通常
 * @value 0
 * @option 加算
 * @value 1
 * @option 乗算
 * @value 2
 * @option スクリーン
 * @value 3
 * @default 0
 */

(() => {
  const pluginName = document.currentScript.src.match(/([^/]+)\.js$/)[1];
  const params = PluginManager.parameters(pluginName);
  const switchPictureSets = JSON.parse(params["SwitchPictureSets"] || "[]").map(d => JSON.parse(d));

  // スイッチ単位のキャッシュ管理
  const _cacheBySwitch = {}; // { switchId: { id: picData, ... }, ... }

  function getSwitchPictures(switchId) {
    const entry = switchPictureSets.find(e => Number(e.switchId) === Number(switchId));
    if (!entry) return [];
    return JSON.parse(entry.pictures || "[]").map(d => JSON.parse(d));
  }

  function createCacheForSwitch(switchId) {
    if (_cacheBySwitch[switchId]) return;
    console.log(`UIPicture: スイッチ${switchId} キャッシュ生成`);

    const dataList = getSwitchPictures(switchId);
    const cache = {};
    for (const data of dataList) {
      const id = Number(data.id);
      cache[id] = {
        name: data.name,
        origin: Number(data.origin || 0),
        x: Number(data.x || 0),
        y: Number(data.y || 0),
        scaleX: Number(data.scaleX || 100),
        scaleY: Number(data.scaleY || 100),
        opacity: Number(data.opacity || 255),
        blendMode: Number(data.blendMode || 0),
        visible: false
      };
      ImageManager.loadPicture(data.name);
    }
    _cacheBySwitch[switchId] = cache;
  }

  function destroyCacheForSwitch(switchId) {
    const cache = _cacheBySwitch[switchId];
    if (!cache) return;
    console.log(`UIPicture: スイッチ${switchId} キャッシュ破棄`);
    for (const id in cache) {
      $gameScreen.erasePicture(Number(id));
    }
    delete _cacheBySwitch[switchId];
  }

  // ============================================================
  // スイッチON/OFF検知
  // ============================================================
  const _Game_Switches_setValue = Game_Switches.prototype.setValue;
  Game_Switches.prototype.setValue = function(switchId, value) {
    const old = this.value(switchId);
    _Game_Switches_setValue.call(this, switchId, value);
    if (value !== old) {
      if (value) {
        createCacheForSwitch(switchId);
      } else {
        destroyCacheForSwitch(switchId);
      }
    }
  };

  // ============================================================
  // マップ移動時のキャッシュ保持
  // ============================================================
  const _Scene_Map_start = Scene_Map.prototype.start;
  Scene_Map.prototype.start = function() {
    _Scene_Map_start.call(this);
    for (const switchId in _cacheBySwitch) {
      const cache = _cacheBySwitch[switchId];
      for (const id in cache) {
        if (cache[id].visible) UIPicture._applyPicture(Number(id), cache[id]);
      }
    }
  };

  // ============================================================
  // グローバル制御オブジェクト
  // ============================================================
  window.UIPicture = {
    show(id, switchId) {
      const cache = _cacheBySwitch[switchId];
      if (!cache) return console.warn(`UIPicture: スイッチ${switchId}のキャッシュ未生成`);
      const pic = cache[id];
      if (!pic) return console.warn(`UIPicture: ID ${id} は未登録`);
      pic.visible = true;
      this._applyPicture(id, pic);
    },
    hide(id, switchId) {
      const cache = _cacheBySwitch[switchId];
      if (!cache) return console.warn(`UIPicture: スイッチ${switchId}のキャッシュ未生成`);
      const pic = cache[id];
      if (!pic) return console.warn(`UIPicture: ID ${id} は未登録`);
      pic.visible = false;
      $gameScreen.erasePicture(id);
    },
    _applyPicture(id, pic) {
      $gameScreen.showPicture(
        id,
        pic.name,
        pic.origin,
        pic.x,
        pic.y,
        pic.scaleX,
        pic.scaleY,
        pic.opacity,
        pic.blendMode
      );
    }
  };
})();
