//=============================================================================
// 100CursorControl.js
//=============================================================================

/*:
 * @plugindesc 条件に応じてマウスカーソル画像を切り替える
 *
 * @param DefaultCursor
 * @text デフォルトカーソル
 * @type file
 * @dir img/pictures
 * @desc 常に使用するカーソル画像。ルールより優先度は低い。
 * @default
 *
 * @param Rules
 * @text カーソル切替ルール
 * @type struct<CursorRule>[]
 * @default []
 * @desc 条件を満たしたときにマウスカーソルを変更するルールを定義します。
 *
 * @help
 * 条件に応じてマウスカーソル画像を切り替えます。
 * カーソル画像は img/pictures/ フォルダに配置してください。
 */

/*~struct~CursorRule:
 * @param SwitchId
 * @text 条件スイッチ
 * @type switch
 * @default 0
 * @desc このスイッチが ON なら変更。0で無視。
 *
 * @param VariableId
 * @text 条件変数
 * @type variable
 * @default 0
 * @desc この変数を条件判定に使用。0で無視。
 *
 * @param Compare
 * @text 変数判定方法
 * @type select
 * @option 無視
 * @value none
 * @option = 指定値と等しい
 * @value equal
 * @option >= 指定値以上
 * @value greaterEqual
 * @option <= 指定値以下
 * @value lessEqual
 * @default none
 *
 * @param Value
 * @text 判定値
 * @type number
 * @default 0
 * @desc 変数判定に使う値
 *
 * @param CursorImage
 * @text カーソル画像
 * @type file
 * @dir img/pictures
 * @desc img/pictures/ に置いたファイル名
 * @default
 */

(function() {
    var pluginName = "100CursorControl";
    var parameters = PluginManager.parameters(pluginName);
    var defaultCursor = String(parameters["DefaultCursor"] || "");
    var rules = JSON.parse(parameters["Rules"] || "[]").map(function(r) {
        return JSON.parse(r);
    });

    var currentCursorName = "";

    function getCursorStyle(name) {
        if (!name) return "auto";
        return "url(img/pictures/" + name + ".png), default";
    }

    function updateMousePointer() {
        if (!document.getElementById('CursorControlPointer')) {
            var canvas = document.getElementById('GameCanvas');
            if (!canvas) return;
            var span = document.createElement('span');
            span.id = 'CursorControlPointer';
            span.style.position = 'absolute';
            span.style.left = '0px';
            span.style.top = '0px';
            span.style.right = '0px';
            span.style.bottom = '0px';
            span.style.width = (Number(canvas.style.width.replace('px', '')) - 16) + 'px';
            span.style.height = (Number(canvas.style.height.replace('px', '')) - 16) + 'px';
            span.style.margin = 'auto';
            span.style.zIndex = 999;
            span.oncontextmenu = function() { return false; };
            document.body.appendChild(span);
        }
        document.getElementById('CursorControlPointer').style.cursor = getCursorStyle(currentCursorName);
    }

    var windowResizeTimeoutId = null;
    window.addEventListener('resize', function(event) {
        var span = document.getElementById('CursorControlPointer');
        if (span) {
            clearTimeout(windowResizeTimeoutId);
            windowResizeTimeoutId = setTimeout(function() {
                var canvas = document.getElementById('GameCanvas');
                if (canvas) {
                    span.style.width = (Number(canvas.style.width.replace('px', '')) - 16) + 'px';
                    span.style.height = (Number(canvas.style.height.replace('px', '')) - 16) + 'px';
                }
            }, 200);
        }
    });

    function setCursor(name) {
        currentCursorName = name;
        updateMousePointer();
    }

    function evaluateCursor() {
        if (!$gameSwitches || !$gameVariables) return;

        for (var i = 0; i < rules.length; i++) {
            var rule = rules[i];
            var ok = true;

            var switchId = Number(rule.SwitchId || 0);
            if (switchId > 0 && !$gameSwitches.value(switchId)) {
                ok = false;
            }

            var variableId = Number(rule.VariableId || 0);
            var compare = rule.Compare || "none";
            var value = Number(rule.Value || 0);
            if (variableId > 0 && compare !== "none") {
                var v = $gameVariables.value(variableId);
                if (compare === "equal" && v !== value) ok = false;
                if (compare === "greaterEqual" && v < value) ok = false;
                if (compare === "lessEqual" && v > value) ok = false;
            }

            if (ok && rule.CursorImage) {
                setCursor(rule.CursorImage);
                return;
            }
        }
        setCursor(defaultCursor);
    }

    if (defaultCursor) {
        setCursor(defaultCursor);
    }

    var _SceneManager_updateScene = SceneManager.updateScene;
    SceneManager.updateScene = function() {
        _SceneManager_updateScene.apply(this, arguments);
        if (this._scene && $gameSwitches) {
            evaluateCursor();
        }
        updateMousePointer();
    };
})();
