//=============================================================================
// NrCustomMessage.js (MV対応版)
//=============================================================================

/*:
 * @plugindesc 独立メッセージ表示プラグイン v1.0.0（MV対応）
 * @author 公也
 *
 * @param FontSize
 * @text フォントサイズ
 * @type number
 * @default 28
 *
 * @param x
 * @text X座標
 * @type number
 * @default 100
 *
 * @param y
 * @text Y座標
 * @type number
 * @default 50
 *
 * @param width
 * @text 幅
 * @type number
 * @default 600
 *
 * @param height
 * @text 高さ
 * @type number
 * @default 120
 *
 * @param autoSwitchId
 * @text オートスイッチID
 * @type switch
 * @default 0
 *
 * @param autoWaitFormula
 * @text オート待機フレーム式
 * @default 100 + text.length * 10
 * @type string
 *
 * @param SpeedVariableId
 * @text 表示速度変数ID
 * @type variable
 * @default 1
 *
 * @help
 * スクリプトから以下のように使用：
 *   NrCustomMessage.create("表示テキスト");
 *   NrCustomMessage.setText("差し替えテキスト");
 *   NrCustomMessage.hide();
 *   NrCustomMessage.show();
 *   NrCustomMessage.destroy();
 *   NrCustomMessage.updateSpeed();
 *   NrCustomMessage.flush();
 */

(function() {
    'use strict';

    var parameters = PluginManager.parameters('NrCustomMessage');
    var posX = Number(parameters['x'] || 100);
    var posY = Number(parameters['y'] || 50);
    var winW = Number(parameters['width'] || 600);
    var winH = Number(parameters['height'] || 120);
    var autoSw = Number(parameters['autoSwitchId'] || 0);
    var autoWaitFormula = String(parameters['autoWaitFormula'] || "100 + text.length * 10");
    var speedVarId = Number(parameters['SpeedVariableId'] || 1);
    var fontSize = Number(parameters['FontSize'] || 28);

    function Window_IndMsg() {
        this.initialize.apply(this, arguments);
    }

    Window_IndMsg.prototype = Object.create(Window_Base.prototype);
    Window_IndMsg.prototype.constructor = Window_IndMsg;

    Window_IndMsg.prototype.initialize = function(x, y, width, height) {
        Window_Base.prototype.initialize.call(this, x, y, width, height);
        this._text = "";
        this._textIndex = 0;
        this._waitCount = 0;
        this._autoWait = 0;
        this._messageSpeed = 1;
        this._drawX = 0;
        this._drawY = 0;
        this._autoSwitchId = autoSw;
        this.hide();
        this.contents.fontSize = fontSize;
    };

    Window_IndMsg.prototype.setText = function(text) {
        this._text = this.convertSpecialCodes(String(text || ""));
        this._textIndex = 0;
        this._waitCount = 0;
        this._autoWait = this.calcAutoWait(this._text);
        this._messageSpeed = $gameVariables.value(speedVarId);
        this.contents.clear();
        this.refresh();
        this.show();
        this.open();
    };

    Window_IndMsg.prototype.convertSpecialCodes = function(text) {
        return text.replace(/\\n/g, "\n");
    };

    Window_IndMsg.prototype.refresh = function() {
        var rect = this.baseTextRect();
        this._drawX = rect.x;
        this._drawY = rect.y;
    };

    Window_IndMsg.prototype.baseTextRect = function() {
        return new Rectangle(0, 0, this.contentsWidth(), this.contentsHeight());
    };

    Window_IndMsg.prototype.calcAutoWait = function(text) {
        try {
            return Math.max(0, Number(Function("text", "return " + autoWaitFormula)(text)) || 0);
        } catch (e) {
            console.error("NrCustomMessage: autoWaitFormula 評価エラー", e);
            return 60;
        }
    };

    Window_IndMsg.prototype.update = function() {
        Window_Base.prototype.update.call(this);
        if (!this.visible) return;

        this._messageSpeed = $gameVariables.value(speedVarId);

        if (this._waitCount > 0) {
            this._waitCount--;
            return;
        }

        if (this._textIndex < this._text.length) {
            var speed = Number(this._messageSpeed) || 0;

            if (speed <= 0) {
                this.flushNow();
                return;
            }

            var c = this._text[this._textIndex++];
            this.drawOneChar(c);
            this._waitCount = (speed <= 1) ? 0 : speed;
        } else if (autoSw > 0 && $gameSwitches.value(autoSw)) {
            this._autoWait--;
            if (this._autoWait <= 0) {
                this._textIndex = 0;
                this.contents.clear();
                this._autoWait = this.calcAutoWait(this._text);
                this.refresh();
            }
        }
    };

    Window_IndMsg.prototype.drawOneChar = function(c) {
        var rect = this.baseTextRect();
        var remaining = Math.max(1, rect.width - (this._drawX - rect.x));
        if (c === "\n") {
            this._drawX = rect.x;
            this._drawY += this.lineHeight();
        } else {
            this.contents.drawText(c, this._drawX, this._drawY, remaining, this.lineHeight());
            this._drawX += this.textWidth(c);
        }
    };

    Window_IndMsg.prototype.flushNow = function() {
        while (this._textIndex < this._text.length) {
            var c = this._text[this._textIndex++];
            this.drawOneChar(c);
        }
        this._waitCount = 0;
    };

    window.NrCustomMessage = {
        _window: null,
        _defaultX: posX,
        _defaultY: posY,
        _defaultOpacity: 255,
        _tempX: null,
        _tempY: null,
        _tempOpacity: null,

        create: function(text) {
            const scene = SceneManager._scene;
            if (!scene) return;

            if (!this._window) {
                const win = new Window_IndMsg(this._defaultX, this._defaultY, winW, winH);
                win.opacity = this._defaultOpacity;
                const layer = scene._windowLayer || (scene._spriteset && scene._spriteset._windowLayer);
                if (!layer) return;
                layer.addChild(win);
                this._window = win;
            }

            const win = this._window;
            win.x = (this._tempX != null) ? this._tempX : this._defaultX;
            win.y = (this._tempY != null) ? this._tempY : this._defaultY;
            win.opacity = (this._tempOpacity != null) ? this._tempOpacity : this._defaultOpacity;

            this.setText(text);
            win.show();
            win.open();
        },

        setText: function(text) {
            if (this._window) {
                this._window.setText(text);
            }
        },

        setPosition: function(x, y) {
            this._tempX = x;
            this._tempY = y;
            if (this._window) {
                this._window.x = x;
                this._window.y = y;
            }
        },

        setOpacity: function(opacity) {
            this._tempOpacity = opacity;
            if (this._window) {
                this._window.opacity = opacity;
            }
        },

        show: function() {
            if (this._window) {
                this._window.x = this._defaultX;
                this._window.y = this._defaultY;
                this._window.opacity = this._defaultOpacity;
                this._tempX = null;
                this._tempY = null;
                this._tempOpacity = null;
                this._window.show();
            }
        },

        hide: function() {
            if (this._window) this._window.hide();
        },

        destroy: function() {
            if (this._window) {
                const scene = SceneManager._scene;
                if (scene && scene.children.contains(this._window)) {
                    scene.removeChild(this._window);
                }
                if (this._window.destroy) {
                    this._window.destroy();
                }
                this._window = null;
            }
        },

        updateSpeed: function() {
            if (this._window) {
                this._window._messageSpeed = $gameVariables.value(speedVarId);
            }
        },

        flush: function() {
            if (this._window) this._window.flushNow();
        }
    };
})();
