//=============================================================================
// NrCustomNumber.js
//=============================================================================

/*:
 * @plugindesc 変数の数字を画像で表示 (MV 対応) v1.0.0
 * @author トリアコンタン MarkⅡ
 *
 * @param configs
 * @text 表示設定リスト
 * @type struct<Config>[]
 * @desc 数字表示の設定リスト
 * @default []
 *
 * @help
 * - NrCustomNumber.move(name, dx, dy);
 * - NrCustomNumber.setPosition(name, x, y);
 * - NrCustomNumber.getPosition(name);
 * - NrCustomNumber.setImage(name, imageName);
 *
 */

/*~struct~Config:
 * @param name
 * @text 名前
 * @type string
 * @default
 *
 * @param image
 * @text 画像ファイル名
 * @type file
 * @dir img/system
 * @default
 *
 * @param x
 * @text X座標
 * @type number
 * @default 0
 *
 * @param y
 * @text Y座標
 * @type number
 *
 * @param switchId
 * @text 表示スイッチID
 * @type switch
 * @default 0
 *
 * @param hideSwitchIds
 * @text 非表示スイッチIDリスト（カンマ区切り）
 * @type string
 * @default
 *
 * @param variableId
 * @text 変数ID
 * @type variable
 * @default 0
 *
 * @param align
 * @text 揃え
 * @type select
 * @option 左揃え
 * @value left
 * @option 中央揃え
 * @value center
 * @option 右揃え
 * @value right
 * @default left
 *
 * @param digits
 * @text 桁数
 * @type number
 * @min 0
 * @default 0
 *
 * @param zeroPad
 * @text 0埋め
 * @type boolean
 * @default false
 *
 * @param spacing
 * @text 数字間の幅
 * @type number
 * @default 0
 */

(function() {
    'use strict';

    var pluginName = 'NrCustomNumber';
    var parameters = PluginManager.parameters(pluginName);
    var configsRaw = parameters['configs'] || '[]';
    var configs = [];
    try {
        configs = JSON.parse(configsRaw).map(function(item) { return JSON.parse(item); });
    } catch(e) {
        configs = [];
    }

    function Sprite_NumberDisplay(config) {
        this.initialize.apply(this, arguments);
    }

    Sprite_NumberDisplay.prototype = Object.create(Sprite.prototype);
    Sprite_NumberDisplay.prototype.constructor = Sprite_NumberDisplay;

    Sprite_NumberDisplay.prototype.initialize = function(config) {
        Sprite.prototype.initialize.call(this);
        this._baseX      = Number(config.x || 0);
        this._baseY      = Number(config.y || 0);
        this.x           = this._baseX;
        this.y           = this._baseY;
        this._name       = config.name || '';
        this._switchId   = Number(config.switchId || 0);
        this._hideSwitchIds = (config.hideSwitchIds || '').split(',').map(function(s){ return Number(s.trim() || 0); }).filter(function(n){ return n > 0; });
        this._variableId = Number(config.variableId || 0);
        this._imageName  = config.image || '';
        this._align      = config.align || 'left';
        this._digits     = Number(config.digits || 0);
        this._zeroPad    = (config.zeroPad === true || config.zeroPad === 'true');
        this._spacing    = Number(config.spacing || 0);
        this._baseBitmap = null;
        this._lastValue  = null;
        this.bitmap      = null;

        if (this._imageName) {
            this._baseBitmap = ImageManager.loadSystem(this._imageName);
            var self = this;
            this._baseBitmap.addLoadListener(function() {
                var v = $gameVariables ? ($gameVariables.value(self._variableId) || 0) : 0;
                self._lastValue = null;
                self._refreshBitmap(v);
            });
        }
        this.visible = false;
    };

    Sprite_NumberDisplay.prototype.update = function() {
        Sprite.prototype.update.call(this);

        var shouldVisible = true;
        if (this._switchId > 0) {
            shouldVisible = !!$gameSwitches.value(this._switchId);
        }
        if (this._hideSwitchIds && this._hideSwitchIds.length) {
            for (var i = 0; i < this._hideSwitchIds.length; i++) {
                if ($gameSwitches.value(this._hideSwitchIds[i])) {
                    shouldVisible = false;
                    break;
                }
            }
        }
        this.visible = shouldVisible;

        if (!this._baseBitmap || !this._baseBitmap.isReady()) {
            return;
        }

        var value = $gameVariables.value(this._variableId) || 0;
        if (this._lastValue !== value || !this.bitmap) {
            this._refreshBitmap(value);
            this._lastValue = value;
        }
    };

    Sprite_NumberDisplay.prototype._refreshBitmap = function(value) {
        if (!this._baseBitmap || !this._baseBitmap.isReady()) return;
        var img    = this._baseBitmap;
        var digitW = Math.floor(img.width / 10);
        var digitH = img.height;
        var digitsArr;

        if (this._digits > 0) {
            if (String(Math.floor(value)).length > this._digits) {
                digitsArr = Array(this._digits).fill('9');
            } else {
                digitsArr = String(Math.max(0, Math.floor(value))).split('');
                while (digitsArr.length < this._digits) {
                    if (this._zeroPad) {
                        digitsArr.unshift('0');
                    } else {
                        digitsArr.unshift(' ');
                    }
                }
            }
        } else {
            digitsArr = String(Math.max(0, Math.floor(value))).split('');
        }

        var drawnDigits = digitsArr.map(function(d,i){ return { char: d, index: i }; }).filter(function(d){ return d.char !== ' '; });
        var totalW = drawnDigits.length > 0 ? (digitW * drawnDigits.length + this._spacing * (drawnDigits.length - 1)) : 1;
        var resultBitmap = new Bitmap(totalW, digitH);

        drawnDigits.forEach((d, drawIndex) => {
            var n = Number(d.char);
            var x = drawIndex * (digitW + this._spacing);
            resultBitmap.blt(img, n * digitW, 0, digitW, digitH, x, 0);
        });

        this.bitmap = resultBitmap;
        this.setFrame(0, 0, totalW, digitH);

        var anchorX = 0;
        if (this._align === 'left') {
            this.x = this._baseX;
        } else if (this._align === 'center') {
            this.x = this._baseX - Math.floor(totalW / 2);
        } else if (this._align === 'right') {
            this.x = this._baseX - totalW;
        }
        this.y = this._baseY;
    };

    // Scene_Map start override to add sprites
    var _Scene_Map_start = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function() {
        _Scene_Map_start.call(this);
        this._numberSprites = [];
        configs.forEach(function(config) {
            var sprite = new Sprite_NumberDisplay(config);
            this._numberSprites.push(sprite);
            this.addChild(sprite);
        }, this);
    };

    // Plugin commands for MV require overriding Game_Interpreter.prototype.pluginCommand
    var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function(command, args) {
        _Game_Interpreter_pluginCommand.call(this, command, args);
        if (command === 'NrCustomNumber_move') {
            var name = String(args[0] || '');
            var dx   = Number(args[1] || 0);
            var dy   = Number(args[2] || 0);
            var scene = SceneManager._scene;
            if (scene && scene._numberSprites) {
                var s = scene._numberSprites.find(function(sp){ return sp._name === name; });
                if (s) {
                    s._baseX += dx;
                    s._baseY += dy;
                    s._refreshBitmap(s._lastValue);
                }
            }
        } else if (command === 'NrCustomNumber_setImage') {
            var name     = String(args[0] || '');
            var imageName= String(args[1] || '');
            var scene    = SceneManager._scene;
            if (scene && scene._numberSprites) {
                var s = scene._numberSprites.find(function(sp){ return sp._name === name; });
                if (s && imageName) {
                    s._imageName  = imageName;
                    s._baseBitmap = ImageManager.loadSystem(imageName);
                    s._baseBitmap.addLoadListener(function(){
                        s._refreshBitmap($gameVariables.value(s._variableId));
                    });
                }
            }
        }
    };

    // Exposed API
    window.NrCustomNumber = {
        move: function(name, dx, dy) {
            var scene = SceneManager._scene;
            if (scene && scene._numberSprites) {
                var s = scene._numberSprites.find(function(sp){ return sp._name === name; });
                if (s) {
                    s._baseX += dx;
                    s._baseY += dy;
                    s._refreshBitmap(s._lastValue);
                }
            }
        },
        setPosition: function(name, x, y) {
            var scene = SceneManager._scene;
            if (scene && scene._numberSprites) {
                var s = scene._numberSprites.find(function(sp){ return sp._name === name; });
                if (s) {
                    s._baseX = x;
                    s._baseY = y;
                    s._refreshBitmap(s._lastValue);
                }
            }
        },
        getPosition: function(name) {
            var scene = SceneManager._scene;
            if (scene && scene._numberSprites) {
                var s = scene._numberSprites.find(function(sp){ return sp._name === name; });
                if (s) {
                    return { x: s._baseX, y: s._baseY };
                }
            }
            return null;
        },
        setImage: function(name, imageName) {
            var scene = SceneManager._scene;
            if (scene && scene._numberSprites) {
                var s = scene._numberSprites.find(function(sp){ return sp._name === name; });
                if (s && imageName) {
                    s._imageName  = imageName;
                    s._baseBitmap = ImageManager.loadSystem(imageName);
                    s._baseBitmap.addLoadListener(function(){
                        s._refreshBitmap($gameVariables.value(s._variableId));
                    });
                }
            }
        }
    };

})();
