//=============================================================================
// NrMyKey.js
//=============================================================================

/*:
 * @target MV
 * @plugindesc 任意キー登録＋キー押下でコモンイベント実行＋プラグイン側メニュー禁止制御
 *
 * @param MenuDisabled
 * @text メニュー呼び出しの禁止
 * @type boolean
 * @default false
 * @desc trueにするとEscキー、Xキー、右クリックなどの通常メニューを無効化します（コモンイベント用のキーは除く）
 *
 * @param ExceptionKeys
 * @text 例外キー
 * @type string[]
 * @default []
 * @desc MenuDisabledがtrueでもコモンイベント呼び出しに使えるキー名（小文字）を指定

 * @param DefineKeys
 * @text 任意キー定義
 * @type struct<DefineKey>[]
 * @default []
 * @desc キー名とキーコードの対応を定義します

 * @param KeyBindings
 * @text キーバインド設定
 * @type struct<KeyBinding>[]
 * @default []
 * @desc キー名と対応するコモンイベント、制御スイッチを指定します
 */

/*~struct~DefineKey:
 * @param key
 * @text キー名
 * @type string
 * @desc 使用したいキーの名前（小文字）

 * @param keyCode
 * @text キーコード
 * @type number
 * @desc JavaScriptのキーコード（例: A=65, S=83）
 */

/*~struct~KeyBinding:
 * @param key
 * @text キー名
 * @type string
 * @desc 反応させるキー名（小文字）

 * @param eventId
 * @text コモンイベントID
 * @type common_event
 * @desc このキーが押された時に実行されるコモンイベント

 * @param allowSwitchIds
 * @text 実行許可スイッチ
 * @type switch[]
 * @default []
 * @desc 全てONで実行許可（空欄なら常に許可）

 * @param denySwitchIds
 * @text 実行禁止スイッチ
 * @type switch[]
 * @default []
 * @desc 1つでもONで実行不可（空欄なら常に許可）
 */

(function() {
    'use strict';

    const p = PluginManager.parameters("NrMyKey");
    const menuDisabled = p["MenuDisabled"] === "true";
    const exceptionKeys = JSON.parse(p["ExceptionKeys"] || "[]").map(k => k.toLowerCase());
    const defineKeys = JSON.parse(p["DefineKeys"] || "[]").map(e => JSON.parse(e));
    const keyBindingsRaw = JSON.parse(p["KeyBindings"] || "[]").map(e => JSON.parse(e));

    defineKeys.forEach(entry => {
        const key = String(entry.key || "").toLowerCase();
        const code = Number(entry.keyCode);
        if (key && code) Input.keyMapper[code] = key;
    });

    function parseSwitchArray(jsonStr) {
        if (!jsonStr) return [];
        try {
            const arr = JSON.parse(jsonStr);
            if (!Array.isArray(arr)) return [];
            return arr.map(id => Number(id)).filter(id => id > 0);
        } catch (e) {
            return [];
        }
    }

    const keyBindings = keyBindingsRaw.map(entry => {
        return {
            key: String(entry.key || "").toLowerCase(),
            eventId: Number(entry.eventId || 0),
            allow: parseSwitchArray(entry.allowSwitchIds),
            deny: parseSwitchArray(entry.denySwitchIds)
        };
    });

    function isBindingEnabled(entry) {
        const isAllowed = entry.allow.length === 0 || 
                         entry.allow.every(id => $gameSwitches.value(id));
        const isDenied = entry.deny.length > 0 && 
                        entry.deny.some(id => $gameSwitches.value(id));
        return isAllowed && !isDenied;
    }

    if (menuDisabled) {
        for (const code in Input.keyMapper) {
            if (Input.keyMapper[code] === 'menu') {
                delete Input.keyMapper[code];
            }
        }

        const _Scene_Map_isMenuCalled = Scene_Map.prototype.isMenuCalled;
        Scene_Map.prototype.isMenuCalled = function() {
            return false;
        };

        const _Scene_Map_updateCallMenu = Scene_Map.prototype.updateCallMenu;
        Scene_Map.prototype.updateCallMenu = function() {
        };
    }

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.call(this);
        if ($gameMap.isEventRunning()) return;

        for (const entry of keyBindings) {
            if (!entry.key || entry.eventId <= 0) continue;
            if (menuDisabled && entry.key === 'menu') continue;
            if (!isBindingEnabled(entry)) continue;

            const key = entry.key;
            if (Input.isTriggered(key)) {
                $gameTemp.reserveCommonEvent(entry.eventId);
            }
        }
    };

})();