//=============================================================================
// NrStockLimitShop.js
//=============================================================================

/*:
 * @plugindesc 在庫制限付きショップ v1.0.0
 * @author 
 * 
 * @param HideSellCommand
 * @text 売却コマンドを非表示
 * @desc ショップの売却コマンドを非表示にするかどうか。
 * trueで非表示/falseで表示。
 * @type boolean
 * @default false
 * 
 * @help
 * ■ 基本設定
 * <Stock:5> で購入在庫数を設定。
 * 
 * ■ プラグインコマンド
 * ResetStock item 5   - アイテムID5の在庫をリセット
 * ResetStock weapon 3 - 武器ID3の在庫をリセット
 * ResetStock armor 2  - 防具ID2の在庫をリセット
 * 
 * ■ スクリプトコマンド
 * StockLimitShop.resetItem(id)   - アイテムの在庫をリセット
 * StockLimitShop.resetWeapon(id) - 武器の在庫をリセット
 * StockLimitShop.resetArmor(id)  - 防具の在庫をリセット
 * StockLimitShop.resetAll()      - 全ての在庫をリセット
 * StockLimitShop.getStock(type, id) - 現在の在庫数を取得（type: 0=item, 1=weapon, 2=armor）
 * StockLimitShop.setStock(type, id, amount) - 在庫数を直接設定
 * 
 * 
 * バージョン:
 * v1.0.0 初回
 *
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可です。
 */

(function() {
    'use strict';

    var parameters = PluginManager.parameters('NrStockLimitShop');
    var hideSellCommand = String(parameters['HideSellCommand'] || 'false') === 'true';
    
    var stockData = {};

    function getItemStock(item) {
        if (!item || !item.meta || !item.meta.Stock) return null;
        return parseInt(item.meta.Stock, 10);
    }

    function getItemCategory(item) {
        if (DataManager.isItem(item)) return 0;
        if (DataManager.isWeapon(item)) return 1;
        if (DataManager.isArmor(item)) return 2;
        return 0;
    }

    function getCurrentStock(item) {
        if (!item) return null;
        var stock = getItemStock(item);
        if (stock === null) return null;
        
        var key = item.id + ':' + getItemCategory(item);
        var scene = SceneManager._scene;
        
        if (scene && scene._customStock && scene._customStock[key] !== undefined) {
            return scene._customStock[key];
        }
        
        if (stockData[key] !== undefined) {
            return stockData[key];
        }
        
        return stock;
    }

    var _Scene_Shop_prepare = Scene_Shop.prototype.prepare;
    Scene_Shop.prototype.prepare = function(goods, purchaseOnly) {
        _Scene_Shop_prepare.call(this, goods, purchaseOnly);
        this._customStock = {};
        for (var i = 0; i < goods.length; i++) {
            var type = goods[i][0];
            var id = goods[i][1];
            var item = [ $dataItems, $dataWeapons, $dataArmors ][type][id];
            var maxStock = getItemStock(item);
            if (maxStock !== null) {
                var key = id + ':' + type;
                if (stockData[key] === undefined) {
                    stockData[key] = maxStock;
                }
                this._customStock[key] = stockData[key];
            }
        }
    };

    var _Scene_Shop_doBuy = Scene_Shop.prototype.doBuy;
    Scene_Shop.prototype.doBuy = function(number) {
        var item = this._item;
        var stock = getItemStock(item);
        if (stock !== null) {
            var key = item.id + ':' + getItemCategory(item);
            var current = stockData[key] || stock;
            stockData[key] = Math.max(current - number, 0);
            if (this._customStock) {
                this._customStock[key] = stockData[key];
            }
        }
        _Scene_Shop_doBuy.call(this, number);
    };

    if (hideSellCommand) {
        var _Window_ShopCommand_makeCommandList = Window_ShopCommand.prototype.makeCommandList;
        Window_ShopCommand.prototype.makeCommandList = function() {
            this.addCommand(TextManager.buy, 'buy');
            this.addCommand(TextManager.cancel, 'cancel');
        };
        
        var _Window_ShopCommand_itemRect = Window_ShopCommand.prototype.itemRect;
        Window_ShopCommand.prototype.itemRect = function(index) {
            var rect = _Window_ShopCommand_itemRect.call(this, index);
            var offset = Math.floor(this.contentsWidth() / 6);
            rect.x += offset;
            return rect;
        };
    }

    var _Window_ShopBuy_isEnabled = Window_ShopBuy.prototype.isEnabled;
    Window_ShopBuy.prototype.isEnabled = function(item) {
        var stock = getItemStock(item);
        if (stock !== null) {
            var current = getCurrentStock(item);
            return current > 0 && _Window_ShopBuy_isEnabled.call(this, item);
        }
        return _Window_ShopBuy_isEnabled.call(this, item);
    };

    // 購入可能数を在庫数で制限
    var _Scene_Shop_maxBuy = Scene_Shop.prototype.maxBuy;
    Scene_Shop.prototype.maxBuy = function() {
        var max = _Scene_Shop_maxBuy.call(this);
        var item = this._item;
        var stock = getItemStock(item);
        if (stock !== null) {
            var current = getCurrentStock(item);
            if (current !== null) {
                max = Math.min(max, current);
            }
        }
        return max;
    };

    var _Window_ShopStatus_refresh = Window_ShopStatus.prototype.refresh;
    
    Window_ShopStatus.prototype.refresh = function() {
        _Window_ShopStatus_refresh.call(this);
                this.drawStockInfo();
    };

    Window_ShopStatus.prototype.drawStockInfo = function() {
        var item = this._item;
        if (!item) return;
        
        var stock = getItemStock(item);
        if (stock === null) return;
        
        var current = getCurrentStock(item);
        if (current === null) return;
        
        var x = this.textPadding();
        var width = this.contentsWidth() - x * 2;
        
        var y = this.lineHeight();
        
        this.changeTextColor(this.systemColor());
        this.drawText('残り在庫', x, y, width);
        this.resetTextColor();
        this.drawText(current + '/' + stock, x, y, width, 'right');
    };

    var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function(command, args) {
        _Game_Interpreter_pluginCommand.call(this, command, args);
        if (command === 'ResetStock') {
            var type = args[0];
            var id = Number(args[1]);
            var item = null;
            var key = null;
            
            if (type === 'item') {
                item = $dataItems[id];
                key = id + ':0';
            } else if (type === 'weapon') {
                item = $dataWeapons[id];
                key = id + ':1';
            } else if (type === 'armor') {
                item = $dataArmors[id];
                key = id + ':2';
            }
            
            if (item && key && getItemStock(item) !== null) {
                var maxStock = getItemStock(item);
                stockData[key] = maxStock;
                
                var scene = SceneManager._scene;
                if (scene && scene._customStock) {
                    scene._customStock[key] = maxStock;
                }
            }
        }
    };

    Window_ShopNumber.prototype.debugDrawLines = function() {
        var width = this.contentsWidth();
        var lh = this.lineHeight();
        
        for (var i = 0; i < 10; i++) {
            this.drawText('行' + i, 0, lh * i, width, 'left');
        }
    };

    function resetStockInternal(type, id) {
        var dataList = [$dataItems, $dataWeapons, $dataArmors];
        var item = dataList[type][id];
        if (!item) return false;
        
        var maxStock = getItemStock(item);
        if (maxStock === null) return false;
        
        var key = id + ':' + type;
        stockData[key] = maxStock;
        
        var scene = SceneManager._scene;
        if (scene && scene._customStock) {
            scene._customStock[key] = maxStock;
        }
        return true;
    }

    window.StockLimitShop = {
        resetItem: function(id) {
            return resetStockInternal(0, id);
        },
        
        resetWeapon: function(id) {
            return resetStockInternal(1, id);
        },
        
        resetArmor: function(id) {
            return resetStockInternal(2, id);
        },
        
        resetAll: function() {
            stockData = {};
            var scene = SceneManager._scene;
            if (scene && scene._customStock) {
                scene._customStock = {};
            }
        },
        
        getStock: function(type, id) {
            var dataList = [$dataItems, $dataWeapons, $dataArmors];
            var item = dataList[type][id];
            return getCurrentStock(item);
        },
        
        setStock: function(type, id, amount) {
            var dataList = [$dataItems, $dataWeapons, $dataArmors];
            var item = dataList[type][id];
            if (!item) return false;
            
            var maxStock = getItemStock(item);
            if (maxStock === null) return false;
            
            var key = id + ':' + type;
            stockData[key] = Math.max(0, Math.min(amount, maxStock));
            
            var scene = SceneManager._scene;
            if (scene && scene._customStock) {
                scene._customStock[key] = stockData[key];
            }
            return true;
        }
    };
})();