//=============================================================================
// AutoReviveItem.js	2017/10/13
// The MIT License (MIT)
//=============================================================================

/*:
 * @plugindesc 特定のアイテムを持っていた場合死んでも蘇生するプラグイン
 * @author 柊菜緒
 * @version 0.0.7
 * 
 * @param RestoreHPtype
 * @type number
 * @min 0
 * @max 1
 * @desc 0の場合最大HPでの割合でHPを回復させます、1の場合指定された数値分HPを回復します。
 * @default 0
 * 
 * @param RestoreHPRatio
 * @type number
 * @min 0
 * @max 100
 * @desc 割合でHPを回復させる場合の割合を設定
 * @default 100
 * 
 * @param RestoreHPnum
 * @type number
 * @min 0
 * @desc 数値を指定してHPを回復させる場合の数値を設定
 * @default 100
 * 
 * @param RestoreMP
 * @type boolean
 * @desc MPを回復させるかの設定
 * @default true
 * 
 * @param RestoreMPtype
 * @type number
 * @min 0
 * @max 1
 * @desc 0の場合最大MPでの割合でMPを回復させます、1の場合指定された数値分MPを回復します。
 * @default 0

 * @param RestoreMPRatio
 * @type number
 * @min 0
 * @max 100
 * @desc 割合でMPを回復させる場合の割合を設定
 * @default 100

 * @param RestoreMPnum
 * @type number
 * @min 0
 * @desc 数値を指定してMPを回復させる場合の数値を設定
 * @default 100
 * 
 * @param MarkStateNotClear
 * @type boolean
 * @desc マークしておいたステートを死亡時に消さないかを設定
 * @default true
 * 
 * @param ExecuteCommonEvent
 * @type boolean
 * @desc 全滅からの回復時にコモンイベントを呼び出すかを設定
 * @default true
 * 
 * @param CallCommonEvent
 * @type number
 * @desc 全滅からの回復時に呼び出されるコモンイベントを設定します。
 * @default 0
 * 
 * @param RemainCountVariable
 * @type number
 * @desc 残機を示す変数を格納する変数の番号
 * @default 0
 * 
 * @param BeforeReviveCommonEvent
 * @type number
 * @desc 蘇生アイテムを使う前に実行するコモンイベント
 * @default 0
 * 
 * @param ReviveDisableSwitch
 * @type number
 * @desc 蘇生を無効化するためのスイッチ(ONで無効化)
 * @default 0
 * 
 * 
 * @help このプラグインにはプラグインコマンドはありません。
 * 敗北時にコモンイベントなどを発生させるタイプのプラグインと
 * 併用する場合出来るだけあとに読み込んでください。
 * 
 * 復活時にMPを回復する場合指定した数値が死亡時のMPよりも小さい場合は
 * 回復しません。
 * 
 * 持っていたら自動的に復活するアイテムのメモ欄に
 * <ReviveItem>
 * と記入してください。
 * 
 * また、回復時に消したくないステートのメモ欄には
 * <ReviveNotClear>
 * と記入してください。
 * 
 * ※シングルアクター環境で動作をチェックしているためその他の環境では未保証です。
 */

'use strict';//厳格なエラーチェック

var Imported = Imported || {};

(function (_global) {
    function toBoolean(str) {
        if (str == "true") return true;
        return false;
    }

    var params = PluginManager.parameters('AutoReviveItem');
    var setting = {};
    setting.HPtype = Number(params['RestoreHPtype']);
    setting.HPratio = Number(params['RestoreHPRatio']);
    setting.HPnum = Number(params['RestoreHPnum']);
    setting.RestoreMP = toBoolean(params['RestoreMP']);
    setting.MPtype = Number(params['RestoreMPtype']);
    setting.MPratio = Number(params['RestoreMPRatio']);
    setting.MPnum = Number(params['RestoreMPnum']);
    setting.MarkStateNotClear = toBoolean(params['MarkStateNotClear']);
    setting.ExeCommonEvent = toBoolean(params['ExecuteCommonEvent']);
    setting.CallEvent = Number(params['CallCommonEvent']);
    setting.remainVariable = Number(params['RemainCountVariable']);
    setting.beforeReviveEvent = Number(params['BeforeReviveCommonEvent']);
    setting.disableSwitch = Number(params['ReviveDisableSwitch']) || 0;

    var BattleManager_checkBattleEnd = BattleManager.checkBattleEnd;
    BattleManager.checkBattleEnd = function () {
        //全滅判定チェック
        if ($gameParty.isAllDead()) {//全滅なら実行
            //所持アイテムをチェック
            if ((setting.remainVariable > 0) &&
                (setting.beforeReviveEvent > 0) &&
                ($gameVariables.value(setting.remainVariable) > 0)) {
                $gameTemp.reserveCommonEvent(setting.beforeReviveEvent);
                return false;
            }
            var dat = Utility.CheckReviveItem();
            if (dat.flag) {
                $gameParty.reviveItemUse(dat.item);
                return false;
            }
        }
        return BattleManager_checkBattleEnd.call(this);
    };

    Game_BattlerBase.prototype.die = function () {
        this._hp = 0;
        //this.clearStates();
        this.dieNotClearStates();
        this.clearBuffs();
    };

    Game_BattlerBase.prototype.dieNotClearStates = function () {
        var states = this._states;
        //アイテムがある場合ステートを残して置く
        var dat = Utility.CheckReviveItem();
        if (!Imported.StateDieNotClear) {
            states.forEach(function (StateId) {
                if (Utility.isMarkState(StateId) && dat.flag) {
                    return;
                }
                this.eraseState(StateId);
            }, this);
        } else {
            states.forEach(function (StateId) {
                if (Utility.isDieNotClear(StateId) || (Utility.isMarkState(StateId) && dat.flag)) {
                    return;
                }
                this.eraseState(StateId);
            }, this);
        }
    };

    Game_BattlerBase.prototype.ReviveNotClearStates = function () {
        var states = this._states;
        var dat = Utility.CheckReviveItem();
        states.forEach(function (StateId) {
            if (Utility.isMarkState(StateId) && dat.flag) {
                return;
            }
            this.eraseState(StateId);
        }, this);
    };

    Game_Party.prototype.reviveItemUse = function (item) {
        this.battleMembers().forEach(function (actor) {
            if (actor._hp === 0) {
                var HP;
                var MP = actor._mp;
                if (setting.HPtype === 0) {
                    HP = actor.mhp * (setting.HPratio / 100);
                } else if (setting.HPtype === 1) {
                    HP = setting.HPnum;
                }
                actor.setHp(HP);
                if (setting.RestoreMP) {
                    if (setting.MPtype === 0) {
                        if (MP < (actor.mmp * (setting.MPratio / 100))) {
                            MP = actor.mmp * (setting.MPratio / 100);
                        }
                    } else if (setting.MPtype === 1) {
                        if (MP < setting.MPnum) {
                            MP = setting.MPnum;
                        }
                    }
                    actor.setMp(MP);
                }
                if (setting.MarkStateNotClear) {
                    actor.ReviveNotClearStates();
                } else {
                    actor.clearStates();
                }
            }
        });

        if (setting.ExeCommonEvent) {
            if (!isNaN(setting.CallEvent)) {
                $gameTemp.reserveCommonEvent(setting.CallEvent);
            }
        }
        $gameParty.consumeItem(item);
    };

    function Utility() { }
    //死んだときに残すかチェック
    Utility.isDieNotClear = function (id) {
        var dat = $dataStates[id].meta;
        if (dat.DieNotClear !== undefined) {
            return true;
        }
        return false;
    };

    //回復時に残すかチェック
    Utility.isMarkState = function (id) {
        var dat = $dataStates[id].meta;
        if (dat.ReviveNotClear !== undefined) {
            return true;
        }
        return false;
    };

    //復活アイテムの確認
    Utility.CheckReviveItem = function () {
        var dat = {};
        dat.flag = false;
        if (setting.disableSwitch > 0 &&
            $gameSwitches.value(setting.disableSwitch)) {
            return dat;
        }
        $gameParty.items().forEach(function (item) {
            if (item.meta.ReviveItem) {
                dat.flag = true;
                dat.item = item;
            } else {
                return;
            }
        }, this);
        return dat;
    };
})(this);