//=============================================================================
// 盾装備可能二刀流プラグイン
// EXC_DualWield.js
// ----------------------------------------------------------------------------
// Copyright (c) 2024 IdiotException
// This software is released under the MIT License.
// http://opensource.org/licenses/mit-license.php
// ----------------------------------------------------------------------------
// Version
// 1.0.0 2024-12-31
//=============================================================================
/*:
 * @target MZ
 * @plugindesc 二刀流でも盾を装備可能にします
 * @author IdiotException
 * @url 
 * 
 * @help 二刀流でも盾を装備可能にします
 * また、職業だけでなく装備アイテムに”スロットタイプ：二刀流”が
 * 設定されている場合にはそちらにも対応します。
 * 
 * 以下、前提プラグイン
 *     KRD_MZ_DivideEquip
 *     EquipScene_Extension
 * 
 * @command equipWeapon
 * @text 武器装備
 * @desc 指定したアクターの武器スロットに武器を装備します
 * 
 * @arg targetActor
 * @text 対象アクター
 * @desc 装備変更をするアクター
 * @type actor
 * 
 * @arg targetWeapon
 * @text 対象武器
 * @desc 装備する武器
 * @type weapon
 * 
 * @arg targetSlot
 * @text 対象スロット
 * @desc 指定した武器を何番目の武器スロットに装備するか
 * @type number
 * @default 1
 * 
 * @command equipArmor
 * @text 防具装備
 * @desc 指定したアクターの防具スロットに防具を装備します
 * 
 * @arg targetActor
 * @text 対象アクター
 * @desc 装備変更をするアクター
 * @type actor
 * 
 * @arg targetArmor
 * @text 対象防具
 * @desc 装備する防具
 * @type armor
 * 
 * @arg targetSlot
 * @text 対象スロット
 * @desc 指定した防具を同装備タイプの中で
 * 何番目に装備するか
 * @type number
 * @default 1
 * 
 * @command removeEquip
 * @text 装備解除
 * @desc 指定したアクターの指定スロットの装備を外します
 * 
 * @arg targetActor
 * @text 対象アクター
 * @desc 装備変更をするアクター
 * @type actor
 * 
 * @arg equipTypeID
 * @text 対象装備タイプID
 * @desc 装備を外す対象の装備タイプID
 * @type number
 * 
 * @arg targetSlot
 * @text 対象スロット
 * @desc 外す装備が同装備タイプの中で
 * 何番目に装備されているか
 * @type number
 * @default 1
 * 
 * @command exchangeWeapon
 * @text 武器交換
 * @desc 指定したアクターの武器を入れ替えます
 * 
 * @arg targetActor
 * @text 対象アクター
 * @desc 装備変更をするアクター
 * @type actor
 * 
 * @arg beforeWeapon
 * @text 変更武器
 * @desc 変更前に装備されている武器
 * @type weapon
 * 
 * @arg afterWeapon
 * @text 装備武器
 * @desc 変更後に装備する武器
 * @type weapon
 * 
 * @command exchangeArmor
 * @text 防具交換
 * @desc 指定したアクターの防具を入れ替えます
 * 
 * @arg targetActor
 * @text 対象アクター
 * @desc 装備変更をするアクター
 * @type actor
 * 
 * @arg beforeArmor
 * @text 変更防具
 * @desc 変更前に装備されている防具
 * @type armor
 * 
 * @arg afterArmor
 * @text 装備防具
 * @desc 変更後に装備する防具
 * @type armor
 * 
*/
const EXCDualWield = document.currentScript.src.match(/^.*\/(.+)\.js$/)[1];

(function() {
	"use strict";
	//--------------------------------------------------
	// 定数設定
	//--------------------------------------------------
	// ゲーム製作途中で変更されるかもしれない定数
	const PARTY_INDEX = 0;			// 対象アクターのパーティ上でのインデックス
	const DIVIDE_WEAPON_MAX = 4;	// 装備コマンド名（前）に含まれる最大の装備タイプID
	const DUAL_WIELD_SLOT = 1;		// 装備スロットにおける2本目の武器装備スロット

	// 変数宣言
	let _allEquipSlots, _weaponSlots, _weaponIndex, _skillSlots , _skillIndex; //上記削除対象が可変になるので内部で保持する
	
	//--------------------------------------------------
	// 装備の対象スロットが有効かチェック
	//--------------------------------------------------
	function Exc_isEquipSlotEnabled(slots, index){
		if (!slots){
			// slotsがnull等の場合は有効
			return true;
		} else if(slots.length <= index){
			// スロット数よりindexが大きい場合は対象スロットの装備は無効
			return false;
		}
		// スロット番号ごとの処理
		switch(index){
			case DUAL_WIELD_SLOT:
				// 画面上は２スロット目の処理
				if(slots[index] != 1){
					// 対象スロットが武器でない場合、有効
					return true;
				} else if($gameParty.members()[PARTY_INDEX].isDualWield()){
					// 二刀流状態の場合、有効
					return true;
				} else {
					return false;
				}
			default :
				return true;
		}
	}
	//--------------------------------------------------
	// 条件で装備スロットを追加
	//--------------------------------------------------
	function EXC_addEquipSlot(slots){
		// 最初が武器かつ２つめが武器以外の場合、二刀流用のスロットを追加
		if(slots.length >= 1 && slots[0] == 1){
			if(slots.length == 1 || (slots.length > DUAL_WIELD_SLOT && slots[DUAL_WIELD_SLOT] != 1)){
				slots.splice(DUAL_WIELD_SLOT, 0, 1);
			}
		}
		return slots;
	}

	//--------------------------------------------------
	// Scene_EquipDivide のオーバーライド
	//--------------------------------------------------
	const _EXC_Scene_EquipDivide_initialize = Scene_EquipDivide.prototype.initialize;
	Scene_EquipDivide.prototype.initialize = function() {
		_EXC_Scene_EquipDivide_initialize.call(this);

		// EquipScene_Extension の情報を利用して装備スロットを設定
		_allEquipSlots = JSON.parse($dataClasses[$gameParty.members()[PARTY_INDEX]._classId].meta.EquipSlots);

		// 最初が武器かつ２つめが武器以外の場合、二刀流用のスロットを追加
		_allEquipSlots = EXC_addEquipSlot(_allEquipSlots);

		_weaponSlots = _allEquipSlots.filter(el => el <= DIVIDE_WEAPON_MAX);
		_skillSlots = _allEquipSlots.filter(el => el > DIVIDE_WEAPON_MAX);
		_weaponIndex = 0;
		_skillIndex = _weaponSlots.length;
		this._allSlots = _allEquipSlots;
	};


	// 装備変更確定時の処理、slotIdで無効スロットを無視するよう修正
	Scene_EquipDivide.prototype.executeEquipChange = function() {
		const actor = this.actor();
		const slotId = this._slotWindow.enableEquipSlot(this._slotWindow.index() + this._slotIndex);
		const item = this._itemWindow.item();
		actor.changeEquip(slotId, item);
	};

	//--------------------------------------------------
	// Scene_EquipWeapon のオーバーライド
	//--------------------------------------------------
	const _EXC_Scene_EquipWeapon_initialize = Scene_EquipWeapon.prototype.initialize;
	Scene_EquipWeapon.prototype.initialize = function() {
		_EXC_Scene_EquipWeapon_initialize.call(this);

		// こちらで再設定
		this._slotIndex = _weaponIndex;
		this._slots = _weaponSlots;
	};

	//--------------------------------------------------
	// Scene_EquipSkill のオーバーライド
	//--------------------------------------------------
	const _EXC_Scene_EquipSkill_initialize = Scene_EquipSkill.prototype.initialize;
	Scene_EquipSkill.prototype.initialize = function() {
		_EXC_Scene_EquipSkill_initialize.call(this);

		// こちらで再設定
		this._slotIndex = _skillIndex;
		this._slots = _skillSlots;
	};

	// アイテム一覧の初期表示が一瞬おかしくなるので修正
	const _EXC_Scene_EquipSkill_create = Scene_EquipSkill.prototype.create;
	Scene_EquipSkill.prototype.create = function() {
		_EXC_Scene_EquipSkill_create.call(this);

		// 初期表示の修正
		this._itemWindow.setSlotId(this._slotWindow.enableEquipSlot(this._slotWindow.index() + this._slotIndex));
		this.refreshActor();
	};

	//--------------------------------------------------
	// Window_EquipSlotDivide のオーバーライド
	//--------------------------------------------------
	const _EXC_Window_EquipSlotDivide_initialize = Window_EquipSlotDivide.prototype.initialize;
	Window_EquipSlotDivide.prototype.initialize = function(rect) {
		_EXC_Window_EquipSlotDivide_initialize.call(this, ...arguments);

		// 装備スロット全体を設定
		this._allSlots = _allEquipSlots;
	};

	// 使用不可スロットを避けて処理するよう修正
	Window_EquipSlotDivide.prototype.update = function() {
		Window_StatusBase.prototype.update.call(this);
		if (this._itemWindow) {
			this._itemWindow.setSlotId(this.enableEquipSlot(this.index() + this._slotIndex));
		}
	};

	Window_EquipSlotDivide.prototype.maxItems = function() {
		// アクター未設定時はデフォルト値を返却
		if(!this._actor){
			return 0;
		}
		// 最大長さから使用不可スロット分減らす
		let ret = this._slots.length;
		const slots = this._slots;
		for(let i = 0; i < slots.length; i++){
			if(!Exc_isEquipSlotEnabled(this._allSlots, i + this._slotIndex)){
				ret--;
			}
		}
		return ret;
	};

	// 使用不可スロットを避けて処理するよう修正
	Window_EquipSlotDivide.prototype.actorSlotName = function(actor, index) {
		const slots = actor.equipSlots();
		return $dataSystem.equipTypes[slots[this.enableEquipSlot(index + this._slotIndex)]];
	};

	// 使用不可スロットを避けて処理するよう修正
	Window_EquipSlotDivide.prototype.itemAt = function(index) {
		return this._actor ? this._actor.equips()[this.enableEquipSlot(index + this._slotIndex)] : null;
	};

	// 使用不可スロットを避けて処理するよう修正
	Window_EquipSlotDivide.prototype.isEnabled = function(index) {
		return this._actor ? this._actor.isEquipChangeOk(this.enableEquipSlot(index + this._slotIndex)) : false;
	};

	// 以下新規追加
	// 使用不可スロットをスキップして行位置と装備スロット位置をずらしたものを返却する
	Window_EquipSlotDivide.prototype.enableEquipSlot = function(index){
		let count = this._slotIndex - 1;	// 確認済み使用可能行数
		let retIndex = this._slotIndex - 1;		// index(行位置)に対応する装備スロット位置
		do {
			retIndex++;
			if(Exc_isEquipSlotEnabled(this._allSlots, retIndex)){
				count++;
			}
		} while (index > count && this._allSlots.length > retIndex)

		return retIndex;
	};

	//--------------------------------------------------
	// Window_EquipSlotWeapon のオーバーライド
	//--------------------------------------------------
	const _EXC_Window_EquipSlotWeapon_initialize = Window_EquipSlotWeapon.prototype.initialize;
	Window_EquipSlotWeapon.prototype.initialize = function(rect) {
		_EXC_Window_EquipSlotWeapon_initialize.call(this, ...arguments);
		
		// こちらで再設定
		this._slotIndex = _weaponIndex;
		this._slots = _weaponSlots;
	};

	//--------------------------------------------------
	// Window_EquipSlotSkill のオーバーライド
	//--------------------------------------------------
	const _EXC_Window_EquipSlotSkill_initialize = Window_EquipSlotSkill.prototype.initialize;
	Window_EquipSlotSkill.prototype.initialize = function(rect) {
		_EXC_Window_EquipSlotSkill_initialize.call(this, ...arguments);
		
		// こちらで再設定
		this._slotIndex = _skillIndex;
		this._slots = _skillSlots;
	};

	//--------------------------------------------------
	// Game_Actor のオーバーライド
	//--------------------------------------------------
	// 二刀流用の武器スロットがない場合、追加するよう処理を追加
	const _EXC_Game_Actor_baseEquipSlots = Game_Actor.prototype.baseEquipSlots;
	Game_Actor.prototype.baseEquipSlots = function() {
		let ret = _EXC_Game_Actor_baseEquipSlots.call(this);

		// 最初が武器かつ２つめが武器以外の場合、二刀流用のスロットを追加
		ret = EXC_addEquipSlot(ret);

		return ret;
	};

	// 最強装備時に二刀流でない場合は２スロット目の装備をスキップする
	const _EXC_Game_Actor_optimizeEquipments = Game_Actor.prototype.optimizeEquipments;
	Game_Actor.prototype.optimizeEquipments = function(index, slots) {
		if (index >= 0 && slots) {
			const maxSlots = index + slots.length;
			this.clearEquipments(...arguments);
			for (let i = index; i < maxSlots; i++) {
				// 使用可スロットのみ最強装備するよう条件追加
				if (Exc_isEquipSlotEnabled(_allEquipSlots, i)){
					if (this.isEquipChangeOk(i)) {
						this.changeEquip(i, this.bestEquipItem(i));
					}
				}
			}
			// 最強装備をすることで使用可スロット状態が変わる場合に備えて、
			// 上記でセットしたのちに使用可となったスロットがあれば再セット
			// 細かいことを言えば変化がなくなるまで再セットを続けるべきか・・・？
			for (let i = index; i < maxSlots; i++) {
				if (this._equips[i]._itemId != 0){
					continue;
				}
				if (Exc_isEquipSlotEnabled(_allEquipSlots, i)){
					if (this.isEquipChangeOk(i)) {
						this.changeEquip(i, this.bestEquipItem(i));
					}
				}
			}
		} else {
			_EXC_Game_Actor_optimizeEquipments.apply(this, arguments);
		}
	};

	// 二刀流じゃない場合は二刀流用のスロットをクリア
	const _EXC_Game_Actor_changeEquip = Game_Actor.prototype.changeEquip;
	Game_Actor.prototype.changeEquip = function(slotId, item) {
		_EXC_Game_Actor_changeEquip.call(this, ...arguments);
		if(this._equips.length > DUAL_WIELD_SLOT && this._equips[DUAL_WIELD_SLOT]._dataClass === "weapon" && !this.isDualWield()){
			_EXC_Game_Actor_changeEquip.call(this, DUAL_WIELD_SLOT, null);
		}
	};

	// 装備変更前後の比較でも二刀流でなくなる場合のステータス変化を含む
	const _EXC_Game_Actor_forceChangeEquip = Game_Actor.prototype.forceChangeEquip;
	Game_Actor.prototype.forceChangeEquip = function(slotId, item) {
		_EXC_Game_Actor_forceChangeEquip.call(this, ...arguments);
		if(this._equips.length > DUAL_WIELD_SLOT && this._equips[DUAL_WIELD_SLOT]._dataClass === "weapon" && !this.isDualWield()){
			_EXC_Game_Actor_forceChangeEquip.call(this, DUAL_WIELD_SLOT, null);
		}
	};

	// ジョブチェンジ他で二刀流でなくなる場合には二刀流スロットをクリア
	const _EXC_Game_Actor_releaseUnequippableItems = Game_Actor.prototype.releaseUnequippableItems;
	Game_Actor.prototype.releaseUnequippableItems = function(forcing) {
		_EXC_Game_Actor_releaseUnequippableItems.call(this, ...arguments);
		if(this._equips.length > DUAL_WIELD_SLOT && this._equips[DUAL_WIELD_SLOT]._dataClass === "weapon" && !this.isDualWield()){

			// 元の処理を参考に二刀流の武器を外す処理を作成
			if (!forcing) {
				this.tradeItemWithParty(null, this.equips()[DUAL_WIELD_SLOT]);
			}
			this._equips[DUAL_WIELD_SLOT].setObject(null);
		}
	};
	
	//--------------------------------------------------
	// プラグインコマンド宣言
	//--------------------------------------------------
	// 武器変更処理
	PluginManager.registerCommand(EXCDualWield, "equipWeapon", function(args) {
		// パラメータの取得
		const actorId	= Number(args['targetActor']);
		const weaponId	= Number(args['targetWeapon']);
		const slotNum	= Number(args['targetSlot']);

		// パラメータをもとにオブジェクトを取得
		const targetActor = $gameActors.actor(actorId);
		const targetWeapon = $dataWeapons[weaponId];

		// 装備できる対象のみ処理
		if(targetActor.canEquip(targetWeapon)){
			// 装備変更の対象となるスロットのインデックスを取得
			const targetSlotIndex = EXC_getTargetSlot(targetActor, targetWeapon.etypeId, slotNum);

			// スロットのインデックスが見つかった場合
			if(targetSlotIndex >= 0){
				targetActor.changeEquip(targetSlotIndex, targetWeapon);
			}
		}
	});

	// 防具変更処理
	PluginManager.registerCommand(EXCDualWield, "equipArmor", function(args) {
		// パラメータの取得
		const actorId	= Number(args['targetActor']);
		const ArmorId	= Number(args['targetArmor']);
		const slotNum	= Number(args['targetSlot']);

		// パラメータをもとにオブジェクトを取得
		const targetActor = $gameActors.actor(actorId);
		const targetArmor = $dataArmors[ArmorId];

		// 装備できる対象のみ処理
		if(targetActor.canEquip(targetArmor)){
			// 装備変更の対象となるスロットのインデックスを取得
			const targetSlotIndex = EXC_getTargetSlot(targetActor, targetArmor.etypeId, slotNum);

			// スロットのインデックスが見つかった場合
			if(targetSlotIndex >= 0){
				targetActor.changeEquip(targetSlotIndex, targetArmor);
			}
		}
	});

	// 装備解除処理
	PluginManager.registerCommand(EXCDualWield, "removeEquip", function(args) {
		// パラメータの取得
		const actorId	= Number(args['targetActor']);
		const etypeId	= Number(args['equipTypeID']);
		const slotNum	= Number(args['targetSlot']);

		// パラメータをもとにオブジェクトを取得
		const targetActor = $gameActors.actor(actorId);

		// 装備変更の対象となるスロットのインデックスを取得
		const targetSlotIndex = EXC_getTargetSlot(targetActor, etypeId, slotNum);

		// スロットのインデックスが見つかった場合
		if(targetSlotIndex >= 0){
			targetActor.changeEquip(targetSlotIndex, null);
		}
	});

	// 装備スロットの取得用
	function EXC_getTargetSlot(targetActor, equipType, slotNum){
		let ret = -1;
		for(let i = 0; i < slotNum; i++){
			ret = targetActor.equipSlots().indexOf(equipType, ret + 1);
			if(ret < 0){
				break;
			}
		}

		return ret;
	};

	// 武器交換処理
	PluginManager.registerCommand(EXCDualWield, "exchangeWeapon", function(args) {
		// パラメータの取得
		const actorId	= Number(args['targetActor']);
		const beforeId	= Number(args['beforeWeapon']);
		const afterId	= Number(args['afterWeapon']);

		// パラメータをもとにオブジェクトを取得
		const targetActor = $gameActors.actor(actorId);
		const afterWeapon = $dataWeapons[afterId];

		// 装備できる対象のみ処理
		if(targetActor.canEquip(afterWeapon)){
			// 変更前武器のインデックス検索
			const targetSlotIndex = EXC_findTargetItem(targetActor, beforeId, "weapon");

			// 変更前装備が見つかった場合
			if(targetSlotIndex >= 0){
				targetActor.changeEquip(targetSlotIndex, afterWeapon);
			}
		}
	});

	// 防具交換処理
	PluginManager.registerCommand(EXCDualWield, "exchangeArmor", function(args) {
		// パラメータの取得
		const actorId	= Number(args['targetActor']);
		const beforeId	= Number(args['beforeArmor']);
		const afterId	= Number(args['afterArmor']);

		// パラメータをもとにオブジェクトを取得
		const targetActor = $gameActors.actor(actorId);
		const afterArmor = $dataArmors[afterId];

		// 装備できる対象のみ処理
		if(targetActor.canEquip(afterArmor)){
			// 変更前防具のインデックス検索
			const targetSlotIndex = EXC_findTargetItem(targetActor, beforeId, "armor");

			// 変更前装備が見つかった場合
			if(targetSlotIndex >= 0){
				targetActor.changeEquip(targetSlotIndex, afterArmor);
			}
		}
	});

	// 対象装備の装備位置確認
	function EXC_findTargetItem(targetActor, beforeId, dataClass){
		let ret = -1;
		if(targetActor._equips){
			for(let i = 0; i <= targetActor._equips.length; i++){
				if(targetActor._equips[i] 
						&& targetActor._equips[i]._itemId == beforeId 
						&& targetActor._equips[i]._dataClass == dataClass){
					// 対象装備のスロット番号を返却
					ret = i;
					break;
				}
			}
		}
		return ret;
	};
})();