//=============================================================================
// メニュー画面変更プラグイン
// EXC_MenuCustom.js
// ----------------------------------------------------------------------------
// Copyright (c) 2023 IdiotException
// This software is released under the MIT License.
// http://opensource.org/licenses/mit-license.php
// ----------------------------------------------------------------------------
// Version
// 1.0.0 2024-12-31
//=============================================================================
/*:
 * @target MZ
 * @plugindesc メニュー画面のデザインを変更します
 * @author IdiotException
 * @url 
 * 
 * @param MoneyTitle
 * @text 所持金タイトル
 * @desc 所持金表示のタイトルテキスト
 * @type text
 * @default 敵から奪ったりしたお金
 * 
 * @param WinCountTitle
 * @text 勝利回数タイトル
 * @desc 戦闘勝利回数表示のタイトルテキスト
 * @type text
 * @default たたきつぶしたむれのかず
 *
 * @param MoneyItemsArray
 * @text ３番目以降項目
 * @desc 所持金、勝利回数以下の表示項目
 * [[項目名,変数ID],[項目名,変数ID]...]
 * @type text[][]
 * @default []
 * 
 * @param LevelTitle
 * @text レベルタイトル
 * @desc レベル表示の数値前につくテキスト
 * @type text
 * @default Lv
 * 
 * @param NextExpTitle
 * @text 経験値タイトル
 * @desc 経験値表示のタイトルテキスト
 * @type text
 * @default 次のレベルまで
* 
 * @param LevelMaxText
 * @text レベル上限テキスト
 * @desc レベル上限時に経験値数値の代わりに表示されるテキスト
 * @type text
 * @default - - - -
 * 
 * @param EquipTitle
 * @text 装備タイトル
 * @desc 装備表示のタイトルテキスト
 * @type text
 * @default 装備
 * 
 * @param SkillTitle
 * @text スキルタイトル
 * @desc スキル表示のタイトルテキスト
 * @type text
 * @default スキル
 * 
 * @param MenuDisableSwitch
 * @text メニュー使用可否設定
 * @desc メニュー項目の使用可否に関する設定
 * @type struct<DisableSwitch>[]
 */
 /*~struct~DisableSwitch:
 *
 * @param MenuNumber
 * @text 項目番号
 * @desc 使用可否設定するメニュー項目の番号
 * 一番上の項目が1になる
 * @type number
 * 
 * @param SwitchID
 * @text 対応するスイッチ
 * @desc 対象のメニュー項目の使用可否を設定するスイッチ
 * @type switch
 */

const EXCMenuCustomPluginName = document.currentScript.src.match(/^.*\/(.+)\.js$/)[1];

//-----------------------------------------------------------------------------
// Scene_Menu
// 基本はScene_Menuのコピー
//
// The scene class of the menu screen.

(function() {
	"use strict";
	//--------------------------------------------------
	// 定数設定
	//--------------------------------------------------
	// ゲーム製作途中で変更されるかもしれない定数
	const ACTOR_ID = 1;
	const PARTY_INDEX = 0;			// 対象アクターのパーティ上でのインデックス
	const SKILL_DIVIDE_ID = 402;	// スキル（アビリティ）の装備扱いでの最小のID

	// 画像素材の名前
	const PICT_PATH				= "img/menuC/";		// 画像ファイル格納フォルダ
	const PICT_BARS_NAME		= "gamemenu_bg";	// メニューの文字背景バー等のファイル名
	const PICT_MENU_ITEM_NAME	= "menu-bar";		// 通し番号をメニュー選択肢絵柄の画像ファイル名
	const PICT_SELECT_TRAILING	= "s";				// メニュー選択中画像につく末尾
	const PICT_DISABLE_TRAILING	= "g";				// メニュー使用不可画像につく末尾
	const PICT_DISABLE_SELECT_TRAILING	= "gs";		// メニュー使用不可選択中画像につく末尾
	const PICT_MONEY_BACK_NAME	= "bar01";			// 背景の画像ファイル名
	const PICT_CHARACTER_NAME	= "menu_character";	// キャラクーの画像ファイル名（ジョブ番号前まで
	const PICT_CHARACTER_TRAIL	= "a";				// メニュー用キャラクター画像の末尾（ジョブ番号以降

	// メニュー選択肢
	const MENU_ITEM_POS_Y	= [160,210,260,310,360,410,460,510,560,610];	// メニュー選択肢の縦位置
	const MENU_ITEM_HEIGHT	= 40;					// メニュー選択肢の高さ
	const MENU_ITEM_WIDTH	= 365;					// メニュー選択肢の幅

	// テキスト関連設定
	const MENU_FONT_NAME			= "AkazukiPOP.woff";	// メニューで使用するフォント名
	const MENU_TEXT_COLOR 			= "#FFFFFF";
	const MENU_TEXT_OUTLINE_COLOR	= "#006699";
	const MENU_TEXT_OUTLINE_WIDTH	= 8;

	// 所持金等項目の設定
	const MONEY_TEXT = {
		x				: 0,
		y				: 748,
		height			: 40,
		width			: 488,
		font_size		: 32,
		LR_Padding		: 20,	// 左右余白
		bottom_margin	: 10,	// 行間余白
		bk_top_padding	: 20,	// 背景ずらし上から
		bk_LR_padding	: -112	// 背景ずらし右から
	} ;

	// マップ名
	const MAP_NAME = {
		x				: 220,
		y				: 55,
		height			: 28,
		width			: 250,
		font_size		: 20,
		LR_Padding		: 4		// 少し設定しておかないと縁取りが見切れる
	} ;

	// キャラクター名
	const NAME_TEXT = {
		x				: 530,
		y				: 105,
		height			: 56,
		width			: 300,
		font_size		: 36,
		LR_Padding		: 4
	} ;

	// ジョブ名とレベル
	const JOB_TEXT = {
		x				: 580,
		y				: 180,
		height			: 34,
		width			: 290,
		font_size		: 26,
		LR_Padding		: 4		
	} ;

	// 経験値タイトル
	const EXP_TITLE = {
		x				: 520,
		y				: 225,
		height			: 56,
		width			: 300,
		font_size		: 20,
		LR_Padding		: 4
	} ;
	
	// 経験値数値
	const EXP_NUMBER = {
		x				: 550,
		y				: 240,
		height			: 40,
		width			: 300,
		font_size		: 28,
		LR_Padding		: 4
	} ;

	// 経験値バー
	const EXP_BAR = {
		x				: 550,
		y				: 270,
		height			: 10,
		width			: 300,
		front_color		: "#c71585",
		back_color		: "#000000"
	} ;


	// 装備タイトル
	const EQUIP_TITLE = {
		x				: 530,
		y				: 345,
		height			: 50,
		width			: 297,
		font_size		: 32,
		LR_Padding		: 4
	} ;

	// 装備項目の設定
	const EQUIP_TEXT = {
		x				: 550,	
		y				: 415,
		height			: 36,
		width			: 300,
		font_size		: 28,
		icon_R_margin	: 7,	// アイコンとテキスト間の余白
		LR_Padding		: 4,
		bottom_margin	: 5
	} ;

	// スキルタイトル
	const SKILL_TITLE = {
		x				: 530,
		y				: 727,
		height			: 50,
		width			: 300,
		font_size		: 32,
		LR_Padding		: 4
	} ;

	// スキル項目の設定
	const SKILL_TEXT = {
		x				: 550,	
		y				: 800,
		height			: 36,
		width			: 300,
		font_size		: 28,
		icon_R_margin	: 7,
		LR_Padding		: 4,
		bottom_margin	: 5
	} ;


	// 内部で利用する定数
	const MENU_FONT = "EXC-menu-font";				// メニューフォント取り回し用
	// メニューアイテムの状態管理
	const MenuSelected = {
		default		: 0,
		selected	: 1,
		disable		: 2,
		disableSel	: 3
	};
	// 所持金より下の項目のパラメータ構成
	const ItemStructure = {
		title	: 0,
		id		: 1
	};
	// テキストのalign指定用
	const TextAlign = {
		left	:"left",
		center  :"center",
		right	:"right"
	}

	//パラメータ受取処理
	let parameters = PluginManager.parameters(EXCMenuCustomPluginName);
	let _moneyTitle		= String(parameters['MoneyTitle'] || "タイトルを設定してください");
	let _winCountTitle	= String(parameters['WinCountTitle'] || "タイトルを設定してください");
	let _mItemsArray	= JSON.parse(parameters['MoneyItemsArray'] || "[]");
	let _mItems;
	_mItems = [];
	for(let i = 0; i < _mItemsArray.length; i++) {
		let tmpItem = JSON.parse(_mItemsArray[i] || "[]");
		_mItems.push(tmpItem);
	}
	let _levelTitle		= String(parameters['LevelTitle'] || "タイトルを設定してください");
	let _expTitle		= String(parameters['NextExpTitle'] || "タイトルを設定してください");
	let _levelMaxText	= String(parameters['LevelMaxText'] || "テキストを設定してください");
	let _equipTitle		= String(parameters['EquipTitle'] || "タイトルを設定してください");
	let _skillTitle		= String(parameters['SkillTitle'] || "タイトルを設定してください");
	const _tempDisable	= JSON.parse(parameters['MenuDisableSwitch'] || "[]");
	let _menuDisable = [];
	_tempDisable.forEach(el => _menuDisable.push(JSON.parse(el)));

	//--------------------------------------------------
	// ツクール側変数取得宣言
	//--------------------------------------------------
	// IDから変数情報取得
	function Exc_getValueFromID(id) {
		return $gameVariables.value(Number(id));
	}

	//--------------------------------------------------
	// Name : Scene_Boot
	// Desc : シーンブートクラス
	//--------------------------------------------------
	const _Scene_Boot_loadGameFonts = Scene_Boot.prototype.loadGameFonts;
	Scene_Boot.prototype.loadGameFonts = function () {
		_Scene_Boot_loadGameFonts.call(this);
		// メニュー用フォントの読み込み（デフォルトと同じで固定？
		FontManager.load(MENU_FONT, MENU_FONT_NAME);
	};

	// 画像用Bitmapの保持
	let _iconSet;
	let _bmpMenuItems,_mitMenuItems;

	let _bmpBars,_bmpMoneyBack,_bmpCharacterImg;

	// コマンドウィンドウの保持
	let _cmdwinMenuItems;

	// メニュー選択肢の状態保持
	let _beforeSelectedIndex = -1;

	//--------------------------------------------------
	// 以下 Scene_Menu のオーバーライド
	//--------------------------------------------------
	Scene_Menu.prototype.create = function() {
		this.loadImageBitmap();
		Scene_MenuBase.prototype.create.call(this);
		this.createCommandWindow();
		_cmdwinMenuItems = this._commandWindow;
		// スイッチからメニュー項目の使用可否設定の呼び出し
		_menuDisable.forEach(el => this._commandWindow._list[el.MenuNumber -1].enabled = $gameSwitches.value(el.SwitchID));
		this.createMenuItem();
		this.createMoneyText();
		this.createCharacterImg();
		this.createStatusText();
	};

	Scene_Menu.prototype.start = function() {
		Scene_MenuBase.prototype.start.call(this);
		// ステータスウィンドウの更新を削除
	};

	// コマンドウィンドウ本体を画面外左側に移動
	const _EXC_Scene_Menu_commandWindowRect = Scene_Menu.prototype.commandWindowRect;
	Scene_Menu.prototype.commandWindowRect = function() {
		let rect = _EXC_Scene_Menu_commandWindowRect.call(this);

		// 画面外に移動(念のため少し多めに飛ばす)
		rect.x = - rect.width - 20;

		return rect;
	};

	Scene_Menu.prototype.commandPersonal = function() {
		// 固定でアクターを引き渡し
		$gameParty.setMenuActor($gameParty.members()[PARTY_INDEX]);
		// ステータスウィンドウの選択をスキップ
		this.onPersonalOk();
	};


	//--------------------------------------------------
	// 以下 Scene_Menu への追加
	//--------------------------------------------------
	// 画面更新処理
	Scene_Menu.prototype.update = function() {
		Scene_MenuBase.prototype.update.call(this);
		let currentIndex = this._commandWindow.index();

		// 現在の対象が選択状態になっていない場合
		if(_mitMenuItems[currentIndex].bitmap != _bmpMenuItems[currentIndex][MenuSelected.selected] &&
					_mitMenuItems[currentIndex].bitmap != _bmpMenuItems[currentIndex][MenuSelected.disableSel]){
			// 初回のみ前がないためスキップ
			if(_beforeSelectedIndex >= 0){
				// 非選択になったものを非選択画像に差し替え
				if(this._commandWindow.isCommandEnabled(_beforeSelectedIndex)){
					_mitMenuItems[_beforeSelectedIndex].bitmap = _bmpMenuItems[_beforeSelectedIndex][MenuSelected.default];
				}else {
					_mitMenuItems[_beforeSelectedIndex].bitmap = _bmpMenuItems[_beforeSelectedIndex][MenuSelected.disable];
				}
			}

			// 選択されたものを選択画像に差し替え
			if(this._commandWindow.isCommandEnabled(currentIndex)){
				_mitMenuItems[currentIndex].bitmap = _bmpMenuItems[currentIndex][MenuSelected.selected];
			} else {
				_mitMenuItems[currentIndex].bitmap = _bmpMenuItems[currentIndex][MenuSelected.disableSel];
			}
			// 現在選択を保持
			_beforeSelectedIndex = currentIndex;
		}
	};

	// 画像準備
	Scene_Menu.prototype.loadImageBitmap = function() {

		// アイコン一覧の読み込み
		_iconSet = ImageManager.loadSystem("IconSet");

		// プラグイン用画像フォルダを設定
		let imgFolder;
		imgFolder = PICT_PATH;

		// 部品の読み込み
		_bmpBars = ImageManager.loadBitmap(imgFolder, PICT_BARS_NAME, 0, false);
		_bmpMoneyBack = ImageManager.loadBitmap(imgFolder, PICT_MONEY_BACK_NAME, 0, false);

		// メニュー部品の読み込み
		_bmpMenuItems = [];
		for(let i = 1; i <= MENU_ITEM_POS_Y.length; i++) {
			let tmpMenuItems = [];
			let tmpNum = ('0' + i).slice(-2);
			tmpMenuItems.push(ImageManager.loadBitmap(imgFolder, PICT_MENU_ITEM_NAME + tmpNum, 0, false));
			tmpMenuItems.push(ImageManager.loadBitmap(imgFolder, PICT_MENU_ITEM_NAME + tmpNum + PICT_SELECT_TRAILING, 0, false));
			if(_menuDisable.find(el => el.MenuNumber == i)){
				// disable設定がある場合のみそのための画像読み込み
				tmpMenuItems.push(ImageManager.loadBitmap(imgFolder, PICT_MENU_ITEM_NAME + tmpNum + PICT_DISABLE_TRAILING, 0, false));
				tmpMenuItems.push(ImageManager.loadBitmap(imgFolder, PICT_MENU_ITEM_NAME + tmpNum + PICT_DISABLE_SELECT_TRAILING, 0, false));
			}
			_bmpMenuItems.push(tmpMenuItems);
		}

		// キャラクター画像の読み込み
		let tmpImgName = PICT_CHARACTER_NAME + ('0' + $gameParty.members()[PARTY_INDEX]._classId).slice(-2) + PICT_CHARACTER_TRAIL;
		_bmpCharacterImg = ImageManager.loadBitmap(imgFolder, tmpImgName, 0, false);
	};

	// メニューの要素の設定
	Scene_Menu.prototype.createMenuItem = function() {
		// タイトルと固定のバーの部品呼び出し
		this.createSprite(_bmpBars,0,0);

		_mitMenuItems = [];
		for(let i = 0; i < MENU_ITEM_POS_Y.length; i++) {
			let tmpMenuItem = new EXC_MenuItem();
			if(this._commandWindow.isCommandEnabled(i)){
				tmpMenuItem.bitmap = _bmpMenuItems[i][MenuSelected.default];
			} else {
				tmpMenuItem.bitmap = _bmpMenuItems[i][MenuSelected.disable];
			}
			tmpMenuItem.y = MENU_ITEM_POS_Y[i];
			tmpMenuItem.setIndex(i);
			_mitMenuItems.push(tmpMenuItem);
			this.addChild(tmpMenuItem);
		}
	};

	// 所持金等の表示設定
	Scene_Menu.prototype.createMoneyText = function() {
		// 所持金の表示設定
		let currentY = MONEY_TEXT.y;
		this.createSprite(_bmpMoneyBack, MONEY_TEXT.x + MONEY_TEXT.bk_LR_padding, currentY + MONEY_TEXT.bk_top_padding);
		this.createTextSprite(_moneyTitle, MONEY_TEXT.x, currentY, MONEY_TEXT.height, MONEY_TEXT.width, MONEY_TEXT.LR_Padding, MONEY_TEXT.font_size, TextAlign.left);
		this.createTextSprite($gameParty.gold(), MONEY_TEXT.x, currentY, MONEY_TEXT.height, MONEY_TEXT.width, MONEY_TEXT.LR_Padding, MONEY_TEXT.font_size, TextAlign.right);

		// 次の行位置
		currentY += MONEY_TEXT.height + MONEY_TEXT.bottom_margin;

		// いっこめだけここに挟み込む
		this.createSprite(_bmpMoneyBack, MONEY_TEXT.x + MONEY_TEXT.bk_LR_padding, currentY + MONEY_TEXT.bk_top_padding);
		this.createTextSprite(_mItems[0][ItemStructure.title], MONEY_TEXT.x, currentY, MONEY_TEXT.height, MONEY_TEXT.width, MONEY_TEXT.LR_Padding, MONEY_TEXT.font_size, TextAlign.left);
		this.createTextSprite(Exc_getValueFromID(_mItems[0][ItemStructure.id]), MONEY_TEXT.x, currentY, MONEY_TEXT.height, MONEY_TEXT.width, MONEY_TEXT.LR_Padding, MONEY_TEXT.font_size, TextAlign.right);

		// 次の行位置
		currentY += MONEY_TEXT.height + MONEY_TEXT.bottom_margin;

		// 戦闘勝利回数の表示設定
		this.createSprite(_bmpMoneyBack, MONEY_TEXT.x + MONEY_TEXT.bk_LR_padding, currentY + MONEY_TEXT.bk_top_padding);
		this.createTextSprite(_winCountTitle, MONEY_TEXT.x, currentY, MONEY_TEXT.height, MONEY_TEXT.width, MONEY_TEXT.LR_Padding, MONEY_TEXT.font_size, TextAlign.left);
		this.createTextSprite($gameSystem.winCount(), MONEY_TEXT.x, currentY, MONEY_TEXT.height, MONEY_TEXT.width, MONEY_TEXT.LR_Padding, MONEY_TEXT.font_size, TextAlign.right);

		// 上記以降の表示設定
		for(let i = 1; i < _mItems.length; i++) {
			currentY += MONEY_TEXT.height + MONEY_TEXT.bottom_margin;
			this.createSprite(_bmpMoneyBack, MONEY_TEXT.x + MONEY_TEXT.bk_LR_padding, currentY + MONEY_TEXT.bk_top_padding);
			this.createTextSprite(_mItems[i][ItemStructure.title], MONEY_TEXT.x, currentY, MONEY_TEXT.height, MONEY_TEXT.width, MONEY_TEXT.LR_Padding, MONEY_TEXT.font_size, TextAlign.left);
			this.createTextSprite(Exc_getValueFromID(_mItems[i][ItemStructure.id]), MONEY_TEXT.x, currentY, MONEY_TEXT.height, MONEY_TEXT.width, MONEY_TEXT.LR_Padding, MONEY_TEXT.font_size, TextAlign.right);
		}

	}

	// キャラクター画像を表示
	Scene_Menu.prototype.createCharacterImg = function() {
		this.createSprite(_bmpCharacterImg, 0, 0);
	};

	// ステータス系の表示の設定
	Scene_Menu.prototype.createStatusText = function() {
		// マップ名
		this.createTextSprite($dataMap.displayName, MAP_NAME.x, MAP_NAME.y, MAP_NAME.height, MAP_NAME.width, MAP_NAME.LR_Padding, MAP_NAME.font_size, TextAlign.center);

		// キャラクター名
		this.createTextSprite($dataActors[ACTOR_ID].name, NAME_TEXT.x, NAME_TEXT.y, NAME_TEXT.height, NAME_TEXT.width, NAME_TEXT.LR_Padding, NAME_TEXT.font_size, TextAlign.left);

		// ジョブ・レベル
		this.createTextSprite($dataClasses[$gameParty.members()[PARTY_INDEX]._classId].name, JOB_TEXT.x, JOB_TEXT.y, JOB_TEXT.height, JOB_TEXT.width, JOB_TEXT.LR_Padding, JOB_TEXT.font_size, TextAlign.left);
		this.createTextSprite(_levelTitle + " " + $gameParty.members()[PARTY_INDEX].level, JOB_TEXT.x, JOB_TEXT.y, JOB_TEXT.height, JOB_TEXT.width, JOB_TEXT.LR_Padding, JOB_TEXT.font_size, TextAlign.right);

		// 経験値バー表示と表示数値の準備
		let nextExp, bmpExp;
		bmpExp = new Bitmap(EXP_BAR.width, EXP_BAR.height);
		if($gameParty.members()[PARTY_INDEX].isMaxLevel()){
			// レベル最大の場合、特殊な表示になる
			nextExp = _levelMaxText;
			bmpExp.fillRect(0, 0, EXP_BAR.width, EXP_BAR.height, EXP_BAR.front_color);

		} else {
			let nextExpALL, backWidth, frontWidth;
			nextExpALL = $gameParty.members()[PARTY_INDEX].nextLevelExp() - $gameParty.members()[PARTY_INDEX].currentLevelExp();
			nextExp = $gameParty.members()[PARTY_INDEX].nextRequiredExp();
			backWidth = Math.floor((nextExp / nextExpALL) * EXP_BAR.width);
			frontWidth = EXP_BAR.width - backWidth;

			bmpExp.fillRect(0, 0, frontWidth, EXP_BAR.height, EXP_BAR.front_color);
			bmpExp.fillRect(frontWidth, 0, backWidth, EXP_BAR.height, EXP_BAR.back_color);
		}
		// 上記をもとにEXPバーを作成
		this.createSprite(bmpExp, EXP_BAR.x, EXP_BAR.y);

		// 経験値テキスト
		this.createTextSprite(_expTitle, EXP_TITLE.x, EXP_TITLE.y, EXP_TITLE.height, EXP_TITLE.width, EXP_TITLE.LR_Padding, EXP_TITLE.font_size, TextAlign.left);
		this.createTextSprite(nextExp, EXP_NUMBER.x, EXP_NUMBER.y, EXP_NUMBER.height, EXP_NUMBER.width, EXP_NUMBER.LR_Padding, EXP_NUMBER.font_size, TextAlign.right);

		// 装備タイトル
		this.createTextSprite(_equipTitle, EQUIP_TITLE.x, EQUIP_TITLE.y, EQUIP_TITLE.height, EQUIP_TITLE.width, EQUIP_TITLE.LR_Padding, EQUIP_TITLE.font_size, TextAlign.left);

		// 装備、スキルで使いまわす変数宣言
		let currentY = EQUIP_TEXT.y;
		let textX = EQUIP_TEXT.x + ImageManager.iconWidth + EQUIP_TEXT.icon_R_margin; // テキスト位置はアイコンと空白分ずれる
		let skillArray = [];
		let equips = $gameParty.members()[PARTY_INDEX].equips();

		// 装備一覧
		for(let i = 0; i < equips.length; i++) {
			if(equips[i] === null){
				// nullの場合は処理対象外
			} else if(equips[i].id >= SKILL_DIVIDE_ID){
				// スキルの場合はいったん退避
				skillArray.push(equips[i]);
			} else {
				this.drawIcon(equips[i].iconIndex, EQUIP_TEXT.x, currentY, EQUIP_TEXT.height);
				this.createTextSprite(equips[i].name, textX, currentY, EQUIP_TEXT.height, EQUIP_TEXT.width, EQUIP_TEXT.LR_Padding, EQUIP_TEXT.font_size, TextAlign.left);
				currentY += EQUIP_TEXT.height + EQUIP_TEXT.bottom_margin;
			}
		}

		// スキルタイトル
		this.createTextSprite(_skillTitle, SKILL_TITLE.x, SKILL_TITLE.y, SKILL_TITLE.height, SKILL_TITLE.width, SKILL_TITLE.LR_Padding, SKILL_TITLE.font_size, TextAlign.left);

		// スキル一覧 //TODO 固定スキルは仕様確定後再調整
		currentY = SKILL_TEXT.y;
		textX = SKILL_TEXT.x + ImageManager.iconWidth + SKILL_TEXT.icon_R_margin;
		for(let i = 0; i < skillArray.length; i++) {
			this.drawIcon(skillArray[i].iconIndex, SKILL_TEXT.x, currentY, SKILL_TEXT.height);
			this.createTextSprite(skillArray[i].name, textX, currentY, SKILL_TEXT.height, SKILL_TEXT.width, SKILL_TEXT.LR_Padding, SKILL_TEXT.font_size, TextAlign.left);
			currentY += SKILL_TEXT.height + SKILL_TEXT.bottom_margin;
		}

	};

	// テキスト表示
	Scene_Menu.prototype.createTextSprite = function(text, x, y, height, width, padding, size, align) {

		let tmpBmp = new Bitmap(width, height);
		tmpBmp.fontFace = MENU_FONT;
		tmpBmp.fontSize = size;
		tmpBmp.textColor = MENU_TEXT_COLOR;
		tmpBmp.outlineColor = MENU_TEXT_OUTLINE_COLOR;
		tmpBmp.outlineWidth = MENU_TEXT_OUTLINE_WIDTH;
		tmpBmp.drawText(text, padding, (height - size) / 2 , width - padding * 2, size, align);
		
		this.createSprite(tmpBmp, x, y);
	};

	// アイコンの表示
	// Window_Baseの同名関数を参考に作成
	Scene_Menu.prototype.drawIcon = function(iconIndex, x, y, lineHeight) {
		const pw = ImageManager.iconWidth;
		const ph = ImageManager.iconHeight;
		const sx = (iconIndex % 16) * pw;
		const sy = Math.floor(iconIndex / 16) * ph;

		let tmpBmp = new Bitmap(pw, ph)
		tmpBmp.blt(_iconSet, sx, sy, pw, ph, 0, 0);
		this.createSprite(tmpBmp, x, y + Math.floor((lineHeight - ph) / 2));
	};

	// スプライトの作成
	Scene_Menu.prototype.createSprite = function(bmp, x, y) {
		let tmpSprite = new Sprite(bmp);
		tmpSprite.x = x;
		tmpSprite.y = y;
		this.addChild(tmpSprite);
	};

	//-----------------------------------------------------------------------------
	// Window_Base のオーバーライド
	//
	const _EXC_Window_Base_resetTextColor = Window_Base.prototype.resetTextColor;
	Window_Base.prototype.resetTextColor = function() {
		_EXC_Window_Base_resetTextColor.call(this);

		// 条件を満たす場合文字縁取り色変更
		if(this.isChangeOutlineColor()){
			this.contents.outlineColor = MENU_TEXT_OUTLINE_COLOR;
		} 
	};

	const _EXC_Window_Base_changeTextColor = Window_Base.prototype.changeTextColor;
	Window_Base.prototype.changeTextColor = function(color) {
		_EXC_Window_Base_changeTextColor.call(this, ...arguments);

		// 条件を満たす場合文字縁取り色変更
		if(this.isChangeOutlineColor()){
			this.contents.outlineColor = MENU_TEXT_OUTLINE_COLOR;
		} 
	};

	const _EXC_Window_Base_drawText = Window_Base.prototype.drawText;
	Window_Base.prototype.drawText = function(text, x, y, maxWidth, align) {
		// 条件を満たす場合文字縁取り色変更
		if(this.isChangeOutlineColor()){
			this.contents.outlineColor = MENU_TEXT_OUTLINE_COLOR;
		}

		_EXC_Window_Base_drawText.call(this, ...arguments);
	};

	// 文字縁取り色変更判定メソッド
	Window_Base.prototype.isChangeOutlineColor = function() {
		
		// Window_EquipCommandは変更
		if(this.constructor.name == "Window_EquipCommand"){return true;}

		// Windows_SelectClassesは変更
		if(this.constructor.name == "Windows_SelectClasses"){return true;}

		// Window_Optionsは変更
		if(this.constructor.name == "Window_Options"){return true;}

		// Window_GameEndは変更
		if(this.constructor.name == "Window_GameEnd"){return true;}

		// 親がない場合変更なし
		if(!this.parent){return false;}

		// 親の親がない場合変更なし
		if(!this.parent.parent){return false;}

		// それぞれのシーンの場合変更
		if(this.parent.parent.constructor.name == "Scene_Item"){return true;}
		if(this.parent.parent.constructor.name == "Scene_Skill"){return true;}
		if(this.parent.parent.constructor.name == "Scene_EquipWeapon"){return true;}
		if(this.parent.parent.constructor.name == "Scene_ClassChange"){return true;}
		if(this.parent.parent.constructor.name == "Scene_EquipSkill"){return true;}
		if(this.parent.parent.constructor.name == "Scene_Mission"){return true;}
		if(this.parent.parent.constructor.name == "Scene_Options"){return true;}
		if(this.parent.parent.constructor.name == "Scene_GameEnd"){return true;}

		return false;
	};

	//-----------------------------------------------------------------------------
	// EXC_MenuItem
	// Sprite_Clickableを継承
	//

	function EXC_MenuItem() {
		this.initialize.apply(this, arguments);
	}

	EXC_MenuItem.prototype = Object.create(Sprite_Clickable.prototype);
	EXC_MenuItem.prototype.constructor = EXC_MenuItem;

	EXC_MenuItem.prototype.initialize = function() {
		Sprite_Clickable.prototype.initialize.call(this);
	};

	// リスト上でのインデックスを保持
	EXC_MenuItem.prototype.setIndex = function(index) {
		this._index = index;
	};

	// マウスオーバーを実体側リストに反映
	EXC_MenuItem.prototype.onMouseEnter = function() {
		_cmdwinMenuItems.select(this._index);
	};

	EXC_MenuItem.prototype.onClick = function() {
		// メニュークリック時のイベント処理
		_cmdwinMenuItems.processOk();
	};

})();

