//=============================================================================
// スロットマシーンプラグイン
// Exc_SlotMashine.js
//=============================================================================

/*:
 * @target MZ
 * @plugindesc スロットマシーンを追加します。
 * @author IdiotException
 * @url https://github.com/IdiotException/EXC_SlotMachine
 * @help 一般的にはパチスロと呼ばれるタイプのスロットマシーンです。
 * 主にキーボードで操作することが想定されています。
 * 
 * 操作方法
 * 　↑キー：掛け金の投入、再度押下するとスロットスタート
 * 　→キー：右リールの停止
 * 　↓キー：中央リールの停止
 * 　←キー：左リールの停止
 * 　決定キー長押し：上記の一連の流れを押している間実行
 * 　キャンセルキー：終了メニュー表示
 *
 * 画像の準備
 * 　実装時には配布Urlにあるimgフォルダの、
 * 　"Slot"フォルダを対象プロジェクトのimgフォルダに配置してください。
 * 　また、各画像ファイルと同名のファイルを用意していただくことで
 *   オリジナルの画像が利用できます。
 * 　他に、起動時コマンドで別フォルダを指定することでマシンごとに
 * 　別な画像を設定することもできます。
 * 
 * 利用規約
 *   MITライセンスです。
 *   作者に無断で改変、再配布が可能で、
 *   利用形態（商用、18禁利用等）についても制限はありません。
 * 
 * @param CoinID
 * @text コイン変数ID
 * @desc カジノで使用するコインの所持数を保管する変数ID
 * @type variable
 * @default 21
 * 
 * @param AutoPlayModeID
 * @text 自動実行モード変数ID
 * @desc スロット自動実行時の停止順モードを保管する変数ID
 * 0:123 1:132 2:213 3:231 4:312 5:321 6:FULL AUTO
 * @type variable
 * @default 22
 * 
 * @param AutoPlayWaitID
 * @text 自動実行待機ID
 * @desc 自動実行時に待機するフレーム数を設定する変数ID
 * @type variable
 * @default 23
 * 
 * @param BonusReachGamesID
 * @text リーチゲーム数ID
 * @desc ボーナス確定時に何ゲーム目に確定揃えをするか
 * を設定する変数ID
 * @type variable
 * @default 24
 * 
 * @param JacAutoSwitchNum
 * @text JACオート可否スイッチ番号
 * @desc JACオートのON/OFFを管理するゲームスイッチの番号
 * @type switch
 * @default 10
 * 
 * @param BonusNoticeSwitchNum
 * @text ボーナス通知音無効化スイッチ番号
 * @desc ボーナス通知音無効化を管理するゲームスイッチの番号
 * @type switch
 * @default 11
 * 
 * @param WinTitle
 * @text Winタイトル
 * @desc Win表示部のタイトルテキスト
 * @type text
 * @default WIN
 *
 * @param BetTitle
 * @text Betタイトル
 * @desc BET数表示部のタイトルテキスト
 * @type text
 * @default BET
 *
 * @param CoinTitle
 * @text Coinタイトル
 * @desc コイン残数表示部のタイトルテキスト
 * @type text
 * @default CREDIT
 *
 * @param MoneyTitle
 * @text Moneyタイトル
 * @desc 所持金表示部のタイトルテキスト
 * @type text
 * @default CASH
 *
 * @param GameCountTitle
 * @text ゲーム数タイトル
 * @desc ゲーム数表示部のタイトルテキスト
 * @type text
 * @default GAME
 *
 * @param BonusGameTitle
 * @text BonusGameタイトル
 * @desc ボーナスゲーム数表示部のタイトルテキスト
 * @type text
 * @default BONUS GAME
 *
 * @param JacGameTitle
 * @text JacGameタイトル
 * @desc JacGame数表示部のタイトルテキスト
 * @type text
 * @default JAC GAME
 *
 * @param BonusStockTitle
 * @text BonusStockタイトル
 * @desc ボーナスのストック数表示部のタイトルテキスト
 * @type text
 * @default BONUS STOCK
 *
 * @param CommandContinue
 * @text 継続の選択肢
 * @desc 終了確認コマンドウィンドウでの継続選択肢テキスト
 * @type text
 * @default 続ける
 * 
 * @param CommandBuyCoin
 * @text コイン購入の選択肢
 * @desc 終了確認コマンドウィンドウでのコイン購入選択肢テキスト
 * @type text
 * @default コインを買う
 *
 * @param CommandExit
 * @text 終了の選択肢
 * @desc 終了確認コマンドウィンドウでの終了選択肢テキスト
 * @type text
 * @default 終了する
 *
 * @param BuyCoinID
 * @text 購入コイン数ID
 * @desc コイン購入選択肢で購入できるコイン数を
 * 指定する変数の番号
 * @type variable
 * @default 90
 *
 * @param PayMoneyID
 * @text 支払金額ID
 * @desc コイン購入選択肢で支払う金額を
 * 指定する変数の番号
 * @type variable
 * @default 91
 *
 * @param MainFontFileName
 * @text 項目名フォントファイル名
 * @desc 各項目の項目名に適用するフォントのファイル名
 * @type text
 *
 * @param NumberFontFileName
 * @text 数字フォントファイル名
 * @desc 各項目の数字に適用するフォントのファイル名
 * @type text
 *
 * @command DisplaySlotMachine
 * @text スロットマシーン画面表示
 * @desc スロットマシーンを開始します。
 *
 * @arg NormalProbability
 * @text 通常時当選確率
 * @desc 通常時の当選確率(XXXX/8192)
 * replay, チェリー, ベル, スイカ, BAR, 青7, 赤7, w青7, w赤7
 * @type number[]
 * @default [1137,99,744,90,27,12,31,4,10]
 *
 * @arg RedSingleBonusProbability
 * @text 赤７シングルボーナス中当選確率
 * @desc 赤７シングルボーナス中の当選確率(XXXX/8192)
 * replay, チェリー, ベル, スイカ, BAR, 青7, 赤7, w青7, w赤7
 * @type number[]
 * @default [1638,1365,1638,410,0,14,42,4,13]
 *
 * @arg RedDoubleBonusProbability
 * @text 赤７ダブルボーナス中当選確率
 * @desc 赤７ダブルボーナス中の当選確率(XXXX/8192)
 * replay, チェリー, ベル, スイカ, BAR, 青7, 赤7, w青7, w赤7
 * @type number[]
 * @default [2340,0,5779,0,0,0,0,18,55]
 *
 * @arg BlueSingleBonusProbability
 * @text 青７シングルボーナス中当選確率
 * @desc 青７シングルボーナス中の当選確率(XXXX/8192)
 * replay, チェリー, ベル, スイカ, BAR, 青7, 赤7, w青7, w赤7
 * @type number[]
 * @default [1638,1365,1638,410,0,205,0,68,0]
 *
 * @arg BlueDoubleBonusProbability
 * @text 青７ダブルボーナス中当選確率
 * @desc 青７ダブルボーナス中の当選確率(XXXX/8192)
 * replay, チェリー, ベル, スイカ, BAR, 青7, 赤7, w青7, w赤7
 * @type number[]
 * @default [2340,0,5579,0,0,0,0,273,0]
 *
 * @arg JacProbability
 * @text JACゲーム中当選確率
 * @desc JACゲーム中の当選確率(XXXX/8192)
 * replay, チェリー, ベル, スイカ, BAR, 青7, 赤7, w青7, w赤7
 * @type number[]
 * @default [4096,0,4096,0,0,0,0,0,0]
 *
 * @arg PromotionProbability
 * @text ボーナス昇格確率
 * @desc ストックされたボーナスの昇格確率(XXXX/8192)
 * 赤7→青7, シングル→ダブル
 * @type number[]
 * @default [4096,4096]
 *
 * @arg AlternativeImageFolder
 * @text 別画像フォルダ名
 * @desc 別な画像取得先フォルダ名 ex.) img/slot2/
 * 指定することで「img/slot/」ではないフォルダから画像を取得する
 * @type Text
 * @default 
 * 
 * @arg RedFanfareFile
 * @text 赤７ファンファーレ
 * @desc 赤７が揃った際に鳴らすファンファーレの
 * ファイルを指定する
 * @type file
 * @dir audio\me
 * @default slot_BB_01a
 *
 * @arg DoubleRedFanfareFile
 * @text 双赤７ファンファーレ
 * @desc 赤７が２つ揃った際に鳴らすファンファーレの
 * ファイルを指定する
 * @type file
 * @dir audio\me
 * @default slot_BB_01a
 * 
 * @arg BlueFanfareFile
 * @text 青７ファンファーレ
 * @desc 青７が揃った際に鳴らすファンファーレの
 * ファイルを指定する
 * @type file
 * @dir audio\me
 * @default slot_BB_01a
 * 
 * @arg DoubleBlueFanfareFile
 * @text 双青７ファンファーレ
 * @desc 青７が２つ揃った際に鳴らすファンファーレの
 * ファイルを指定する
 * @type file
 * @dir audio\me
 * @default slot_BB_01a
 */

const ExcSlotMachinePluginName = document.currentScript.src.match(/^.*\/(.+)\.js$/)[1];

(function() {
	"use strict";

	//--------------------------------------------------
	// 定数設定
	//--------------------------------------------------
	// 画像素材の名前
	const PICT_PATH				= "img/slot/";		// スロットの画像ファイル格納フォルダ
	const PICT_FRAME_NAME		= "slot_frame_a";	// 通し番号を除くフレームの画像ファイル名
	const PICT_BACKGROUND_NAME	= "slot_bg";		// 通し番号を除く背景の画像ファイル名
	const PICT_PATTERN_NAME		= "slot_pattern_";	// 通し番号を除く絵柄の画像ファイル名
	const PICT_BLUR_TRAILING	= "_blur";			// 回転中画像につく末尾
	const PICT_GLOW_TRAILING	= "_glow";			// 点滅用画像につく末尾
	const PICT_JACLAMP_NAME		= "slot_jaclamp_a";	// 通し番号を除くJacLamp画像のファイル名
	const PICT_TEXT_BACK_NAME	= "slot_bar_";		// l,rを除くテキスト項目背景画像のファイル名
	const PICT_LOG_PATTERN_NAME	= "slot_icon_a";	// 通し番号を除く履歴用絵柄の画像ファイル名
	const PICT_ARROW_NAME		= "zyunoshi";		// 順押し矢印の画像ファイル名
	const PICT_JAC_AUTO_NAME	= "jac_auto";		// JACオートの画像ファイル名
	const PICT_REEL_LIST_NAME	= "reel_array";		// リール配列の画像ファイル名
	const VIDEO_BONUS_NAME		= "effect_kirakira01";	// ボーナスゲーム中のエフェクト動画
	
	// BGM音量
	const BGM_VOLUME = 90;

	// リール設定の固定値
	const REEL_POS_X = [224, 512, 800];	// 各リールの左側の横位置
	const REEL_POS_Y = 190;				// リールの上部の縦位置
	const REEL_DISP_COUNT = 3;			// 各リールの高さ指定（絵柄何個分か
	const REEL_CENTER_INDEX = 1;		// 中段絵柄のインデックス

	// リールごとの絵柄の並び管理
	const REEL_PATTERN =  [[4,3,7,4,7,1,3,2,1,5,1,3,6,6,4,1,3,2,5,3,1],
						   [4,1,3,2,1,7,3,5,2,1,3,4,6,4,1,3,2,1,7,3,2],
						   [1,5,2,4,3,1,5,3,1,4,6,6,3,1,5,3,7,1,7,2,3]];

	// リールの絵柄の画像ファイル関連の設定
	const REEL_PATTERN_HEIGHT	= 130;		// 絵柄の縦の間隔
	const REEL_PATTERN_WIDTH	= 256;		// 絵柄の横幅（使ってない
	const REEL_ROTATION_SPEED	= 60;		// 絵柄の移動速度（px/f）※絵柄の縦の間隔が上限
	const REEL_START_NUMBER		= [4,5,16];	// リールの初期表示絵柄位置（１～２１）
	const REEL_FLASH_FRAME		= {	veryFast	:1,
									fast		:2,
									middle		:3,
									slow		:6,
									verySlow	:12,
									glow		:999};	// リールの結果の絵柄の点滅切り替わり待ちフレーム数
	const REEL_BACK_FLASH_WAIT	= 1;	// ボーナス中7揃いでのリール点滅の切り替わり待ちフレーム数
	const REEL_BACK_FLASHING	= 20;	// ボーナス中7揃いでのリール点滅するフレーム数

	// キーの入力不可時間
	const INIT_DISABLE_FRAME		= 15	// 開始直後等のボタン誤入力を防ぐ
	const INS_COIN_DISABLE_FRAME	= 8;	// コイン投入からレバー操作の入力不可フレーム数
	const STARTING_DISABLE_FRAME	= 30;	// 回転直後の停止入力不可時間のフレーム数
	const STOP_DISABLE_FRAME		= 6;	// 停止同時押し不可時間のフレーム数
	const JAC_AUTO_DISABLE_FRAME	= 3;	// JACオートモード中の操作間フレーム数
	const RESULT_DISABLE_FRAME		={  none  : 5,
										coin2 : 8,
										coin4 : 16,
										coin6 : 24,
										coin8 : 32,
										coin10: 40,
										coin12: 48,
										coin15: 60,
										replay: 40};	// コイン払い戻し効果音中入力不可時間
	const ME_FANFARE_FRAME			= 390;	// ボーナスファンファーレのフレーム数
	const ME_ENDING_FRAME			= 200;	// ボーナスエンディングのフレーム数

	// ７ボーナス外のJacゲーム終了時BGMフェードアウト時間(秒)
	// フェードアウト中はキー入力不可
	const JAC_BONUS_FADE_OUT_SEC = 2; 

	// テキスト表示
	const TEXT_POS_Y	= [688, 752, 816, 880];	// テキスト表示の縦位置
	const TEXT_HEIGHT	= 40;					// テキスト表示の高さ
	const TEXT_WIDTH	= 370;					// テキスト表示の幅
	const TEXT_SIZE		= 32;					// テキストのフォントサイズ
	const TEXT_PADDING	= 20;					// テキストの左右余白
	const NUM_UPDATE_WAIT = 4;					// 数値テキストの更新待ちフレーム数

	// JACランプ
	const JAC_LAMP_POS_X = [513, 603, 693];
	const JAC_LAMP_POS_Y = 620;
	const JAC_LAMP_FLASH_FRAME = 30;

	// ボーナス履歴
	const BONUS_LOG_POS_X		= 1100;	// ボーナス履歴のX座標
	const BONUS_LOG_POS_Y		= 220;	// ボーナス履歴のY座標
	const BONUS_LOG_ICON_HEIGHT	= 27;	// ボーナス履歴のアイコン高さ
	const BONUS_LOG_HEIGHT		= 32;	// ボーナス履歴の行高さ
	const BONUS_LOG_WIDTH		= 168;	// ボーナス履歴の幅
	const BONUS_LOG_MAX			= 12;	// ボーナス履歴の最大件数
	const BONUS_LOG_FONT_SIZE	= 18;	// ボーナス履歴のフォントサイズ
	const BONUS_LOG_PADDING		= 10;	// ボーナス履歴の内部左右空白

	// ボーナス回数テキスト表示
	const COUNT_BONUS_POS_X		= 1220;		// ボーナス回数の横位置
	const COUNT_BONUS_POS_Y		= [28, 82, 55, 109, 136];	// ボーナス回数の縦位置(赤7、青7、赤7W、青7W、Jac)
	const COUNT_BONUS_HEIGHT	= 20;		// ボーナス回数の行高さ
	const COUNT_BONUS_WIDTH		= 40;		// ボーナス回数の幅
	const COUNT_BONUS_SIZE		= 18;		// ボーナス回数のフォントサイズ
	const COUNT_BONUS_PADDING	= 2;		// ボーナス回数の左右余白
	const COUNT_BONUS_MAX		= 999;		// ボーナス回数のカウント最大値

	// 順押し矢印画像
	const IN_ORDER_ARROW_X	= 180;
	const IN_ORDER_ARROW_Y	= 580;

	// JACオート画像
	const JAC_AUTO_IMG_X	= 415;
	const JAC_AUTO_IMG_Y	= 350;
	const JAC_AUTO_FLASH_FRAME = 50000;

	// リール配列画像
	const REEL_LIST_X	= 10;
	const REEL_LIST_Y	= 10;

	// エフェクト動画
	const BONUS_EFFECT_BLEND = 1;	// ボーナス時エフェクト動画のブレンドモード

	// 乱数基底設定
	const RANDOM_MAX = 8192;

	// 停止禁止絵柄位置（１～２１）(下段絵柄番号)
	const NO_STOP_NUMBER = [[3,9,13,14,20],[12,13],[2,10,12,17,19]];	// 通常時停止禁止絵柄位置
	const NO_STOP_REACH_NUMBER = [[1,11,14],[4,13,17],[9,12]];		// リーチ時停止禁止絵柄位置
	const NO_STOP_BAR_NUMBER =[[19],[8],[7]]; 				//BAR揃え時に不可の組み合わせ

	// 赤7、青7のwが成立しうる絵柄位置（１～２１）
	const RED7_W_NUMBER		= [[3],[5,18],[17]];			// 順不同
	const BLUE7_W_NUMBER	= [[12,13],[12],[11,10]];		// 並び順も重要

	// リーチ中最大滑り
	const REEL_SLIDE_MAX = 4;

	// ボーナスの回数
	const BONUS_GAMES_MAX	= 30;	// ボーナスゲームのゲーム数
	const BONUS_JAC_MAX		= 3;	// ボーナス中にストックできるJacの最大数
	const JAC_GAMES_MAX		= 8;	// Jacゲームのゲーム数

	// コイン投入数
	const INS_DEFAULT = 3;
	const INS_JAC = 1;

	// ボーナス中最大獲得枚数
	const MAX_BONUS_WIN = 711;

	// リール滑り数想定外判定用最大滑りのバッファ
	const ERROR_MAX_SLIDE_COUNT_BUFFER = 5

	// 以下スロットマシンの状態管理（こちらはいじらないで！）
	const MachineMode = {
		stop	: 0,
		run		: 1,
		result	: 2,
		replay	: 3
	};
	const BackMode = {
		normal	:0,
		red		:1,
		blue	:2
	};
	const PatternCode = {
		replay	: 1,
		cherry	: 2,
		bell	: 3,
		melon	: 4,
		bar		: 5,
		blue7	: 6,
		red7	: 7,
		blue7_w	: 8,
		red7_w	: 9
	};
	const PatternState = {
		stop	:0,
		glow	:1,
		blur	:2
	};
	const ReelIndexes = {
		left	:0,
		center	:1,
		right	:2
	};
	const JacLampState = {
		wait	:0,
		stock	:1,
		end		:2,
		flash	:3
	};
	// jacゲームは他のボーナスと重なる可能性があるため、_bonusModeには格納されない…
	const BonusMode = {
		none		: 0,
		redSingle	: 1,
		blueSingle	: 2,
		redDouble	: 3,
		blueDouble	: 4,
		jac			: 5
	};
	const FontType ={
		mainFont	:"slot-mainfont",
		numberFont	:"slot-numberfont"
	};
	const textBackState = {
		normal	:0,
		glow	:1
	};
	const LogSprites = {
		number	:0,
		icon	:1
	};
	//パラメータ受取処理
	let parameters = PluginManager.parameters(ExcSlotMachinePluginName);
	let _coinId				= Number(parameters['CoinID'] || 21);
	let _autoPlayId			= Number(parameters['AutoPlayModeID'] || 22);
	let _autoPlayWId		= Number(parameters['AutoPlayWaitID'] || 23);
	let _reachGamesId		= Number(parameters['BonusReachGamesID'] || 24);
	let _jacAutoSw			= Number(parameters['JacAutoSwitchNum'] || 10);
	let _bonusNoticeSw		= Number(parameters['BonusNoticeSwitchNum'] || 11);
	let _buyCoinId			= Number(parameters['BuyCoinID']|| 90);
	let _payMoneyId			= Number(parameters['PayMoneyID'] || 91);
	let _winTitle			= String(parameters['WinTitle'] || "WIN");
	let _betTitle			= String(parameters['BetTitle'] || "BET");
	let _coinTitle			= String(parameters['CoinTitle'] || "CREDIT");
	let _moneyTitle			= String(parameters['MoneyTitle'] || "CASH");
	let _gameCntTitle		= String(parameters['GameCountTitle'] || "GAME");
	let _bonusGameTitle		= String(parameters['BonusGameTitle'] || "BONUS GAME");
	let _jacGameTitle		= String(parameters['JacGameTitle'] || "JAC GAME");
	let _bonusStockTitle	= String(parameters['BonusStockTitle'] || "BONUS STOCK");
	let _commandContinue	= String(parameters['CommandContinue'] || "続ける");
	let _commandBuyCoin		= String(parameters['CommandBuyCoin'] || "コインを買う");
	let _commandExit		= String(parameters['CommandExit'] || "終了する");
	let _mainFontName		= String(parameters['MainFontFileName'] || "");
	let _numberFontName		= String(parameters['NumberFontFileName'] || "");
	
	//--------------------------------------------------
	// 変数宣言
	//--------------------------------------------------
	// コマンド起動時の引数受け取り変数
	// 各役の確率格納用
	let _normalProb, _redSingleProb, _redDoubleProb, _blueSingleProb, _blueDoubleProb, _jacProb, _promoProb;
	// 画像差し替え用
	let _altImageFolder;
	// ファンファーレ指定用
	let _fanFareMap;

	// スロット内部管理
	// スロットマシーンの状態
	let _machineMode = MachineMode.stop;
	let _machineDisableCount = 0;
	let _bonusMEWait = 0;

	// スロットのフレーム画像管理用スプライト
	let _sprFrame;

	// リールの状態管理変数
	let _reelObj =[];
	let _resultIndexes = [[],[],[]];
	let _resultFlash = {mode:0, count:0, speed:REEL_FLASH_FRAME.verySlow};
	let _reelFlash = {time:0, mode:0, count:0, backMode:BackMode.normal};
	let _winSet = -1;

	// 画像用Bitmapの保持
	let _bmpBackGround, _bmpPatterns, _bmpFrame, _bmpJacLamp, _bmpTextBackR, _bmpTextBackL, _bmpLogIcon, _bmpArrow, _bmpJacAuto, _bmpReelList;
	let _bmpBonusVideo;
	// テキスト表示Bitmapの保持
	let _txtCntBonusLog;
	let _txtWin, _txtBet, _txtCoin, _txtMoney, _txtGameCnt, _txtBonusCnt, _txtJacCnt, _txtBonusStk;
	// テキスト背景のSpriteの保持
	let _txtBackWin, _txtBackGameCnt, _txtBackBonusCnt, _txtBackJacCnt, _txtBackBonusStk;

	// 順押し画像のSpriteの保持
	let _inOrderArrow;
	
	// JACオート画像のSpriteの保持
	let _jacAutoSprite;

	// リール配列画像のSpriteの保持
	let _reelList;

	// エフェクト動画用Spriteの保持
	let _sprBonusEffect;

	// テキスト表示項目関連
	let _valueWin,_valueBet,_valueCoin;
	let _gameCount = 0;
	let _gameCountAll = 0;

	// リール滑り数想定外処理用
	let _reelSlideCount = 0;
	let _errorWin = -1;

	// ボーナス関連
	let _bonusMode = BonusMode.none;	// 現在のボーナスゲーム
	let _fanfare;			// 鳴らすファンファーレの指定
	let _jacCount = 0;		// JACゲームの残りゲーム数（表示と逆で減算している
	let _bonusCount = 0;	// ボーナスゲームの残りゲーム数
	let _jacStock = 0;		// JACゲームストック数
	let _jacAlready = 0;	// 消化済みJACゲーム数
	let _bonusStock = [];	// ストックしているボーナスの一覧
	let _jacBellCount = 0;	// ジャックゲーム中にベルがそろった回数
	let _bonusLog = [];		// ボーナスのログ

	// ボーナス入り前のリーチ状態の保持
	let _bonusReachCount = 0;
	let _bonusReachWin = -1;

	// 各ボーナス発生回数
	let _cntBonusLog = [0, 0, 0, 0, 0];

	// ジャックオートのフラグ格納用
	let _enableJacAuto = true;
	let _jacAutoFlash = -1;

	// jacストックのランプ管理（functionに頼りすぎでは？）
	let _jacLamp = {
		lamp: [],
		sprite:[],
		reset:function() {
			this.lamp = [];
			for(let i = 0; i < BONUS_JAC_MAX; i++) {
				let tempJac = {state:JacLampState.end, flashCount:0};
				this.lamp.push(tempJac);
			}

			// スプライトも再セット
			for(let i = 0; i <this.sprite.length; i++) {
				this.sprite[i].bitmap = _bmpJacLamp[JacLampState.end];
			}
		},
		setWait:function() {
			// ウェイトにリセット
			for(let i = 0; i < this.lamp.length; i++) {
				this.lamp[i].state = JacLampState.wait;
			}
		},
		addStock:function() {
			// ストック追加
			let target = this.lamp.find(el => el.state == JacLampState.wait);
			if(target != null) {
				target.state = JacLampState.stock;
			}
		},
		decStock:function() {
			let target = this.lamp.find(el => el.state == JacLampState.stock);
			if(target != null) {
				target.state = JacLampState.wait;
			}
		},
		flashStock:function() {
			let target = this.lamp.find(el => el.state == JacLampState.stock);
			if(target != null) {
				target.state = JacLampState.flash;
			}
		},
		endStock:function() {
			let target = this.lamp.find(el => el.state == JacLampState.flash);
			if(target != null) {
				target.state = JacLampState.end;
			}
		}
	};

	
	//--------------------------------------------------
	// ツクール側変数取得宣言
	//--------------------------------------------------
	// コイン
	function Exc_getCoin() {
		return $gameVariables.value(_coinId);
	}
	function Exc_setCoin(value) {
		$gameVariables.setValue(_coinId, value);
	}
	function Exc_addCoin(value) {
		Exc_setCoin(Number(Exc_getCoin() || 0) + Number(value || 0));
	}

	// 購入時に獲得できるコインの数量
	function Exc_getBuyCoins() {
		return Number($gameVariables.value(_buyCoinId) || 0);
	}

	// 購入時に支払う所持金の数量
	function Exc_getPayMoney() {
		return Number($gameVariables.value(_payMoneyId) || 0);
	}

	// 自動プレイモード
	function Exc_getAutoPlayMode() {
		return $gameVariables.value(_autoPlayId);
	}

	// 自動プレイ待機フレーム
	function Exc_getAutoPlayWait() {
		return Number($gameVariables.value(_autoPlayWId) || 0);
	}

	// ボーナスリーチゲーム数
	function Exc_getReachGames() {
		return Number($gameVariables.value(_reachGamesId) || 1);
	}

	// JACオートON/OFF
	function Exc_getEnableJacAuto() {
		return $gameSwitches.value(_jacAutoSw);
	}

	// ボーナス通知音無効化
	function Exc_getDisbleBonusNotice() {
		return $gameSwitches.value(_bonusNoticeSw);
	}

	//--------------------------------------------------
	// プラグインコマンド宣言
	//--------------------------------------------------
	// スロット開始コマンド設定
	PluginManager.registerCommand(ExcSlotMachinePluginName, "DisplaySlotMachine", function(args) {
		// パラメータの取得
		_normalProb = JSON.parse(args['NormalProbability'] || "[0,0,0,0,0,0,0,0,0]");
		_redSingleProb = JSON.parse(args['RedSingleBonusProbability'] || "[0,0,0,0,0,0,0,0,0]");
		_redDoubleProb = JSON.parse(args['RedDoubleBonusProbability'] || "[0,0,0,0,0,0,0,0,0]");
		_blueSingleProb = JSON.parse(args['BlueSingleBonusProbability'] || "[0,0,0,0,0,0,0,0,0]");
		_blueDoubleProb = JSON.parse(args['BlueDoubleBonusProbability'] || "[0,0,0,0,0,0,0,0,0]");
		_jacProb = JSON.parse(args['JacProbability'] || "[0,0,0,0,0,0,0,0,0]");
		_promoProb = JSON.parse(args['PromotionProbability'] || "[0,0]");
		_altImageFolder = String(args['AlternativeImageFolder'] || "");
		_fanFareMap = new Map();
		_fanFareMap.set(BonusMode.redSingle, String(args['RedFanfareFile'] || "slot_BB_01a"));
		_fanFareMap.set(BonusMode.redDouble, String(args['DoubleRedFanfareFile'] || "slot_BB_01a"));
		_fanFareMap.set(BonusMode.blueSingle, String(args['BlueFanfareFile'] || "slot_BB_01a"));
		_fanFareMap.set(BonusMode.blueDouble, String(args['DoubleBlueFanfareFile'] || "slot_BB_01a"));
		SceneManager.push(Exc_SlotMachine);
	});

	//--------------------------------------------------
	// Name : Scene_Boot
	// Desc : シーンブートクラス
	//--------------------------------------------------
	const _Scene_Boot_loadGameFonts = Scene_Boot.prototype.loadGameFonts;
	Scene_Boot.prototype.loadGameFonts = function () {
		_Scene_Boot_loadGameFonts.call(this);
		FontManager.load(FontType.mainFont, _mainFontName);
		FontManager.load(FontType.numberFont, _numberFontName);
	};

	//--------------------------------------------------
	// Name : Exc_SlotMachine
	// Desc : スロットマシーンクラス
	//--------------------------------------------------
	// シーンの作成
	function Exc_SlotMachine() {
		this.initialize.apply(this, arguments);
	};
	Exc_SlotMachine.prototype = Object.create(Scene_Base.prototype);
	Exc_SlotMachine.prototype.constructor = Exc_SlotMachine;
	
	// init
	Exc_SlotMachine.prototype.initialize = function() {
		Scene_Base.prototype.initialize.call(this);
	};

	// スプライトの作成（画像の準備）
	Exc_SlotMachine.prototype.create = function() {
		Scene_Base.prototype.create.call(this);
		this.loadImageBitmap();
		this.createBackground();
		this.createReel();
		this.createFrame();
		this.createText();
		this.createJacLamp();
		this.createEtcImgSprite();
		// 変数の初期化
		this.machineReset();
		this.resultReset();
		// デフォルトBGMの取得
		$gameSystem.saveBgm();
		// 開始直後の入力不可設定
		_machineDisableCount = INIT_DISABLE_FRAME;
	};
	
	//--------------------------------------------------
	// 画面初期表示
	//--------------------------------------------------
	// 画像準備（マジックナンバーが多すぎやしないかね
	Exc_SlotMachine.prototype.loadImageBitmap = function() {

		let imgFolder = _altImageFolder;
		if(imgFolder == ""){
			imgFolder = PICT_PATH;
		}
		
		// 背景画像の読み込み
		_bmpBackGround = [];
		for(let i = 1; i <= 3; i++) {
			_bmpBackGround.push(ImageManager.loadBitmap(imgFolder, PICT_BACKGROUND_NAME + i, 0, false));
		}
		// フレームの読み込み
		_bmpFrame = [];
		for(let i = 1; i <= 6; i++) {
			_bmpFrame.push(ImageManager.loadBitmap(imgFolder, PICT_FRAME_NAME + i, 0, false));
		}
		// 絵柄画像の作成(絵柄パターンの組み合わせ、数に変更があった場合はここも修正)
		_bmpPatterns = [];
		for(let i = 1; i <= 7; i++) {
			let tmpPatterns = [];
			tmpPatterns.push(ImageManager.loadBitmap(imgFolder, PICT_PATTERN_NAME + i, 0, false));
			tmpPatterns.push(ImageManager.loadBitmap(imgFolder, PICT_PATTERN_NAME + i + PICT_GLOW_TRAILING, 0, false));
			tmpPatterns.push(ImageManager.loadBitmap(imgFolder, PICT_PATTERN_NAME + i + PICT_BLUR_TRAILING, 0, false));
			_bmpPatterns.push(tmpPatterns);
		}
		// JACランプの読み込み
		_bmpJacLamp = [];
		for(let i = 1; i <= 3; i++) {
			_bmpJacLamp.push(ImageManager.loadBitmap(imgFolder, PICT_JACLAMP_NAME + i, 0, false));
		}
		// テキスト項目背景の読み込み
		_bmpTextBackR = [];
		_bmpTextBackR.push(ImageManager.loadBitmap(imgFolder, PICT_TEXT_BACK_NAME + "r", 0, false));
		_bmpTextBackR.push(ImageManager.loadBitmap(imgFolder, PICT_TEXT_BACK_NAME + "r" + PICT_GLOW_TRAILING, 0, false));
		_bmpTextBackL = [];
		_bmpTextBackL.push(ImageManager.loadBitmap(imgFolder, PICT_TEXT_BACK_NAME + "l", 0, false));
		_bmpTextBackL.push(ImageManager.loadBitmap(imgFolder, PICT_TEXT_BACK_NAME + "l" + PICT_GLOW_TRAILING, 0, false));

		// 履歴用アイコンの読み込み
		_bmpLogIcon = [];
		_bmpLogIcon.push(new Bitmap(10, 10)); // 0番を使わないようにダミーを入れておく
		for(let i = 1; i <= 5; i++) {
			_bmpLogIcon.push(ImageManager.loadBitmap(imgFolder, PICT_LOG_PATTERN_NAME + i, 0, false));
		}
	 
		// 順押し矢印画像の読み込み
		_bmpArrow = ImageManager.loadBitmap(imgFolder, PICT_ARROW_NAME, 0, false)

		// JACオート画像の読み込み
		_bmpJacAuto = ImageManager.loadBitmap(imgFolder, PICT_JAC_AUTO_NAME, 0, false)

		// リール配列画像の読み込み
		_bmpReelList = ImageManager.loadBitmap(imgFolder, PICT_REEL_LIST_NAME, 0, false)

		// ビデオの読み込み(loadVideoはMoviePictureで追加される関数
		_bmpBonusVideo = ImageManager.loadVideo(VIDEO_BONUS_NAME, true);
		_bmpBonusVideo.setVideoLoop(true);
	};

	// 背景設定
	Exc_SlotMachine.prototype.createBackground = function() {
		this._backgroundSprite = new Sprite(_bmpBackGround[BackMode.normal]);
		this.addChild(this._backgroundSprite);
	};
	
	// リール初期設定
	Exc_SlotMachine.prototype.createReel = function() {
		// 保持変数初期化
		_reelObj = [];
		for(let i = 0; i < REEL_PATTERN.length; i++) {
			// リール単位でオブジェクト設定
			let workObj = {
				reelNum: i,
				sprites: [],
				rotDistance: 0,
				rotSpeed: 0,
				rotation: false,
				slideCount:0,
				bottomIndex: REEL_START_NUMBER[i] - 1,
				nextIndex: function(add) {
					return (this.bottomIndex + add) % REEL_PATTERN[this.reelNum].length;
				}
			};
			_reelObj.push(workObj);
			for(let j = REEL_DISP_COUNT; j >= 0; j--) {
				// リールの絵柄のパラメータ初期化
				let workSprite = new Sprite(_bmpPatterns[REEL_PATTERN[i][workObj.nextIndex(j)]-1][PatternState.stop]);
				workSprite.x = REEL_POS_X[i];
				// 上部見切れている分が必要なので絵柄の初期位置は実際のリールの頭位置より絵柄一つ分上になる
				// リールの上から絵柄を詰めている
				workSprite.y = REEL_POS_Y + REEL_PATTERN_HEIGHT * (REEL_DISP_COUNT - 1 - j);
				
				// 初期化した絵柄をそれぞれにセット
				_reelObj[i].sprites.unshift(workSprite);
				this.addChild(workSprite);
			}
		}
	};
	
	// スロットのフレームをセット
	Exc_SlotMachine.prototype.createFrame = function() {
		_sprFrame = new Sprite(_bmpFrame[BonusMode.none]);
		this.addChild(_sprFrame);
	};
	
	// スロットのテキスト表示をセット
	Exc_SlotMachine.prototype.createText = function() {

		// ボーナス回数テキスト
		_txtCntBonusLog = [];
		for(let i = 0; i < 5; i++){
			_txtCntBonusLog.push(this.createCntBonusTextSprite(COUNT_BONUS_POS_Y[i]));
		}
		// 画面下部各種テキスト
		// 左側背景
		_txtBackWin = this.createSprite(_bmpTextBackL[textBackState.normal], 0, TEXT_POS_Y[0], 0);
		this.createSprite(_bmpTextBackL[textBackState.normal], 0, TEXT_POS_Y[1], 0);
		this.createSprite(_bmpTextBackL[textBackState.normal], 0, TEXT_POS_Y[2], 0);
		this.createSprite(_bmpTextBackL[textBackState.normal], 0, TEXT_POS_Y[3], 0);

		// 左側タイトル
		this.createTextSprite(_winTitle, 0, TEXT_POS_Y[0], 0, FontType.mainFont, "left");
		this.createTextSprite(_betTitle, 0, TEXT_POS_Y[1], 0, FontType.mainFont, "left");
		this.createTextSprite(_coinTitle, 0, TEXT_POS_Y[2], 0, FontType.mainFont, "left");
		this.createTextSprite(_moneyTitle, 0, TEXT_POS_Y[3], 0, FontType.mainFont, "left");

		// 左側値
		_txtWin = this.createTextSprite(0, 0, TEXT_POS_Y[0], 0, FontType.numberFont, "right");
		_txtBet = this.createTextSprite(0, 0, TEXT_POS_Y[1], 0, FontType.numberFont, "right");
		_txtCoin = this.createTextSprite(Exc_getCoin(), 0, TEXT_POS_Y[2], 0, FontType.numberFont, "right");
		_txtMoney = this.createTextSprite($gameParty.gold(), 0, TEXT_POS_Y[3], 0, FontType.numberFont, "right");
		
		// 右側背景
		_txtBackGameCnt = this.createSprite(_bmpTextBackR[textBackState.normal], Graphics.width, TEXT_POS_Y[0], 1);
		_txtBackBonusCnt = this.createSprite(_bmpTextBackR[textBackState.normal], Graphics.width, TEXT_POS_Y[1], 1);
		_txtBackJacCnt = this.createSprite(_bmpTextBackR[textBackState.normal], Graphics.width, TEXT_POS_Y[2], 1);
		_txtBackBonusStk = this.createSprite(_bmpTextBackR[textBackState.normal], Graphics.width, TEXT_POS_Y[3], 1);

		// 右側タイトル
		this.createTextSprite(_gameCntTitle, Graphics.width, TEXT_POS_Y[0], 1, FontType.mainFont, "left");
		this.createTextSprite(_bonusGameTitle, Graphics.width, TEXT_POS_Y[1], 1, FontType.mainFont, "left");
		this.createTextSprite(_jacGameTitle, Graphics.width, TEXT_POS_Y[2], 1, FontType.mainFont, "left");
		this.createTextSprite(_bonusStockTitle, Graphics.width, TEXT_POS_Y[3], 1, FontType.mainFont, "left");
		
		// 右側値
		_txtGameCnt = this.createTextSprite("0", Graphics.width, TEXT_POS_Y[0], 1, FontType.numberFont, "right");
		_txtBonusCnt = this.createTextSprite("0/" + BONUS_GAMES_MAX, Graphics.width, TEXT_POS_Y[1], 1, FontType.numberFont, "right");
		_txtJacCnt = this.createTextSprite("0/" + JAC_GAMES_MAX, Graphics.width, TEXT_POS_Y[2], 1, FontType.numberFont, "right");
		_txtBonusStk = this.createTextSprite(0, Graphics.width, TEXT_POS_Y[3], 1, FontType.numberFont, "right");
	};
	
	// スロットのテキスト表示をセット（その際に使用したビットマップを返す
	Exc_SlotMachine.prototype.createTextSprite = function(text, x, y, anchorX, fontType, align) {

		let tmpBmp = new Bitmap(TEXT_WIDTH, TEXT_HEIGHT);
		tmpBmp.fontFace = fontType;
		tmpBmp.fontSize = TEXT_SIZE;
		tmpBmp.drawText(text, TEXT_PADDING, (TEXT_HEIGHT - TEXT_SIZE) / 2 , TEXT_WIDTH - TEXT_PADDING * 2, TEXT_SIZE, align);
		
		this.createSprite(tmpBmp, x, y, anchorX);

		return tmpBmp;
	};

	// スロットのボーナス回数表示をセット（その際に使用したビットマップを返す
	Exc_SlotMachine.prototype.createCntBonusTextSprite = function(y) {

		let tmpBmp = new Bitmap(COUNT_BONUS_WIDTH, COUNT_BONUS_HEIGHT);
		tmpBmp.fontFace = FontType.numberFont;
		tmpBmp.fontSize = COUNT_BONUS_SIZE;
		tmpBmp.drawText(0, COUNT_BONUS_PADDING, (COUNT_BONUS_HEIGHT - COUNT_BONUS_SIZE) / 2 , COUNT_BONUS_WIDTH - COUNT_BONUS_PADDING * 2, COUNT_BONUS_SIZE, "right");
		
		this.createSprite(tmpBmp, COUNT_BONUS_POS_X, y, 0);

		return tmpBmp;
	};

	// スプライトの作成
	Exc_SlotMachine.prototype.createSprite = function(bmp, x, y, anchorX) {
		let tmpSprite = new Sprite(bmp);
		tmpSprite.x = x;
		tmpSprite.y = y;
		tmpSprite.anchor.x = anchorX;
		tmpSprite.anchor.y = 0;
		this.addChild(tmpSprite);
		return tmpSprite;
	};
	
	// テキスト部のテキスト再セット
	Exc_SlotMachine.prototype.redrawText = function(bmp, text) {
		bmp.clear();
		bmp.drawText(text, TEXT_PADDING, (TEXT_HEIGHT - TEXT_SIZE) / 2, TEXT_WIDTH - TEXT_PADDING * 2, TEXT_SIZE, "right");
	};
	
	// スロットのボーナス回数表示再セット
	Exc_SlotMachine.prototype.redrawCntBonusText = function(bmp, text) {
		bmp.clear();
		bmp.drawText(text, COUNT_BONUS_PADDING, (COUNT_BONUS_HEIGHT - COUNT_BONUS_SIZE) / 2 , COUNT_BONUS_WIDTH - COUNT_BONUS_PADDING * 2, COUNT_BONUS_SIZE, "right");
	};

	// JACゲームのランプ作成
	Exc_SlotMachine.prototype.createJacLamp = function() {
		_jacLamp.sprite = [];
		for(let i = 0; i < BONUS_JAC_MAX;i++) {
			let lampSprite = new Sprite(_bmpJacLamp[JacLampState.end]);
			lampSprite.x = JAC_LAMP_POS_X[i];
			lampSprite.y = JAC_LAMP_POS_Y;
			_jacLamp.sprite.push(lampSprite);
			this.addChild(lampSprite);
		}
	};

	// そのほか装飾画像のSprite作成
	Exc_SlotMachine.prototype.createEtcImgSprite= function() {
		// 順押し矢印
		_inOrderArrow = new Sprite(_bmpArrow);
		_inOrderArrow.x = IN_ORDER_ARROW_X;
		_inOrderArrow.y = IN_ORDER_ARROW_Y;
		this.addChild(_inOrderArrow);

		// ジャックオート
		_jacAutoSprite = new Sprite(_bmpJacAuto);
		_jacAutoSprite.x = JAC_AUTO_IMG_X;
		_jacAutoSprite.y = JAC_AUTO_IMG_Y;
		this.addChild(_jacAutoSprite);

		// リール配列
		_reelList = new Sprite(_bmpReelList);
		_reelList.x = REEL_LIST_X;
		_reelList.y = REEL_LIST_Y;
		_reelList.hide();
		this.addChild(_reelList);

		// ボーナス中エフェクト動画
		_sprBonusEffect = new Sprite(_bmpBonusVideo);
		_sprBonusEffect.blendMode = BONUS_EFFECT_BLEND;
		this.addChild(_sprBonusEffect);
	};
	//--------------------------------------------------
	// 画面更新処理
	//--------------------------------------------------
	// 1フレームごとの画面更新処理
	Exc_SlotMachine.prototype.update = function() {
		Scene_Base.prototype.update.call(this);
		
		// キー入力処理
		if(this.enableKeyCheck()) {
			this.keyControll();
		}
		
		// リール位置再計算
		this.reelRotation();

		// 役チェック
		this.reelResultCheck();

		// ボーナス中７揃い予告背景点滅
		this.bonusReelFlash();

		// リザルト点滅
		this.reelResultShow();

		// JACランプ点滅
		this.jacLampFlash();

		// コイン表示更新
		this.displayCoinUpdate();

		// JACオート点滅
		this.jacAutoFlash();

		// リール配列表示
		this.showReelList();
	};
	
	// キー入力処理の前処理
	Exc_SlotMachine.prototype.enableKeyCheck = function() {
		let ret = true;

		// 確認ウィンドウ表示中はキー入力を処理しない
		if(this._exitWindow != null && this._exitWindow.isOpen()) {
			ret = false;
		}

		// キー入力不可カウント内の場合
		if(_machineDisableCount > 0) {
			_machineDisableCount--;
			if(_machineDisableCount == 0 &&
					(_machineMode == MachineMode.result || _machineMode == MachineMode.stop)) {
				// キー入力可能になった直後の処理、BGM切り替えなど
				if(_bonusMEWait > 0) {
					if(_bonusCount == 0) {
						// エンディングを鳴らす処理
						AudioManager.playMe({"name":"slot_BB_06a","volume":BGM_VOLUME,"pitch":100,"pan":0});
						AudioManager.stopBgm();
					} else {
						// ファンファーレを鳴らす処理
						AudioManager.playMe({"name":_fanfare,"volume":BGM_VOLUME,"pitch":100,"pan":0});
						AudioManager.stopBgm();
						_resultFlash.speed = REEL_FLASH_FRAME.fast;
					}
				} else { 
					_resultFlash.speed = REEL_FLASH_FRAME.verySlow;
					if(_jacCount == JAC_GAMES_MAX) {
						// JacゲームのBGMに切り替え
						AudioManager.playBgm({"name":"slot_BB_05a","volume":90,"pitch":100,"pan":0});
						// BIG外JACのフレーム切り替え処理
						if(_bonusCount == 0){
							_sprFrame.bitmap = _bmpFrame[BonusMode.jac];
						} else {
							// Jacに入るときは矢印を消す
							_inOrderArrow.hide();
						}
					} else if(_jacCount == 0){
						// Jacゲーム終了時の処理
						if(_bonusCount > 0 ){
							// ボーナス中Jacゲーム後のBGMに切り替え
							switch(_jacAlready){
								case 1:
									AudioManager.playBgm({"name":"slot_BB_03a","volume":90,"pitch":100,"pan":0});
									break;
								case 2:
									AudioManager.playBgm({"name":"slot_BB_04a","volume":90,"pitch":100,"pan":0});
									break;
							}
							// 順押し矢印表示処理
							if(BONUS_JAC_MAX -_jacAlready == _bonusCount){
								_inOrderArrow.show();
							}
						} else {
							// 通常BGMに切り替え
							$gameSystem.replayBgm();
							// Jacランプもリセット
							_jacLamp.reset();
						}
					}
				}
			}
			ret = false;
		}else if(_bonusMEWait > 0) {
			// ファンファーレ待ちの処理
			_bonusMEWait--;
			if(_bonusMEWait == 0) {
				if(_bonusCount > 0) {
					// ファンファーレ開始後の処理
					_txtBackBonusCnt.bitmap = _bmpTextBackR[textBackState.glow];
					_sprFrame.bitmap = _bmpFrame[_bonusMode];
					_resultFlash.speed = REEL_FLASH_FRAME.verySlow;
					AudioManager.playBgm({"name":"slot_BB_02a","volume":BGM_VOLUME,"pitch":100,"pan":0});
					// 青７ボーナスの場合エフェクトも表示
					if(_bonusMode == BonusMode.blueSingle || _bonusMode == BonusMode.blueDouble){
						_sprBonusEffect.show();
					}
				} else {
					// エンディング停止後の処理
					$gameSystem.replayBgm();
					_resultFlash.speed = REEL_FLASH_FRAME.verySlow;
					// ボーナス終了時は矢印を消す
					_inOrderArrow.hide();
					// Jacランプもリセット
					_jacLamp.reset();

				}
			} 
			ret = false;
		}

		return ret;
	};

	// キー入力受付処理
	Exc_SlotMachine.prototype.keyControll = function() {
		
		// コイン投入後は停止不可
		if (_valueBet.internal == 0 && Input.isTriggered('cancel')) {
			this.openExitWindow();
			return;
		}

		switch (_machineMode) {
			case MachineMode.stop:
			case MachineMode.result:
				if (Input.isTriggered('up') || Input.isLongPressed('ok') ||
						(_enableJacAuto && _jacCount > 0)) {
					if(_valueBet.internal > 0) {
						// コイン投入済みの場合、リールを回す
						this.reelRun();
					} else {
						// コイン未投入の場合、投入処理
						this.insertCoin();
					}
				}
				break;
			case MachineMode.run:
				this.reelStopButton();
				break;
			case MachineMode.replay:
				this.insertCoin();
				_machineMode = MachineMode.result;
				_resultFlash.speed = REEL_FLASH_FRAME.verySlow;
				break;
		}
		// スペースで操作した場合は待ち時間が長くなる
		if (!(_enableJacAuto && _jacCount > 0)){
			if (_machineDisableCount != 0 && Input.isLongPressed('ok')) {
				_machineDisableCount = Exc_getAutoPlayWait();
			}
		}
	};
	
	Exc_SlotMachine.prototype.insertCoin = function() {
		// リプレイの時は音と見た目のみ
		let se = "slot_11a";
		if (_machineMode == MachineMode.replay) {
			_valueBet.internal = INS_DEFAULT;
		} else {
			// コイン枚数のチェック
			if(Exc_getCoin() < INS_DEFAULT || (_jacCount > 0 && Exc_getCoin() < INS_JAC)) {
				// コイン不足
				se = "buzzer1";
			} else {
				// winの初期化
				_valueWin.internal = 0;

				// コイン投入処理
				if(_jacCount > 0 ) {
					_valueBet.internal = INS_JAC;
					se = "slot_12a";
				} else {
					_valueBet.internal = INS_DEFAULT;
					// JACランプの処理
					_jacLamp.endStock();
					if(_bonusCount == 0) {
						_jacLamp.reset();
						// ボーナスwinの初期化
						_valueWin.bonusInternal = 0;
						_valueWin.bonusDisplay = 1;
						_valueWin.bonusHide = true;
						_txtBackWin.bitmap = _bmpTextBackL[textBackState.normal];
					}
				}
				// 投入コイン数を所持数から減らす
				Exc_addCoin(-1 * _valueBet.internal);
				
				// ボーナス中の場合、累計獲得から減算
				if(_jacCount > 0 || _bonusCount > 0) {
					_valueWin.bonusInternal -= _valueBet.internal;
				}

				// 点滅速度の初期化
				_resultFlash.speed = REEL_FLASH_FRAME.verySlow;
			}
		}
		if (_enableJacAuto && _jacCount > 0){
			_machineDisableCount = JAC_AUTO_DISABLE_FRAME;
		}else {
			_machineDisableCount = INS_COIN_DISABLE_FRAME;
		}
		AudioManager.playSe({"name":se,"volume":90,"pitch":100,"pan":0});
	};
	
	// リールの回転開始処理
	Exc_SlotMachine.prototype.reelRun = function() {
	
		if(_errorWin != -1){
			// 前回想定外リール配置のために揃えられなかった場合
			_winSet = _errorWin;
			_errorWin = -1;
		}else if(_bonusReachWin != -1){
			// ボーナスリーチ内の場合
			_winSet = _bonusReachWin;
			_bonusReachWin = -1;
		} else {
			// 役取得
			_winSet = this.getWin();

			// リーチモードセット
			if(_bonusReachCount == 0 && _bonusCount == 0 && _jacCount == 0){
				switch(_winSet){
					case PatternCode.bar:
					case PatternCode.blue7:
					case PatternCode.red7:
					case PatternCode.blue7_w:
					case PatternCode.red7_w:
						_bonusReachCount = Exc_getReachGames();
						break;
				}
			} 
		}

		// 結果表示状態のリセット
		this.resultReset();

		// スロット状態の更新
		_machineMode = MachineMode.run;

		// リールの状態を回転中に設定
		for(let i = 0; i < REEL_PATTERN.length; i++) {
			_reelObj[i].rotation = true;
			_reelObj[i].rotSpeed = REEL_ROTATION_SPEED;
			if(_bonusReachCount > 0){
				_reelObj[i].slideCount = REEL_SLIDE_MAX;
			}
			for(let j = 0; j <= REEL_DISP_COUNT; j++) {
				// 回転中絵柄に置換
				_reelObj[i].sprites[j].bitmap = _bmpPatterns[REEL_PATTERN[i][_reelObj[i].nextIndex(j)]-1][PatternState.blur];
			}
		}
		// 開始後のキー入力不可セット
		if(_enableJacAuto && _jacCount > 0){
			_machineDisableCount = JAC_AUTO_DISABLE_FRAME;
		} else {
			_machineDisableCount = STARTING_DISABLE_FRAME;
		}

		let se ="slot_09a";
		if(_bonusCount == 0 && _jacCount == 0) {
			// 無効化チェック
			if(!Exc_getDisbleBonusNotice()){
				// 通常時のチェリー、スイカ、ボーナス成立時にリール始動音を変更する。
				if(_winSet > 0 && _winSet != PatternCode.replay && _winSet != PatternCode.bell) {
					se ="slot_09b";
				}
			}
		} else if(_winSet == PatternCode.blue7 || _winSet == PatternCode.red7 ||
				_winSet == PatternCode.blue7_w || _winSet == PatternCode.red7_w) {
			// ボーナス中7揃い当選時
			se ="slot_09c";
			_reelFlash.time = REEL_BACK_FLASHING;
			// リール背景の点灯色切り分け
			switch(_winSet){
				case PatternCode.red7:
				case PatternCode.red7_w:
					_reelFlash.backMode = BackMode.red;
					break;
				case PatternCode.blue7:
				case PatternCode.blue7_w:
					_reelFlash.backMode = BackMode.blue;
					break;
			}
		}
		// SEを鳴らす
		AudioManager.playSe({"name":se,"volume":90,"pitch":100,"pan":0});

		// ボーナス表示を表示のみ１進める
		if(_jacCount > 0) {
			this.redrawText(_txtJacCnt, (JAC_GAMES_MAX - _jacCount + 1) + "/" + JAC_GAMES_MAX);
			if(_jacCount == JAC_GAMES_MAX) {
				_jacLamp.flashStock();
			}
		} else {
			// JACランプの更新
			_jacLamp.endStock();
			this.jacLampUpdate();

			this.redrawText(_txtJacCnt, "0/" + JAC_GAMES_MAX);
			if(_bonusCount > 0) {
				this.redrawText(_txtBonusCnt, (BONUS_GAMES_MAX - _bonusCount + 1) + "/" + BONUS_GAMES_MAX);
			} else {
				this.redrawText(_txtBonusCnt, "0/" + BONUS_GAMES_MAX);
				// ゲーム数カウント
				_gameCount++;
				_gameCountAll++;
				this.redrawText(_txtGameCnt, _gameCount + "/" +_gameCountAll);

				// JACランプの初期化
				_jacLamp.reset();
			}
		}


		// テキスト項目背景・フレームをリセット
		if(_bonusCount == 0 && _jacCount == 0) {
			_txtBackGameCnt.bitmap = _bmpTextBackR[textBackState.glow];
			_sprFrame.bitmap = _bmpFrame[BonusMode.none];
			if(_sprBonusEffect.visible){
				_sprBonusEffect.hide();
			}
		} else {
			_txtBackGameCnt.bitmap = _bmpTextBackR[textBackState.normal];
		}
		if(_jacCount == 0) {
			_txtBackJacCnt.bitmap = _bmpTextBackR[textBackState.normal];
		}
		if(_bonusCount == 0) {
			_txtBackBonusCnt.bitmap = _bmpTextBackR[textBackState.normal];
		}

		// リール背景のリセット
		this._backgroundSprite.bitmap = _bmpBackGround[BackMode.normal];
		_reelFlash.mode = 0;

		_txtBackBonusStk.bitmap = _bmpTextBackR[textBackState.normal];
	};

	// リールの停止ボタン押下処理
	Exc_SlotMachine.prototype.reelStopButton= function() {

		let stopTarget = -1;

		// 処理対象振り分け
		if(_enableJacAuto && _jacCount > 0){
			if(_reelObj[ReelIndexes.left].rotation) {
				stopTarget = ReelIndexes.left;
			} else if(_reelObj[ReelIndexes.center].rotation) {
				stopTarget = ReelIndexes.center;
			} else if(_reelObj[ReelIndexes.right].rotation) {
				stopTarget = ReelIndexes.right;
			}
		}else if(Input.isTriggered('left') && _reelObj[ReelIndexes.left].rotation) {
			stopTarget = ReelIndexes.left;
		} else if(Input.isTriggered('down') && _reelObj[ReelIndexes.center].rotation) {
			stopTarget = ReelIndexes.center;
		} else if(Input.isTriggered('right') && _reelObj[ReelIndexes.right].rotation) {
			stopTarget = ReelIndexes.right;
		} else if(Input.isLongPressed('ok')) {
			switch(Number(Exc_getAutoPlayMode() || 0)) {
				case 0:
					if(_reelObj[ReelIndexes.left].rotation) {
						stopTarget = ReelIndexes.left;
					} else if(_reelObj[ReelIndexes.center].rotation) {
						stopTarget = ReelIndexes.center;
					} else if(_reelObj[ReelIndexes.right].rotation) {
						stopTarget = ReelIndexes.right;
					}
					break;
				case 1:
					if(_reelObj[ReelIndexes.left].rotation) {
						stopTarget = ReelIndexes.left;
					} else if(_reelObj[ReelIndexes.right].rotation) {
						stopTarget = ReelIndexes.right;
					} else if(_reelObj[ReelIndexes.center].rotation) {
						stopTarget = ReelIndexes.center;
					}
					break;
				case 2:
					if(_reelObj[ReelIndexes.center].rotation) {
						stopTarget = ReelIndexes.center;
					} else if(_reelObj[ReelIndexes.left].rotation) {
						stopTarget = ReelIndexes.left;
					} else if(_reelObj[ReelIndexes.right].rotation) {
						stopTarget = ReelIndexes.right;
					}
					break;
				case 3:
					if(_reelObj[ReelIndexes.center].rotation) {
						stopTarget = ReelIndexes.center;
					} else if(_reelObj[ReelIndexes.right].rotation) {
						stopTarget = ReelIndexes.right;
					} else if(_reelObj[ReelIndexes.left].rotation) {
						stopTarget = ReelIndexes.left;
					}
					break;
				case 4:
					if(_reelObj[ReelIndexes.right].rotation) {
						stopTarget = ReelIndexes.right;
					} else if(_reelObj[ReelIndexes.left].rotation) {
						stopTarget = ReelIndexes.left;
					} else if(_reelObj[ReelIndexes.center].rotation) {
						stopTarget = ReelIndexes.center;
					}
					break;
				case 5:
					if(_reelObj[ReelIndexes.right].rotation) {
						stopTarget = ReelIndexes.right;
					} else if(_reelObj[ReelIndexes.center].rotation) {
						stopTarget = ReelIndexes.center;
					} else if(_reelObj[ReelIndexes.left].rotation) {
						stopTarget = ReelIndexes.left;
					}
					break;
				case 6:
					if(_bonusCount > BONUS_JAC_MAX && _jacCount == 0) {
						if(_reelObj[ReelIndexes.right].rotation) {
							stopTarget = ReelIndexes.right;
						} else if(_reelObj[ReelIndexes.center].rotation) {
							stopTarget = ReelIndexes.center;
						} else if(_reelObj[ReelIndexes.left].rotation) {
							stopTarget = ReelIndexes.left;
						}
					} else {
						if(_reelObj[ReelIndexes.left].rotation) {
							stopTarget = ReelIndexes.left;
						} else if(_reelObj[ReelIndexes.center].rotation) {
							stopTarget = ReelIndexes.center;
						} else if(_reelObj[ReelIndexes.right].rotation) {
							stopTarget = ReelIndexes.right;
						}
					}
					break;
			}
		}

		if(stopTarget < 0) {
			return;
		}

		let irregular = false;
		switch(stopTarget) {
			case ReelIndexes.left:
				// 左キー処理
				_reelObj[ReelIndexes.left].rotation = false;
				_machineDisableCount = STOP_DISABLE_FRAME;
				// JACゲームではないボーナスゲーム中かつ他のリール回転中かつリプレイもしくはBAR以外がそろう場合
				// →ボーナスゲーム中の順押し処理
				if(_reelObj[ReelIndexes.center].rotation && _reelObj[ReelIndexes.right].rotation && _jacCount == 0 && _bonusCount > 0 && _jacStock > 0 ) {
						// ７が上書きされている場合はストックする
						if(_winSet == PatternCode.red7 || _winSet == PatternCode.red7_w ||
								_winSet == PatternCode.blue7 || _winSet == PatternCode.blue7_w) {
							this.stockBonus(_winSet);
						}
						// ストックを減らしてリプレイをそろえる
						_winSet = PatternCode.replay;
						_jacStock--;
						_jacLamp.decStock();
				}
				break;
			case ReelIndexes.center:
				// 下キー処理
				_reelObj[ReelIndexes.center].rotation = false;
				_machineDisableCount = STOP_DISABLE_FRAME;
				// 変則押しチェック
				if(_reelObj[ReelIndexes.left].rotation && _reelObj[ReelIndexes.right].rotation) {
					irregular = true;
				}
				break;
			case ReelIndexes.right:
				// 右キー処理
				_reelObj[ReelIndexes.right].rotation = false;
				_machineDisableCount = STOP_DISABLE_FRAME;
				// 変則押しチェック
				if(_reelObj[ReelIndexes.left].rotation && _reelObj[ReelIndexes.center].rotation) {
					irregular = true;
				}
				break;
		}
		// 変則押し処理
		if(irregular) {
			if(_jacCount > 0) {
				
				switch(_winSet) {
					case PatternCode.bell:
						// JACゲーム中のベルは変則押しでリプレイに化ける
						_winSet = PatternCode.replay;
						break;
					case PatternCode.replay:
					case PatternCode.bar:
						// JACゲーム中のリプレイまたはBARは変則押しでベルに化ける
						_winSet = PatternCode.bell;
						break;
				}
			} else if(_bonusCount > 0) {
				// ボーナスゲーム中のリプレイまたはBARの時はストックを増やしてベルをそろえる
				if(_winSet == PatternCode.replay || _winSet == PatternCode.bar) {
					_winSet = PatternCode.bell;
					if((_jacStock + _jacAlready) < BONUS_JAC_MAX) {
						_jacStock++;
						_jacLamp.addStock();
					}
				}
			}
		}

		if (_enableJacAuto && _jacCount > 0 && _machineDisableCount > 0){
			_machineDisableCount = JAC_AUTO_DISABLE_FRAME;
		}
	}
	// リールの回転処理
	Exc_SlotMachine.prototype.reelRotation = function() {
		// 回転処理中でない場合処理しない
		if( _machineMode != MachineMode.run) {
			return;
		}
		
		for(let i = 0; i < REEL_PATTERN.length; i++) {
			// すでにスピードが０になっているものは再描画対象外
			if(_reelObj[i].rotSpeed >= 0) {
				// 回転による移動距離の計算
				_reelObj[i].rotDistance += _reelObj[i].rotSpeed;
				
				// 移動距離が絵柄一つ分を超過＝流れた絵柄の再設定と次の絵柄準備
				if (_reelObj[i].rotDistance > REEL_PATTERN_HEIGHT) {
					// 絵柄のインデックスを一つ分進める
					_reelObj[i].bottomIndex = _reelObj[i].nextIndex(1);
					
					// 絵柄を一つ下にずらす
					for(let j = 0; j < REEL_DISP_COUNT; j++) {
						_reelObj[i].sprites[j].bitmap = _reelObj[i].sprites[j + 1].bitmap;
					}
					_reelObj[i].sprites[REEL_DISP_COUNT].bitmap = _bmpPatterns[REEL_PATTERN[i][_reelObj[i].nextIndex(REEL_DISP_COUNT)]-1][PatternState.blur];

					// 回転停止処理中かどうかチェック
					if(this.isStopReel(i)) {
						// 回転を停止
						_reelObj[i].rotDistance = 0;
						_reelObj[i].rotSpeed = 0;
						
						// 停止絵柄に置き換え
						for(let j = 0; j <= REEL_DISP_COUNT; j++) {
							_reelObj[i].sprites[j].bitmap = _bmpPatterns[REEL_PATTERN[i][_reelObj[i].nextIndex(j)]-1][PatternState.stop];
						}

						// 停止音の変更
						let se = "slot_10a"
						if(_bonusCount > 0 &&
								(_winSet >= PatternCode.red7 || _winSet >= PatternCode.red7_w ||
									_winSet >= PatternCode.blue7 || _winSet >= PatternCode.blue7_w)){
							se = "slot_10b"
						}
						// 停止音
						AudioManager.playSe({"name":se,"volume":90,"pitch":100,"pan":0});
					} else {
						// 移動距離を再計算
						_reelObj[i].rotDistance -= REEL_PATTERN_HEIGHT;

						// 最後のリール滑り数のチェック
						let stopReelsCount = 0
						if(_reelObj[ReelIndexes.left].rotSpeed == 0){stopReelsCount++;}
						if(_reelObj[ReelIndexes.center].rotSpeed == 0){stopReelsCount++;}
						if(_reelObj[ReelIndexes.right].rotSpeed == 0){stopReelsCount++;}
						// 他リールが停止かつ自リールが停止処理中の場合にリールの滑り数チェック
						if(stopReelsCount >= 2 && !_reelObj[i].rotation){
							_reelSlideCount++;
							if(_winSet != -1 && _reelSlideCount > REEL_PATTERN[i].length + ERROR_MAX_SLIDE_COUNT_BUFFER){
								// 滑り数がリールの長さ+バッファを超える場合、何らかのエラーと判断、外れに止める
								// その際に当選していた役はそのまま持ち越し
								_errorWin = _winSet;
								_winSet = -1;
							}
						}
					}
				}
				// 表示範囲内の絵柄の縦位置再計算
				for(let j = 0; j <= REEL_DISP_COUNT; j++) {
					_reelObj[i].sprites[j].y = REEL_POS_Y + REEL_PATTERN_HEIGHT * (REEL_DISP_COUNT - j - 1) + _reelObj[i].rotDistance;
				}
			}
		}
	};
	
	// リールがそろってるかチェック
	Exc_SlotMachine.prototype.reelResultCheck = function() {
		// 回転処理中のみチェック
		// 回転処理中でない場合処理しない
		if( _machineMode != MachineMode.run) {
			return;
		}
		// 速度が0でない=回転中のリールがあるかチェック
		let sumSpeed = 0;
		for(let i = 0; i < REEL_PATTERN.length; i++) {
			sumSpeed += _reelObj[i].rotSpeed;
		}
		if(sumSpeed > 0) {
			// まだ回転中のリールがある場合はリザルトチェックしない
			return;
		}
		
		// モードを停止モードに
		_machineMode = MachineMode.stop;

		// 結果の保持用
		_valueWin.internal = 0;
		let nextBonus = BonusMode.none;
		let tmpBefBonusStk = _bonusStock.length;

		// 役のチェック
		let pattern = this.resultCheck(-1);

		// リーチゲームカウントの処理
		if(_bonusReachCount > 0){
			if(pattern.length > 0) {
				// 役がそろっていればカウントリセット
				_bonusReachCount = 0;
			} else {
				// リーチゲームカウントをデクリメント
				_bonusReachCount--;
			}
		}

		if(pattern.length > 0) {
			// モードをリザルトモードに
			_machineMode = MachineMode.result;
			
			// 点滅速度初期値設定
			_resultFlash.speed = REEL_FLASH_FRAME.slow;

			 // 役ごとの処理
			switch(pattern[0]) {
				case PatternCode.replay:
					if(_bonusCount > 0 || _jacCount > 0) {
						// ボーナス中
						_valueWin.internal = 15;
						if(_jacCount == 0) {
							nextBonus = BonusMode.jac;
						}
					} else {
						// 通常
						_machineMode = MachineMode.replay;
						_resultFlash.speed = REEL_FLASH_FRAME.glow;
					}
					break;
				case PatternCode.cherry:
					_valueWin.internal = 4;
					break;
				case PatternCode.bell:
					if(_jacCount > 0) {
						_valueWin.internal = 15;
						// jacゲーム中のベルをカウント
						_jacBellCount++;
					} else if(_bonusMode == BonusMode.redDouble || _bonusMode == BonusMode.blueDouble) {
						// ダブルボーナス中
						_valueWin.internal = 15;
					} else {
						// 通常・シングルボーナス
						_valueWin.internal = 8;
					}
					break;
				case PatternCode.melon:
					_valueWin.internal = 12;
					break;
				case PatternCode.bar:
					_valueWin.internal = 15;
					if(_jacCount == 0) {
						nextBonus = BonusMode.jac;
					}
					break;
				case PatternCode.blue7:
					_valueWin.internal = 15;
					if(pattern.length == 1) {
						if(_bonusCount > 0 ) {
							this.stockBonus(PatternCode.blue7);
							_resultFlash.speed = REEL_FLASH_FRAME.veryFast;
						} else {
							nextBonus = BonusMode.blueSingle;
							_resultFlash.speed = REEL_FLASH_FRAME.middle;
						}
					} else {
						if(_bonusCount > 0 ) {
							this.stockBonus(PatternCode.blue7_w);
							_resultFlash.speed = REEL_FLASH_FRAME.veryFast;
						} else {
							nextBonus = BonusMode.blueDouble;
							_resultFlash.speed = REEL_FLASH_FRAME.middle;
						}
					}
					break;
				
				case PatternCode.red7:
					_valueWin.internal = 15;
					if(pattern.length == 1) {
						if(_bonusCount > 0 ) {
							this.stockBonus(PatternCode.red7);
							_resultFlash.speed = REEL_FLASH_FRAME.veryFast;
						} else {
							nextBonus = BonusMode.redSingle;
							_resultFlash.speed = REEL_FLASH_FRAME.middle;
						}
					} else {
						if(_bonusCount > 0 ) {
							this.stockBonus(PatternCode.red7_w);
							_resultFlash.speed = REEL_FLASH_FRAME.veryFast;
						} else {
							nextBonus = BonusMode.redDouble;
							_resultFlash.speed = REEL_FLASH_FRAME.middle;
						}
					}
					break;
			}
			
			// 払い戻し枚数に合わせてSEを鳴らす
			let seName = "";
			switch(_valueWin.internal) {
				case 0:
					_machineDisableCount = RESULT_DISABLE_FRAME.replay;
					seName = "slot_08a";
					break;
				case 2:
					_machineDisableCount = RESULT_DISABLE_FRAME.coin2;
					if(_jacCount > 0 || _bonusCount > 0) {
						seName = "slot_01b";
					} else {
						seName = "slot_01a";
					}
					break;
				case 4:
					if(_jacCount > 0 || _bonusCount > 0) {
					_machineDisableCount = RESULT_DISABLE_FRAME.coin4;
					seName = "slot_02b";
					} else {
						seName = "slot_02a";
					}
					break;
				case 6:
					_machineDisableCount = RESULT_DISABLE_FRAME.coin6;
					if(_jacCount > 0 || _bonusCount > 0) {
						seName = "slot_03b";
					} else {
						seName = "slot_03a";
					}
					break;
				case 8:
					_machineDisableCount = RESULT_DISABLE_FRAME.coin8;
					if(_jacCount > 0 || _bonusCount > 0) {
						seName = "slot_04b";
					} else {
						seName = "slot_04a";
					}
					break;
				case 10:
					_machineDisableCount = RESULT_DISABLE_FRAME.coin10;
					if(_jacCount > 0 || _bonusCount > 0) {
						seName = "slot_05b";
					} else {
						seName = "slot_05a";
					}
					break;
				case 12:
					_machineDisableCount = RESULT_DISABLE_FRAME.coin12;
					if(_jacCount > 0 || _bonusCount > 0) {
						seName = "slot_06b";
					} else {
						seName = "slot_06a";
					}
					break;
				case 15:
					_machineDisableCount = RESULT_DISABLE_FRAME.coin15;
					if(_jacCount > 0 || _bonusCount > 0) {
						if(pattern[0] == PatternCode.blue7 ||
								pattern[0] == PatternCode.red7) {
							seName = "slot_08b";
						} else {
							seName = "slot_07b";
						}
					} else {
						seName = "slot_07a";
					}
					break;
			}
			AudioManager.playSe({"name":seName,"volume":90,"pitch":100,"pan":0});
			
			// リプレイ以外なら結果の処理
			if(_valueWin.internal != 0) {
				Exc_addCoin(_valueWin.internal);
				
				if(nextBonus != BonusMode.none && _bonusCount == 0) {
					// ボーナス開始時は累計獲得をセット
					_valueWin.bonusInternal = _valueWin.internal;
				} else if(_bonusCount > 0 || _jacCount > 0) {
					// ボーナス中は累計獲得をセット
					_valueWin.bonusInternal += _valueWin.internal;
				}
			}
		} else {
			_machineDisableCount = RESULT_DISABLE_FRAME.none;
		}
		// 投入済みコインを０に
		_valueBet.internal = 0;

		// ボーナスカウントの減少
		if(_jacCount > 0) {
			_jacCount--;
			// JACゲーム終了時
			if(_jacCount == 0) {
				// JACゲーム中すべてベルだった場合
				if(_jacBellCount >= JAC_GAMES_MAX) {
					this.stockBonus(PatternCode.blue7_w);
				}
				_jacBellCount = 0;
				
				if(_bonusCount == 0) {
					// ボーナス外でのJACはゲーム数カウント初期化
					_gameCount = 0;
					_jacAlready = 0;
					// BGMフェードアウト
					AudioManager.fadeOutBgm(JAC_BONUS_FADE_OUT_SEC);
					_machineDisableCount = JAC_BONUS_FADE_OUT_SEC * 60;
				} else if(_jacAlready >= BONUS_JAC_MAX) {
					// JAC回数上限の場合はそこでボーナス終了
					_bonusCount = 0;
					this.endBonus();
				} else {
					_bonusCount--;
					this.endBonus();
				}
			}		
		} else if(_bonusCount > 0 && nextBonus != BonusMode.jac ) {
			// ボーナスゲーム減少、jacゲーム開始時はjac終了時に減らすのでスキップ
			_bonusCount--;
			this.endBonus();
		}
		switch(nextBonus) {
			case BonusMode.jac:
				// JACゲーム開始
				_jacCount = JAC_GAMES_MAX;
				_txtBackJacCnt.bitmap = _bmpTextBackR[textBackState.glow];
				if(_bonusCount == 0) {
					_jacLamp.lamp[0].state = JacLampState.stock;
					_valueWin.bonusHide = false;
					this.bonusLogAdd(BonusMode.jac, _gameCount);
				} else {
					_jacAlready++;
					_jacLamp.addStock();
				}
				break;
			case BonusMode.redSingle:
			case BonusMode.redDouble:
			case BonusMode.blueSingle:
			case BonusMode.blueDouble:
				// それぞれのボーナス開始
				_bonusMode = nextBonus;
				_bonusCount = BONUS_GAMES_MAX;
				_jacLamp.setWait();
				_bonusMEWait = ME_FANFARE_FRAME;
				_resultFlash.speed = REEL_FLASH_FRAME.glow;
				_valueWin.bonusHide = false;
				this.bonusLogAdd(_bonusMode, _gameCount);

				// ファンファーレの指定
				_fanfare = _fanFareMap.get(nextBonus);
				break;
		}

		// 表示の更新
		this.redrawText(_txtBonusStk, _bonusStock.length);
		this.jacLampUpdate();
	
		// ボーナスストックが増加していた場合は背景に色を付ける
		if(tmpBefBonusStk < _bonusStock.length) {
			_txtBackBonusStk.bitmap = _bmpTextBackR[textBackState.glow];
		}

		// スペース押下中は待ち時間が長くなる
		if (_machineDisableCount < Exc_getAutoPlayWait() && Input.isLongPressed('ok')) {
			_machineDisableCount = Exc_getAutoPlayWait();
		}
	};
	
	// ボーナスストック処理（格納時ランクアップ）
	Exc_SlotMachine.prototype.stockBonus = function(bonus) {
		let tmpPattern = bonus;

		//赤→青ランクアップチェック
		if(tmpPattern == PatternCode.red7 || tmpPattern == PatternCode.red7_w) {
			let promoRnd = Math.floor(Math.random() * RANDOM_MAX);
			if(_promoProb[0] > promoRnd) {
				switch(tmpPattern) {
					case PatternCode.red7:
						tmpPattern = PatternCode.blue7;
						break;
					case PatternCode.red7_w:
						tmpPattern = PatternCode.blue7_w;
						break;
				}
			}
		}
		//シングル→ダブルランクアップチェック
		if(tmpPattern == PatternCode.red7 || tmpPattern == PatternCode.blue7) {
			let promoRnd = Math.floor(Math.random() * RANDOM_MAX);
			if(_promoProb[1] > promoRnd) {
				switch(tmpPattern) {
					case PatternCode.red7:
						tmpPattern = PatternCode.red7_w;
						break;
					case PatternCode.blue7:
						tmpPattern = PatternCode.blue7_w;
						break;
				}
			}
		}
		// ランクアップ結果を格納
		_bonusStock.push(tmpPattern);
	}

	// ボーナス終了処理
	Exc_SlotMachine.prototype.endBonus = function() {
		// ボーナスゲーム終了時
		if(_bonusCount == 0) {
			// JACゲーム回数が不足した場合、ボーナス付与
			if(_jacStock + _jacAlready < BONUS_JAC_MAX) {
				let tmpPattern = -1;
				switch(_bonusMode) {
					case BonusMode.redSingle:
						tmpPattern = PatternCode.red7;
						break;
					case BonusMode.redDouble:
						tmpPattern = PatternCode.red7_w;
						break;
					case BonusMode.blueSingle:
						tmpPattern = PatternCode.blue7;
						break;
					case BonusMode.blueDouble:
						tmpPattern = PatternCode.blue7_w;
						break;
				}
				if(tmpPattern != -1) {
					for(let i = _jacStock + _jacAlready; i < BONUS_JAC_MAX; i++) {
						this.stockBonus(tmpPattern);
					}
				}
			}

			// エンディングを鳴らす準備
			_bonusMEWait = ME_ENDING_FRAME;

			// ボーナス中のwin累計が最大の場合のみ背景に色を付ける
			if(_valueWin.bonusInternal == MAX_BONUS_WIN) {
				_txtBackWin.bitmap = _bmpTextBackL[textBackState.glow];
			}

			// ボーナス終了時に初期化
			_gameCount = 0;
			_bonusMode = BonusMode.none;
			_jacStock = 0;
			_jacAlready = 0;
		}
	};
	
	// ボーナス履歴の追加
	Exc_SlotMachine.prototype.bonusLogAdd = function(bonusType, gameCount) {

		// テキストbmpの作成
		let tmpBmp = new Bitmap(BONUS_LOG_WIDTH, BONUS_LOG_HEIGHT);
		tmpBmp.fontFace = FontType.numberFont;
		tmpBmp.fontSize = BONUS_LOG_FONT_SIZE;
		tmpBmp.drawText(gameCount, BONUS_LOG_PADDING, (BONUS_LOG_HEIGHT - BONUS_LOG_FONT_SIZE) / 2 , BONUS_LOG_WIDTH - BONUS_LOG_PADDING * 2, BONUS_LOG_FONT_SIZE, "right");

		let iconPadding = (BONUS_LOG_HEIGHT - BONUS_LOG_ICON_HEIGHT) / 2;
		if(_bonusLog.length >= BONUS_LOG_MAX){
			// 上限超過は再利用
			let tmpArray = _bonusLog.pop();
			tmpArray[LogSprites.number].bitmap = tmpBmp;
			tmpArray[LogSprites.icon].bitmap = _bmpLogIcon[bonusType];
			_bonusLog.unshift(tmpArray);

		}else {
			// スプライトの新規追加
			let tmpNumberSprite = new Sprite(tmpBmp);
			tmpNumberSprite.x = BONUS_LOG_POS_X;
			this.addChild(tmpNumberSprite);
	
			let tmpIconSprite = new Sprite(_bmpLogIcon[bonusType]);
			tmpIconSprite.x = BONUS_LOG_POS_X + BONUS_LOG_PADDING;
			this.addChild(tmpIconSprite);
	
			_bonusLog.unshift([tmpNumberSprite, tmpIconSprite]);
		}
		
		// Y座標の再計算
		for(let i = 0; i < _bonusLog.length; i++){
			// 縦位置の再計算
			_bonusLog[i][LogSprites.number].y = BONUS_LOG_POS_Y + (BONUS_LOG_HEIGHT * i);
			_bonusLog[i][LogSprites.icon].y = BONUS_LOG_POS_Y + (BONUS_LOG_HEIGHT * i) + iconPadding;
		}

		// ボーナス回数の追加(noneはカウントしてないのでその分インデックスが前にずれている)
		if(_cntBonusLog[bonusType - 1] < COUNT_BONUS_MAX){
			_cntBonusLog[bonusType - 1]++;
			this.redrawCntBonusText(_txtCntBonusLog[bonusType - 1], _cntBonusLog[bonusType - 1]);
		}
	}

	// ボーナス中７のリール背景点滅
	Exc_SlotMachine.prototype.bonusReelFlash = function() {

		// 切り替わり時間内　もしくは　切り替わり時間後、色なし状態
		if(_reelFlash.time > 0){
			_reelFlash.time--;

			// 最後は点灯状態を維持するよう設定
			if(_reelFlash.time == 0){
				_reelFlash.count = 0;
				_reelFlash.mode = 0;
			}

			if(_reelFlash.count > 0){
				_reelFlash.count--;
			} else {
				// 切り替わりモードのチェック
				let backBmp;
				switch(_reelFlash.mode){
					case 0:
						// 色付き背景の処理
						backBmp = _bmpBackGround[_reelFlash.backMode];
						_reelFlash.mode = 1;
						break;
					case 1:
						// 通常背景の処理
						backBmp = _bmpBackGround[BackMode.normal];
						_reelFlash.mode = 0;
						break;
				}
				this._backgroundSprite.bitmap = backBmp;
				_reelFlash.count = REEL_BACK_FLASH_WAIT;
			}
			
		}	 
	}

	// リザルト表示(絵柄点滅処理)
	Exc_SlotMachine.prototype.reelResultShow = function() {
		// リザルト表示もしくはリプレイ待機モードでない場合処理しない
		if(!(_machineMode == MachineMode.result || _machineMode == MachineMode.replay)) {
			return;
		}
		// リザルトが格納されていない場合処理しない
		if(_resultIndexes[0].length == 0) {
			return;
		}

		if(_resultFlash.speed == REEL_FLASH_FRAME.glow){
			// 点灯モードの処理
			if(_resultFlash.mode == 0){
				_resultFlash.mode = 1;
				for(let i = 0; i < _resultIndexes.length; i++) {
					for (let j = 0; j < _resultIndexes[i].length; j++) {
						_reelObj[i].sprites[_resultIndexes[i][j]].bitmap = _bmpPatterns[REEL_PATTERN[i][_reelObj[i].nextIndex(_resultIndexes[i][j])]-1][PatternState.glow];
					}
				}
			}
		}else {
			// カウント0の場合のみ状態切り替え
			if(_resultFlash.count == 0) {
				// カウントセット
				_resultFlash.count = _resultFlash.speed;
				// 発光表現
				let tmpState = PatternState.stop;
				switch(_resultFlash.mode) {
					case 0:
						tmpState = PatternState.glow;
						_resultFlash.mode = 1;
						break;
					case 1:
						_resultFlash.mode = 0;
						break;
				}
				for(let i = 0; i < _resultIndexes.length; i++) {
					for (let j = 0; j < _resultIndexes[i].length; j++) {
						_reelObj[i].sprites[_resultIndexes[i][j]].bitmap = _bmpPatterns[REEL_PATTERN[i][_reelObj[i].nextIndex(_resultIndexes[i][j])]-1][tmpState];
					}
				}
			} else {
				_resultFlash.count--;
			}
		}
	};

	// JACランプ表示更新
	Exc_SlotMachine.prototype.jacLampUpdate = function() {
		for(let i = 0; i < _jacLamp.sprite.length; i++) {
			switch(_jacLamp.lamp[i].state) {
				case JacLampState.wait:
				case JacLampState.stock:
				case JacLampState.end:
					_jacLamp.sprite[i].bitmap = _bmpJacLamp[_jacLamp.lamp[i].state];
					break;
			}
		}
	};

	// JACランプ点滅処理
	Exc_SlotMachine.prototype.jacLampFlash = function() {
		for(let i = 0; i < _jacLamp.sprite.length; i++) {
			if(_jacLamp.lamp[i].state == JacLampState.flash) {
				if(_jacLamp.lamp[i].flashCount > 0) {
					_jacLamp.lamp[i].flashCount--;
				} else {
					_jacLamp.lamp[i].flashCount = JAC_LAMP_FLASH_FRAME;
					if(_jacLamp.sprite[i].bitmap === _bmpJacLamp[JacLampState.wait]) {
						_jacLamp.sprite[i].bitmap = _bmpJacLamp[JacLampState.stock];
					}else{
						_jacLamp.sprite[i].bitmap = _bmpJacLamp[JacLampState.wait];
					}
				}
			}
		}
	};

	// win/bet/coinの表示処理
	Exc_SlotMachine.prototype.displayCoinUpdate = function() {
			// テキスト再セット
			// win
			if(_valueWin.wait > 0) {
				_valueWin.wait--;
			} else if(_valueWin.internal != _valueWin.display ||
					_valueWin.bonusInternal != _valueWin.bonusDisplay) {
				_valueWin.wait = NUM_UPDATE_WAIT;
				if(_valueWin.internal != _valueWin.display) {
					if(_valueWin.internal == 0) {
						_valueWin.display = 0;
					} else {
						_valueWin.display++;
					}
				}
				let tmpWin = "" + _valueWin.display;
				if(!_valueWin.bonusHide) {
					if(_valueWin.bonusInternal > _valueWin.bonusDisplay) {
						_valueWin.bonusDisplay++;
					} else if(_valueWin.bonusInternal < _valueWin.bonusDisplay) {
						_valueWin.bonusDisplay--;
					}
					tmpWin += "/" + _valueWin.bonusDisplay;
				}
				this.redrawText(_txtWin, tmpWin);
			}

			// bet
			if(_valueBet.wait > 0) {
				_valueBet.wait--;
			} else if(_valueBet.internal != _valueBet.display) {
				_valueBet.wait = NUM_UPDATE_WAIT;
				if(_valueBet.internal == 0) {
					_valueBet.display = 0;
				} else {
					_valueBet.display++;
				}
				this.redrawText(_txtBet, _valueBet.display);
			}
			
			// Coin
			let coin = Exc_getCoin()
			if(_valueCoin.wait > 0) {
				_valueCoin.wait--;
			} else if(coin != _valueCoin.display) {
				_valueCoin.wait = NUM_UPDATE_WAIT;
				if(coin < _valueCoin.display) {
					_valueCoin.display--;
				} else {
					_valueCoin.display++;
				}
				this.redrawText(_txtCoin, _valueCoin.display);
			}
	}

	// JACオート点滅処理
	Exc_SlotMachine.prototype.jacAutoFlash = function() {
		// ジャックオートの表示終了処理
		if(_jacCount <= 0 && _jacAutoFlash >= 0 && _machineDisableCount <= 0 && _bonusMEWait <= 0) {
			_jacAutoSprite.hide();
			_jacAutoFlash = -1;
		} else if((_enableJacAuto && _jacCount > 0) || _jacAutoFlash >= 0){
			// ジャックオートに入った直後、もしくはすでに点滅中の場合、点滅処理
			if(_jacAutoFlash < 0 ){
				// 最初の処理自体はボーナスに入った直後のSE後
				if(_machineDisableCount <= 0 && _bonusMEWait <= 0){
					_jacAutoFlash = JAC_AUTO_FLASH_FRAME;
					_jacAutoSprite.show();
				}
			} else if (_jacAutoFlash > 0) {
				_jacAutoFlash--;
			} else {
				_jacAutoFlash = JAC_AUTO_FLASH_FRAME;
				if(_jacAutoSprite.visible){
					_jacAutoSprite.hide();
				} else {
					_jacAutoSprite.show();
				}
			}
		}
	}

	// リール配列の表示・非表示
	Exc_SlotMachine.prototype.showReelList = function() {
		// シフト入力で切り替え
		if(Input.isTriggered("shift")){
			if(_reelList.visible){
				_reelList.hide();
			} else {
				_reelList.show();
			}
		}
	};

	//--------------------------------------------------
	// 内部での処理
	//--------------------------------------------------
	// 内部の状態リセット
	Exc_SlotMachine.prototype.machineReset = function() {
		_valueWin = {internal:0,display:0,bonusInternal:0,bonusDisplay:0,bonusHide:true,wait:0};
		_valueBet = {internal:0,display:0,wait:0};
		_valueCoin = {display:Exc_getCoin()-1,wait:0};
		_gameCount = 0;
		_gameCountAll = 0;
		_fanfare = "";
		_errorWin = -1;
		_bonusMode = BonusMode.none;
		_jacCount= 0;
		_bonusCount = 0;
		_jacAlready = 0;
		_jacStock = 0;
		_bonusStock = [];
		_bonusReachCount = 0;
		_bonusReachWin = -1;
		_cntBonusLog = [0, 0, 0, 0, 0];
		_jacBellCount = 0;
		_jacLamp.reset();
		_bonusLog = [];
		_inOrderArrow.hide();
		_jacAutoSprite.hide();
		_enableJacAuto = Exc_getEnableJacAuto();
		_jacAutoFlash = -1;
		_sprBonusEffect.hide();
	};

	// 画面の状態リセット
	Exc_SlotMachine.prototype.resultReset = function() {
		_machineMode = MachineMode.stop;
		_resultIndexes = [[],[],[]];
		_resultFlash.mode = 0;
		_resultFlash.count = 0;
		_resultFlash.speed = REEL_FLASH_FRAME.verySlow;
		_reelFlash.time = 0;
		_reelFlash.mode = 0;
		_reelFlash.count = 0;
		_reelFlash.backMode = BackMode.normal;
		_reelSlideCount = 0;
	};
	
	// リール停止可否
	Exc_SlotMachine.prototype.isStopReel = function(reelIndex) {
		// 停止状態になっていなければ不可
		if(_reelObj[reelIndex].rotation) {
			return false;
		}
		
		// リーチ滑りの処理、すでにハズレになっていれば処理不要
		if(_bonusReachCount > 1 && _winSet != -1){
			if(_reelObj[reelIndex].slideCount > 0){
				if(_reelObj[reelIndex].slideCount == REEL_SLIDE_MAX){
					// 初回のみ滑り範囲内に対象が存在するかチェック
					// w揃いの場合の確認対象置換
					let targetPtn;
					switch(_winSet){
						case PatternCode.blue7_w:
							targetPtn = PatternCode.blue7;
							break;
						case PatternCode.red7_w:
							targetPtn = PatternCode.red7;
							break;
						default:
							targetPtn = _winSet;
							break;
					}
					let exists = 0;
					// MAX⇒(滑り最大値) + (画面内表示数 - 1)
					for(let i = 0; i < REEL_SLIDE_MAX + REEL_DISP_COUNT - 1; i++){
						if(targetPtn == REEL_PATTERN[reelIndex][_reelObj[reelIndex].nextIndex(i)]){
							exists++;
						}
					}
					if(exists == 0 || ( reelIndex != ReelIndexes.center && (_winSet == PatternCode.blue7_w || _winSet == PatternCode.red7_w) && exists < 2)){
						// 存在しない場合外れに
						_reelObj[reelIndex].slideCount = 1; //下のデクリメントで0になる
						_bonusReachWin = _winSet;
						_winSet = -1;
					}
				}
				// リーチ時の滑りデクリメント
				_reelObj[reelIndex].slideCount--;
			} else {
				// リーチ時の滑りの外れ処理
				_bonusReachWin = _winSet;
				_winSet = -1;
			}
		}

		// 既停止リール数のカウント
		let rotStopCnt = 0;
		for(let i = 0; i < _reelObj.length; i++) {
			if(_reelObj[i].rotSpeed == 0) {
				rotStopCnt++;
			}
		}
		if(_winSet == -1) {
			// 役なしの場合
			if(_bonusReachCount == 0 && _bonusCount == 0 && _jacCount == 0 && NO_STOP_NUMBER[reelIndex].includes(_reelObj[reelIndex].bottomIndex + 1)) {
				// 通常時停止不可位置の場合
				return false;
			} else if(_bonusReachCount != 0 && NO_STOP_REACH_NUMBER[reelIndex].includes(_reelObj[reelIndex].bottomIndex + 1)) {
				// リーチ時停止不可位置の場合
				return false;
			} else if(rotStopCnt < REEL_PATTERN.length - 1) {
				// 左リールの場合のみ
				if(reelIndex == ReelIndexes.left) {
					// 停止範囲内にチェリーを含まない
					for(let i = 0; i < REEL_PATTERN.length; i++) {
						if(REEL_PATTERN[reelIndex][_reelObj[reelIndex].nextIndex(i)] == PatternCode.cherry) {
							return false;
						}
					}
				}

				// 通常時は３役リーチは不可
				if(_bonusReachCount == 0 && _bonusCount == 0 && _jacCount == 0 && rotStopCnt != 0){
					let tmpResult = this.resultCheck(reelIndex);
					if(tmpResult.length >= REEL_DISP_COUNT){
						return false;
					}
				}

				// 上記以外で最後ではなければ特に条件なし
				return true;
			} else {
				// 通常時は異色7揃いは不可
				if(_bonusReachCount == 0 && _bonusCount == 0 && _jacCount == 0 && this.mix7Check()){
					return false
				}
				// 最後の一つの場合のみ役不成立位置で止める
				let tmpResult = this.resultCheck(reelIndex);
				return (tmpResult.length == 0);
			}
		} else if(_winSet == PatternCode.blue7_w) {
			// 青7の２重 ほぼ決め打ち
			if(reelIndex == ReelIndexes.left && _reelObj[ReelIndexes.right].rotSpeed == 0) {
				if(_reelObj[ReelIndexes.right].bottomIndex + 1 == BLUE7_W_NUMBER[ReelIndexes.right][0]) {
					return BLUE7_W_NUMBER[ReelIndexes.left][0] == _reelObj[ReelIndexes.left].bottomIndex + 1;
				} else {
					return BLUE7_W_NUMBER[ReelIndexes.left][1] == _reelObj[ReelIndexes.left].bottomIndex + 1;
				}
			} else if(reelIndex == ReelIndexes.right && _reelObj[ReelIndexes.left].rotSpeed == 0) {
				if(_reelObj[ReelIndexes.left].bottomIndex + 1 == BLUE7_W_NUMBER[ReelIndexes.left][0]) {
					return BLUE7_W_NUMBER[ReelIndexes.right][0] == _reelObj[ReelIndexes.right].bottomIndex + 1;
				} else {
					return BLUE7_W_NUMBER[ReelIndexes.right][1] == _reelObj[ReelIndexes.right].bottomIndex + 1;
				}
			} else {
				return BLUE7_W_NUMBER[reelIndex].includes(_reelObj[reelIndex].bottomIndex + 1);
			}
		} else if(_winSet == PatternCode.red7_w) {
			// 赤7の２重
			return RED7_W_NUMBER[reelIndex].includes(_reelObj[reelIndex].bottomIndex + 1);
		} else {
			// 順押しではない、もしくはJAC GAMEの場合、役を中段にそろえる
			if((_reelObj[ReelIndexes.left].rotSpeed != 0 || _jacCount > 0) && 
					reelIndex != ReelIndexes.left && rotStopCnt < REEL_PATTERN.length - 1) {
				if (_winSet == PatternCode.cherry) {
					// チェリーの場合は引き込み不要
					return true;
				} else {
					return _winSet == REEL_PATTERN[reelIndex][_reelObj[reelIndex].nextIndex(REEL_CENTER_INDEX)];
				}
			} else if(reelIndex == 0 && _winSet == PatternCode.cherry && _jacCount == 0 &&
					REEL_PATTERN[reelIndex][_reelObj[reelIndex].nextIndex(REEL_CENTER_INDEX)] == PatternCode.cherry) {
				// 中段チェリー不可
				return false;
			} else if(_winSet == PatternCode.bar && 
					NO_STOP_BAR_NUMBER[ReelIndexes.left].includes(_reelObj[ReelIndexes.left].bottomIndex + 1) &&
					NO_STOP_BAR_NUMBER[ReelIndexes.right].includes(_reelObj[ReelIndexes.right].bottomIndex + 1) &&
					_reelObj[ReelIndexes.center].rotSpeed != 0 &&
					((reelIndex == ReelIndexes.left && !_reelObj[ReelIndexes.right].rotSpeed == 0) ||
						(reelIndex == ReelIndexes.right && !_reelObj[ReelIndexes.left].rotSpeed == 0))) {
				//はさみ押しの時に特定並びのBARは不可（絵柄の並び順の問題で複数役が成立する箇所でしか停止できなくなる
				return false;
			} else if(reelIndex == ReelIndexes.left && (rotStopCnt == 0 || _jacCount > 0 && rotStopCnt < ReelIndexes.right)) {
				// 最初の停止リールかつ左リールの場合
				let check = false;
				for(let i = 0; i < REEL_PATTERN.length; i++) {
					let tmpTarget = REEL_PATTERN[reelIndex][_reelObj[reelIndex].nextIndex(i)];
					if(_winSet != PatternCode.cherry && tmpTarget == PatternCode.cherry) {
						// チェリーをそろえようとしていないのにチェリーが含まれる場合は不可
						return false;
					} else if(_winSet == tmpTarget) {
						if (_jacCount > 0 && i != REEL_CENTER_INDEX) {
							// JACゲーム中は中段のみ絵柄と一致するかチェック
						} else {
							// いずれかに対象の絵柄が含まれること
							check = true;
						}
					}
				}
				return check;
			} else {
				// ２つ目以降の場合
				let tmpResult = this.resultCheck(reelIndex);
				if(rotStopCnt < REEL_PATTERN.length - 1 || tmpResult.length == 1) {
					// 最後のリールでない場合、対象がそろう可能性があればOK
					// ただし、チェリーはチェックする
					// また、青７が左右の停止位置がダブルになるパターンの停止は不可
					// 最後のリールの場合は、揃うのが一つである必要あり
					if(_winSet != PatternCode.cherry && tmpResult.includes(PatternCode.cherry)) {
						return false;
					}else if(_winSet == PatternCode.blue7 && reelIndex == ReelIndexes.left && _reelObj[ReelIndexes.right].rotSpeed == 0) {
						if(_reelObj[ReelIndexes.right].bottomIndex + 1 == BLUE7_W_NUMBER[ReelIndexes.right][0] ) {
							if(_reelObj[ReelIndexes.left].bottomIndex + 1 == BLUE7_W_NUMBER[ReelIndexes.left][0]) {
								return false;
							}
						} else {
							if(_reelObj[ReelIndexes.left].bottomIndex + 1 == BLUE7_W_NUMBER[ReelIndexes.left][1]) {
								return false;
							}
						}
					} else if(_winSet == PatternCode.blue7 && reelIndex == ReelIndexes.right && _reelObj[ReelIndexes.left].rotSpeed == 0) {
						if(_reelObj[ReelIndexes.left].bottomIndex + 1 == BLUE7_W_NUMBER[ReelIndexes.left][0]) {
							if(_reelObj[ReelIndexes.right].bottomIndex + 1 == BLUE7_W_NUMBER[ReelIndexes.right][0]) {
								return false;
							}
						} else {
							if(_reelObj[ReelIndexes.right].bottomIndex + 1 == BLUE7_W_NUMBER[ReelIndexes.right][1]) {
								return false;
							}
						}
					}
					return tmpResult.includes(_winSet);
				} else if(_winSet == PatternCode.cherry) {
					// チェリーが複数そろっている場合はすべて同一絵柄であること
					for(let i = 0; i< tmpResult.length; i++) {
						if(_winSet != tmpResult[i]) {
							return false;
						}
					}
					return true;
				}
			}
		}
		return false;
	};

	// 当選役取得
	Exc_SlotMachine.prototype.getWin = function() {
		let prob = [];
		
		if(_jacCount > 0) {
			// ジャックゲーム中
			prob = _jacProb;
		} else if(_bonusCount == 0 && _bonusStock.length > 0) {
			// ボーナス外で7のストックがある場合はすぐにボーナス準備
			let tmpPattern = -1;
			let workIndex = _bonusStock.indexOf(PatternCode.red7);
			if(workIndex == -1) {
				workIndex = _bonusStock.indexOf(PatternCode.red7_w);
				if(workIndex == -1) {
					workIndex = _bonusStock.indexOf(PatternCode.blue7);
					if(workIndex == -1) {
						workIndex = _bonusStock.indexOf(PatternCode.blue7_w);
					}
				}
			}
			if(workIndex != -1) {
				tmpPattern = _bonusStock.splice(workIndex,1)[0];
			}
			return tmpPattern;
		} else {
			//ボーナス状態によって振り分け
			switch(_bonusMode) {
				case BonusMode.none:
					prob = _normalProb;
					break;
				case BonusMode.redSingle:
					prob = _redSingleProb;
					break;
				case BonusMode.redDouble:
					prob = _redDoubleProb;
					break;
				case BonusMode.blueSingle:
					prob = _blueSingleProb;
					break;
				case BonusMode.blueDouble:
					prob = _blueDoubleProb;
					break;
			}
		}
		
		// ランダム値をもとに役を確定
		let rnd = Math.floor(Math.random() * RANDOM_MAX);
		let workSum = 0;
		let ret = -1;
		for(let i = 0; i < prob.length; i++) {
			workSum += Number(prob[i] || 0);
			if(rnd < workSum) {
				ret = i + 1;
				break;
			}
		}
		return ret;
	};
	
	// リザルトのチェック
	Exc_SlotMachine.prototype.resultCheck = function(checkIndex) {
		let ret = [];
		
		let baseReel = -1;
		// 最初のリール取得
		for(let i = 0; i< REEL_PATTERN.length; i++) {
			if(_reelObj[i].rotSpeed == 0 || checkIndex == i) {
				baseReel = i;
				break;
			}
		}
		
		// チェック用変数
		let aligned = false;
		_resultIndexes = [[],[],[]];
		// リールの回転処理と違って横方向に探査する点に注意
		for(let i = 0; i < REEL_DISP_COUNT; i++) {
			aligned = true;
			let work = REEL_PATTERN[baseReel][_reelObj[baseReel].nextIndex(i)];
			for(let j = baseReel + 1; j < REEL_PATTERN.length; j++) {
				if(_reelObj[j].rotSpeed != 0 && checkIndex != j) {
					// リールが停止しておらず、確認対象でもない場合はスキップ
					continue;
				} else if( work != REEL_PATTERN[j][_reelObj[j].nextIndex(i)]) {
					aligned = false;
					break;
				}
			}
			if(aligned) {
				ret.push(work);
				// インデックスを保持
				for(let j = 0; j < REEL_PATTERN.length; j++) {
					if(baseReel <= j) { 
						_resultIndexes[j].push(i);
					} else {
						_resultIndexes[j].push(-1);
					}
				}
			}
		}
		// 斜め右上
		aligned = true;
		let work = REEL_PATTERN[baseReel][_reelObj[baseReel].bottomIndex];
		for(let i = baseReel + 1; i < REEL_PATTERN.length; i++) {
			if(_reelObj[i].rotSpeed != 0 && checkIndex != i) {
				// リールが停止しておらず、確認対象でもない場合はスキップ
				continue;
			} else if( work != REEL_PATTERN[i][_reelObj[i].nextIndex(i)]) {
				aligned = false;
				break;
			}
		}
		if(aligned) {
			ret.push(work);
			// インデックスを保持
			for(let i = 0; i < REEL_PATTERN.length; i++) {
				if(_reelObj[i].rotSpeed != 0 && checkIndex != i) { 
					_resultIndexes[i].push(-1);
				} else {
					_resultIndexes[i].push(i);
				}
			}
		}
		// 斜め右下
		aligned = true;
		work = REEL_PATTERN[baseReel][_reelObj[baseReel].nextIndex(REEL_DISP_COUNT - 1)];
		for(let i = baseReel + 1; i < REEL_PATTERN.length; i++) {
			if(_reelObj[i].rotSpeed != 0 && checkIndex != i) {
				// リールが停止しておらず、確認対象でもない場合はスキップ
				continue;
			} else if( work != REEL_PATTERN[i][_reelObj[i].nextIndex(REEL_DISP_COUNT - 1 - i )]) {
				aligned = false;
				break;
			}
		}
		if(aligned) {
			ret.push(work);
			// インデックスを保持
			for(let i = 0; i < REEL_PATTERN.length ; i++) {
				if(_reelObj[i].rotSpeed != 0 && checkIndex != i) { 
					_resultIndexes[i].push(-1);
				} else {
					_resultIndexes[i].push(REEL_DISP_COUNT - 1 - i);
				}
			}
		}
		
		// チェリーチェック
		if (_reelObj[ReelIndexes.left].rotSpeed == 0 || checkIndex == 0) {
			for(let i = 0; i < REEL_DISP_COUNT; i++) {
				if(REEL_PATTERN[ReelIndexes.left][_reelObj[ReelIndexes.left].nextIndex(i)] == PatternCode.cherry) {
					ret.push(PatternCode.cherry);
					_resultIndexes[ReelIndexes.left].push(i);
				}
			}
		}
		
		return ret;
	};

	// 異色７揃いのチェック
	Exc_SlotMachine.prototype.mix7Check = function() {
		// 全リール停止が前提,異色７揃いがある場合戻り値true
		
		// チェック用変数
		let aligned;

		// リールの回転処理と違って横方向に探査する点に注意
		for(let i = 0; i < REEL_DISP_COUNT; i++) {
			aligned = true;
			for(let j = 0; j < REEL_PATTERN.length; j++) {
				let target = REEL_PATTERN[j][_reelObj[j].nextIndex(i)];
				if(target != PatternCode.blue7 && target != PatternCode.red7) {
					aligned = false;
					break;
				}
			}
			if(aligned) {
				return true;
			}
		}
		// 斜め右上
		aligned = true;
		for(let i = 0; i < REEL_PATTERN.length; i++) {
			let target = REEL_PATTERN[i][_reelObj[i].nextIndex(i)];
			if(target != PatternCode.blue7 && target != PatternCode.red7) {
				aligned = false;
				break;
			}
		}
		if(aligned) {
			return true;
		}
		// 斜め右下
		aligned = true;
		for(let i = 0; i < REEL_PATTERN.length; i++) {
			let target = REEL_PATTERN[i][_reelObj[i].nextIndex(REEL_DISP_COUNT - 1 - i )]
			if(target != PatternCode.blue7 && target != PatternCode.red7) {
				aligned = false;
				break;
			}
		}
		if(aligned) {
			return true;
		}
		
		return false;
	};
	
	
	//--------------------------------------------------
	// 確認ウィンドウ処理
	//--------------------------------------------------
	// 子ウィンドウの作成
	Exc_SlotMachine.prototype.openExitWindow = function() {
		if(this._exitWindow == null) {
			let windowHeight = 160;
			let windowWidth = 350;
			this._exitWindow = new Exc_ExitWindow(new Rectangle(
							(Graphics.width - windowWidth)/2, (Graphics.height - windowHeight)/2, 
								windowWidth, windowHeight));
			this._exitWindow.setHandler("continue", this.continueSlot.bind(this));
			this._exitWindow.setHandler("buyCoin", this.buyCoinSlot.bind(this));
			this._exitWindow.setHandler("exit", this.exitSlot.bind(this));
			this.addChild(this._exitWindow);
		} else {
			this._exitWindow.select(0);
			this._exitWindow.activate();
			this._exitWindow.open();
		}
	};
	
	// コマンドウィンドウで続けるを押したときの処理
	Exc_SlotMachine.prototype.continueSlot = function() {
		_machineDisableCount = INIT_DISABLE_FRAME;
		this._exitWindow.close();
	};

	// コマンドウィンドウでコインを買うを押したときの処理
	Exc_SlotMachine.prototype.buyCoinSlot = function() {
		// 所持金不足チェック
		if($gameParty.gold() < Exc_getPayMoney()) {
			AudioManager.playSe({"name":"buzzer1","volume":90,"pitch":100,"pan":0});
		} else {
			Exc_addCoin(Exc_getBuyCoins());
			$gameParty.loseGold(Exc_getPayMoney());
			this.redrawText(_txtCoin, Exc_getCoin());
			this.redrawText(_txtMoney, $gameParty.gold());
			_valueCoin.display = Exc_getCoin();

			AudioManager.playSe({"name":"Coin","volume":90,"pitch":100,"pan":0});
		}
		_machineDisableCount = INIT_DISABLE_FRAME;
		this._exitWindow.close();
	};

	// コマンドウィンドウで終了を押したときの処理
	Exc_SlotMachine.prototype.exitSlot = function() {
		$gameSystem.replayBgm();
		this.popScene();
	};
	
	//--------------------------------------------------
	// 確認ウィンドウ定義
	//--------------------------------------------------
	// 確認ウィンドウの作成
	function Exc_ExitWindow() {
		this.initialize.apply(this, arguments);
	}
	
	Exc_ExitWindow.prototype = Object.create(Window_Command.prototype);
	Exc_ExitWindow.prototype.constructor = Exc_ExitWindow;

	Exc_ExitWindow.prototype.initialize = function(rect) {
		Window_Command.prototype.initialize.call(this, rect);
	};

	// コマンドを設定
	Exc_ExitWindow.prototype.makeCommandList = function() {
		this.addCommand(_commandContinue, "continue", true);
		this.addCommand(_commandBuyCoin, "buyCoin", true);
		this.addCommand(_commandExit, "exit", true);
	};

})();