//=============================================================================
// HSMZ_CustomStatus.js
// ----------------------------------------------------------------------------
// Copyright (c) 2021 n2naokun(柊菜緒)
// This software is released under the MIT License.
// http://opensource.org/licenses/mit-license.php
// ----------------------------------------------------------------------------
// Version
// 1.0.0 2024/05/02 初版
// 1.0.1 2024/05/10 ステータス画面のpageup/pagedownを無効化
// 1.0.2 2024/12/17 属性耐性の計算方法を変更
// ----------------------------------------------------------------------------
// [Twitter]: https://twitter.com/n2naokun/
// [GitHub] : https://github.com/n2naokun/
//=============================================================================

/*:
 * @plugindesc ステータス画面カスタムプラグイン
 * @target MZ
 * @url 
 * @base FOTA3_MenuCommon
 * @orderAfter FOTA3_MenuCommon
 * @author n2naokun(柊菜緒)
 *
 * @help 説明
 * 
 * 
 * 
 * 利用規約：
 *  作者に無断で改変、再配布が可能で、利用形態（商用、18禁利用等）
 *  についても制限はありません。
 *  このプラグインはもうあなたのものです。
 * 
 * @param TextBaseColor
 * @text 文字色
 * @desc ステータス画面の文字色
 * @default #FFFFFF
 * 
 * @param TextOutlineColor
 * @text 文字外形色
 * @desc ステータス画面の文字外形色
 * @default #006699
 * 
 * @param TextOutlineWidth
 * @text 文字外形幅
 * @desc ステータス画面の文字外形幅
 * @default 8
 * 
 * @param NameTextBlock
 * @text 名前テキスト
 * @desc 名前のテキスト
 * @type struct<BlockParam>
 * @default {"FontSize":"40","LineHeight":"36","TextColor":"#FFFFFF","OutlineColor":"#006699","OutlineWidth":"8","BasePosX":"865","BasePosY":"565"}
 * 
 * @param ClassTextBlock
 * @text ジョブ名テキスト
 * @desc ジョブ名のテキスト
 * @type struct<BlockParam>
 * @default {"FontSize":"28","LineHeight":"36","TextColor":"#FFFFFF","OutlineColor":"#006699","OutlineWidth":"8","BasePosX":"930","BasePosY":"625"}
 * 
 * @param HpMpSpTextBlock
 * @text HP・MP・SPテキスト
 * @desc HP・MP・SPのテキスト
 * @type struct<BlockParam>
 * @default {"FontSize":"24","LineHeight":"36","TextColor":"#FFFFFF","OutlineColor":"#006699","OutlineWidth":"6","BasePosX":"960","BasePosY":"680"}
 * 
 * @param EquipTextBlock
 * @text 装備一覧表示ブロック
 * @desc 装備一覧の表示ブロック
 * @type struct<BlockParam>
 * @default {"FontSize":"28","LineHeight":"44","TextColor":"#FFFFFF","OutlineColor":"#006699","OutlineWidth":"8","BasePosX":"60","BasePosY":"160"}
 * 
 * @param ElementTextBlock
 * @text 属性値一覧表示ブロック
 * @desc 属性値一覧の表示ブロック
 * @type struct<BlockParam>
 * @default {"FontSize":"24","LineHeight":"28","TextColor":"#FFFFFF","OutlineColor":"#006699","OutlineWidth":"8","BasePosX":"590","BasePosY":"170"}
 * 
 * @param ElementMargin
 * @text 属性強化率と耐性率の間のスペース
 * @desc 属性強化率と耐性率の間のスペース
 * @type number
 * @default 120
 * 
 * @param ElementDatas
 * @text 表示する属性IDの定義
 * @desc 表示する属性IDを列挙します
 * @type struct<ElementData>[]
 * @default ["{\"ElementId\":\"3\",\"ElementIconId\":\"65\"}","{\"ElementId\":\"4\",\"ElementIconId\":\"66\"}","{\"ElementId\":\"9\",\"ElementIconId\":\"67\"}","{\"ElementId\":\"6\",\"ElementIconId\":\"68\"}","{\"ElementId\":\"5\",\"ElementIconId\":\"69\"}","{\"ElementId\":\"10\",\"ElementIconId\":\"70\"}","{\"ElementId\":\"7\",\"ElementIconId\":\"71\"}","{\"ElementId\":\"11\",\"ElementIconId\":\"72\"}","{\"ElementId\":\"12\",\"ElementIconId\":\"73\"}","{\"ElementId\":\"1\",\"ElementIconId\":\"75\"}","{\"ElementId\":\"17\",\"ElementIconId\":\"76\"}","{\"ElementId\":\"2\",\"ElementIconId\":\"77\"}","{\"ElementId\":\"15\",\"ElementIconId\":\"63\"}"]
 * 
 * @param AbilityTextBlock
 * @text アビリティ一覧表示ブロック
 * @desc アビリティ一覧の表示ブロック
 * @type struct<BlockParam>
 * @default {"FontSize":"28","LineHeight":"44","TextColor":"#FFFFFF","OutlineColor":"#006699","OutlineWidth":"8","BasePosX":"815","BasePosY":"785"}
 * 
 * @param AbilitySlotNumber
 * @text アビリティスロット番号
 * @desc アビリティスロットとして使用する装備タイプ番号
 * @type number
 * @default 5
 * 
 * @param AbilityValueBlock
 * @text 能力値表示ブロック
 * @desc 能力値の表示ブロック
 * @type struct<BlockParam>
 * @default {"FontSize":"28","LineHeight":"36","TextColor":"#FFFFFF","OutlineColor":"#006699","OutlineWidth":"8","BasePosX":"60","BasePosY":"545"}
 * 
 * @param AbilityValueSpace
 * @text 能力値の項目と数値の間のスペース
 * @desc 能力値の項目と数値の間のスペース
 * @type number
 * @default 150
 * 
 * @param EffectiveRateBlock
 * @text 効果率表示ブロック
 * @desc 効果率の表示ブロック
 * @type struct<BlockParam>
 * @default {"FontSize":"22","LineHeight":"28","TextColor":"#FFFFFF","OutlineColor":"#006699","OutlineWidth":"6","BasePosX":"440","BasePosY":"600"}
 * 
 * @param EffectiveRateSpace
 * @text 効果率の項目と数値の間のスペース
 * @desc 効果率の項目と数値の間のスペース
 * @type number
 * @default 150
 * 
 * @param UseCustomFont
 * @text カスタムフォント指定
 * @desc FOTA3_MenuCommonであらかじめ読み込んだフォントを使用するか。
 * @default false
 * @type boolean
 * 
 * @param CustomFontName
 * @text カスタムフォント名
 * @desc FOTA3_MenuCommonであらかじめ読み込んだフォント名を指定できます。
 * 
 * 
 */
/*~struct~BlockParam:
 * @param FontSize
 * @text フォントサイズ
 * @desc 文字の大きさ
 * @type number
 * @default 28
 * 
 * @param LineHeight
 * @text 行の高さ
 * @desc 1行の高さ
 * @type number
 * @default 36
 * 
 * @param TextColor
 * @text 文字色
 * @desc 文字のベースカラー
 * @default #FFFFFF
 * 
 * @param OutlineColor
 * @text 文字外形色
 * @desc 文字の縁取りの色
 * @default #006699
 * 
 * @param OutlineWidth
 * @text 文字外形幅
 * @desc 文字の縁取りの幅
 * @type number
 * @default 8
 * 
 * @param BasePosX
 * @text 基準X座標
 * @desc 表示の基準X位置
 * @type number
 * @default 0
 * 
 * @param BasePosY
 * @text 基準Y座標
 * @desc 表示の基準Y位置
 * @type number
 * @default 0
 */
/*~struct~ElementData:
 * @param ElementId
 * @text 属性ID
 * @desc 属性ID
 * @type number
 * @default 0
 * 
 * @param ElementIconId
 * @text 属性アイコンID
 * @desc 属性アイコンID
 * @type number
 * @default 0
 */

// ESLint向けグローバル変数宣言
/*global */

"use strict";//厳格なエラーチェック

var Imported = Imported || {};
Imported.HSMZ_CustomStatus = true;
// 他のプラグインとの連携用シンボル

(function (_global) {
   const params = PluginManager.parameters('HSMZ_CustomStatus');

   const ColorCodePattern = /#([0-9a-fA-F]{2})([0-9a-fA-F]{2})([0-9a-fA-F]{2})/;
   // ステータス画面の文字色
   var TextBaseColor
      = params['TextBaseColor'][0] === '#' ?
         params['TextBaseColor'] :
         // #が無ければ自動挿入する
         '#' + params['TextBaseColor'];
   if (!ColorCodePattern.test(TextBaseColor)) {
      TextBaseColor = '#000000';
   }

   // ステータス画面の文字外形色
   var TextOutlineColor
      = params['TextOutlineColor'][0] === '#' ?
         params['TextOutlineColor'] :
         // #が無ければ自動挿入する
         '#' + params['TextOutlineColor'];
   if (!ColorCodePattern.test(TextOutlineColor)) {
      TextOutlineColor = '#006699';
   }

   // ステータス画面の文字外形太さ
   var TextOutlineWidth = Number(params['TextOutlineWidth']);
   if (isNaN(TextOutlineWidth)) {
      TextOutlineWidth = 8;
   }

   // BlockParamのパース
   function BPParse(BlockParam) {
      var param = JSON.parse(BlockParam);
      param.FontSize = Number(param.FontSize);
      param.LineHeight = Number(param.LineHeight);
      param.OutlineWidth = Number(param.OutlineWidth);
      param.BasePosX = Number(param.BasePosX);
      param.BasePosY = Number(param.BasePosY);

      return param;
   }

   function AttDataParse(AttDatas) {
      var param = JSON.parse(AttDatas);
      param = param.map(function (AttData) {
         let tmp = JSON.parse(AttData);
         tmp.ElementId = Number(tmp.ElementId);
         tmp.ElementIconId = Number(tmp.ElementIconId);
         return tmp;
      });
      return param;
   }

   var NameTextBlock = BPParse(params['NameTextBlock']);

   var ClassTextBlock = BPParse(params['ClassTextBlock']);

   //2024/11/11 Start HP・MP・SP表示エリア描画
   var HpMpSpTextBlock = BPParse(params['HpMpSpTextBlock']);
   //2024/11/11 End

   var EquipTextBlock = BPParse(params['EquipTextBlock']);

   var ElementTextBlock = BPParse(params['ElementTextBlock']);

   var ElementMargin = Number(params['ElementMargin']);
   if (isNaN(ElementMargin)) {
      ElementMargin = 0;
   }

   var ElementDatas = AttDataParse(params['ElementDatas']);

   var AbilityTextBlock = BPParse(params['AbilityTextBlock']);

   var AbilitySlotNumber = Number(params['AbilitySlotNumber']);

   var AbilityValueBlock = BPParse(params['AbilityValueBlock']);

   var AbilityValueSpace = Number(params['AbilityValueSpace']);
   if (isNaN(AbilityValueSpace)) {
      AbilityValueSpace = 0;
   }

   var EffectiveRateBlock = BPParse(params['EffectiveRateBlock']);
   var EffectiveRateSpace = Number(params['EffectiveRateSpace']);
   if (isNaN(EffectiveRateSpace)) {
      EffectiveRateSpace = 0;
   }



   // カスタムフォントの使用
   var UseCustomFont = params['UseCustomFont'] === 'true';

   // カスタムフォント名
   var CustomFontName = params['CustomFontName'];

   // カスタムフォントを使用する場合使用するフォントファミリーにカスタムフォント名を追加
   var MainFont = UseCustomFont ? CustomFontName + ', ' : null;
   // $gameSystemが初期化されるのはScene_Boot移行のためScene_Boot内でフォントファミリーを取得
   const _Scene_Boot_startNormalGame = Scene_Boot.prototype.startNormalGame;
   Scene_Boot.prototype.startNormalGame = function () {
      _Scene_Boot_startNormalGame.call(this);
      // ツクールのシステム設定で既にAkazukiPOP.woffが指定されているためシステムから取得
      if (MainFont) {
         MainFont = MainFont += $gameSystem.mainFontFace();
      } else {
         MainFont = $gameSystem.mainFontFace();
      }
   };


   // ステータス画面背景を追加
   const _Scene_Status_createBackground = Scene_Status.prototype.createBackground;
   Scene_Status.prototype.createBackground = function () {
      _Scene_Status_createBackground.call(this);

      this._statusBGSprite = new Sprite();
      this._statusBGSprite.bitmap = ImageManager.loadBitmap("img/menuC/", "status_bg");
      this.addChild(this._statusBGSprite);
   };

   const _Scene_Status_createWindowLayer = Scene_Status.prototype.createWindowLayer;
   Scene_Status.prototype.createWindowLayer = function () {
      this._windowLayer = new WindowLayer();
      this._windowLayer.x = 0;
      this._windowLayer.y = 0;
      this.addChild(this._windowLayer);
   };

   const _Scene_Status_create = Scene_Status.prototype.create;
   Scene_Status.prototype.create = function () {
      Scene_MenuBase.prototype.create.call(this);
      this.createStatusWindow();
   };

   Scene_Status.prototype.createStatusWindow = function () {
      const rect = this.statusWindowRect();
      this._statusWindow = new Window_CustomStatus(rect);
      this._statusWindow.setHandler("cancel", this.popScene.bind(this));
      // 2024/05/10 ステータス画面のpageup/pagedownを無効化
      // this._statusWindow.setHandler("pagedown", this.nextActor.bind(this));
      // this._statusWindow.setHandler("pageup", this.previousActor.bind(this));
      // 2024/05/10 ここまで
      this.addWindow(this._statusWindow);
   };

   const _Scene_Status_statusWindowRect = Scene_Status.prototype.statusWindowRect;
   Scene_Status.prototype.statusWindowRect = function () {
      const wx = 0;
      const wy = 0;
      const ww = Graphics.width;
      const wh = Graphics.height;
      return new Rectangle(wx, wy, ww, wh);
   };

   const _Scene_Status_refreshActor = Scene_Status.prototype.refreshActor;
   Scene_Status.prototype.refreshActor = function () {
      const actor = this.actor();
      this._statusWindow.setActor(actor);
   };

   function Window_CustomStatus() {
      this.initialize(...arguments);
   }

   Window_CustomStatus.prototype = Object.create(Window_Status.prototype);
   Window_CustomStatus.prototype.constructor = Window_CustomStatus;
   Window_CustomStatus.prototype.initialize = function (rect) {
      Window_Status.prototype.initialize.apply(this, arguments);
      this.frameVisible = false;
      this.padding = 0;
      this.margin = 0;
      this.changeLineHeight(36);
      this.createContents();
      this.refresh();
   };

   Window_CustomStatus.prototype.changeLineHeight = function (lineHeight) {
      this._lineHeight = lineHeight || 36;
   };

   Window_CustomStatus.prototype.lineHeight = function () {
      return this._lineHeight || 36;
   };

   Window_CustomStatus.prototype.changeFontSize = function (fontSize) {
      this.contents.fontSize = fontSize;
   };

   Window_CustomStatus.prototype.resetFontSettings = function () {
      this.contents.fontFace = MainFont;
      this.contents.fontSize = $gameSystem.mainFontSize();
      this.changeOutlineWidth(TextOutlineWidth);
      this.resetTextColor();
   };

   Window_CustomStatus.prototype.resetTextColor = function () {
      this.changeTextColor(TextBaseColor);
      this.changeOutlineColor(TextOutlineColor);
   };

   Window_CustomStatus.prototype.changeOutlineWidth = function (width) {
      this.contents.outlineWidth = width;
   };

   Window_CustomStatus.prototype.updateBackOpacity = function () {
      this.backOpacity = 0;
   };

   Window_CustomStatus.prototype.refresh = function () {
      Window_StatusBase.prototype.refresh.call(this);
      if (this._actor) {
         this.drawNameBlock();
         this.drawClassBlock();
         //2024/11/11 Start HP・MP・SP表示エリア描画
         this.drawHpMpSpBlock();
         //2024/11/11 End
         this.drawEquipBlock();
         this.drawElementBlock();
         this.drawAbilityListBlock();
         this.drawAbilityValueBlock();
         this.drawEffectiveRateBlock();
      }
   };

   Window_CustomStatus.prototype.drawNameBlock = function () {
      const info = NameTextBlock;
      const fontSize = info.FontSize;
      this.changeFontSize(fontSize);
      const lineHeight = info.LineHeight;
      const textColor = info.TextColor;
      this.changeTextColor(textColor);
      const outlineWidth = info.OutlineWidth;
      this.changeOutlineWidth(outlineWidth);
      const outlineColor = info.OutlineColor;
      this.changeOutlineColor(outlineColor);
      const x = info.BasePosX;
      const y = info.BasePosY;
      const actor = this._actor;


      this.changeLineHeight(lineHeight);
      this.drawActorName(actor, x, y, this.textWidth(actor.name()));
      this.changeLineHeight();

      this.resetFontSettings();
   };

   Window_CustomStatus.prototype.drawActorName = function (actor, x, y, width) {
      width = width || 168;
      this.drawText(actor.name(), x, y, width);
   };

   Window_CustomStatus.prototype.drawClassBlock = function () {
      const info = ClassTextBlock;
      const fontSize = info.FontSize;
      this.changeFontSize(fontSize);
      const lineHeight = info.LineHeight;
      const textColor = info.TextColor;
      this.changeTextColor(textColor);
      const outlineWidth = info.OutlineWidth;
      this.changeOutlineWidth(outlineWidth);
      const outlineColor = info.OutlineColor;
      this.changeOutlineColor(outlineColor);
      const x = info.BasePosX;
      const y = info.BasePosY;
      const actor = this._actor;
      const className = actor.currentClass().name;

      this.changeLineHeight(lineHeight);
      this.drawActorClass(actor, x, y, this.textWidth(className));
      //2024/11/11 Start クラス名とレベルの間隔調整
      //this.drawActorLevel(actor, x + this.textWidth(className + '  '), y);
      this.drawActorLevel(actor, x + this.textWidth(className + '     '), y);
      //2024/11/11 End
      this.changeLineHeight();

      this.resetFontSettings();
   };

   Window_CustomStatus.prototype.drawActorClass = function (actor, x, y, width) {
      width = width || 168;
      this.drawText(actor.currentClass().name, x, y, width);
   };

   Window_CustomStatus.prototype.drawActorLevel = function (actor, x, y) {
      const lvText = TextManager.levelA;
      const tw = this.textWidth(lvText);
      const level = actor.level;
      this.drawText(lvText, x, y, tw);
      this.drawText(level, x + tw, y, this.textWidth(('000' + level).slice(-3)), "right");
   };

   //2024/11/11 Start HP・MP・SP表示エリア
   Window_CustomStatus.prototype.drawHpMpSpBlock = function () {
      const info = HpMpSpTextBlock;
      const fontSize = info.FontSize;
      this.changeFontSize(fontSize);
      const lineHeight = info.LineHeight;
      const textColor = info.TextColor;
      this.changeTextColor(textColor);
      const outlineWidth = info.OutlineWidth;
      this.changeOutlineWidth(outlineWidth);
      const outlineColor = info.OutlineColor;
      this.changeOutlineColor(outlineColor);
      const x = info.BasePosX;
      const y = info.BasePosY;

      this.changeLineHeight(lineHeight);
      this.drawHpClass(this._actor, x, y);
      this.changeLineHeight();

      this.resetFontSettings();
   };
   
   Window_CustomStatus.prototype.drawHpClass = function (actor, x, y) {
      let headWidth = this.textWidth(TextManager.mpA);
      let valueWidth = this.textWidth("999");
      for (let i = 0; i <= 2; i++) {
         this.drawText(i == 0 ? TextManager.hpA : i == 1 ? TextManager.mpA : TextManager.tpA, x, y, headWidth);
         x += headWidth + 3;
         this.drawText(i == 0 ? actor.mhp : i == 1 ? actor.mmp : actor.maxTp(), x, y, valueWidth, "right");
         x += valueWidth + 20;
      }
   };
   //2024/11/11 End

   Window_CustomStatus.prototype.drawEquipBlock = function () {
      const info = EquipTextBlock;
      const fontSize = info.FontSize;
      this.changeFontSize(fontSize);
      const lineHeight = info.LineHeight;
      const textColor = info.TextColor;
      this.changeTextColor(textColor);
      const outlineWidth = info.OutlineWidth;
      this.changeOutlineWidth(outlineWidth);
      const outlineColor = info.OutlineColor;
      this.changeOutlineColor(outlineColor);
      const x = info.BasePosX;
      const y = info.BasePosY;
      const actor = this._actor;

      this.changeLineHeight(lineHeight);
      this.drawEquipItems(actor, x, y);
      this.changeLineHeight();

      this.resetFontSettings();
   };

   Window_CustomStatus.prototype.drawEquipItems = function (actor, x, y) {
      if (actor) {
         const equips = actor.equips();
         let lineCount = 0;
         const abilitySlotId = AbilitySlotNumber;
         // var maxCount = weaponSlots ? weaponSlots.length : actor.equipSlots().length;
         for (let index = 0; index < equips.length; index++) {
            const equip = equips[index];
            const slots = actor.equipSlots();
            if (equip && slots[index] !== AbilitySlotNumber) {
               const iconSpace = ImageManager.iconWidth + 4;
               this.drawItemName(equip, x, y + this.lineHeight() * lineCount, iconSpace + this.textWidth(equip.name));
               lineCount++;
            }
         }
      }
   };

   Window_CustomStatus.prototype.drawItemName = function (item, x, y, width) {
      if (item) {
         const iconY = y + (this.lineHeight() - ImageManager.iconHeight) / 2;
         const textMargin = ImageManager.iconWidth + 4;
         const itemWidth = Math.max(0, width - textMargin);
         this.drawIcon(item.iconIndex, x, iconY);
         this.drawText(item.name, x + textMargin, y, itemWidth);
      }
   };

   Window_CustomStatus.prototype.drawElementBlock = function () {
      const info = ElementTextBlock;
      const fontSize = info.FontSize;
      this.changeFontSize(fontSize);
      const lineHeight = info.LineHeight;
      const textColor = info.TextColor;
      this.changeTextColor(textColor);
      const outlineWidth = info.OutlineWidth;
      this.changeOutlineWidth(outlineWidth);
      const outlineColor = info.OutlineColor;
      this.changeOutlineColor(outlineColor);
      const x = info.BasePosX;
      const y = info.BasePosY;
      const actor = this._actor;

      this.changeLineHeight(lineHeight);
      this.drawElementRates(actor, x, y);
      this.changeLineHeight();

      this.resetFontSettings();
   };

   let elements = {};
   let boost = [];
   Window_CustomStatus.prototype.drawElementRates = function (actor, x, y) {
      const elmCount = ElementDatas.length;
      const elmDatas = ElementDatas;
      const offset = this.textWidth('100%');
      var margin = ElementMargin;
      const lineHeight = this.lineHeight();

      $dataSystem.elements.forEach(function (element, cnt, obj) {
         if (element) {
            elements[element] = cnt;
            boost[cnt] = 100;
         }
      }.bind(this));
      const equips = this._actor.equips();
      equips.forEach(function (equip, cnt, equips) {
         if (!equip || !equip.note) return;
         let params = equip.note.split("\n");
         params.forEach(function (elem) {
            if (elem.match(/^<(.+)の知識:([+-]?\d+)[%％]>/)) {
               let id = -1;
               if (elements[RegExp.$1]) {
                  id = elements[RegExp.$1];
               } else {
                  id = -1;
               }
               if (id > 0) {
                  boost[id] += Number(RegExp.$2);
               }
            }
         }.bind(this));
      }.bind(this));

      // ElementDatas
      // ElementId,ElementIconId
      // ElementMargin
      for (let i = 0; i < elmCount; i++) {

         this.drawElementIcons(elmDatas[i].ElementIconId, x, y + ((i * lineHeight)));
         this.drawElementBoosts(actor, elmDatas[i].ElementId, x - ((margin / 2) + offset), y + ((i * lineHeight)));
         this.drawElementResists(actor, elmDatas[i].ElementId, x + (margin / 2), y + ((i * lineHeight)));
      }
   };

   Window_CustomStatus.prototype.drawElementBoosts = function (actor, elmId, x, y) {
      const width = this.textWidth('100%');
      const elmRate = boost[elmId];
      this.drawText(elmRate.toFixed(0) + "%", x, y, width, "right");
   };

   Window_CustomStatus.prototype.drawElementResists = function (actor, elmId, x, y) {
      const width = this.textWidth('100%');
      // 2024/12/17 属性耐性の計算方法を変更
      const elmResistRate = Math.round(actor.elementRate(elmId) * 100);
      // 2024/12/17 ここまで
      this.drawText(elmResistRate.toFixed(0) + "%", x, y, width, "right");
   };

   Window_CustomStatus.prototype.drawElementIcons = function (iconId, x, y) {
      const fontSize = $gameSystem.mainFontSize();
      const sizeRate = this.contents.fontSize / fontSize;
      this.customDrawIcon(iconId, x, y, sizeRate);
   };

   Window_CustomStatus.prototype.customDrawIcon = function (iconIndex, x, y, sizeRate) {
      const bitmap = ImageManager.loadSystem("IconSet");
      const pw = ImageManager.iconWidth;
      const ph = ImageManager.iconHeight;
      const sx = (iconIndex % 16) * pw;
      const sy = Math.floor(iconIndex / 16) * ph;
      const dw = pw * sizeRate;
      const dh = ph * sizeRate;
      this.contents.blt(bitmap, sx, sy, pw, ph, x - dw / 2, y, dw, dh);
   };

   Window_CustomStatus.prototype.drawAbilityListBlock = function () {
      const info = AbilityTextBlock;
      const fontSize = info.FontSize;
      this.changeFontSize(fontSize);
      const lineHeight = info.LineHeight;
      const textColor = info.TextColor;
      this.changeTextColor(textColor);
      const outlineWidth = info.OutlineWidth;
      this.changeOutlineWidth(outlineWidth);
      const outlineColor = info.OutlineColor;
      this.changeOutlineColor(outlineColor);
      const x = info.BasePosX;
      const y = info.BasePosY;
      const actor = this._actor;

      this.changeLineHeight(lineHeight);
      this.drawAbilityList(actor, x, y);
      this.changeLineHeight();

      this.resetFontSettings();
   };

   Window_CustomStatus.prototype.drawAbilityList = function (actor, x, y) {
      if (actor) {
         const equips = actor.equips();
         const slots = actor.equipSlots();
         let lineCount = 0;
         for (let index = 0; index < slots.length; index++) {
            const equip = equips[index];
            if (equip && slots[index] === AbilitySlotNumber) {
               const iconSpace = ImageManager.iconWidth + 4;
               this.drawAbilityItem(equip, x, y + this.lineHeight() * lineCount, iconSpace + this.textWidth(equip.name));
               lineCount++;
            }
         }
      }
   };

   Window_CustomStatus.prototype.drawAbilityItem = function (ability, x, y, width) {
      if (ability) {
         const iconY = y + (this.lineHeight() - ImageManager.iconHeight) / 2;
         const textMargin = ImageManager.iconWidth + 4;
         const itemWidth = Math.max(0, width - textMargin);
         this.drawIcon(ability.iconIndex, x, iconY);
         this.drawText(ability.name, x + textMargin, y, itemWidth);
      }
   };

   Window_CustomStatus.prototype.drawAbilityValueBlock = function () {
      const info = AbilityValueBlock;
      const fontSize = info.FontSize;
      this.changeFontSize(fontSize);
      const lineHeight = info.LineHeight;
      const textColor = info.TextColor;
      this.changeTextColor(textColor);
      const outlineWidth = info.OutlineWidth;
      this.changeOutlineWidth(outlineWidth);
      const outlineColor = info.OutlineColor;
      this.changeOutlineColor(outlineColor);
      const x = info.BasePosX;
      const y = info.BasePosY;
      const actor = this._actor;

      this.changeLineHeight(lineHeight);
      this.drawAbilityValueList(actor, x, y);
      this.changeLineHeight();

      this.resetFontSettings();
   };

   const xParamNames = ['命中', '回避', '会心', null, '魔法回避', null, null, 'HP再生', 'MP再生', 'SP再生'];
   Window_CustomStatus.prototype.drawAbilityValueList = function (actor, x, y) {
      const lineHeight = this.lineHeight();
      var abilityValueSpace = AbilityValueSpace;
      var y2 = y;
      for (var i = 0; i < 6; i++) {
         var paramId = i + 2;
         this.drawText(TextManager.param(paramId), x, y2, abilityValueSpace);
         this.drawText(actor.param(paramId), x + abilityValueSpace, y2, abilityValueSpace, 'right');
         y2 += lineHeight;
      }
      for (let i = 0; i < 6; i++) {
         let paramId = i;
         if (!xParamNames[paramId]) continue;
         this.drawText(xParamNames[paramId], x, y2, abilityValueSpace);
         this.drawText((actor.xparam(paramId) * 100).toFixed(0), x + abilityValueSpace, y2, abilityValueSpace, 'right');
         y2 += lineHeight;
      }
   };

   Window_CustomStatus.prototype.drawEffectiveRateBlock = function () {
      const info = EffectiveRateBlock;
      const fontSize = info.FontSize;
      this.changeFontSize(fontSize);
      const lineHeight = info.LineHeight;
      const textColor = info.TextColor;
      this.changeTextColor(textColor);
      const outlineWidth = info.OutlineWidth;
      this.changeOutlineWidth(outlineWidth);
      const outlineColor = info.OutlineColor;
      this.changeOutlineColor(outlineColor);
      const x = info.BasePosX;
      const y = info.BasePosY;
      const actor = this._actor;

      this.changeLineHeight(lineHeight);
      this.drawEffectiveRateList(actor, x, y);
      this.changeLineHeight();

      this.resetFontSettings();
   };

   const sParamNames = [null, '防御効果', '回復効果', null,
      'MP消費', 'SP増加', '物理ダメージ', '魔法ダメージ', '床ダメージ', '経験値獲得'];
   Window_CustomStatus.prototype.drawEffectiveRateList = function (actor, x, y) {
      const lineHeight = this.lineHeight();
      var effectiveRateSpace = EffectiveRateSpace;
      var y2 = y;
      for (var i = 7; i < 10; i++) {
         var paramId = i;
         if (!xParamNames[paramId]) continue;
         this.drawText(xParamNames[paramId], x, y2, effectiveRateSpace);
         this.drawText((this._actor.xparam(paramId) * 100).toFixed(0) + "%", x + effectiveRateSpace, y2, effectiveRateSpace, 'right');
         y2 += lineHeight;
      }
      for (let i = 0; i < 10; i++) {
         let paramId = i;
         if (!sParamNames[paramId]) continue;
         this.drawText(sParamNames[paramId], x, y2, effectiveRateSpace);
         this.drawText((this._actor.sparam(paramId) * 100).toFixed(0) + "%", x + effectiveRateSpace, y2, effectiveRateSpace, 'right');
         y2 += lineHeight;
      }

   };

})(this);
