//=============================================================================
// HSMZ_SceneMovieTransition.js
// ----------------------------------------------------------------------------
// Copyright (c) 2021 n2naokun(柊菜緒)
// This software is released under the MIT License.
// http://opensource.org/licenses/mit-license.php
// ----------------------------------------------------------------------------
// Version
// 1.4.3 2024/11/28 トランジションが終わる前にシーン移動があると直前の
//                  トランジションが再び再生されるバグを修正
// 1.4.2 2024/11/16 外部プラグイン用に再生APIを公開
// 1.4.1 2024/10/26 マップとショップの変数指定が逆だったのを修正
// 1.4.0 2024/10/26 マップ・ショップ間でトランジションを指定できるように変更
// 1.3.1 2024/08/09 プラグインコマンドからでも確率再生を使用できるように変更
// 1.3.0 2024/08/09 敗北時に専用トランジションを再生する機能を追加
//                  確率で複数のトランジションを切り替えできるように変更
// 1.2.2 2024/07/17 マップ復帰時にフェードが表示されるのを修正
// 1.2.1 2024/06/18 ウェイトが正常に機能しない問題を修正
// 1.2.0 2024/06/11 バトル突入/終了時のエフェクト/フェード削除機能を追加
// 1.1.0 2024/05/22 各種トランジション再生トリガーを実装&ウェイト機能を改善
// 1.0.0 2024/05/20 最低限の機能を実装
// 0.1.0 2024/05/18 プレビュー版
// ----------------------------------------------------------------------------
// [Twitter]: https://twitter.com/n2naokun/
// [GitHub] : https://github.com/n2naokun/
//=============================================================================

/*:
 * @plugindesc 動画トランジションプラグイン
 * @target MV MZ
 * @url 
 * @author n2naokun(柊菜緒)
 *
 * @help 説明
 * プラグインコマンド
 * 動画選択
 * transitionset 動画ファイル名 ウェイト時間(フレーム)
 * 例: transitionset star.webm 20
 * 
 * トランジション再生
 * transitionstart
 * 
 * MZの場合はエディタに従ってください。
 * 
 * 
 * 利用規約：
 *  作者に無断で改変、再配布が可能で、利用形態（商用、18禁利用等）
 *  についても制限はありません。
 *  このプラグインはもうあなたのものです。
 * 
 * 
 * @param moviesDir
 * @text 動画フォルダ
 * @desc トランジションに使用する動画のフォルダ名
 * @type text
 * @default movies
 * 
 * @param transitionData
 * @text トランジションデータ
 * @desc 動画を拡張子付きで登録しておきます
 * この動画は起動時にすべてメモリー上に読み込まれます
 * @type text[]
 * @default []
 * 
 * @param probDataList
 * @text 確率分岐データリスト
 * @desc トランジションを確率で分岐させるためのデータセット
 * @type struct<probData>[][]
 * @default []
 * 
 * @param MapToMenu
 * @text マップからメニュー
 * @desc マップからメニューに移る時のトランジション指定用変数
 * @type number
 * @default 0
 * 
 * @param MenuToMap
 * @text メニューからマップ
 * @desc メニューからマップに移る時のトランジション指定用変数
 * @type number
 * @default 0
 * 
 * @param MapToBattle
 * @text メニューからバトル
 * @desc メニューからバトルに移る時のトランジション指定用変数
 * @type number
 * @default 0
 * 
 * @param BattleToMap
 * @text バトルからメニュー
 * @desc バトルからメニューに移る時のトランジション指定用変数
 * @type number
 * @default 0
 * 
 * @param MapToShop
 * @text メニューからショップ
 * @desc メニューからショップに移る時のトランジション指定用変数
 * @type number
 * @default 0
 * 
 * @param ShopToMap
 * @text ショップからメニュー
 * @desc ショップからメニューに移る時のトランジション指定用変数
 * @type number
 * @default 0
 * 
 * @param BadEnd
 * @text バッドエンド
 * @desc バッドエンド時のトランジション指定用変数
 * @type number
 * @default 0
 * 
 * @param DisableEncounterEffect
 * @text エンカウントエフェクトを無効化
 * @desc エンカウントエフェクトを完全に無効化します
 * @type boolean
 * @default false
 * 
 * @param ControlEncounterEffect
 * @text エンカウントエフェクト制御
 * @desc エンカウントエフェクトをスイッチで有効/無効化します
 * @type boolean
 * @default false
 * 
 * @param EffectControlSwitch
 * @text コントロール用スイッチ
 * @desc エンカウントエフェクト制御用のスイッチ
 * @type number
 * @default 0
 * 
 * @param DisableBattleEndFade
 * @text バトル終了フェードを無効化
 * @desc バトル終了フェードを完全に無効化します
 * @type boolean
 * @default false
 * 
 * @param ControlBattleEndFade
 * @text バトル終了フェード制御
 * @desc バトル終了フェードをスイッチで有効/無効化します
 * @type boolean
 * @default false
 * 
 * @param FadeControlSwitch
 * @text コントロール用スイッチ
 * @desc バトル終了フェード制御用のスイッチ
 * @type number
 * @default 0
 * 
 * 
 * 
 * @command transitionset
 * @text 動画選択
 * @desc 再生する動画を準備します
 * 
 * @arg fileName
 * @text ファイル名
 * @desc 再生する動画ファイル名
 * 
 * @arg wait
 * @text 待ち時間
 * @desc 再生終了を待つ待ち時間をフレーム単位で設定します
 * @type number
 * @default 0
 * 
 * @command probset
 * @text 確率動画選択
 * @desc 確率で再生する動画を準備します
 * 指定したSEは再生されます
 * 
 * @arg probDataIndex
 * @text 確率データ番号
 * @desc 確率で再生するデータを指定します
 * @type number
 * @default 0
 * 
 * @command transitionstart
 * @text トランジション再生
 * @desc トランジションを開始します
 * 
 */
/*~struct~transitionSetting:
 * @param name
 * @text ファイル名
 * @desc 再生するファイル名
 * 
 * @param wait
 * @text ウェイト(フレーム)
 * @desc シーンチェンジ前のウェイト
 * @type number
 * @default 0
 * 
 * @param se
 * @text SE
 * @desc トランジション時に再生するSE
 * @type struct<seData>
 * 
 */
/*~struct~seData:
 * @param name
 * @text SE
 * @desc トランジション時に再生するSE
 * @type file
 * @dir audio/se
 * 
 * @param volume
 * @text 音量
 * @desc SE音量
 * @type number
 * @min 0
 * @max 100
 * @default 100
 * 
 * @param pitch
 * @text ピッチ
 * @desc SEピッチ
 * @type number
 * @min 50
 * @max 150
 * @default 100
 * 
 * @param pan
 * @text パン
 * @desc SE位相
 * @type number
 * @min -100
 * @max 100
 * @default 0
 */
/*~struct~probData:
 * @param prob
 * @text 確率
 * @desc トランジション分岐の確率(比率)
 * @type number
 * @default 1
 * 
 * @param trData
 * @text トランジションデータ
 * @desc 再生するトランジションとSE
 * @type struct<transitionSetting>
 * @default {"name":"","wait":"0","se":"{\"name\":\"\",\"volume\":\"100\",\"pitch\":\"100\",\"pan\":\"0\"}"}
 */

// ESLint向けグローバル変数宣言
/*global */

"use strict";//厳格なエラーチェック

var Imported = Imported || {};
// 他のプラグインとの連携用シンボル

var transition;
(function (_global) {
   const PluginName = 'HSMZ_SceneMovieTransition';
   Imported[PluginName] = true;
   const pname = PluginName;

   const params = PluginManager.parameters(PluginName);
   const movieList = JSON.parse(params['transitionData']);
   const movieDir = params['moviesDir'] + '/';

   const MapToMenu = Number(params['MapToMenu']);
   const MenuToMap = Number(params['MenuToMap']);
   const MapToBattle = Number(params['MapToBattle']);
   const MapToShop = Number(params['MapToShop']);
   const ShopToMap = Number(params['ShopToMap']);
   const BattleToMap = Number(params['BattleToMap']);
   const BadEnd = Number(params['BadEnd']);

   const probDataList = pDataParse(params['probDataList']);

   const DisableEncounterEffect = params['DisableEncounterEffect'] === 'true' ? true : false;
   const ControlEncounterEffect = params['ControlEncounterEffect'] === 'true' ? true : false;
   const EffectControlSwitch = Number(params['EffectControlSwitch'] || 0);

   const DisableBattleEndFade = params['DisableBattleEndFade'] === 'true' ? true : false;
   const ControlBattleEndFade = params['ControlBattleEndFade'] === 'true' ? true : false;
   const FadeControlSwitch = Number(params['FadeControlSwitch'] || 0);



   function tDataParse(struct) {
      let dat = { name: "", wait: 0 };
      if (struct) {
         try {
            struct = JSON.parse(struct);
            dat.name = struct.name;
            dat.wait = Number(struct.wait);
            let se = JSON.parse(struct.se);
            se.volume = Number(se.volume);
            se.pitch = Number(se.pitch);
            se.pan = Number(se.pan);
            dat.se = se;
         } catch (e) {
            console.error(e);
         }
      }
      return dat;
   }

   function pDataParse(struct) {
      let dat = [];
      if (struct) {
         try {
            let tmp = JSON.parse(struct);
            dat = tmp.map(function (probDatas) {
               probDatas = JSON.parse(probDatas);
               return probDatas.map(function (probData) {
                  try {
                     probData = JSON.parse(probData);
                     let prob = Number(probData.prob);
                     let trData = tDataParse(probData.trData);
                     return { prob: prob, trData: trData };
                  } catch (error) {
                     console.error(error);
                  }
               });
            });
         } catch (error) {
            console.error(error);
         }
      }
      return dat;
   }

   function getProbData(index) {
      return probDataList[index];
   }

   function pickTrData(probDatas) {
      let max = 0;
      probDatas.forEach(function (probData) {
         max += probData.prob;
      });
      let prob = max * Math.random();
      let cnt = 0;
      let maxCnt = probDatas.length;
      let threshold = 0;
      let result = null;
      while (cnt < maxCnt) {
         threshold += probDatas[cnt].prob;
         if (prob <= threshold) {
            result = probDatas[cnt].trData;
            break;
         }
         cnt++;
      }
      return result;
   }

   var gameCanvas;
   var transitions = {};
   var videoDiv;
   var playing = null;
   var playingSet = null;
   var wait = 0;
   var standby = false;
   var manualMode = false;
   var battleLosed = false;

   const Scene_Boot_initialize = Scene_Boot.prototype.initialize;
   Scene_Boot.prototype.initialize = function () {
      Scene_Boot_initialize.call(this);
      gameCanvas = document.getElementById('gameCanvas');

      // HTMLに動画再生用のdivを挿入
      const div = document.createElement('div');
      videoDiv = div;
      div.id = 'videoDiv';
      div.style.zIndex = '2';
      div.style.position = 'absolute';
      div.style.margin = 'auto';
      div.style.top = '0';
      div.style.left = '0';
      div.style.right = '0px';
      div.style.bottom = '0px';
      div.hidden = true;
      Graphics._updateVideoDiv();
      gameCanvas.parentNode.insertBefore(videoDiv, gameCanvas);

      // トランジション用の動画を読み込み
      movieList.forEach(name => {
         var video = document.createElement('video');
         video.id = 'videoElement';
         video.width = gameCanvas.width;
         video.height = gameCanvas.height;
         video.src = movieDir + name;
         video.textContent = 'Your browser does not support the video tag.';
         video.addEventListener('ended', function () {
            videoDiv.hidden = true;
            standby = false;
            manualMode = false;
         });
         transitions[name] = video;
      });

   };

   function Utility() { }
   transition = Utility;
   Utility.play = function () {
      if (playing && playing.play) {
         videoDiv.hidden = false;
         SceneManager._scene.applyTransWait();
         if (playing.currentTime > 0)
            playing.currentTime = 0;
         playing.play();
      }
   };
   const pattern = /.+\/(.+\/.+\..+)/;
   Utility.set = function (fileName, waitTime) {
      // console.log('set wait:' + waitTime);
      var videoElement = document.getElementById('videoElement');
      if (videoElement) {
         const src = videoElement.src.match(pattern);
         // videoオブジェクトが存在していてなおかつ同じ動画の場合はdivから消さない
         if (src && src[1] && src[1] === movieDir + fileName) {
            standby = true;
            if (transitions[fileName]) {
               standby = true;
               playing = transitions[fileName];
               wait = Number(waitTime);
            } else {
               console.error(fileName + 'は読み込まれていません');
            }
            return;
         }
         videoDiv.removeChild(videoElement);
      }
      if (transitions[fileName]) {
         standby = true;
         videoDiv.appendChild(transitions[fileName]);
         playing = transitions[fileName];
         wait = Number(waitTime);
      } else {
         console.error(fileName + 'は読み込まれていません');
      }
   };
   Utility.initPlay = function (index) {
      var trData = null;
      let se = { name: '', volume: 100, pitch: 100, pan: 0 };
      if (index > 0) {
         let probDatas = getProbData(index - 1);
         if (probDatas) trData = pickTrData(probDatas);
      }

      if (trData) {
         Utility.set(trData.name, trData.wait);
         se = trData.se;
      }

      if (standby) {
         if (se.name != '') AudioManager.playSe(se);
         standby = false;
         Utility.play();
      }
   };

   var Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
   Game_Interpreter.prototype.pluginCommand = function (command, args) {
      Game_Interpreter_pluginCommand.call(this, command, args);
      command = command.toLowerCase();
      if (command === 'transitionstart') {
         const trData = pickTrData(playingSet);
         const name = trData.name;
         const wait = trData.wait;
         const se = trData.se;
         Utility.set(name, wait);
         standby = false;
         manualMode = true;
         if (se.name != '') AudioManager.playSe(se);
         Utility.play();
      }
      if (command === 'transitionset') {
         // Utility.set(args[0], args[1]);
         playingSet = [{
            prob: 1,
            trData: {
               name: args[0],
               wait: args[1],
               se: { name: '', volume: 100, pitch: 100, pan: 0 }
            }
         }];
         // standby = false;
         // manualMode = true;
      }
      if (command === 'probset') {
         const index = Number(args[0]);
         if (index > 0) {
            playingSet = getProbData(index);
            // standby = false;
            // manualMode = true;
         }
      }
   };
   if (Utils.RPGMAKER_NAME === 'MZ') {
      PluginManager.registerCommand(pname, "transitionstart", args => {
         const trData = pickTrData(playingSet);
         const name = trData.name;
         const wait = trData.wait;
         const se = trData.se;
         Utility.set(name, wait);
         standby = false;
         manualMode = true;
         if (se.name != '') AudioManager.playSe(se);
         Utility.play();
      });
      PluginManager.registerCommand(pname, "transitionset", args => {
         // Utility.set(args['fileName'], args['wait']);
         playingSet = [{
            prob: 1,
            trData: {
               name: args['fileName'],
               wait: args['wait'],
               se: { name: '', volume: 100, pitch: 100, pan: 0 }
            }
         }];
         // standby = false;
         // manualMode = true;
      });
      PluginManager.registerCommand(pname, 'probset', args => {
         const index = Number(args['probDataIndex']);
         if (index > 0) {
            playingSet = getProbData(index);
            // standby = false;
            // manualMode = true;
         }
      });
   }

   const Scene_Base_initialize = Scene_Base.prototype.initialize;
   Scene_Base.prototype.initialize = function () {
      Scene_Base_initialize.call(this);
      this._transWait = 0;
   };

   Scene_Base.prototype.updateTransWait = function () {
      if (this._transWait > 0) {
         this._transWait--;
         return true;
      }
      return false;
   };

   const Scene_Base_isBusy = Scene_Base.prototype.isBusy;
   Scene_Base.prototype.isBusy = function () {
      return Scene_Base_isBusy.call(this) || this._transWait > 0;
   };

   const Scene_Base_update = Scene_Base.prototype.update;
   Scene_Base.prototype.update = function () {
      Scene_Base_update.call(this);
      this.updateTransWait();
   };

   const Scene_Base_stop = Scene_Base.prototype.stop;
   Scene_Base.prototype.stop = function () {
      Scene_Base_stop.call(this);
      const sm = SceneManager;
      const sceneName = sm._scene.constructor.name;
      const nextSceneName = sm._nextScene.constructor.name;
      // let se = { name: '', volume: 100, pitch: 100, pan: 0 };
      // let trData = null;
      let dataSetIndex = 0;

      if (sceneName === 'Scene_Map' && nextSceneName === 'Scene_Menu') {
         dataSetIndex = $gameVariables.value(MapToMenu);

      } else if (sceneName === 'Scene_Menu' && nextSceneName === 'Scene_Map') {
         dataSetIndex = $gameVariables.value(MenuToMap);

      } else if (sceneName === 'Scene_Map' && nextSceneName === 'Scene_Battle') {
         dataSetIndex = $gameVariables.value(MapToBattle);

      } else if (sceneName === 'Scene_Battle' && nextSceneName === 'Scene_Map') {
         if (!battleLosed) {
            dataSetIndex = $gameVariables.value(BattleToMap);
         } else {
            dataSetIndex = $gameVariables.value(BadEnd);
         }

      } else if (sceneName === 'Scene_Battle' && nextSceneName === 'Scene_Gameover') {

      } else if (sceneName === 'Scene_Map' && nextSceneName === 'Scene_Gameover') {

      } else if (sceneName === 'Scene_Map' && nextSceneName === 'Scene_Shop') {
         dataSetIndex = $gameVariables.value(MapToShop);

      } else if (sceneName === 'Scene_Shop' && nextSceneName === 'Scene_Map') {
         dataSetIndex = $gameVariables.value(ShopToMap);

      }

      Utility.initPlay(dataSetIndex);
      // if (dataSetIndex > 0) {
      //    let probDatas = getProbData(dataSetIndex - 1);
      //    if (probDatas) trData = pickTrData(probDatas);
      // }

      // if (trData) {
      //    Utility.set(trData.name, trData.wait);
      //    se = trData.se;
      // }

      // if (standby) {
      //    if (se.name != '') AudioManager.playSe(se);
      //    Utility.play();
      // }
   };

   Scene_Base.prototype.applyTransWait = function () {
      this._transWait = wait;
   };

   const Scene_Base_isActive = Scene_Base.prototype.isActive;
   Scene_Base.prototype.isActive = function () {
      return Scene_Base_isActive.call(this) &&
         (!playing || playing && (playing.ended || playing.paused || manualMode));
   };

   const Scene_Map_updateTransferPlayer = Scene_Map.prototype.updateTransferPlayer;
   Scene_Map.prototype.updateTransferPlayer = function () {
      Scene_Map_updateTransferPlayer.call(this);
      if ($gamePlayer.isTransferring()) {
         manualMode = false;
      }
   };

   // エンカウントエフェクトを削除する処理
   const Scene_Map_launchBattle = Scene_Map.prototype.launchBattle;
   Scene_Map.prototype.launchBattle = function () {
      if (DisableEncounterEffect ||
         (ControlEncounterEffect && $gameSwitches.value(EffectControlSwitch))) {
         BattleManager.saveBgmAndBgs();
         this.stopAudioOnBattleStart();
         SoundManager.playBattleStart();
         // エンカウントエフェクトを削除
         // this.startEncounterEffect();
         this._mapNameWindow.hide();
      } else {
         // 元の処理
         Scene_Map_launchBattle.call(this);
      }
   };

   // マップ復帰時のフェードを削除する処理
   const Scene_Map_needsFadeIn = Scene_Map.prototype.needsFadeIn;
   Scene_Map.prototype.needsFadeIn = function () {
      if (DisableBattleEndFade ||
         (ControlBattleEndFade && $gameSwitches.value(FadeControlSwitch))) {
         return Scene_Map_needsFadeIn.call(this)
            && !SceneManager.isPreviousScene(Scene_Battle);
      } else {
         return Scene_Map_needsFadeIn.call(this);
      }
   };

   // 戦闘終了時のフェードを削除する処理
   const Scene_Battle_stop = Scene_Battle.prototype.stop;
   Scene_Battle.prototype.stop = function () {
      if (DisableBattleEndFade ||
         (ControlBattleEndFade && $gameSwitches.value(FadeControlSwitch))) {
         Scene_Message.prototype.stop.call(this);
         // フェードを削除
         // if (this.needsSlowFadeOut()) {
         //    this.startFadeOut(this.slowFadeSpeed(), false);
         // } else {
         //    this.startFadeOut(this.fadeSpeed(), false);
         // }
         this._statusWindow.close();
         this._partyCommandWindow.close();
         this._actorCommandWindow.close();
      } else {
         // 元の処理
         Scene_Battle_stop.call(this);
      }
   };

   const Window_MenuCommand_update = Window_MenuCommand.prototype.update;
   Window_MenuCommand.prototype.update = function () {
      if (playing && !playing.ended) return;
      Window_MenuCommand_update.call(this);
   };



   const BattleManager_initMembers = BattleManager.initMembers;
   BattleManager.initMembers = function () {
      BattleManager_initMembers.call(this);
      battleLosed = false;
   };

   const BattleManager_processDefeat = BattleManager.processDefeat;
   BattleManager.processDefeat = function () {
      BattleManager_processDefeat.call(this);
      battleLosed = true;
   };



   const Graphics__updateAllElements = Graphics._updateAllElements;
   Graphics._updateAllElements = function () {
      Graphics__updateAllElements.call(this);
      this._updateVideoDiv();
   };

   Graphics._updateVideoDiv = function () {
      videoDiv.width = this._width;
      videoDiv.height = this._height;
      videoDiv.style.zIndex = 2;
      for (var key in transitions) {
         let video = transitions[key];
         video.width = gameCanvas.width;
         video.height = gameCanvas.height;
         this._centerElement(video);
      }
      this._centerElement(videoDiv);
   };

})(this);
