//=============================================================================
// HS_MultiSoundPlayer.js
// ----------------------------------------------------------------------------
// Copyright (c) 2021 n2naokun(柊菜緒)
// This software is released under the MIT License.
// http://opensource.org/licenses/mit-license.php
// ----------------------------------------------------------------------------
// Version
// 1.0.1 2024/10/18 グローバル変数関連のミスを修正
// 1.0.0 2024/05/16 初版
// ----------------------------------------------------------------------------
// [Twitter]: https://twitter.com/n2naokun/
// [GitHub] : https://github.com/n2naokun/
//=============================================================================

/*:
 * @plugindesc 通常のBGMやBGSとは別に複数のサウンドを同時再生するプラグイン
 * MV/MZ両対応
 * @target MV MZ
 * @author n2naokun(柊菜緒)
 *
 * @help 説明
 * プラグインコマンドから
 * 事前読み込みする場合
 * SetSound 再生識別子 BGM/BGS フォルダに存在するファイル名 音量
 * 音量は省略出来ます。省略した場合は100%に設定されます。
 * セットサウンドであらかじめ読み込んでいた場合PlaySoundで
 * 再生識別子を指定するだけで再生できます。
 * 
 * 読み込みデータを削除する場合
 * DelSound 再生識別子
 * StopSoundを使用した場合でもメモリー上にデータを残し続けるので
 * 使用しなくなった場合DelSoundを実行することをおすすめします。
 * StopSoundを実行せずにDelSoundを実行することもできます。
 * 
 * 再生する場合
 * PlaySound 再生識別子 BGM/BGS フォルダに存在するファイル名 音量
 *  事前読み込みせずにここで読み込むこともできます。
 *  事前読み込みされていた場合は再生識別子以外のパラメーターは無視されます。
 *  音量は省略出来ます。省略した場合は100%に設定されます。
 *  ピッチとパンは固定です。
 * 
 * 音量を変える場合
 * SoundVolume 再生識別子 音量
 *  再生識別子で指定したサウンドの音量を変更します。
 * 
 * パンを変える場合
 * SoundPan 再生識別子 パン
 * 
 * ピッチを変える場合
 * SoundPitch 再生識別子 ピッチ（ツクールよりも広範囲に設定可能）
 * ※あまり極端な値にすると動作の保証ができません（笑）
 *  また、再生中に変更すると最初から再生しなおされます。
 *  SetSoundとPlaySoundの間で変更した方が精神衛生上安全でしょう。
 *  MZではツクールと同じ50～100に制限していますがテキストとして
 *  手動で指定する場合はこの限りではありません。
 * 
 * 停止する場合
 * StopSound 再生識別子
 * ※停止しただけではデータは削除されません。
 *  使用しない場合はDelSoundを使用してください。
 * 
 * 全て停止する場合
 * StopAllSound
 * 
 * 全て削除する場合
 * DelAllSound
 * 
 * 全てのBGMを停止する場合
 * StopAllBgm
 * ※ツクール内蔵のBGM再生機能には影響しません。
 * 
 * 全てのBGMを削除する場合
 * DelAllBgm
 * ※ツクール内蔵のBGM再生機能には影響しません。
 *  また、データ本体にも影響はありません。
 * 
 * 全てのBGSを停止する場合
 * StopAllBgs
 * ※ツクール内蔵のBGS再生機能には影響しません。
 * 
 * 全てのBGSを削除する場合
 * DelAllBgs
 * ※ツクール内蔵のBGS再生機能には影響しません。
 *  また、データ本体にも影響はありません。
 *
 * ※再生識別子とは
 *  再生中のサウンド自体に付ける名前です。
 *  これがあることによって同じ名前のファイルも同時再生ができます。
 * 
 * ※指定したファイルが存在しないとエラーが発生します。
 * 
 * 利用規約：
 *  作者に無断で改変、再配布が可能で、利用形態（商用、18禁利用等）
 *  についても制限はありません。
 *  このプラグインはもうあなたのものです。
 * 
 * @command SetSound
 * @text サウンドの再生の準備 (ファイル名直接指定)
 * @desc 指定したBGM/BGSをメモリー上に準備します
 * 音量は空欄で省略出来ます。省略した場合は100%に設定されます
 * 
 * @arg SoundId
 * @text 再生識別子
 * @desc 再生しているサウンドのトラックを指定するための識別子
 * @type text
 * 
 * @arg SoundType
 * @text 再生タイプ
 * @desc BGMかBGSを指定します
 * @type select
 * @option BGM
 * @option BGS
 * @default BGM
 * 
 * @arg Name
 * @text ファイル名
 * @desc 再生するサウンドのファイル名(BGMかBGSのフォルダに存在する必要があります)
 * @type text
 * 
 * @arg Volume
 * @text 音量
 * @desc 再生するサウンドの音量
 * 未入力の場合は100%になります
 * @type number
 * @min 0
 * @max 100
 * @default 
 * 
 * 
 * @command SetBGMSound
 * @text BGMサウンドの再生の準備 (ファイルブラウザ対応)
 * @desc 指定したBGMをメモリー上に準備します
 * 音量は空欄で省略出来ます。省略した場合は100%に設定されます
 * 
 * @arg SoundId
 * @text 再生識別子
 * @desc 再生しているサウンドのトラックを指定するための識別子
 * @type text
 * 
 * @arg Name
 * @text BGMファイル
 * @desc 再生するBGMのファイル
 * ファイルブラウザから指定できます
 * @type file
 * @dir audio/bgm
 * 
 * @arg Volume
 * @text 音量
 * @desc 再生するサウンドの音量
 * 未入力の場合は100%になります
 * @type number
 * @min 0
 * @max 100
 * @default 
 * 
 * 
 * @command SetBGSSound
 * @text BGSサウンドの再生の準備 (ファイルブラウザ対応)
 * @desc 指定したBGSをメモリー上に準備します
 * 音量は空欄で省略出来ます。省略した場合は100%に設定されます
 * 
 * @arg SoundId
 * @text 再生識別子
 * @desc 再生しているサウンドのトラックを指定するための識別子
 * @type text
 * 
 * @arg Name
 * @text BGSファイル
 * @desc 再生するサウンドのファイル名(BGMかBGSのフォルダに存在する必要があります)
 * @type file
 * @dir audio/bgs
 * 
 * @arg Volume
 * @text 音量
 * @desc 再生するサウンドの音量
 * 未入力の場合は100%になります
 * @type number
 * @min 0
 * @max 100
 * @default 
 * 
 * 
 * @command PlaySound
 * @text サウンドの再生 (ファイル名直接指定)
 * @desc 指定したBGM/BGSを再生します ここで直接指定もできます
 * 準備済みの場合は再生識別子以外のパラメーターは無視されます
 * 
 * @arg SoundId
 * @text 再生識別子
 * @desc 再生しているサウンドのトラックを指定するための識別子
 * @type text
 * 
 * @arg SoundType
 * @text 再生タイプ
 * @desc BGMかBGSを指定します
 * @type select
 * @option BGM
 * @option BGS
 * @default BGM
 * 
 * @arg Name
 * @text ファイル名
 * @desc 再生するサウンドのファイル名(BGMかBGSのフォルダに存在する必要があります)
 * @type text
 * 
 * @arg Volume
 * @text 音量
 * @desc 再生するサウンドの音量
 * 未入力の場合は100%になります
 * @type number
 * @min 0
 * @max 100
 * @default 
 * 
 * 
 * @command PlayBGMSound
 * @text BGMサウンドの再生 (ファイルブラウザ対応)
 * @desc 指定したBGMを再生します ここで直接指定もできます
 * 準備済みの場合は再生識別子以外のパラメーターは無視されます
 * 
 * @arg SoundId
 * @text 再生識別子
 * @desc 再生しているサウンドのトラックを指定するための識別子
 * @type text
 * 
 * @arg Name
 * @text ファイル名
 * @desc 再生するサウンドのファイル名(BGMかBGSのフォルダに存在する必要があります)
 * @type text
 * 
 * @arg Volume
 * @text 音量
 * @desc 再生するサウンドの音量
 * 未入力の場合は100%になります
 * @type number
 * @min 0
 * @max 100
 * @default 
 * 
 * 
 * @command PlayBGSSound
 * @text BGSサウンドの再生 (ファイルブラウザ対応)
 * @desc 指定したBGSを再生します ここで直接指定もできます
 * 準備済みの場合は再生識別子以外のパラメーターは無視されます
 * 
 * @arg SoundId
 * @text 再生識別子
 * @desc 再生しているサウンドのトラックを指定するための識別子
 * @type text
 * 
 * @arg Name
 * @text ファイル名
 * @desc 再生するサウンドのファイル名(BGMかBGSのフォルダに存在する必要があります)
 * @type text
 * 
 * @arg Volume
 * @text 音量
 * @desc 再生するサウンドの音量
 * 未入力の場合は100%になります
 * @type number
 * @min 0
 * @max 100
 * @default 
 * 
 * 
 * @command StopSound
 * @text 再生停止
 * @desc 再生識別子を指定して再生中のサウンドを停止します
 * 
 * @arg SoundId
 * @text 再生識別子
 * @desc 再生しているサウンドのトラックを指定するための識別子
 * @type text
 * 
 * 
 * @command StopAllSound
 * @text 再生全停止
 * @desc このプラグインで再生しているサウンドを全て停止します
 * 
 * 
 * @command StopAllBgm
 * @text BGM全停止
 * @desc このプラグインで再生しているBGMカテゴリのサウンドを全て停止します
 * 
 * 
 * @command StopAllBgs
 * @text BGS全停止
 * @desc このプラグインで再生しているBGSカテゴリのサウンドを全て停止します
 * 
 * 
 * @command SoundVolume
 * @text 音量変更
 * @desc 再生識別子を指定してトラックの音量を変更します
 * 
 * @arg SoundId
 * @text 再生識別子
 * @desc 再生しているサウンドのトラックを指定するための識別子
 * @type text
 * 
 * @arg Volume
 * @text 音量
 * @desc 再生するサウンドの音量
 * @type number
 * @min 0
 * @max 100
 * @default 100
 * 
 * 
 * @command SoundPan
 * @text 位相(パン)の変更
 * @desc サウンドの左右の音量の偏りを変更します
 * 
 * @arg SoundId
 * @text 再生識別子
 * @desc 再生しているサウンドのトラックを指定するための識別子
 * @type text 
 * 
 * @arg Pan
 * @text パン
 * @desc 左右の音量の偏り-100～100の値で指定します
 * @type number
 * @min -100
 * @max 100
 * @default 0
 * 
 * 
 * @command SoundPitch
 * @text ピッチの変更
 * @desc サウンドの再生ピッチを変更します
 * 
 * @arg SoundId
 * @text 再生識別子
 * @desc 再生しているサウンドのトラックを指定するための識別子
 * @type text
 * 
 * @arg Pitch
 * @text ピッチ
 * @desc 再生ピッチの値50%～150%で指定します。
 * @type number
 * @min 50
 * @max 150
 * @default 100
 * 
 * 
 * @command DelSound
 * @text サウンド削除
 * @desc 再生識別子を指定して準備済みのサウンド情報をメモリ上から削除します
 * 
 * @arg SoundId
 * @text 再生識別子
 * @desc 再生しているサウンドのトラックを指定するための識別子
 * @type text
 * 
 * 
 * @command DelAllBgm
 * @text BGM全削除
 * @desc このプラグインで再生しているBGMサウンドを全てメモリー上から削除します
 * 
 * 
 * @command DelAllBgs
 * @text BGS全削除
 * @desc このプラグインで再生しているBGSサウンドを全てメモリー上から削除します
 * 
 * 
 * @command DelAllSound
 * @text 全サウンド削除
 * @desc このプラグインで再生しているサウンドを全てメモリー上から削除します
 * 
 * 
 */

// ESLint向けグローバル変数宣言
/*global */

"use strict";//厳格なエラーチェック

// グローバル変数
var ExSoundBuffer = {};
var ExSound = {};
var ExSoundType = {};


(function (_global) {
   const PluginName = "HS_MultiSoundPlayer";
   _global.Imported = _global.Imported || {};
   _global.Imported[PluginName] = true;
   // 他のプラグインとの連携用シンボル

   const mgr = AudioManager;
   const pname = PluginName;

   function isMV() {
      return Utils.RPGMAKER_NAME !== 'MZ';
   }

   //プラグインコマンド定義
   // MVの場合
   var Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
   Game_Interpreter.prototype.pluginCommand = function (command, args) {
      switch (command) {
         case "SetSound":
            Utility.setSound(args[0], args[1], args[2], args[3]);
            break;

         case "PlaySound":
            Utility.playSound(args[0], args[1], args[2], args[3]);
            break;

         case "StopSound":
            Utility.stopSound(args[0]);
            break;

         case "StopAllBgm":
            Utility.stopAllBgm();
            break;

         case "StopAllBgs":
            Utility.stopAllBgs();
            break;

         case "StopAllSound":
            Utility.stopAllSound();
            break;

         case "SoundVolume":
            Utility.soundVolume(args[0], args[1]);
            break;

         case "SoundPan":
            Utility.soundPan(args[0], args[1]);
            break;

         case "SoundPitch":
            Utility.soundPitch(args[0], args[1]);
            break;

         case "DelSound":
            Utility.delSound(args[0]);
            break;

         case "DelAllBgm":
            Utility.delAllBgm();
            break;

         case "DelAllBgs":
            Utility.delAllBgs();
            break;

         case "DelAllSound":
            Utility.delAllSound();
            break;
      }
      Game_Interpreter_pluginCommand.call(this, command, args);
   };
   if (!isMV()) {
      // MZの場合はプラグインコマンドを登録する
      PluginManager.registerCommand(pname, "SetSound", args => {
         Utility.setSound(args['SoundId'], args['SoundType'], args['Name'], args['Volume']);
      });

      PluginManager.registerCommand(pname, "SetBGMSound", args => {
         Utility.setSound(args['SoundId'], 'BGM', args['Name'], args['Volume']);
      });

      PluginManager.registerCommand(pname, "SetBGSSound", args => {
         Utility.setSound(args['SoundId'], 'BGS', args['Name'], args['Volume']);
      });

      PluginManager.registerCommand(pname, "PlaySound", args => {
         Utility.playSound(args['SoundId'], args['SoundType'], args['Name'], args['Volume']);
      });

      PluginManager.registerCommand(pname, "PlayBGMSound", args => {
         Utility.playSound(args['SoundId'], 'BGM', args['Name'], args['Volume']);
      });

      PluginManager.registerCommand(pname, "PlayBGSSound", args => {
         Utility.playSound(args['SoundId'], 'BGS', args['Name'], args['Volume']);
      });

      PluginManager.registerCommand(pname, "StopSound", args => {
         Utility.stopSound(args['SoundId']);
      });

      PluginManager.registerCommand(pname, "StopAllBgm", args => {
         Utility.stopAllBgm();
      });

      PluginManager.registerCommand(pname, "StopAllBgs", args => {
         Utility.stopAllBgs();
      });

      PluginManager.registerCommand(pname, "StopAllSound", args => {
         Utility.stopAllSound();
      });

      PluginManager.registerCommand(pname, "SoundVolume", args => {
         Utility.soundVolume(args['SoundId'], args['Volume']);
      });

      PluginManager.registerCommand(pname, "SoundPan", args => {
         Utility.soundPan(args['SoundId'], args['Pan']);
      });

      PluginManager.registerCommand(pname, "SoundPitch", args => {
         Utility.soundPitch(args['SoundId'], args['Pitch']);
      });

      PluginManager.registerCommand(pname, "DelSound", args => {
         Utility.delSound(args['SoundId']);
      });

      PluginManager.registerCommand(pname, "DelAllBgm", args => {
         Utility.delAllBgm();
      });

      PluginManager.registerCommand(pname, "DelAllBgs", args => {
         Utility.delAllBgs();
      });

      PluginManager.registerCommand(pname, "DelAllSound", args => {
         Utility.delAllSound();
      });
   }


   function Utility() { }
   Utility.setSound = function (soundId, soundType, soundName, volume) {
      // パラメーターが無ければ実行しない
      if (!soundId || !soundType || !soundName) return;
      if (volume === '') volume = undefined;
      var type;
      // サウンドタイプを設定
      if (soundType == "BGM") {
         type = "BGM";
      } else {
         type = "BGS";
      }

      // 古いサウンドバッファを削除
      Utility.delSound(soundId);

      // サウンド情報を構築
      var sound = {};
      sound.name = String(soundName);
      sound.pan = 0;
      sound.pitch = 100;

      // ボリュームが指定されていない場合100に固定
      if (!isNaN(Number(volume))) {
         sound.volume = Number(volume).clamp(0, 100);
      } else {
         sound.volume = 100;
      }

      // バッファの作成とパラメーター設定
      ExSoundBuffer[String(soundId)] = mgr.createBuffer(type.toLocaleLowerCase() + (isMV() ? '' : '/'), sound.name);
      Utility.updateSoundParameters(ExSoundBuffer[String(soundId)], sound, type);
      // サウンドの情報の登録
      ExSound[String(soundId)] = Object.assign({}, sound);
      // サウンドタイプの登録
      ExSoundType[String(soundId)] = type;
   };

   Utility.delSound = function (soundId) {
      if (ExSoundBuffer[String(soundId)]) {
         // バッファ削除
         ExSoundBuffer[String(soundId)].stop();
         ExSoundBuffer[String(soundId)] = null;
         delete ExSoundBuffer[String(soundId)];
         // サウンド情報の削除
         ExSound[String(soundId)] = null;
         delete ExSound[String(soundId)];
         // サウンドタイプの削除
         ExSoundType[String(soundId)] = null;
         delete ExSoundType[String(soundId)];
      }
   };

   Utility.soundVolume = function (soundId, volume) {
      if (ExSoundBuffer[soundId && String(soundId)] && volume) {
         if (!isNaN(Number(volume))) {
            ExSound[String(soundId)].volume = Number(volume).clamp(0, 100);
         } else {
            return;
         }
         Utility.updateSoundParameters(
            ExSoundBuffer[String(soundId)],
            ExSound[String(soundId)],
            ExSoundType[String(soundId)]);
      }
   };

   Utility.soundPan = function (soundId, pan) {
      if (ExSoundBuffer[soundId && String(soundId)] && pan) {
         if (!isNaN(Number(pan))) {
            ExSound[String(soundId)].pan = Number(pan).clamp(-100, 100);
         } else {
            return;
         }
         Utility.updateSoundParameters(
            ExSoundBuffer[String(soundId)],
            ExSound[String(soundId)],
            ExSoundType[String(soundId)]);
      }
   };

   Utility.soundPitch = function (soundId, pitch) {
      if (ExSoundBuffer[soundId && String(soundId)] && pitch) {
         if (!isNaN(Number(pitch))) {
            ExSound[String(soundId)].pitch = Math.round(pitch);
         } else {
            return;
         }
         Utility.updateSoundParameters(
            ExSoundBuffer[String(soundId)],
            ExSound[String(soundId)],
            ExSoundType[String(soundId)]);
      }
   };

   Utility.playSound = function (soundId, soundType, soundName, volume) {
      if (ExSoundBuffer[String(soundId)]) {
         ExSoundBuffer[String(soundId)].play(true, 0);
      } else if (soundId && soundType && soundName) {
         Utility.setSound(soundId, soundType, soundName, volume);
         if (ExSoundBuffer[String(soundId)]) {
            ExSoundBuffer[String(soundId)].play(true, 0);
         }
      }
   };

   Utility.stopSound = function (soundId) {
      if (ExSoundBuffer[String(soundId)]) {
         ExSoundBuffer[String(soundId)].stop();
      }
   };

   Utility.updateSoundParameters = function (buffer, sound, soundType) {
      if (soundType == "BGS") {
         mgr.updateBufferParameters(buffer, mgr._bgsVolume, sound);
      } else if (soundType == "BGM") {
         mgr.updateBufferParameters(buffer, mgr._bgmVolume, sound);
      }
   };

   Utility.stopAllSound = function () {
      for (var soundId in ExSoundBuffer) {
         this.stopSound(soundId);
      }
   };

   Utility.delAllSound = function () {
      for (var soundId in ExSoundBuffer) {
         this.delSound(soundId);
      }
   };

   Utility.stopAllBgm = function () {
      for (var soundId in ExSoundBuffer) {
         if (ExSoundType[soundId] == "BGM") {
            this.stopSound(soundId);
         }
      }
   };

   Utility.delAllBgm = function () {
      for (var soundId in ExSoundBuffer) {
         if (ExSoundType[soundId] == "BGM") {
            this.delSound(soundId);
         }
      }
   };

   Utility.stopAllBgs = function () {
      for (var soundId in ExSoundBuffer) {
         if (ExSoundType[soundId] == "BGS") {
            this.stopSound(soundId);
         }
      }
   };

   Utility.delAllBgs = function () {
      for (var soundId in ExSoundBuffer) {
         if (ExSoundType[soundId] == "BGS") {
            this.delSound(soundId);
         }
      }
   };

   Object.defineProperty(AudioManager, "bgmVolume", {
      get: function () {
         return this._bgmVolume;
      },
      set: function (value) {
         this._bgmVolume = value;
         this.updateBgmParameters(this._currentBgm);
         // MultiSoundPlayerの音量を変更
         for (var soundId in ExSoundType) {
            if (ExSoundType[String(soundId)] == "BGM") {
               Utility.updateSoundParameters(
                  ExSoundBuffer[String(soundId)],
                  ExSound[String(soundId)],
                  ExSoundType[String(soundId)]);
            }
         }
      },
      configurable: true
   });

   Object.defineProperty(AudioManager, "bgsVolume", {
      get: function () {
         return this._bgsVolume;
      },
      set: function (value) {
         this._bgsVolume = value;
         this.updateBgsParameters(this._currentBgs);
         // MultiSoundPlayerの音量を変更
         for (var soundId in ExSoundType) {
            if (ExSoundType[String(soundId)] == "BGS") {
               Utility.updateSoundParameters(
                  ExSoundBuffer[String(soundId)],
                  ExSound[String(soundId)],
                  ExSoundType[String(soundId)]);
            }
         }
      },
      configurable: true
   });

})(this);
