//=============================================================================
// HS_OnlineUpdateSystem_v2.js
// ----------------------------------------------------------------------------
// Copyright (c) 2021 n2naokun(柊菜緒)
// This software is released under the MIT License.
// http://opensource.org/licenses/mit-license.php
// ----------------------------------------------------------------------------
// Version
// 1.1.3 2025/06/17 バージョン表記が同じでもデプロイ番号が違ったらアップデート判定
//                  していた仕様を変更
// 1.1.2 2025/06/17 アップデート確認のメッセージをマウスでも消せるように変更
// 1.1.1 2025/06/17 不具合の修正
// 1.1.0 2025/06/16 日本語設定以外ではアップデートを確認しないように変更
//                  通信量削減のための仕様変更、並行ダウンロードするように仕様変更
// 1.0.0 2025/05/29 初版
// ----------------------------------------------------------------------------
// [Twitter]: https://twitter.com/n2naokun/
// [GitHub] : https://github.com/n2naokun/
//=============================================================================

/*:
 * @plugindesc アップデートプラグイン
 * @target MZ
 * @author n2naokun(柊菜緒)
 *
 * @help 説明
 * 
 * 
 * 
 * 利用規約：
 *  作者に無断で改変、再配布が可能で、利用形態（商用、18禁利用等）
 *  についても制限はありません。
 *  このプラグインはもうあなたのものです。
 *
 */

// ESLint向けグローバル変数宣言
/*global */

"use strict";//厳格なエラーチェック

var Imported = Imported || {};
// 他のプラグインとの連携用シンボル

(function (_global) {
   const PluginName = "HS_OnlineUpdateSystem_v2";
   Imported[PluginName] = true;

   // ライブラリ読み込み
   const fs = require('fs-extra');
   const path = require('path');
   const crypto = require('crypto');
   const { PassThrough } = require('stream');

   const { ZSTDDecompress } = require('simple-zstd');
   const iconv = require('iconv-lite');


   // 定数宣言
   const logDir = './log/';
   const tempDir = './temp/';
   const dataDir = './data/';

   const configFile = 'update_config.json';

   const languageTable =
      [
         'jp',
         'en'
      ];



   // 変数宣言
   var config = null;
   var key = null;
   var remoteVersioninfo = null;
   var localVersioninfo = null;



   // フォルダチェック
   if (!fs.existsSync(logDir)) {
      fs.ensureDirSync(logDir);
   }


   // ツクール関連の処理
   function Scene_Update() {
      this.initialize.apply(this, arguments);
   }

   Scene_Update.prototype = Object.create(Scene_Base.prototype);
   Scene_Update.prototype.constructor = Scene_Update;

   Scene_Update.prototype.initialize = function () {
      Scene_Base.prototype.initialize.call(this);
      this._updateInfo = false;
      this._size = 0;
      this._speedCalc = false;
   };

   Scene_Update.prototype.create = function () {
      Scene_Base.prototype.create.call(this);
      this.createWindowLayer();
      this.createHelpWindow();
      this.createLogWindow();
   };

   Scene_Update.prototype.start = function () {
      Scene_Base.prototype.start.call(this);
      this._helpWindow.setText("アップデート情報を確認しています。");
      this.updateExecute();

   };

   Scene_Update.prototype.createHelpWindow = function () {
      if (Utils.RPGMAKER_NAME !== 'MZ') {
         this._helpWindow = new Window_Help();
      } else {
         const rect = this.helpWindowRect();
         this._helpWindow = new Window_Help(rect);
      }
      this.addWindow(this._helpWindow);
   };

   Scene_Update.prototype.helpWindowRect = function () {
      const wx = 0;
      const wy = 0;
      const ww = Graphics.boxWidth;
      const wh = this.helpAreaHeight();
      return new Rectangle(wx, wy, ww, wh);
   };

   Scene_Update.prototype.helpAreaHeight = function () {
      return this.calcWindowHeight(2, false);
   };

   Scene_Update.prototype.createLogWindow = function () {
      if (Utils.RPGMAKER_NAME !== 'MZ') {
         this._updateLog = new Window_UpdateLog(
            0,
            this._helpWindow.height,
            Graphics.boxWidth,
            Graphics.boxHeight - this._helpWindow.height);
      } else {
         this._updateLog = new Window_UpdateLog(
            new Rectangle(
               0,
               this._helpWindow.height,
               Graphics.boxWidth,
               Graphics.boxHeight - this._helpWindow.height));
      }
      this.addWindow(this._updateLog);
   };

   Scene_Update.prototype.update = function () {
      Scene_Base.prototype.update.call(this);
   };

   // アップデート実行
   var fileIo = [];
   const logText = [];
   var isCancel = false;
   var slotCount = 0;
   const parallel = 5;
   Scene_Update.prototype.updateExecute = function () {
      this._helpWindow.setText("アップデートを開始します。\nファイルリストをダウンロード中です。");
      const localFileInfo = getVersionInfo();
      const fetchUrl = `${config.endpoint}info/filelist_remote.json`;

      // filelist_remote.jsonの署名検証が必要
      download(fetchUrl)
         .then(res => {
            const originalUrl = res.url;
            const signatureUrl = `${originalUrl}.sig`;

            return Promise.all([
               res.arrayBuffer().then(buffer => Buffer.from(buffer)), // 元のデータをBufferとして取得
               download(signatureUrl).then(sigRes => sigRes.arrayBuffer().then(buffer => Buffer.from(buffer))) // 署名データをBufferとして取得
            ]);

         })
         .then(([fileDataBuffer, signatureBuffer]) => {
            this._helpWindow.setText('ファイルリストをダウンロードしました。');
            // 署名検証
            const isVerified = verifySignature(fileDataBuffer, signatureBuffer, config.publicKey);

            if (!isVerified) {
               this._textWindow.setText('ファイルリストの署名検証に失敗しました。\nアップデートを中止します。');
               log.addErrMsg('ファイルリストの署名検証に失敗しました。');
               return; // 署名検証失敗で処理を中断
            }

            // 署名検証が成功した場合のみ、ファイルリストをJSONとして解析
            const remoteFileInfo = JSON.parse(fileDataBuffer.toString('utf8'));

            // info/filelist_remote.jsonをtemp/data/filelist_local.jsonとして保存
            fs.outputFileSync(path.join(tempDir, 'data/', 'filelist_local.json'), fileDataBuffer);

            const changedFiles = compareFilelists(localFileInfo, remoteFileInfo);
            function processResponse(res, info) {
               // リトライ回数の初期化（初回のみ）
               if (info.retryCount === undefined) {
                  info.retryCount = 0;
               }

               // エラーが上限回数発生した場合は並行しているダウンロードの継続を停止
               if (isCancel) {
                  return;
               }

               // ダウンロードされたWeb ReadableStreamをNode.js ReadableStreamに変換
               const nodeStream = webStreamToNodeStream(res.body);

               // decryptFileの実行（nodeStreamからデータを取得）
               const decryptPromise = decryptFile(nodeStream, path.join(tempDir, info.path), key);

               // decryptFileが完了したら、そのファイルのハッシュを計算して検証
               fileIo.push(decryptPromise
                  .then(() => {
                     // ファイルがtempDirに保存された後、そのファイルのハッシュを計算
                     return generateFileHash(path.join(tempDir, info.path));
                  })
                  .then(hashResult => {
                     const calculatedHash = hashResult.sha256;
                     // ここでハッシュ値を検証
                     if (calculatedHash !== info.sha256) {
                        const errMsg = `ファイルのハッシュ値が一致しません: ${info.path}\n期待値: ${info.sha256}\n計算値: ${calculatedHash}`;
                        log.addErrMsg(errMsg);
                        throw new Error(errMsg); // エラーをスローしてcatchブロックへ
                     }
                     log.addMsg(`ファイルのハッシュ値が一致しました: ${info.path}`);

                     // 成功したら次のファイルを処理
                     const nextInfo = changedFiles.shift();
                     if (nextInfo) {
                        const url = `${config.endpoint}files/${nextInfo.path}`;
                        logText.push(`${nextInfo.path}をダウンロード`);
                        if (logText.length > 15) {
                           logText.shift();
                        }
                        this._updateLog.setText(logText.join('\n'));
                        download(url)
                           .then(res => {
                              processResponse.call(this, res, nextInfo);
                           })
                           .catch(err => {
                              errCatch.call(this, err, nextInfo); // エラーハンドリングにinfoを渡す
                           });
                     } else {
                        slotCount--;
                        if (slotCount === 0) {
                           // 全てのファイルが成功したら最終処理
                           Promise.all(fileIo) // fileIoはPromiseの配列なので、ここで解決を待つ
                              .then(() => {
                                 // アップデート前にセーブをバックアップ
                                 UpdateManager.saveBackup();
                                 this._updateLog.logAddLine("セーブデータをバックアップしました。\n");

                                 var bat =
                                    "@echo off\r\n" +
                                    "cls\r\n" +
                                    "echo 5秒後にアップデートを適用します。\r\n" +
                                    "timeout /t 5 /nobreak\r\n" +
                                    "cls\r\n" +
                                    "if exist .\\TEMP\\www\\save (rd .\\TEMP\\www\\save /s /q)\r\n" +
                                    "if exist .\\TEMP\\www\\captures (rd .\\TEMP\\www\\captures /s /q)\r\n" +
                                    "echo 適用中です。\r\n" +
                                    "xcopy .\\TEMP\\* .\\ /e /y /c /q\r\n" +
                                    "cls\r\n" +
                                    "echo 適用完了\r\n" +
                                    "echo 不要な一時ファイルを削除します。\r\n" +
                                    "rd .\\TEMP /s /q\r\n" +
                                    "cls\r\n" +
                                    "echo 5秒後にゲームを起動します。\r\n" +
                                    "timeout /t 5 /nobreak\r\n" +
                                    "start game.exe\r\n" +
                                    "timeout /t 1 /nobreak\r\n" +
                                    "del updateFile.bat\r\n";

                                 saveSJISFile(bat, "./updateFile.bat");

                                 this.downloadEnd();
                              })
                              .catch(err => {
                                 log.addErrMsg(err);
                                 // ここで全体のエラーハンドリング
                                 this._helpWindow.setText("アップデート中にエラーが発生しました。\nタイトルに戻ります。");
                                 setTimeout(() => {
                                    SceneManager.goto(Scene_Title);
                                 }, 3000);
                              });
                        }
                     }
                  })
                  .catch(err => {
                     errCatch.call(this, err, info); // エラーハンドリングにinfoを渡す
                  }));
            }

            // エラーハンドリング関数を修正し、リトライロジックを追加
            function errCatch(err, info) {
               log.addErrMsg(err);
               info.retryCount++;
               if (info.retryCount < 3) {
                  log.addMsg(`${info.path} のダウンロードに失敗しました。リトライします (${info.retryCount}回目)。`);
                  this._helpWindow.setText(`${info.path} のダウンロードに失敗しました。\nリトライ中... (${info.retryCount}/3)`);
                  const url = `${config.endpoint}files/${info.path}`;
                  download(url)
                     .then(res => {
                        processResponse.call(this, res, info); // 同じinfoオブジェクトで再試行
                     })
                     .catch(retryErr => {
                        errCatch.call(this, retryErr, info); // リトライも失敗したら再度エラーハンドリング
                     });
               } else {
                  isCancel = true;
                  log.addErrMsg(`${info.path} のダウンロードが3回失敗しました。アップデートを中止します。`);
                  this._helpWindow.setText("アップデートに失敗しました。\nタイトルに戻ります。");
                  setTimeout(() => {
                     SceneManager.goto(Scene_Title);
                  }, 3000);
               }
            }

            this._updateLog.setLogText("ファイルをダウンロードしています。\n\n");
            for (let i = 0; i < parallel; i++) {
               if (changedFiles.length > 0) {
                  slotCount++;
                  let info = changedFiles.shift();
                  if (info) {
                     const url = `${config.endpoint}files/${info.path}`;
                     download(url)
                        .then(res => {
                           processResponse.call(this, res, info);
                        })
                        .catch(err => {
                           errCatch.call(this, err, info); // 初回ダウンロード失敗時もリトライロジックへ
                        });
                  }
               } else {
                  if (slotCount === 0) {
                     SceneManager.goto(Scene_Title);
                  }
               }
            }

         })
         .catch(err => {
            log.addErrMsg(err);
            SceneManager.goto(Scene_Title);
         });

   };

   Scene_Update.prototype.downloadEnd = function () {
      this._size = 0;
      this._speedCalc = false;
      this._updateLog.clear();
      this._helpWindow.setText("ダウンロードが完了しました。\nファイルを適用します。");
      this._updateLog.logAddLine("ダウンロード完了\n");

      var time = 5;
      this._updateLog.setText(time + "秒後自動的にゲームを終了しファイルの更新を行います。\n※更新中は黒いウィンドウが表示されます。");
      var timer = setInterval(() => {
         time--;
         if (time === 0) {
            clearInterval(timer);
            // ファイル更新開始
            const { spawn } = require('child_process');
            const child = spawn('start', ['cmd.exe', '/c', 'updateFile.bat'], {
               shell: true,
               detached: true,
               stdio: 'ignore'
            });

            child.on('error', (err) => {
               log.addErrMsg(`Failed to start subprocess: ${err}`);
            });

            // spawnは非同期でプロセスを起動するため、すぐに次の行が実行される
            log.addMsg('ゲームを終了します。');
            window.process.exit();
         }
         this._updateLog.setText(time + "秒後自動的にゲームを終了しファイルの更新を行います。\n※更新中は黒いウィンドウが表示されます。");
      }, 1000);


   };



   // updateLogWindow
   function Window_UpdateLog() {
      this.initialize.apply(this, arguments);
   }

   Window_UpdateLog.prototype = Object.create(Window_Base.prototype);
   Window_UpdateLog.prototype.constructor = Window_UpdateLog;

   Window_UpdateLog.prototype.initialize = function (x, y, width, height) {
      Window_Base.prototype.initialize.apply(this, arguments);
      this._text = "";
      this._tmpText = "";
      this._logText = "";
   };

   Window_UpdateLog.prototype.setText = function (text) {
      if (this._tmpText !== text) {
         this._tmpText = text;
         this._text = this._logText + text;
         this.refresh();
      }
   };

   Window_UpdateLog.prototype.setLogText = function (text) {
      if (this._logText !== text) {
         this._logText = text;
         this._text = this._logText + this._tmpText;
         this.refresh();
      }
   };

   Window_UpdateLog.prototype.logAddLine = function (text) {
      this._logText += text + "\n";
      this._text = this._logText + this._tmpText;
      this.refresh();
   };

   Window_UpdateLog.prototype.getLogText = function () {
      return this._logText;
   };

   Window_UpdateLog.prototype.clear = function () {
      this.setText('');
   };

   Window_UpdateLog.prototype.clearLog = function () {
      this.setLogText("");
   };

   Window_UpdateLog.prototype.refresh = function () {
      this.contents.clear();
      if (Utils.RPGMAKER_NAME !== 'MZ') {
         this.drawTextEx(this._text, this.textPadding(), 0);
      } else {
         this.drawTextEx(this._text, this.itemPadding(), 0);
      }
   };


   function Window_Text(rect) {
      this.initialize.call(this, rect);
   }

   Window_Text.prototype = Object.create(Window_Base.prototype);
   Window_Text.prototype.constructor = Window_Text;

   Window_Text.prototype.initialize = function (rect) {
      Window_Base.prototype.initialize.call(this, rect);
      this._text = '';
      this.deactivate();
      this.hide();
      this._commandDeactivate = null;
      this._commandActivate = null;
   };

   Window_Text.prototype.setText = function (text) {
      this.callDeactivate();
      this._text = text;
      this.contents.clear();
      this.resetFontSettings();

      const padding = this.itemPadding();
      let lines = this._text.trim().split('\n');
      this.height = this.fittingHeight(lines.length) + padding * 2;

      // linesの要素にtextSizeを適用して一番大きな数字をtextWidthに代入
      let textWidth = 0;
      lines.forEach(line => {
         const width = this.textSize(line);
         if (width > textWidth) {
            textWidth = width;
         }
      });
      this.width = textWidth + $gameSystem.windowPadding() * 2 + padding * 2;

      this.createContents();

      this.x = (Graphics.width - this.width) / 2;
      this.y = (Graphics.height - this.height) / 2;
      const lh = this.lineHeight();
      for (let i = 0; i < lines.length; i++) {
         const line = lines[i];
         const y = padding + i * lh;
         this.drawText(line, padding, y, textWidth, 'center');
      }

      this.show();
      this.activate();
   };

   Window_Text.prototype.update = function () {
      Window_Base.prototype.update.call(this);
      if (this.active &&
         // 2025/06/17 ウィンドウを消すボタンを増加
         (Input.isTriggered('ok') || Input.isTriggered('cancel') ||
            TouchInput.isTriggered() || TouchInput.isCancelled())) {
         // 2025/06/17 ここまで
         this.setText('');
         this.deactivate();
         this.callActivate();
         this.hide();
      }
   };

   Window_Text.prototype.textSize = function (text) {
      return this.contents.measureTextWidth(text);
   };

   Window_Text.prototype.setCommandActivate = function (func) {
      this._commandActivate = func;
   };

   Window_Text.prototype.setCommandDeactivate = function (func) {
      this._commandDeactivate = func;
   };

   Window_Text.prototype.callActivate = function () {
      if (this._commandActivate) this._commandActivate();
   };

   Window_Text.prototype.callDeactivate = function () {
      if (this._commandDeactivate) this._commandDeactivate();
   };


   // var updateName = String(params["updateName"] || "アップデート確認");

   // var Window_TitleCommand_makeCommandList = Window_TitleCommand.prototype.makeCommandList;
   // Window_TitleCommand.prototype.makeCommandList = function () {
   //    Window_TitleCommand_makeCommandList.call(this);
   //    this.addCommand(updateName, 'checkUpdate');
   // };

   // var Scene_Title_createCommandWindow = Scene_Title.prototype.createCommandWindow;
   // Scene_Title.prototype.createCommandWindow = function () {
   //    Scene_Title_createCommandWindow.call(this);
   //    this._commandWindow.setHandler('checkUpdate', this.commandCheckUpdate.bind(this));
   // };

   var Scene_Title_initialize = Scene_Title.prototype.initialize;
   Scene_Title.prototype.initialize = function () {
      Scene_Title_initialize.apply(this, arguments);
      createVersionInfo();
      config = loadConfig();
      key = Buffer.from(config.encryptKey, 'base64');
   };

   var updateChecked = false;
   var Scene_Title_create = Scene_Title.prototype.create;
   Scene_Title.prototype.create = function () {
      Scene_Title_create.apply(this, arguments);

      this._textWindow = new Window_Text(new Rectangle(0, 0, 0, 0));
      this._textWindow.setCommandActivate(this.commandActivate.bind(this));
      this._textWindow.setCommandDeactivate(this.commandDeactivate.bind(this));

      this.addWindow(this._textWindow);
      let lang = languageTable[ConfigManager['multilingual'] || 0];

      if (!updateChecked && lang === 'jp') {
         updateChecked = true;
         const configExists = fs.existsSync(path.join(dataDir, configFile));
         const filelistExists = fs.existsSync(path.join(dataDir, 'filelist_local.json'));
         if (config && configExists && filelistExists) {
            this._textWindow.setText('アップデートを確認中');
            this._textWindow.deactivate();
            UpdateManager.chkUpdate()
               .then(res => {
                  const originalUrl = res.url;
                  const signatureUrl = `${originalUrl}.sig`;

                  // 元のデータと署名データを並行してダウンロード
                  return Promise.all([
                     res.arrayBuffer().then(buffer => Buffer.from(buffer)), // 元のデータをBufferとして取得
                     download(signatureUrl).then(sigRes => sigRes.arrayBuffer().then(buffer => Buffer.from(buffer))) // 署名データをBufferとして取得
                  ]);
               })
               .then(([fileDataBuffer, signatureBuffer]) => {
                  // 署名検証
                  const isVerified = verifySignature(fileDataBuffer, signatureBuffer, config.publicKey);

                  if (!isVerified) {
                     this._textWindow.setText('バージョン情報の署名検証に失敗しました。\nアップデートを中止します。');
                     log.addErrMsg('バージョン情報の署名検証に失敗しました。');
                     return; // 署名検証失敗で処理を中断
                  }

                  // 署名検証が成功した場合のみ、バージョン情報をJSONとして解析
                  remoteVersioninfo = JSON.parse(fileDataBuffer.toString('utf8'));

                  localVersioninfo = getVersionInfo(path.join(dataDir, 'versionInfo.json'));

                  if (localVersioninfo) {
                     const result = compareVersion(localVersioninfo, remoteVersioninfo);
                     if (result === 'new') {
                        this._textWindow.setText('アップデートがありました。\n3秒後にアップデートを開始します。');
                        this._textWindow.deactivate();
                        const time = 3 * 1000;
                        const timer = setTimeout(() => {
                           clearTimeout(timer);
                           this.gotoUpdate();
                        }, time);
                     } else if (result === 'nochange' || result === 'old') {
                        this._textWindow.setText('アップデートはありませんでした。');
                     }
                     return;
                  } else {
                     this._textWindow.setText('アップデートの確認に失敗しました。');
                     return;
                  }
               })
               .catch(err => {
                  this._textWindow.setText('アップデートの確認に失敗しました。\nしばらくしてから再度お試しください。');
                  log.addErrMsg(err);
               });
         } else {
            this._textWindow.setText(`アップデートシステムに必要なファイルが足りません。\n開発者にこのメッセージをお問い合わせください。`);
            if (!configExists) log.addErrMsg(`update_config.json not found.`);
            if (!filelistExists) log.addErrMsg(`filelist_local.json not found`);
         }
      }
   };

   Scene_Title.prototype.commandActivate = function () {
      this._commandWindow.activate();
      this._picComE = false;
   };

   Scene_Title.prototype.commandDeactivate = function () {
      this._commandWindow.deactivate();
      this._picComE = true;
   };

   Scene_Title.prototype.gotoUpdate = function () {
      SceneManager.push(Scene_Update);
   };


   // UpdateManager
   const UpdateManager = {};
   UpdateManager.chkUpdate = function () {
      const fetchUrl = `${config.endpoint}info/fileinfo.json`;
      // テスト用ローカルサーバー
      // const fetchUrl = `${config.endpoint}data/fileinfo.json`;
      return download(fetchUrl);
   };

   UpdateManager.saveBackup = function () {
      var date = new Date();
      var dirName = date.getFullYear() + "-" +
         this.zeroPadding(date.getMonth() + 1, 2) + "-" + this.zeroPadding(date.getDate(), 2) + "-" +
         this.zeroPadding(date.getHours(), 2) + "-" + this.zeroPadding(date.getMinutes(), 2);

      // セーブをバックアップ
      fs.copy("./save/", "./saveBackup/" + dirName + "/save/", (err) => {
         if (err) log.addErrMsg(err);
      });
   };

   UpdateManager.zeroPadding = function (num, length) {
      return ("0000" + num).slice(-length);
   };



   // ログ処理
   const logDate = new Date();
   const logFileName = 'log_' +
      logDate.getFullYear() +
      ('00' + logDate.getMonth()).slice(-2) +
      ('00' + logDate.getDate()).slice(-2) +
      ('00' + logDate.getHours()).slice(-2) +
      ('00' + logDate.getMinutes()).slice(-2) +
      ('00' + logDate.getSeconds()).slice(-2) +
      '.txt';
   var log = {};

   log.addMsg = function (msg) {
      const date = new Date();
      const time = date.getFullYear() + '/' +
         ('00' + date.getMonth()).slice(-2) + '/' +
         ('00' + date.getDate()).slice(-2) + ' ' +
         ('00' + date.getHours()).slice(-2) + ':' +
         ('00' + date.getMinutes()).slice(-2) + ':' +
         ('00' + date.getSeconds()).slice(-2);
      fs.outputFileSync(path.join(logDir, logFileName), `[info] ${time} ` + msg + '\n', { flag: 'a' });
      console.log(msg);
   };

   // エラーメッセージを赤色で追加する関数
   log.addErrMsg = function (msg) {
      const date = new Date();
      const time = date.getFullYear() + '/' +
         ('00' + date.getMonth()).slice(-2) + '/' +
         ('00' + date.getDate()).slice(-2) + ' ' +
         ('00' + date.getHours()).slice(-2) + ':' +
         ('00' + date.getMinutes()).slice(-2) + ':' +
         ('00' + date.getSeconds()).slice(-2);
      fs.outputFileSync(path.join(logDir, logFileName), `[error] ${time} ` + msg + '\n', { flag: 'a' });
      console.error(msg);
   };


   // 関数定義
   // 指定されたファイルのSHA-256ハッシュを生成
   function generateFileHash(filePath) {
      const stream = fs.createReadStream(filePath);
      return generateHash(stream);
   }

   // 指定されたストリームのSHA-256ハッシュを生成
   function generateHash(stream) {
      return new Promise((resolve, reject) => {
         const hash = crypto.createHash('sha256'); // SHA-256を暗黙的に使用

         stream.on('data', (data) => hash.update(data));
         stream.on('end', () => {
            resolve({
               sha256: hash.digest('hex')
            });
         });
         stream.on('error', (err) => {
            reject({
               err: err
            });
         });
      });
   }

   // 署名を検証する関数 (同期版)
   function verifySignature(data, signature, publicKey) {
      try {
         const isVerified = crypto.verify('sha256WithRSAEncryption', data, publicKey, signature);
         if (isVerified) {
            log.addMsg('署名が正常に検証されました。');
         } else {
            log.addErrMsg('署名の検証に失敗しました。');
         }
         return isVerified;
      } catch (err) {
         log.addErrMsg(`署名の検証中にエラーが発生しました: ${err.message}`);
         return false;
      }
   }


   // 暗号化されたファイルを復号する
   function decryptFile(inputStream, outputFile, key) {
      return new Promise((resolve, reject) => {
         sliceStream(inputStream, 16)
            .then(({ iv, body }) => {
               // IVを読み取ったら、残りのデータを復号
               const decipher = crypto.createDecipheriv('aes-256-ctr', key, iv);

               fs.ensureDirSync(path.dirname(outputFile));
               const output = fs.createWriteStream(outputFile);

               // 暗号化された部分を復号しながら書き込む
               body
                  // AESを復号化
                  .pipe(decipher)
                  // ZSTDの圧縮を解凍
                  .pipe(ZSTDDecompress())
                  .pipe(output);
               const msg = `ファイルの復号を開始しました: ${outputFile}`;
               log.addMsg(msg);


               output.on('finish', () => {
                  const msg = `ファイルが復号されました: ${outputFile}`;
                  log.addMsg(msg);
                  resolve();
               });

               output.on('error', (err) => {
                  log.addErrMsg(err);
                  reject(err);
               });
            })
            .catch(err => {
               log.addErrMsg(err);
               reject(err);
            });
      });
   }


   function sliceStream(stream, byteCount) {
      return new Promise((resolve, reject) => {
         let ivChunks = [];
         let collected = 0;
         let ivExtracted = false;
         const body = new PassThrough();

         // 1) readable→readループで先頭部分を集める
         const onReadable = () => {
            let chunk;
            while (!ivExtracted && (chunk = stream.read())) {
               const need = byteCount - collected;

               if (chunk.length < need) {
                  ivChunks.push(chunk);
                  collected += chunk.length;
               } else {
                  // 切り出し完了
                  ivChunks.push(chunk.slice(0, need));
                  const rest = chunk.slice(need);
                  ivExtracted = true;

                  // 【A】残りを先行して書き込む
                  if (rest.length) body.write(rest);

                  // ここで一気にリスナーを外す
                  stream
                     .off('readable', onReadable)
                     .off('end', onEnd)
                     .off('error', onError);

                  // 【B】残りを pipe で body に流す
                  stream.pipe(body);
                  // v14 では error は転送されないので手動で
                  stream.once('error', err => body.emit('error', err));

                  // 最終的に解決
                  resolve({ iv: Buffer.concat(ivChunks), body });
                  return;
               }
            }
         };

         // 2) end 前に byteCount に達しなかった場合
         const onEnd = () => {
            if (!ivExtracted) {
               // 全部 iv にまとめる
               resolve({ iv: Buffer.concat(ivChunks), body });
            }
            // body は pipe されていれば自動 end、そうでなければ手動で
            if (!ivExtracted) body.end();
         };

         // 3) 元ストリームのエラーを拒否
         const onError = err => reject(err);

         // --- イベント登録 ---
         stream.on('readable', onReadable);
         stream.on('end', onEnd);
         stream.on('error', onError);
      });
   }


   // Web ReadableStreamをNode.js ReadableStreamに変換するヘルパー関数
   function webStreamToNodeStream(webReadableStream) {
      const reader = webReadableStream.getReader();
      const nodeStream = new PassThrough();

      function pump() {
         reader.read().then(({ done, value }) => {
            if (done) {
               nodeStream.end();
               return;
            }
            nodeStream.write(value);
            pump();
         }).catch(err => {
            nodeStream.emit('error', err);
         });
      }

      pump();
      return nodeStream;
   }

   function getVersionInfo(infoPath) {
      const versionInfoPath = infoPath || path.join(dataDir, 'filelist_local.json');
      if (fs.existsSync(versionInfoPath)) {
         const versionInfo = JSON.parse(fs.readFileSync(versionInfoPath, 'utf-8'));
         return versionInfo;
      } else {
         log.addErrMsg(`バージョン情報ファイルが見つかりませんでした。\n初回動作では問題ありません。\n${versionInfoPath}`);
         return null;
      }
   }

   function compareVersion(oldVersion, newVersion) {
      var result = 'nochange';
      if ( // oldVersion.versionId !== newVersion.versionId ||
         oldVersion.version.toString() !== newVersion.version.toString()) {
         const ov = oldVersion.version;
         const nv = newVersion.version;
         if (nv[0] > ov[0]) {
            result = 'new';
         } else if (nv[0] === ov[0] && nv[1] > ov[1]) {
            result = 'new';
         } else if (nv[0] === ov[0] && nv[1] === ov[1] && nv[2] > ov[2]) {
            result = 'new';
         }
         // if (result === 'nochange' && oldVersion.versionId !== newVersion.versionId) {
         //    result = 'new';
         // }
      }
      return result;
   }


   // // copy files
   // function copyFiles(sourceList, destDir) {
   //    sourceList.forEach(info => {
   //       const sourcePath = path.join(tempDir, info.path);
   //       const destPath = path.join(destDir, info.path);
   //       fs.copyFileSync(sourcePath, destPath);
   //    });
   // }


   // Web Path to Windows Path
   function webPathToWindowsPath(webPath) {
      // Webのパス区切り文字をWindowsのパス区切り文字に変換
      let windowsPath = webPath.replace(/\//g, '\\');
      return windowsPath;
   }


   function compareFilelists(oldFilelist, newFilelist) {
      const changedFiles = [];
      const newFiles = newFilelist.files;
      const oldFiles = oldFilelist.files;

      for (const key in newFiles) {
         const newFile = newFiles[key];

         if (!oldFiles[key] || oldFiles[key].sha256 !== newFile.sha256) {
            changedFiles.push({
               "path": key,
               "sha256": newFile.sha256,
               "size": newFile.size
            });
         }
      }

      return changedFiles;
   }


   function readClientSetting(filePath) {
      let setting = null;
      if (fs.existsSync(filePath)) {
         setting = JSON.parse(fs.readFileSync(filePath));
      } else {
         log.addErrMsg('ゲームフォルダにアップデートシステム設定がありません。\nupdate_config.jsonをコピーしてください。');
      }
      return setting;
   }

   /**
    * URLを渡すとfetch APIでダウンロードしてPromiseを返す
    * @param {string} url ダウンロードするURL
    * @returns {Promise<Response>}
    */
   function download(url) {
      return new Promise((resolve, reject) => {
         fetch(url)
            .then(res => {
               if (!res.ok) {
                  // res.ok でない場合もログに記録
                  log.addErrMsg(`Download failed for URL ${url}: HTTP error! status: ${res.status}`);
                  return reject(new Error(`HTTP error! status: ${res.status}`));
               }
               resolve(res);
            })
            .catch(error => {
               log.addErrMsg(`Download failed for URL ${url}: ${error.message}`); // console.errorから変更
               reject(error); // Promiseは引き続きrejectする
            });
      });
   }

   /**
    * textをSJISでファイルに保存する
    * @param {string} text 保存するテキスト
    * @param {string} path 保存先のパス
    */
   function saveSJISFile(text, filePath) {
      try {
         const buffer = iconv.encode(text, 'Shift_JIS');

         fs.ensureDirSync(path.dirname(filePath));
         const writeStream = fs.createWriteStream(filePath);

         writeStream.write(buffer);
         writeStream.end();

         writeStream.on('finish', () => {
            log.addMsg(`ファイル ${filePath} をSJISで保存しました。`);
         });

         writeStream.on('error', (error) => {
            log.addErrMsg(`ファイル ${filePath} の保存に失敗しました: ${error}`);
            throw error;
         });
      } catch (error) {
         log.addErrMsg(`ファイル ${filePath} の保存に失敗しました: ${error}`);
         throw error;
      }
   }


   function loadConfig() {
      let config = null;
      if (fs.existsSync(path.join(dataDir, configFile))) {
         config = fs.readFileSync(path.join(dataDir, configFile), 'utf-8');
         config = JSON.parse(config);
      }
      return config;
   }

   function createVersionInfo() {
      const filePath = path.join(dataDir, 'versionInfo.json');
      const version =
         $dataSystem.gameTitle
            .match(/([0-9]+)\.([0-9]+)\.([0-9]+)/)
            .slice(1, 4).map(
               (num) => {
                  return Number(num);
               });
      fs.outputFileSync(filePath, JSON.stringify({
         version: version,
         versionId: $dataSystem.versionId
      }, null, 3));
   }

})(this);
