//=============================================================================
// MPP_BtStWindowEX_TypeA.js
//=============================================================================
// Copyright (c) 2018-2019 Mokusei Penguin
// Released under the MIT license
// http://opensource.org/licenses/mit-license.php
//=============================================================================

/*:
 * @plugindesc 【ver.1.10】戦闘画面でステータスウィンドウのデザインを変更します。
 * @author 木星ペンギン
 * 
 * @help ※アクティブタイムバトル（MPP_ActiveTimeBattle.js）と併用する場合、
 * 　こちらのプラグインが下になるように導入してください。
 * 
 * ================================================================
 * ▼詳細
 * --------------------------------
 *  〇【ステータスウィンドウの行数】を変更した場合、
 * 　 パーティコマンドやアクターコマンドの行数も変わります。
 * 
 *  〇ゲージは縦のグラデーションになります。
 * 
 *  〇【MPとTPを重ねて表示】する場合、MPゲージの表示位置に描写されます。
 * 
 *  ※注意
 *   このプラグインは処理が少し重いです。
 *   環境によって大きく処理落ちすることがあるため、
 *   気になるようであれば導入しないでください。
 * 
 * ================================
 * 制作 : 木星ペンギン
 * URL : http://woodpenguin.blog.fc2.com/
 * 
 * @param Window Row
 * @type number
 * @min 1
 * @desc ウィンドウの行数
 * @default 4
 *
 * @param HP Gauge Width
 * @type number
 * @min 1
 * @desc HPゲージの幅
 * @default 108
 *
 * @param MP Gauge Width
 * @type number
 * @min 1
 * @desc MPゲージの幅
 * @default 88
 *
 * @param TP Gauge Width
 * @type number
 * @min 1
 * @desc TPゲージの幅
 * @default 80
 *
 * @param Gauge Height
 * @type number
 * @min 1
 * @desc ゲージの高さ
 * @default 14
 *
 * @param Gauge Slope
 * @desc ゲージの傾き
 * (-1.0～1.0で指定 / 0で傾きなし)
 * @default 1.0
 *
 * @param MP TP Pile?
 * @type boolean
 * @desc MPとTPを重ねて表示するかどうか
 * @default true
 *
 * @param Gauge Outline Color
 * @desc ゲージの縁取り色
 * @default 48,48,96
 *
 * @param === ATB ===
 *
 * @param AT Gauge Pos
 * @type number
 * @min -1
 * @max 1
 * @desc ATゲージの位置
 * (-1:非表示, 0:MP/TPの右, 1:名前の下)
 * @default 0
 * 
 * @param AT Gauge Oy
 * @type number
 * @desc ATゲージのY座標補正
 * @default 0
 * 
 * @param AT Gauge Opacity
 * @type number
 * @desc キャラの下に表示されるATゲージの不透明度
 * @default 192
 * 
 */

//=============================================================================
// Main
//=============================================================================

(function() {

const MPPlugin = {};

{
    
    MPPlugin.contains = {};
    MPPlugin.contains['ATB'] = $plugins.some(function(plugin) {
        return (plugin.name === 'MPP_ActiveTimeBattle' && plugin.status);
    });
    MPPlugin.contains['TypeB'] = $plugins.some(function(plugin) {
        return (plugin.name === 'MPP_BtStWindowEX_TypeB' && plugin.status);
    });

    // MPP_ActiveTimeBattle.js
    if (MPPlugin.contains['ATB']) {
        let parameters = PluginManager.parameters('MPP_ActiveTimeBattle');

        MPPlugin.atGaugeWidth = Number(parameters['AT Gauge Width'] || 80);
        MPPlugin.atGaugeHeight = Number(parameters['AT Gauge Height'] || 14);
    }

    // MPP_BtStWindowEX_TypeA.js
    let parameters = PluginManager.parameters('MPP_BtStWindowEX_TypeA');

    MPPlugin.windowRow = Math.max(Number(parameters['Window Row'] || 4), 1);
    MPPlugin.hpGaugeWidth = Number(parameters['HP Gauge Width'] || 108);
    MPPlugin.mpGaugeWidth = Number(parameters['MP Gauge Width'] || 88);
    MPPlugin.tpGaugeWidth = Number(parameters['TP Gauge Width'] || 80);
    MPPlugin.gaugeHeight = Number(parameters['Gauge Height'] || 14);
    MPPlugin.gaugeSlope = Number(parameters['Gauge Slope'] || 1.0);
    MPPlugin.pileMpTp = !!eval(parameters['MP TP Pile?']);
    MPPlugin.gaugeOutlineColor = 'rgb(%1)'.format(parameters['Gauge Outline Color'] || '48,48,96');

    // === ATB ===
    MPPlugin.atGaugePos = Number(parameters['AT Gauge Pos'] || 0);
    MPPlugin.atGaugeOy = Number(parameters['AT Gauge Oy'] || 0);
    MPPlugin.atGaugeBackColor = 'rgb(32,32,64)';
    
}

if (MPPlugin.contains['TypeB']) {
    alert('MPP_BtStWindowEX_TypeA.js にてエラー。\nMPP_BtStWindowEX_TypeB との併用はできません。');
    return;
}

const Alias = {};

//-----------------------------------------------------------------------------
// Game_Actor

//
if (Game_Actor.prototype.hasOwnProperty('onBattleEnd')) {
    Alias.GaAc_onBattleEnd = Game_Actor.prototype.onBattleEnd;
}
Game_Actor.prototype.onBattleEnd = function() {
    var _super = Alias.GaAc_onBattleEnd || Game_Battler.prototype.onBattleEnd;
    _super.apply(this, arguments);
    delete this._mppDrawer;
};

Game_Actor.prototype.getMppDrawer = function() {
    if (!this._mppDrawer) {
        this._mppDrawer = new DrawActor(this);
    }
    return this._mppDrawer;
};

Game_Actor.prototype.updateDrawer = function() {
    if (this._mppDrawer) this._mppDrawer.update();
};

Game_Actor.prototype.startDrawer = function() {
    if (this._mppDrawer) this._mppDrawer.start();
};

//-----------------------------------------------------------------------------
// Game_Party

Game_Party.prototype.updateDrawer = function() {
    this.members().forEach( actor => actor.updateDrawer() );
};

Game_Party.prototype.startDrawer = function() {
    this.members().forEach( actor => actor.startDrawer() );
};

//-----------------------------------------------------------------------------
// Window

Window.prototype.addChildToContentsBack = function(child) {
    var index = this.children.indexOf(this._windowCursorSprite);
    return this.addChildAt(child, index + 1);
};

//-----------------------------------------------------------------------------
// DrawActor

function DrawActor() {
    this.initialize.apply(this, arguments);
}

DrawActor.params = ["hp", "mp", "tp", "mhp", "mmp"];

Object.defineProperties(DrawActor.prototype, {
    hp: { get: function() { return this._params.hp.value; }, configurable: true },
    mp: { get: function() { return this._params.mp.value; }, configurable: true },
    tp: { get: function() { return this._params.tp.value; }, configurable: true },
    mhp: { get: function() { return this._params.mhp.value; }, configurable: true },
    mmp: { get: function() { return this._params.mmp.value; }, configurable: true }
});

DrawActor.prototype.initialize = function(actor) {
    this._actor = actor;
    this._params = {};
    for (var i = 0; i < DrawActor.params.length; i++) {
        var param = DrawActor.params[i];
        var value = (param === "tp" ? 0 : actor[param]);
        this._params[param] = new DrawActorParam(value);
    }
};

DrawActor.prototype.start = function() {
    var actor = this._actor;
    for (var i = 0; i < DrawActor.params.length; i++) {
        var param = DrawActor.params[i];
        this._params[param].start(actor[param]);
    }
};

DrawActor.prototype.update = function() {
    for (var i = 0; i < DrawActor.params.length; i++) {
        var param = DrawActor.params[i];
        this._params[param].update();
    }
};

DrawActor.prototype.isDead = function() {
    return this._actor.isDead();
};

DrawActor.prototype.isDying = function() {
    return this._actor.isDying();
};

DrawActor.prototype.hpRate = function() {
    return this.hp / this.mhp;
};

DrawActor.prototype.mpRate = function() {
    return this.mmp > 0 ? this.mp / this.mmp : 0;
};

DrawActor.prototype.tpRate = function() {
    return this.tp / 100;
};

DrawActor.prototype.isMoveHp = function() {
    return this._params.hp.isMove() || this._params.mhp.isMove();
};

DrawActor.prototype.isMoveMp = function() {
    return this._params.mp.isMove() || this._params.mmp.isMove();
};

DrawActor.prototype.isMoveTp = function() {
    return this._params.tp.isMove();
};

DrawActor.prototype.needUpdateHp = function() {
    return this._params.hp.needUpdate() || this._params.mhp.needUpdate();
};

DrawActor.prototype.needUpdateMp = function() {
    return this._params.mp.needUpdate() || this._params.mmp.needUpdate();
};

DrawActor.prototype.needUpdateTp = function() {
    return this._params.tp.needUpdate();
};

//-----------------------------------------------------------------------------
// DrawActorParam

function DrawActorParam() {
    this.initialize.apply(this, arguments);
}

DrawActorParam.prototype.initialize = function(value) {
    this.value = value;
    this._target = value;
    this._duration = 0;
};

DrawActorParam.prototype.start = function(value) {
    if (this._target !== value) {
        this._target = value;
        this._duration = 60;
    } 
};

DrawActorParam.prototype.update = function() {
    if (this._duration > 0) {
        var d = this._duration;
        var sv = this._target - this.value;
        var max = d * d;
        this.value += Math.round(sv * (max - Math.pow(d - 1, 2)) / max);
        this._duration--;
    }
};

DrawActorParam.prototype.isMove = function() {
    return this.value !== this._target;
};

DrawActorParam.prototype.needUpdate = function() {
    return this._duration % 2 === 0;
};

//-----------------------------------------------------------------------------
// Sprite_GradGauge

function Sprite_GradGauge() {
    this.initialize.apply(this, arguments);
}

Sprite_GradGauge.prototype = Object.create(Sprite.prototype);
Sprite_GradGauge.prototype.constructor = Sprite_GradGauge;

Sprite_GradGauge.prototype.initialize = function(width, height, color1, color2, backColor, mirror) {
    Sprite.prototype.initialize.call(this);
    this.bitmap = new Bitmap(width, height);
    this._mirror = mirror || false;
    this._gaugeWidth = width - height - 2;
    this._color1 = color1;
    this._color2 = color2;
    this._backColor = backColor;
    this._slope = MPPlugin.gaugeSlope * (mirror ? -1 : 1);
    this._animeCount = 0;
    this._drawCount = 0;
    this._rate = 0;
};

Sprite_GradGauge.prototype.update = function() {
    Sprite.prototype.update.call(this);
    if (this.visible) {
        this.updateAnimeCount();
        this.updateDrawCount();
    }
};

Sprite_GradGauge.prototype.updateAnimeCount = function() {
    if (this._animeCount > 0) {
        this._animeCount--;
    } else {
        this._animeCount = Math.floor(this._gaugeWidth * (1 + this._rate) / 2) + 32;
    }
};

Sprite_GradGauge.prototype.updateDrawCount = function() {
    this._drawCount = (this._drawCount + 1) % 4;
    if (this._drawCount === 0) {
        var fillW = (this._gaugeWidth - 2) * this._rate;
        var gx = fillW - this._animeCount * 2 + 64;
        if (gx > 0) this.refresh();
    }
};

Sprite_GradGauge.prototype.refresh = function() {
    var bitmap = this.bitmap;
    bitmap.clear();
    var context = bitmap.context;
    context.save();
    
    var slope = this._slope;
    var bh = bitmap.height;
    var bw = this._gaugeWidth;
    var sx = bh / 2 + bh / 2 * slope;
    context.setTransform(1, 0, -slope, 1, sx, 0);
    context.strokeStyle = MPPlugin.gaugeOutlineColor;
    context.lineWidth = 2 + Math.abs(slope) * 2;
    context.strokeRect(1, 1, bw-2, bh-2);
    
    context.fillStyle = this._backColor;
    context.fillRect(1, 1, bw-2, bh-2);
    
    var gradient = context.createLinearGradient(0, 1, 0, bh - 1);
    gradient.addColorStop(0, this._color1);
    gradient.addColorStop(0.5, this._color2);
    gradient.addColorStop(1, this._color1);
    context.fillStyle = gradient;
    var fillW = (bw - 2) * this._rate;
    context.fillRect(1, 1, fillW, bh - 2);
    
    var gx = fillW - this._animeCount * 2;
    var gw = 64;
    var gradient = context.createLinearGradient(gx, 0, gx + gw, 0);
    gradient.addColorStop(0, 'rgba(255,255,255,0)');
    gradient.addColorStop(0.5, 'rgba(255,255,255,1.0)');
    gradient.addColorStop(1, 'rgba(255,255,255,0)');
    context.fillStyle = gradient;
    context.fillRect(1, 1, fillW, bh - 2);
    
    context.restore();
    
    var cx = bh / 2 - bh / 2 * Math.abs(slope) + Math.abs(slope);
    context.clearRect(0,0,cx,bh);
    context.clearRect(bitmap.width-cx,0,cx,bh);
    bitmap._setDirty();
};

Sprite_GradGauge.prototype.setRate = function(rate) {
    this.visible = true;
    if (this._rate !== rate) {
        this._rate = rate;
        this.refresh();
        this._drawCount = 0;
    }
};

//-----------------------------------------------------------------------------
// Sprite_AtGauge

function Sprite_AtGauge() {
    this.initialize.apply(this, arguments);
}

Sprite_AtGauge.prototype = Object.create(Sprite.prototype);
Sprite_AtGauge.prototype.constructor = Sprite_AtGauge;

Sprite_AtGauge.prototype.initialize = function(width, height) {
    Sprite.prototype.initialize.call(this);
    this.bitmap = new Bitmap(width, height);
    this._subject = null;
    this._blinkCount = 0;
};

Sprite_AtGauge.prototype.setBattler = function(subject) {
    if (this._subject !== subject) {
        this._subject = subject;
        this.refresh();
    }
};

Sprite_AtGauge.prototype.update = function() {
    Sprite.prototype.update.call(this);
    if (this._subject && Graphics.frameCount % 2 === 0) {
        if (this._subject.isStandby()) {
            this._blinkCount = (this._blinkCount + 1) % 16;
        } else {
            this._blinkCount = 0;
        }
        if (this.visible) this.refresh();
    }
};

Sprite_AtGauge.prototype.refresh = function() {
    this.bitmap.clear();
    if (this._subject) {
        var color1 = this._subject.atGaugeColor1();
        var color2 = this._subject.atGaugeColor2();
        this.drawGauge(this._subject.atGaugeRate(), color1, color2);
    }
};

Sprite_AtGauge.prototype.drawGauge = function(rate, color1, color2) {
    var gaugeW = this.width - 2;
    var gaugeH = this.height - 2;
    var fillW = gaugeW * rate;
    
    var context = this.bitmap.context;
    context.save();
    
    context.strokeStyle = MPPlugin.gaugeOutlineColor;
    context.lineWidth = 2;
    context.strokeRect(0, 0, this.width, this.height);
    context.fillStyle = MPPlugin.atGaugeBackColor;
    context.fillRect(1, 1, gaugeW, gaugeH);
    
    context.globalAlpha = (192 + Math.abs(8 - this._blinkCount) * 8) / 255;
    var gradient = context.createLinearGradient(1, 1, 1, gaugeH);
    gradient.addColorStop(0, color1);
    gradient.addColorStop(0.5, color2);
    gradient.addColorStop(1, color1);
    context.fillStyle = gradient;
    context.fillRect(1, 1, fillW, gaugeH);
    
    context.restore();
    this.bitmap._setDirty();
};

//-----------------------------------------------------------------------------
// Sprite_Icons

function Sprite_Icons() {
    this.initialize.apply(this, arguments);
}

Sprite_Icons.prototype = Object.create(Sprite.prototype);
Sprite_Icons.prototype.constructor = Sprite_Icons;

Sprite_Icons.prototype.initialize = function(width) {
    Sprite.prototype.initialize.call(this);
    this.bitmap = new Bitmap(1, 1);
    this._icons = [];
    this._iconsMax = Math.floor(width / Window_Base._iconWidth);
    this._animeCount = 0;
};

Sprite_Icons.prototype.setIcons = function(icons) {
    this.visible = true;
    if (!this._icons.equals(icons)) {
        this._icons = icons;
        this._animeCount = 0;
        this.refresh();
    }
};

Sprite_Icons.prototype.update = function() {
    Sprite.prototype.update.call(this);
    if (!this.visible) {
        this._animeCount = 0;
    } else if (Graphics.frameCount % 2 === 0) {
        this.updateAnime();
    }
};

Sprite_Icons.prototype.updateAnime = function() {
    var size = this._icons.length;
    if (size > this._iconsMax) {
        this._animeCount++;
        var max = (size - this._iconsMax + 2) * Window_Base._iconWidth;
        if (this._animeCount >= max) {
            this._animeCount = 0;
        }
        this.updateFrame();
    }
};

Sprite_Icons.prototype.updateFrame = function() {
    var maxItems = this._icons.length;
    var x = this._animeCount - Window_Base._iconWidth;
    x = Math.min(x, (maxItems - this._iconsMax) * Window_Base._iconWidth);
    x = Math.max(x, 0);
    var width = Math.min(this._iconsMax, maxItems) * Window_Base._iconWidth;
    this.setFrame(x, 0, width, Window_Base._iconHeight);
};

Sprite_Icons.prototype.refresh = function() {
    var maxItems = this._icons.length;
    var width = Window_Base._iconWidth * maxItems;
    if (width === 0 || this.bitmap.width === width) {
        this.bitmap.clear();
    } else {
        this.bitmap = new Bitmap(width, Window_Base._iconHeight);
    }
    for (var i = 0; i < maxItems; i++) {
        this.drawIcon(this._icons[i], Window_Base._iconWidth * i);
    }
    this.updateFrame();
};

Sprite_Icons.prototype.drawIcon = function(iconIndex, x) {
    var iconSet = ImageManager.loadSystem('IconSet');
    var pw = Window_Base._iconWidth;
    var ph = Window_Base._iconHeight;
    var sx = iconIndex % 16 * pw;
    var sy = Math.floor(iconIndex / 16) * ph;
    this.bitmap.blt(iconSet, sx, sy, pw, ph, x, 0);
};

//-----------------------------------------------------------------------------
// Window_PartyCommand

//24
Window_PartyCommand.prototype.numVisibleRows = function() {
    return MPPlugin.windowRow;
};

//-----------------------------------------------------------------------------
// Window_ActorCommand

//25
Window_ActorCommand.prototype.numVisibleRows = function() {
    return MPPlugin.windowRow;
};

//-----------------------------------------------------------------------------
// Window_BattleStatus

//13
Alias.WiBaSt_initialize = Window_BattleStatus.prototype.initialize;
Window_BattleStatus.prototype.initialize = function() {
    this._mppContentsSprites = {};
    this._createDrawer = false;
    Alias.WiBaSt_initialize.apply(this, arguments);
};

//
if (Window_BattleStatus.prototype.hasOwnProperty('drawActorName')) {
    Alias.WiBaSt_drawActorName = Window_BattleStatus.prototype.drawActorName;
}
Window_BattleStatus.prototype.drawActorName = function(actor, x, y, width) {
    var _super = Alias.WiBaSt_drawActorName || Window_Selectable.prototype.drawActorName;
    _super.apply(this, arguments);
    if (this.isDrawAt(1)) {
        this.drawActorAt(actor, x, y, MPPlugin.atGaugeWidth);
    }
};

// 2024/07/19 NUUN_IconSideBySideとの競合対策のためコメントアウト
// Window_BattleStatus.prototype.drawActorIcons = function(actor, x, y, width) {
//     width = width || 144;
//     var sprite = this.getIconsSprite(x, y, width);
//     sprite.setIcons(actor.allIcons());
//     this.setupSpritePos(sprite, x, y);
// };
// 2024/07/19 ここまで

Window_BattleStatus.prototype.getIconsSprite = function(x, y, width) {
    var sprite = this.getMppSprite(x, y);
    if (!sprite) {
        sprite = new Sprite_Icons(width);
        this.addMppSprite(x, y, sprite);
    }
    return sprite;
};

Window_BattleStatus.prototype.getMppSpriteKey = function(x, y) {
    return x + ':' + y;
};

Window_BattleStatus.prototype.getMppSprite = function(x, y) {
    var key = this.getMppSpriteKey(x, y);
    return this._mppContentsSprites[key];
};

Window_BattleStatus.prototype.addMppSprite = function(x, y, sprite) {
    var key = this.getMppSpriteKey(x, y);
    this._mppContentsSprites[key] = sprite;
    this.addChildToContentsBack(sprite);
};

//31
Window_BattleStatus.prototype.numVisibleRows = function() {
    return MPPlugin.windowRow;
};

Window_BattleStatus.prototype.hideMppContentsSprites = function() {
    for (var key in this._mppContentsSprites) {
        this._mppContentsSprites[key].visible = false;
    }
};

//39
Alias.WiBaSt_refresh = Window_BattleStatus.prototype.refresh;
Window_BattleStatus.prototype.refresh = function() {
    this.hideMppContentsSprites();
    $gameParty.startDrawer();
    this.clearUpdateDrawer();
    this._createDrawer = true;
    Alias.WiBaSt_refresh.call(this);
    this._createDrawer = false;
};

//63
Window_BattleStatus.prototype.gaugeAreaWidth = function() {
    var width = 0;
    if (this.isDrawHp())  width += MPPlugin.hpGaugeWidth + 15;
    if (this.isDrawMp())  width += MPPlugin.mpGaugeWidth + 15;
    if (this.isDrawTp())  width += MPPlugin.tpGaugeWidth + 15;
    if (this.isDrawAt(0)) width += MPPlugin.atGaugeWidth + 15;
    return Math.max(width - 15, 0);
};

Window_BattleStatus.prototype.isDrawHp = function() {
    return MPPlugin.hpGaugeWidth > 0;
};

Window_BattleStatus.prototype.isDrawMp = function() {
    return MPPlugin.mpGaugeWidth > 0;
};

Window_BattleStatus.prototype.isDrawTp = function() {
    return $dataSystem.optDisplayTp && MPPlugin.tpGaugeWidth > 0 &&
            !MPPlugin.pileMpTp;
};

Window_BattleStatus.prototype.isDrawAt = function(pos) {
    return MPPlugin.contains['ATB'] && MPPlugin.atGaugeWidth > 0 &&
            MPPlugin.atGaugePos === pos;
};

//72
Window_BattleStatus.prototype.drawGaugeArea = function(rect, actor) {
    var dx = rect.x;
    var drawActor = actor.getMppDrawer();
    if (this.isDrawHp()) {
        this.drawActorHp(drawActor, dx, rect.y, MPPlugin.hpGaugeWidth);
        dx += MPPlugin.hpGaugeWidth + 15;
    }
    if (this.isDrawMp()) {
        if ($dataSystem.optDisplayTp && MPPlugin.pileMpTp) {
            this.drawActorMpTp(drawActor, dx, rect.y, MPPlugin.mpGaugeWidth);
        } else {
            this.drawActorMp(drawActor, dx, rect.y, MPPlugin.mpGaugeWidth);
        }
        dx += MPPlugin.mpGaugeWidth + 15;
    }
    if (this.isDrawTp()) {
        this.drawActorTp(drawActor, dx, rect.y, MPPlugin.tpGaugeWidth);
        dx += MPPlugin.tpGaugeWidth + 15;
    }
    if (this.isDrawAt(0)) {
        this.drawActorAt(actor, dx, rect.y, MPPlugin.atGaugeWidth);
    }
};

//
Window_BattleStatus.prototype.drawGauge = function(x, y, width, rate, color1, color2, mirror) {
    mirror = mirror || false;
    var sprite = this.getMppSprite(x, y);
    if (!sprite) {
        var height = MPPlugin.gaugeHeight;
        sprite = new Sprite_GradGauge(width + 16, height, color1, color2,
                                        this.gaugeBackColor(), mirror);
        this.addMppSprite(x, y, sprite);
    }
    this.setupSpritePos(sprite, x - 8, y);
    sprite.setRate(rate);
};

Window_BattleStatus.prototype.setupSpritePos = function(sprite, x, y) {
    var padding = this.standardPadding();
    sprite.x = x + padding;
    sprite.y = y + padding + this.lineHeight() - 2 - sprite.height;
};

//
if (Window_BattleStatus.prototype.hasOwnProperty('drawActorHp')) {
    Alias.WiBaSt_drawActorHp = Window_BattleStatus.prototype.drawActorHp;
}
Window_BattleStatus.prototype.drawActorHp = function(actor, x, y, width) {
    var _super = Alias.WiBaSt_drawActorHp || Window_Selectable.prototype.drawActorHp;
    _super.apply(this, arguments);
    
    if (this._createDrawer && actor.isMoveHp()) {
        this.createActorHpDrawer(actor, x, y, width);
    }
};

Window_BattleStatus.prototype.createActorHpDrawer = function(actor, x, y, width) {
    this.addUpdateDrawer(() => {
        if (actor.needUpdateHp()) {
            this.contents.clearRect(x-1, y, width+2, this.lineHeight());
            this.drawActorHp(actor, x, y, width);
        }
        return actor.isMoveHp();
    });
};

//
if (Window_BattleStatus.prototype.hasOwnProperty('drawActorMp')) {
    Alias.WiBaSt_drawActorMp = Window_BattleStatus.prototype.drawActorMp;
}
Window_BattleStatus.prototype.drawActorMp = function(actor, x, y, width) {
    var _super = Alias.WiBaSt_drawActorMp = Window_Selectable.prototype.drawActorMp;
    _super.apply(this, arguments);
    
    if (this._createDrawer && actor.isMoveMp()) {
        this.createActorMpDrawer(actor, x, y, width);
    }
};

Window_BattleStatus.prototype.createActorMpDrawer = function(actor, x, y, width) {
    this.addUpdateDrawer(() => {
        if (actor.needUpdateMp()) {
            this.contents.clearRect(x-1, y, width+2, this.lineHeight());
            this.drawActorMp(actor, x, y, width);
        }
        return actor.isMoveMp();
    });
};

//
if (Window_BattleStatus.prototype.hasOwnProperty('drawActorTp')) {
    Alias.WiBaSt_drawActorTp = Window_BattleStatus.prototype.drawActorTp;
}
Window_BattleStatus.prototype.drawActorTp = function(actor, x, y, width) {
    var _super = Alias.WiBaSt_drawActorTp || Window_Selectable.prototype.drawActorTp;
    _super.apply(this, arguments);
    
    if (this._createDrawer && actor.isMoveTp()) {
        this.createActorTpDrawer(actor, x, y, width);
    }
};

Window_BattleStatus.prototype.createActorTpDrawer = function(actor, x, y, width) {
    this.addUpdateDrawer(() => {
        if (actor.needUpdateTp()) {
            this.contents.clearRect(x-1, y, width+2, this.lineHeight());
            this.drawActorTp(actor, x, y, width);
        }
        return actor.isMoveTp();
    });
};

Window_BattleStatus.prototype.drawActorMpTp = function(actor, x, y, width) {
    width = (width || 186) - 10;
    var color1 = this.mpGaugeColor1();
    var color2 = this.mpGaugeColor2();
    var drawY = y - MPPlugin.gaugeHeight + 6;
    this.drawGauge(x, drawY, width, actor.mpRate(), color1, color2);
    color1 = this.tpGaugeColor1();
    color2 = this.tpGaugeColor2();
    this.drawGauge(x + 10, y, width, actor.tpRate(), color1, color2, true);
    this.contents.fontSize = 24;
    this.changeTextColor(this.systemColor());
    this.contents.drawText(TextManager.mpA, x, y + 1, 32, 24);
    this.contents.drawText(TextManager.tpA, x + 22, y + 10, 32, 24);
    this.contents.fontSize = this.standardFontSize();
    
    if (this._createDrawer && (actor.isMoveMp() || actor.isMoveTp())) {
        this.createActorMpTpDrawer(actor, x, y, width);
    }
};

Window_BattleStatus.prototype.createActorMpTpDrawer = function(actor, x, y, width) {
    this.addUpdateDrawer(() => {
        if (actor.needUpdateMp()) {
            var sprite = this.getMppSprite(x, y - MPPlugin.gaugeHeight + 6);
            if (sprite) sprite.setRate(actor.mpRate());
        }
        if (actor.needUpdateTp()) {
            var sprite = this.getMppSprite(x + 10, y);
            if (sprite) sprite.setRate(actor.tpRate());
        }
        return actor.isMoveMp() || actor.isMoveTp();
    });
};

Alias.WiBaSt_drawActorAt = Window_BattleStatus.prototype.drawActorAt;
Window_BattleStatus.prototype.drawActorAt = function(actor, x, y, width) {
    Alias.WiBaSt_drawActorAt.apply(this, arguments);
    var sprite = this.getMppSprite(x, y);
    sprite.visible = true;
    sprite.setBattler(actor);
};

Window_BattleStatus.prototype.drawAtGauge = function(x, y, width) {
    var sprite = this.getMppSprite(x, y);
    if (!sprite) {
        sprite = new Sprite_AtGauge(width, MPPlugin.atGaugeHeight);
        this.addMppSprite(x, y, sprite);
    }
    this.setupSpritePos(sprite, x, y);
};

//-----------------------------------------------------------------------------
// Window_BattleEnemy

//30
Window_BattleEnemy.prototype.numVisibleRows = function() {
    return MPPlugin.windowRow;
};

//-----------------------------------------------------------------------------
// Window_AtbSkillStatus

if (MPPlugin.contains['ATB']) {

    Alias.WiAtSkSt_initialize = Window_AtbSkillStatus.prototype.initialize;
    Window_AtbSkillStatus.prototype.initialize = function(x, y, width, height) {
        this._mppContentsSprites = {};
        Alias.WiAtSkSt_initialize.apply(this, arguments);
    };
    
    let AtbSkillStatus = Window_AtbSkillStatus.prototype;
    let BattleStatus = Window_BattleStatus.prototype;

    AtbSkillStatus.getMppSpriteKey = BattleStatus.getMppSpriteKey;
    AtbSkillStatus.getMppSprite = BattleStatus.getMppSprite;
    AtbSkillStatus.addMppSprite = BattleStatus.addMppSprite;

    Alias.WiAtSkSt_setActor = Window_AtbSkillStatus.prototype.setActor;
    Window_AtbSkillStatus.prototype.setActor = function(actor) {
        Alias.WiAtSkSt_setActor.call(this, actor.getMppDrawer());
    };

    AtbSkillStatus.drawGauge = BattleStatus.drawGauge;
    AtbSkillStatus.setupSpritePos = BattleStatus.setupSpritePos;
    AtbSkillStatus.drawActorHp = BattleStatus.drawActorHp;
    AtbSkillStatus.createActorHpDrawer = BattleStatus.createActorHpDrawer;
    AtbSkillStatus.drawActorMp = BattleStatus.drawActorMp;
    AtbSkillStatus.createActorMpDrawer = BattleStatus.createActorMpDrawer;
    AtbSkillStatus.drawActorTp = BattleStatus.drawActorTp;
    AtbSkillStatus.createActorTpDrawer = BattleStatus.createActorTpDrawer;

} // if (MPPlugin.contains['ATB'])

//-----------------------------------------------------------------------------
// Scene_Battle

//29
Alias.ScBa_update = Scene_Battle.prototype.update;
Scene_Battle.prototype.update = function() {
    $gameParty.updateDrawer();
    Alias.ScBa_update.apply(this, arguments);
};



})();


//=============================================================================
// UpdateDrawer
//=============================================================================

(function() {

if (!Window_Base.Mpp_UpdateDrawer) {

    const Alias = {};

    //-----------------------------------------------------------------------------
    // Window_Base

    //13
    Alias.WiBa_initialize = Window_Base.prototype.initialize;
    Window_Base.prototype.initialize = function(x, y, width, height) {
        Alias.WiBa_initialize.apply(this, arguments);
        this._updateDrawers = [];
    };

    //105
    Alias.WiBa_update = Window_Base.prototype.update;
    Window_Base.prototype.update = function() {
        Alias.WiBa_update.apply(this, arguments);
        this.updateDrawer();
    };

    Window_Base.prototype.updateDrawer = function() {
        if (this.isOpen() && this.visible && this._updateDrawers.length > 0) {
            this._updateDrawers = this._updateDrawers.filter( process => process() );
        }
    };

    Window_Base.prototype.addUpdateDrawer = function(process) {
        this._updateDrawers.push(process);
    };

    Window_Base.prototype.clearUpdateDrawer = function() {
        this._updateDrawers = [];
    };

    Window_Base.Mpp_UpdateDrawer = 1;
} //if (!Window_Base.MPP_UpdateDrawer)


})();
