/*:
 * @target MZ
 * @plugindesc [Fixed] Weighted random selection plugin (save-compatible)
 * @author YourName (Fixed)
 * @url https://your-website.com
 *
 * @help
 * ============================================================================
 * Usage
 * ============================================================================
 * 
 * Randomly selects one value from 1,2,3,4 with custom probability weights.
 * Once selected, the value is locked for that specific event.
 * 
 * **Key Features**
 * - Map ID and Event ID are automatically appended to event keys
 * - Copied events will have different results from each other
 * - Save/Load compatible
 * 
 * ============================================================================
 * 
 * @param autoAppendEventId
 * @text Auto Append Event ID
 * @desc true: Automatically append map/event ID to keys (recommended)
 * @type boolean
 * @default true
 *
 * @param enableDebugLog
 * @text Enable Debug Logging
 * @desc Output console logs (use only during development)
 * @type boolean
 * @default false
 *
 * @command setAndExecute
 * @text Set Weights + Execute
 * @desc Set probabilities and execute random selection in one command
 *
 * @arg eventKey
 * @text Event Key
 * @desc Unique key to identify the event
 * @type string
 * @default event1
 *
 * @arg weight1
 * @text Weight 1 (%)
 * @desc Probability for selecting 1
 * @type number
 * @min 0
 * @max 100
 * @default 25
 *
 * @arg weight2
 * @text Weight 2 (%)
 * @desc Probability for selecting 2
 * @type number
 * @min 0
 * @max 100
 * @default 25
 *
 * @arg weight3
 * @text Weight 3 (%)
 * @desc Probability for selecting 3
 * @type number
 * @min 0
 * @max 100
 * @default 25
 *
 * @arg weight4
 * @text Weight 4 (%)
 * @desc Probability for selecting 4
 * @type number
 * @min 0
 * @max 100
 * @default 25
 *
 * @arg variableId
 * @text Result Variable ID
 * @desc Variable to store the selected value (1,2,3,4)
 * @type variable
 * @default 1
 *
 * @command setWeights
 * @text Set Probability Weights
 * @desc Set selection probabilities for 1,2,3,4
 *
 * @arg eventKey
 * @text Event Key
 * @desc Unique key to identify the event
 * @type string
 * @default event1
 *
 * @arg weight1
 * @text Weight 1 (%)
 * @type number
 * @min 0
 * @max 100
 * @default 25
 *
 * @arg weight2
 * @text Weight 2 (%)
 * @type number
 * @min 0
 * @max 100
 * @default 25
 *
 * @arg weight3
 * @text Weight 3 (%)
 * @type number
 * @min 0
 * @max 100
 * @default 25
 *
 * @arg weight4
 * @text Weight 4 (%)
 * @type number
 * @min 0
 * @max 100
 * @default 25
 *
 * @command executeWeightedRandom
 * @text Execute Weighted Random
 * @desc Select one from 1,2,3,4 using set probabilities and store in variable
 *
 * @arg eventKey
 * @text Event Key
 * @type string
 * @default event1
 *
 * @arg variableId
 * @text Result Variable ID
 * @type variable
 * @default 1
 *
 * @command resetEvent
 * @text Reset Event
 * @desc Unlock the fixed value and allow random selection again
 *
 * @arg eventKey
 * @text Event Key
 * @type string
 * @default event1
 *
 * @command checkEventStatus
 * @text Check Event Status
 * @desc Check currently selected value and lock status
 *
 * @arg eventKey
 * @text Event Key
 * @type string
 * @default event1
 *
 * @arg variableId
 * @text Result Variable ID
 * @type variable
 * @default 1
 */

(() => {
    'use strict';

    const pluginName = 'CustomProbabilityPlugin';
    const parameters = PluginManager.parameters(pluginName);
    const autoAppendEventId = parameters['autoAppendEventId'] !== 'false';
    const enableDebugLog = parameters['enableDebugLog'] === 'true';

    function log(message) {
        if (enableDebugLog) {
            console.log(`[CustomProb] ${message}`);
        }
    }

    function generateUniqueKey(baseKey, interpreter) {
        if (!autoAppendEventId) return baseKey;

        if (interpreter && interpreter._eventId > 0) {
            const mapId = $gameMap.mapId();
            const eventId = interpreter._eventId;
            return `${baseKey}_m${mapId}_e${eventId}`;
        }

        return baseKey;
    }

    // Save compatible generic object creation functions
    function createEventData(w1, w2, w3, w4) {
        return {
            weights: [w1, w2, w3, w4],
            selectedValue: null,
            locked: false
        };
    }

    const _DataManager_createGameObjects = DataManager.createGameObjects;
    DataManager.createGameObjects = function() {
        _DataManager_createGameObjects.call(this);
        $gameSystem.initWeightedRandomData();
    };

    const _DataManager_makeSaveContents = DataManager.makeSaveContents;
    DataManager.makeSaveContents = function() {
        const contents = _DataManager_makeSaveContents.call(this);
        contents.weightedRandomData = $gameSystem._weightedRandomData;
        log('Save data created');
        return contents;
    };

    const _DataManager_extractSaveContents = DataManager.extractSaveContents;
    DataManager.extractSaveContents = function(contents) {
        _DataManager_extractSaveContents.call(this, contents);
        $gameSystem._weightedRandomData = contents.weightedRandomData || {};
        log('Data loaded');
    };

    Game_System.prototype.initWeightedRandomData = function() {
        if (!this._weightedRandomData) {
            this._weightedRandomData = {};
        }
    };

    Game_System.prototype.setWeights = function(eventKey, w1, w2, w3, w4) {
        this.initWeightedRandomData();
        
        if (!this._weightedRandomData[eventKey]) {
            this._weightedRandomData[eventKey] = createEventData(w1, w2, w3, w4);
            return true;
        }
        
        const data = this._weightedRandomData[eventKey];
        if (!data.locked) {
            data.weights = [w1, w2, w3, w4];
            return true;
        }
        return false;
    };

    Game_System.prototype.executeWeightedRandom = function(eventKey) {
        this.initWeightedRandomData();
        
        if (!this._weightedRandomData[eventKey]) {
            this._weightedRandomData[eventKey] = createEventData(25, 25, 25, 25);
        }
        
        const data = this._weightedRandomData[eventKey];
        
        if (data.locked && data.selectedValue !== null) {
            return data.selectedValue;
        }
        
        const weights = data.weights;
        const totalWeight = weights[0] + weights[1] + weights[2] + weights[3];
        
        if (totalWeight === 0) {
            data.selectedValue = Math.floor(Math.random() * 4) + 1;
        } else {
            let random = Math.random() * totalWeight;
            
            if (random < weights[0]) {
                data.selectedValue = 1;
            } else if (random < weights[0] + weights[1]) {
                data.selectedValue = 2;
            } else if (random < weights[0] + weights[1] + weights[2]) {
                data.selectedValue = 3;
            } else {
                data.selectedValue = 4;
            }
        }
        
        data.locked = true;
        return data.selectedValue;
    };

    Game_System.prototype.resetWeightedRandom = function(eventKey) {
        this.initWeightedRandomData();
        
        if (this._weightedRandomData[eventKey]) {
            this._weightedRandomData[eventKey].selectedValue = null;
            this._weightedRandomData[eventKey].locked = false;
        }
    };

    Game_System.prototype.getWeightedRandomStatus = function(eventKey) {
        this.initWeightedRandomData();
        
        const data = this._weightedRandomData[eventKey];
        if (!data) {
            return { value: 0, locked: false, weights: [25, 25, 25, 25] };
        }
        
        return {
            value: data.selectedValue || 0,
            locked: data.locked,
            weights: data.weights
        };
    };

    PluginManager.registerCommand(pluginName, 'setAndExecute', function(args) {
        const baseKey = args.eventKey;
        const eventKey = generateUniqueKey(baseKey, this);
        const w1 = Number(args.weight1);
        const w2 = Number(args.weight2);
        const w3 = Number(args.weight3);
        const w4 = Number(args.weight4);
        const variableId = Number(args.variableId);
        
        $gameSystem.setWeights(eventKey, w1, w2, w3, w4);
        const result = $gameSystem.executeWeightedRandom(eventKey);
        $gameVariables.setValue(variableId, result);
        
        log(`[Set+Execute] ${eventKey}: result=${result} (${w1}/${w2}/${w3}/${w4}%)`);
    });

    PluginManager.registerCommand(pluginName, 'setWeights', function(args) {
        const baseKey = args.eventKey;
        const eventKey = generateUniqueKey(baseKey, this);
        const w1 = Number(args.weight1);
        const w2 = Number(args.weight2);
        const w3 = Number(args.weight3);
        const w4 = Number(args.weight4);
        
        const success = $gameSystem.setWeights(eventKey, w1, w2, w3, w4);
        log(success ? `[Set] ${eventKey}` : `[Set Failed] ${eventKey} locked`);
    });

    PluginManager.registerCommand(pluginName, 'executeWeightedRandom', function(args) {
        const baseKey = args.eventKey;
        const eventKey = generateUniqueKey(baseKey, this);
        const variableId = Number(args.variableId);
        
        const result = $gameSystem.executeWeightedRandom(eventKey);
        $gameVariables.setValue(variableId, result);
        
        log(`[Execute] ${eventKey}: result=${result}`);
    });

    PluginManager.registerCommand(pluginName, 'resetEvent', function(args) {
        const baseKey = args.eventKey;
        const eventKey = generateUniqueKey(baseKey, this);
        $gameSystem.resetWeightedRandom(eventKey);
        log(`[Reset] ${eventKey}`);
    });

    PluginManager.registerCommand(pluginName, 'checkEventStatus', function(args) {
        const baseKey = args.eventKey;
        const eventKey = generateUniqueKey(baseKey, this);
        const variableId = Number(args.variableId);
        
        const status = $gameSystem.getWeightedRandomStatus(eventKey);
        $gameVariables.setValue(variableId, status.value);
        
        log(`[Check] ${eventKey}: value=${status.value}, locked=${status.locked}`);
    });

})();