/*:
@target MZ
@plugindesc QTE timing bar with dynamic difficulty control
@author hiz,munokura,enhanced

@param bar width
@type number
@default 500
@min 200
@max 800

@param hit area size
@type number
@default 20
@min 10
@max 50

@param critical area size
@type number
@default 10
@min 5
@max 20

@param cursor speed
@type number
@decimals 1
@default 1.0
@min 0.1
@max 5.0

@param random seed
@type number
@default -1

@param easy switch
@type switch
@default 0

@param hard switch
@type switch
@default 0

@param expert switch
@type switch
@default 0

@param hit size variable
@type variable
@default 0

@param critical size variable
@type variable
@default 0

@param easy multiplier
@type number
@decimals 2
@default 1.5

@param hard multiplier
@type number
@decimals 2
@default 0.7

@param expert multiplier
@type number
@decimals 2
@default 0.5

@param enable caching
@type boolean
@default true

@param max cached bitmaps
@type number
@default 10

@param required SE
@type file
@default Decision2
@dir audio/se/

@param hit SE
@type file
@default Attack2
@dir audio/se/

@param critical SE
@type file
@default Attack3
@dir audio/se/

@param miss SE
@type file
@default Buzzer1
@dir audio/se/

@command HzTimingBar
@text Execute Timing Bar

@arg varNumber
@type variable
@default 0

@arg requireArea
@default

@arg x
@type number
@default -1

@arg y
@type number
@default -1

@arg useRandomAreas
@type boolean
@default true

@arg hitArea
@default 60-80

@arg criticalArea
@default 68-72

@arg customSpeed
@type number
@decimals 1

@arg checkPrevious
@type boolean
@default false

@arg ignoreDifficulty
@type boolean
@default false
*/

(() => {
    'use strict';
    
    const PLUGIN_NAME = document.currentScript.src.split("/").pop().replace(/\.js$/, "");
    const p = PluginManager.parameters(PLUGIN_NAME);
    
    const CFG = {
        W: Number(p['bar width']) || 500,
        HIT: Number(p['hit area size']) || 20,
        CRIT: Number(p['critical area size']) || 10,
        SPD: Number(p['cursor speed']) || 1.0,
        SEED: Number(p['random seed']),
        CACHE: p['enable caching'] !== 'false',
        MAX_CACHE: Number(p['max cached bitmaps']) || 10,
        SW: {
            EASY: Number(p['easy switch']) || 0,
            HARD: Number(p['hard switch']) || 0,
            EXP: Number(p['expert switch']) || 0
        },
        VAR: {
            HIT: Number(p['hit size variable']) || 0,
            CRIT: Number(p['critical size variable']) || 0
        },
        MULT: {
            EASY: Number(p['easy multiplier']) || 1.5,
            HARD: Number(p['hard multiplier']) || 0.7,
            EXP: Number(p['expert multiplier']) || 0.5
        },
        SE: {
            REQ: p['required SE'] || 'Decision2',
            HIT: p['hit SE'] || 'Attack2',
            CRIT: p['critical SE'] || 'Attack3',
            MISS: p['miss SE'] || 'Buzzer1'
        }
    };
    
    const C = {
        MAX: 100,
        H: 10,
        R: 4,
        CW: 18,
        CH: 32,
        COL: {
            FRAME: { F: "#FFF", S: "#000" },
            REQ: "#43D197",
            HIT: "#EBCE41",
            CRIT: "#E47237",
            CUR: { F: "#000", S: "#FFF" }
        }
    };

    class BitmapCache {
        constructor() {
            this._c = new Map();
            this._u = new Map();
        }
        
        get(t, w, h, col, fn) {
            if (!CFG.CACHE) return fn();
            const k = `${t}_${w}_${h}_${col||''}`;
            if (this._c.has(k)) {
                this._u.set(k, (this._u.get(k) || 0) + 1);
                return this._c.get(k);
            }
            if (this._c.size >= CFG.MAX_CACHE) {
                let min = Infinity, lk = null;
                for (const [key, u] of this._u) {
                    if (u < min) { min = u; lk = key; }
                }
                if (lk) { this._c.delete(lk); this._u.delete(lk); }
            }
            const b = fn();
            this._c.set(k, b);
            this._u.set(k, 1);
            return b;
        }
        
        clear() { this._c.clear(); this._u.clear(); }
    }

    const cache = new BitmapCache();

    class SeededRandom {
        constructor(s = -1) {
            this._s = s === -1 ? (Date.now() + Math.random() * 1e6) % 2147483647 : s;
            this._c = this._s;
        }
        next() { return (this._c = (this._c * 16807) % 2147483647) / 2147483647; }
        range(min, max) { return Math.floor(this.next() * (max - min + 1)) + min; }
    }

    const U = {
        pa: (s) => s ? s.split("-").map(Number) : null,
        pra: (s) => s ? s.split(",").map(a => a.trim().split("-").map(Number)) : [],
        ovl: (a1, as) => as.some(a2 => !(a1[1] <= a2[0] || a1[0] >= a2[1])),
        
        gm: (t, ig) => {
            if (ig) return 1.0;
            let m = 1.0;
            if (CFG.SW.EXP > 0 && $gameSwitches.value(CFG.SW.EXP)) m = CFG.MULT.EXP;
            else if (CFG.SW.HARD > 0 && $gameSwitches.value(CFG.SW.HARD)) m = CFG.MULT.HARD;
            else if (CFG.SW.EASY > 0 && $gameSwitches.value(CFG.SW.EASY)) m = CFG.MULT.EASY;
            return Math.max(0.2, Math.min(3.0, m));
        },
        
        ads: (b, t, ig) => {
            const m = U.gm(t, ig);
            let result = b * m;
            
            const v = t === 'hit' ? CFG.VAR.HIT : CFG.VAR.CRIT;
            if (v > 0) {
                const val = $gameVariables.value(v);
                if (val > 0) {
                    result = result * (val / 100);
                }
            }
            
            const n = Math.floor(result);
            // Min values: hit 2, critical 1
            return t === 'hit' ? Math.max(2, Math.min(80, n)) : Math.max(1, Math.min(40, n));
        },
        
        gnoa: (oc, sz, r) => {
            for (let i = 0; i < 50; i++) {
                const st = r.range(0, C.MAX - sz);
                const a = [st, st + sz];
                if (!U.ovl(a, oc)) return a;
            }
            return [Math.max(0, C.MAX - sz - 5), C.MAX - 5];
        },
        
        gra: (req, hs, cs, r) => {
            const ha = U.gnoa(req, hs, r);
            const hc = (ha[0] + ha[1]) / 2;
            const ch = cs / 2;
            let ca = [Math.floor(hc - ch), Math.ceil(hc + ch)];
            ca[0] = Math.max(ha[0], ca[0]);
            ca[1] = Math.min(ha[1], ca[1]);
            if (ca[1] - ca[0] < 1) {
                const c = (ca[0] + ca[1]) / 2;
                ca = [Math.max(ha[0], Math.round(c - 1)), Math.min(ha[1], Math.round(c + 1))];
            }
            return { hitArea: ha, criticalArea: ca };
        },
        
        ps: (n) => {
            if (n) AudioManager.playSe({ name: n, volume: 90, pitch: 100, pan: 0 });
        }
    };

    class TimingBar {
        constructor(x, y, ha, ca, ra, vn, sp, cp) {
            this._ha = ha;
            this._ca = ca || [];
            this._ra = ra || [];
            this._vn = vn;
            this._sp = sp || CFG.SPD;
            this._f = 0;
            this._d = 1;
            this._hr = new Array(this._ra.length).fill(false);
            this._ct = null;
            this._cu = null;
            this._lx = -1;
            this._cp = cp;
            this._pm = false;
            
            if (this._cp) {
                this._pm = ($gameVariables.value(this._vn) === 0);
            }
            $gameVariables.setValue(this._vn, 0);
            this._cd(x, y);
        }
        
        _cd(x, y) {
            this._ct = new Sprite();
            this._ct.position.set(x - CFG.W / 2, y - C.H / 2);
            this._cf();
            this._cas();
            this._ccu();
            SceneManager._scene._spriteset.addChild(this._ct);
        }
        
        _cf() {
            const b = cache.get('f', CFG.W + 4, C.H + 4, null, () => {
                const bm = new Bitmap(CFG.W + 4, C.H + 4);
                const cx = bm.context;
                this._rr(cx, 2, 2, CFG.W, C.H);
                cx.fillStyle = C.COL.FRAME.F;
                cx.strokeStyle = C.COL.FRAME.S;
                cx.lineWidth = 2;
                cx.fill();
                cx.stroke();
                return bm;
            });
            const s = new Sprite(b);
            s.position.set(-2, -2);
            this._ct.addChild(s);
        }
        
        _cas() {
            this._ra.forEach(a => this._ca2(a, C.COL.REQ));
            if (this._ha && this._ha.length === 2) this._ca2(this._ha, C.COL.HIT);
            if (this._ca && this._ca.length === 2 && this._ca[1] > this._ca[0]) {
                this._ca2(this._ca, C.COL.CRIT);
            }
        }
        
        _ca2(a, col) {
            const w = this._gaw(a);
            if (w <= 0) return;
            const b = cache.get('a', Math.round(w), C.H - 1, col, () => {
                const bm = new Bitmap(Math.ceil(w), C.H - 1);
                bm.context.fillStyle = col;
                bm.context.fillRect(0, 1, Math.ceil(w), C.H - 1);
                return bm;
            });
            const s = new Sprite(b);
            s.position.set(this._gap(a[0]), 0);
            this._ct.addChild(s);
        }
        
        _ccu() {
            const th = C.CH + C.H + 2;
            const b = cache.get('c', C.CW, th, null, () => {
                const bm = new Bitmap(C.CW, th);
                const cx = bm.context;
                cx.fillStyle = C.COL.CUR.F;
                cx.strokeStyle = C.COL.CUR.S;
                cx.lineWidth = 1;
                this._tr(cx, 1, 1, 17, 1, 9, 17);
                const by = C.H + 17;
                this._tr(cx, 9, by, 17, by + 16, 1, by + 16);
                return bm;
            });
            this._cu = new Sprite(b);
            this._cu.position.set(-9, -17);
            this._ct.addChild(this._cu);
        }
        
        _tr(cx, x1, y1, x2, y2, x3, y3) {
            cx.beginPath();
            cx.moveTo(x1, y1);
            cx.lineTo(x2, y2);
            cx.lineTo(x3, y3);
            cx.closePath();
            cx.fill();
            cx.stroke();
        }
        
        _rr(cx, x, y, w, h) {
            const r = C.R;
            cx.beginPath();
            cx.moveTo(x + r, y);
            cx.arcTo(x + w, y, x + w, y + h, r);
            cx.arcTo(x + w, y + h, x, y + h, r);
            cx.arcTo(x, y + h, x, y, r);
            cx.arcTo(x, y, x + w, y, r);
            cx.closePath();
        }
        
        _gaw(a) { return CFG.W * (a[1] - a[0]) / C.MAX; }
        _gap(st) { return CFG.W * st / C.MAX; }
        _ia(a) { return a && a.length === 2 && a[0] <= this._f && this._f < a[1]; }
        _arh() { return this._hr.every(h => h); }
        
        update() {
            if ($gameTimer.isWorking() && $gameTimer._frames === 0) {
                U.ps(CFG.SE.MISS);
                $gameVariables.setValue(this._vn, 0);
                return false;
            }
            this._uf();
            this._uc();
            if (Input.isTriggered('ok') || TouchInput.isTriggered()) {
                return this._pi();
            }
            return true;
        }
        
        _uf() {
            this._f += this._sp * this._d;
            if (this._f >= C.MAX) { this._d = -1; this._f = C.MAX; }
            else if (this._f <= 0) { this._d = 1; this._f = 0; }
        }
        
        _uc() {
            const p = this._gap(this._f) - C.CW / 2;
            const nx = Math.max(-C.CW / 2, Math.min(p, CFG.W - C.CW / 2));
            if (Math.abs(nx - this._lx) > 0.5) {
                this._cu.x = nx;
                this._lx = nx;
            }
        }
        
        _pi() {
            for (let i = 0; i < this._ra.length; i++) {
                if (this._ia(this._ra[i])) {
                    this._hr[i] = true;
                    U.ps(CFG.SE.REQ);
                    return true;
                }
            }
            const rc = this._ra.length === 0 || this._arh();
            
            const hasCritical = this._ca && this._ca.length === 2 && this._ca[1] > this._ca[0];
            
            if (hasCritical && this._ia(this._ca)) {
                if (rc) {
                    U.ps(CFG.SE.CRIT);
                    $gameVariables.setValue(this._vn, (this._cp && this._pm) ? 3 : 2);
                    return false;
                }
            } else if (this._ia(this._ha)) {
                if (rc) {
                    U.ps(CFG.SE.HIT);
                    $gameVariables.setValue(this._vn, (this._cp && this._pm) ? 3 : 1);
                    return false;
                }
            }
            U.ps(CFG.SE.MISS);
            $gameVariables.setValue(this._vn, 0);
            return false;
        }
        
        terminate() {
            if (this._ct && this._ct.parent) {
                while (this._ct.children.length > 0) this._ct.removeChildAt(0);
                SceneManager._scene._spriteset.removeChild(this._ct);
                this._ct = null;
                this._cu = null;
            }
        }
    }

    PluginManager.registerCommand(PLUGIN_NAME, "HzTimingBar", function(args) {
        this.setWaitMode("hzTimingBar");
        const vn = Number(args.varNumber) || 0;
        const ra = U.pra(args.requireArea);
        const x = Number(args.x) < 0 ? Graphics.width / 2 : Number(args.x);
        const y = Number(args.y) < 0 ? Graphics.height / 2 : Number(args.y);
        const ur = args.useRandomAreas === 'true';
        const cs = args.customSpeed ? Number(args.customSpeed) : CFG.SPD;
        const cp = args.checkPrevious === 'true';
        const ig = args.ignoreDifficulty === 'true';
        
        const ahs = U.ads(CFG.HIT, 'hit', ig);
        const acs = U.ads(CFG.CRIT, 'critical', ig);
        
        let ha, ca;
        if (ur) {
            const r = new SeededRandom(CFG.SEED);
            const as = U.gra(ra, ahs, acs, r);
            ha = as.hitArea;
            ca = as.criticalArea;
        } else {
            ha = U.pa(args.hitArea) || [60, 80];
            const pc = U.pa(args.criticalArea);
            if (pc) {
                ca = pc;
            } else {
                const hc = (ha[0] + ha[1]) / 2;
                const ch = acs / 2;
                ca = [Math.floor(hc - ch), Math.ceil(hc + ch)];
                ca[0] = Math.max(ha[0], ca[0]);
                ca[1] = Math.min(ha[1], ca[1]);
            }
        }
        this._timingBar = new TimingBar(x, y, ha, ca, ra, vn, cs, cp);
    });

    const _GI_uwm = Game_Interpreter.prototype.updateWaitMode;
    Game_Interpreter.prototype.updateWaitMode = function() {
        if (this._waitMode === 'hzTimingBar' && this._timingBar) {
            const w = this._timingBar.update();
            if (!w) {
                this._timingBar.terminate();
                this._timingBar = null;
                this._waitMode = '';
            }
            return w;
        }
        return _GI_uwm.call(this);
    };

    const _SB_t = Scene_Base.prototype.terminate;
    Scene_Base.prototype.terminate = function() {
        _SB_t.call(this);
        if (this.constructor === Scene_Map) cache.clear();
    };
})();