/*:
 * @target MZ
 * @plugindesc Randomly spawns events from other maps
 * @author YourName
 *
 * @command spawnRandomEvent
 * @text Spawn Random Event
 * @desc Spawn a random event from the specified list
 *
 * @arg eventList
 * @text Event List
 * @desc Format: MapID:EventID (comma separated). Example: 1:5,2:10,3:1-50
 * @default 1:1-50
 *
 * @arg spawnRate
 * @text Spawn Rate (%)
 * @type number
 * @min 0
 * @max 100
 * @default 100
 *
 * @arg pageSwitch
 * @text Page Switch Settings
 * @desc Format: SwitchID:PageNum:SelfSwitch. Example: 41:4:C
 * @default
 *
 * @command resetAllEvents
 * @text Reset All Events
 * @desc Completely resets all dynamic events and self switches (for lobby return)
 *
 * @help
 * ============================================================================
 * Plugin Commands
 * 
 * 1. Spawn Random Event
 *    - Randomly selects and spawns an event from the list
 * 
 * 2. Reset All Events
 *    - Use when returning to lobby with 0 HP
 *    - Completely removes all dynamic events and self switches
 * 
 * ============================================================================
 */
(function() {
    'use strict';
    
    const mapCache = {};
    const spawning = new Set();
    
    function loadMap(mapId, callback) {
        if (mapId === $gameMap._mapId) {
            callback($dataMap);
            return;
        }
        if (mapCache[mapId]) {
            callback(mapCache[mapId]);
            return;
        }
        
        const xhr = new XMLHttpRequest();
        xhr.open('GET', 'data/Map%1.json'.format(mapId.padZero(3)));
        xhr.overrideMimeType('application/json');
        xhr.onload = () => {
            if (xhr.status < 400) {
                mapCache[mapId] = JSON.parse(xhr.responseText);
                callback(mapCache[mapId]);
            }
        };
        xhr.send();
    }
    
    function parseList(str) {
        const result = [];
        str.split(',').forEach(item => {
            item = item.trim();
            if (item.includes(':')) {
                const [map, event] = item.split(':');
                if (event.includes('-')) {
                    const [start, end] = event.split('-').map(Number);
                    for (let i = start; i <= end; i++) {
                        result.push(map + ':' + i);
                    }
                } else {
                    result.push(item);
                }
            }
        });
        return result;
    }
    
    function parsePageSwitch(str) {
        const result = {};
        if (!str || str.trim() === '') return result;
        
        str.split(',').forEach(item => {
            item = item.trim();
            if (item.includes(':')) {
                const parts = item.split(':');
                const switchId = Number(parts[0]);
                const pageNum = Number(parts[1]);
                const selfSwitch = parts[2] || null;
                
                if (switchId > 0 && pageNum > 0) {
                    result[switchId] = {
                        pageIndex: pageNum - 1,
                        selfSwitch: selfSwitch
                    };
                }
            }
        });
        return result;
    }
    
    const _Game_Map_setupEvents = Game_Map.prototype.setupEvents;
    Game_Map.prototype.setupEvents = function() {
        _Game_Map_setupEvents.call(this);
        
        if (!this._dynamicEventIds) {
            this._dynamicEventIds = [];
        }
        if (!this._dynamicEventData) {
            this._dynamicEventData = {};
        }
        if (!this._dynamicEventPageSwitch) {
            this._dynamicEventPageSwitch = {};
        }
        if (!$gameSystem._dynamicEventsByMap) {
            $gameSystem._dynamicEventsByMap = {};
        }
        if (!$gameSystem._nextDynamicEventId) {
            $gameSystem._nextDynamicEventId = 1000;
        }
    };
    
    const _Game_Map_setup = Game_Map.prototype.setup;
    Game_Map.prototype.setup = function(mapId) {
        if (this._mapId && this._dynamicEventIds && this._dynamicEventIds.length > 0) {
            const oldMapId = this._mapId;
            console.log(`[Map Change] Removing ${this._dynamicEventIds.length} dynamic events from map ${oldMapId}`);
            
            for (let i = this._dynamicEventIds.length - 1; i >= 0; i--) {
                const eventId = this._dynamicEventIds[i];
                
                const keys = Object.keys($gameSelfSwitches._data);
                for (const key of keys) {
                    if (key.startsWith(`[${oldMapId},${eventId},`)) {
                        delete $gameSelfSwitches._data[key];
                    }
                }
                
                delete $dataMap.events[eventId];
                delete this._events[eventId];
            }
        }
        
        this._dynamicEventIds = [];
        this._dynamicEventData = {};
        this._dynamicEventPageSwitch = {};
        
        _Game_Map_setup.call(this, mapId);
        
        if (!$gameSystem._dynamicEventsByMap) {
            $gameSystem._dynamicEventsByMap = {};
        }
        
        const previousDynamicIds = $gameSystem._dynamicEventsByMap[mapId] || [];
        if (previousDynamicIds.length > 0) {
            console.log(`[Map Load] Clearing ${previousDynamicIds.length} previous dynamic events from map ${mapId}`);
            
            for (const eventId of previousDynamicIds) {
                const keys = Object.keys($gameSelfSwitches._data);
                for (const key of keys) {
                    if (key.startsWith(`[${mapId},${eventId},`)) {
                        delete $gameSelfSwitches._data[key];
                        console.log(`[Map Load] Deleted self switch: ${key}`);
                    }
                }
            }
        }
        
        $gameSystem._dynamicEventsByMap[mapId] = [];
        
        console.log(`[Map Load] Map ${mapId} fully initialized`);
    };
    
    const _Game_Event_event = Game_Event.prototype.event;
    Game_Event.prototype.event = function() {
        if ($gameMap._dynamicEventData && $gameMap._dynamicEventData[this._eventId]) {
            return $gameMap._dynamicEventData[this._eventId];
        }
        
        const event = _Game_Event_event.call(this);
        if (!event) {
            return { id: this._eventId, name: "", x: 0, y: 0, pages: [] };
        }
        if (!event.pages) {
            event.pages = [];
        }
        return event;
    };
    
    const _Game_Event_initialize = Game_Event.prototype.initialize;
    Game_Event.prototype.initialize = function(mapId, eventId) {
        _Game_Event_initialize.call(this, mapId, eventId);
        
        if ($gameMap._dynamicEventData && $gameMap._dynamicEventData[eventId]) {
            this._isDynamicEvent = true;
        }
    };
    
    const _Game_Event_findProperPageIndex = Game_Event.prototype.findProperPageIndex;
    Game_Event.prototype.findProperPageIndex = function() {
        const event = this.event();
        if (!event || !event.pages || event.pages.length === 0) {
            return -1;
        }
        
        if ($gameMap._dynamicEventPageSwitch && $gameMap._dynamicEventPageSwitch[this._eventId]) {
            const pageSwitches = $gameMap._dynamicEventPageSwitch[this._eventId];
            const sortedSwitches = Object.keys(pageSwitches).map(Number).sort((a, b) => b - a);
            
            for (const switchId of sortedSwitches) {
                if ($gameSwitches && $gameSwitches.value(switchId) === true) {
                    const config = pageSwitches[switchId];
                    const pageIndex = config.pageIndex;
                    
                    if (pageIndex >= 0 && pageIndex < event.pages.length) {
                        if (config.selfSwitch) {
                            continue;
                        }
                        return pageIndex;
                    }
                }
            }
        }
        
        return _Game_Event_findProperPageIndex.call(this);
    };
    
    const _Game_Event_refresh = Game_Event.prototype.refresh;
    Game_Event.prototype.refresh = function() {
        const event = this.event();
        if (!event || !event.pages) {
            return;
        }
        _Game_Event_refresh.call(this);
    };
    
    PluginManager.registerCommand("RandomEventSpawn", "spawnRandomEvent", function(args) {
        const trigger = $gameMap.event(this._eventId);
        if (!trigger) {
            console.log('[Spawn] No trigger found');
            return;
        }
        
        const x = trigger.x;
        const y = trigger.y;
        const key = x + ',' + y;
        
        console.log(`[Spawn] Start: (${x},${y}), TriggerID: ${this._eventId}`);
        
        if (Math.random() * 100 > Number(args.spawnRate || 100)) {
            console.log('[Spawn] Spawn rate check failed');
            trigger.erase();
            return;
        }
        
        const allEvents = $gameMap.eventsXy(x, y);
        console.log(`[Spawn] Events at (${x},${y}): ${allEvents.length}`);
        
        allEvents.forEach(e => {
            const isDynamic = $gameMap._dynamicEventIds && $gameMap._dynamicEventIds.includes(e._eventId);
            console.log(`  - ID:${e._eventId}, Dynamic:${isDynamic}, Trigger:${e._eventId === this._eventId}`);
        });
        
        const existing = allEvents.filter(e => {
            return e._eventId !== this._eventId && 
                   $gameMap._dynamicEventIds && 
                   $gameMap._dynamicEventIds.includes(e._eventId);
        });
        
        if (existing.length > 0) {
            console.log(`[Spawn] Removing ${existing.length} existing dynamic events`);
            for (const event of existing) {
                const eventId = event._eventId;
                console.log(`  - Removing: ID ${eventId}`);
                
                ['A', 'B', 'C', 'D'].forEach(letter => {
                    const delKey = [$gameMap._mapId, eventId, letter];
                    delete $gameSelfSwitches._data[delKey];
                });
                
                delete $dataMap.events[eventId];
                delete $gameMap._events[eventId];
                
                const idx = $gameMap._dynamicEventIds.indexOf(eventId);
                if (idx > -1) {
                    $gameMap._dynamicEventIds.splice(idx, 1);
                }
                delete $gameMap._dynamicEventData[eventId];
                delete $gameMap._dynamicEventPageSwitch[eventId];
            }
            console.log('[Spawn] Existing events removed');
        }
        
        if (spawning.has(key)) {
            console.log('[Spawn] Already spawning');
            trigger.erase();
            return;
        }
        
        spawning.add(key);
        
        const list = parseList(args.eventList);
        const pick = list[Math.floor(Math.random() * list.length)];
        const [mapId, eventId] = pick.split(':').map(Number);
        
        console.log(`[Spawn] Selected event: Map${mapId}, Event${eventId}`);
        
        const keys = Object.keys($gameSelfSwitches._data);
        for (const key of keys) {
            if (key.startsWith(`[${mapId},${eventId},`)) {
                delete $gameSelfSwitches._data[key];
                console.log(`[Spawn] Deleted original self switch: ${key}`);
            }
        }
        
        loadMap(mapId, mapData => {
            const template = mapData?.events?.[eventId];
            if (!template || !template.pages) {
                console.warn('RandomEventSpawn: Invalid event', mapId, eventId);
                spawning.delete(key);
                trigger.erase();
                return;
            }
            
            try {
                const newId = $gameSystem._nextDynamicEventId++;
                
                console.log(`[Spawn] New event ID: ${newId}, Original: Map${mapId}-Event${eventId}`);
                
                const keys = Object.keys($gameSelfSwitches._data);
                const keysToDelete = [];
                
                for (const key of keys) {
                    if (key.startsWith(`[${$gameMap._mapId},${newId},`)) {
                        keysToDelete.push(key);
                    }
                    if (key.startsWith(`[${mapId},${eventId},`)) {
                        keysToDelete.push(key);
                    }
                }
                
                for (const key of keysToDelete) {
                    delete $gameSelfSwitches._data[key];
                    console.log(`[Spawn] Deleted self switch: ${key}`);
                }
                
                console.log(`[Spawn] Total ${keysToDelete.length} self switches cleared`);
                
                const newEvent = JSON.parse(JSON.stringify(template));
                newEvent.id = newId;
                newEvent.x = x;
                newEvent.y = y;
                
                if (!newEvent.pages) {
                    newEvent.pages = [];
                }
                
                if (!$gameMap._dynamicEventData) {
                    $gameMap._dynamicEventData = {};
                }
                $gameMap._dynamicEventData[newId] = newEvent;
                
                $dataMap.events[newId] = newEvent;
                
                const gameEvent = new Game_Event($gameMap._mapId, newId);
                $gameMap._events[newId] = gameEvent;
                
                if (!$gameMap._dynamicEventIds) {
                    $gameMap._dynamicEventIds = [];
                }
                $gameMap._dynamicEventIds.push(newId);
                
                if (!$gameSystem._dynamicEventsByMap) {
                    $gameSystem._dynamicEventsByMap = {};
                }
                if (!$gameSystem._dynamicEventsByMap[$gameMap._mapId]) {
                    $gameSystem._dynamicEventsByMap[$gameMap._mapId] = [];
                }
                $gameSystem._dynamicEventsByMap[$gameMap._mapId].push(newId);
                console.log(`[Spawn] Recorded dynamic event ID ${newId} for map ${$gameMap._mapId}`);
                
                const pageSwitchStr = args.pageSwitch || '';
                if (pageSwitchStr.trim() !== '') {
                    const pageSwitches = parsePageSwitch(pageSwitchStr);
                    if (Object.keys(pageSwitches).length > 0) {
                        if (!$gameMap._dynamicEventPageSwitch) {
                            $gameMap._dynamicEventPageSwitch = {};
                        }
                        $gameMap._dynamicEventPageSwitch[newId] = pageSwitches;
                        
                        for (const switchId in pageSwitches) {
                            const id = Number(switchId);
                            if ($gameSwitches.value(id) === true) {
                                const config = pageSwitches[id];
                                if (config.selfSwitch) {
                                    const selfKey = [$gameMap._mapId, newId, config.selfSwitch];
                                    $gameSelfSwitches.setValue(selfKey, true);
                                    console.log(`[Switch #${id} ON] Event ${newId} self switch ${config.selfSwitch} turned on`);
                                }
                            }
                        }
                    }
                }
                
                gameEvent.refresh();
                
                const scene = SceneManager._scene;
                if (scene instanceof Scene_Map && scene._spriteset) {
                    const sprites = scene._spriteset._characterSprites;
                    if (sprites) {
                        const sprite = new Sprite_Character(gameEvent);
                        sprites.push(sprite);
                        scene._spriteset._tilemap.addChild(sprite);
                    }
                }
                
                trigger.erase();
            } catch (e) {
                console.error('RandomEventSpawn Error:', e);
            } finally {
                spawning.delete(key);
            }
        });
    });
    
    const _Game_Switches_setValue = Game_Switches.prototype.setValue;
    Game_Switches.prototype.setValue = function(switchId, value) {
        const oldValue = this._data[switchId];
        
        _Game_Switches_setValue.call(this, switchId, value);
        
        if (oldValue === value) return;
        
        if (value === true && $gameMap && $gameMap._dynamicEventPageSwitch) {
            for (const eventId in $gameMap._dynamicEventPageSwitch) {
                const pageSwitches = $gameMap._dynamicEventPageSwitch[eventId];
                
                if (pageSwitches[switchId]) {
                    const config = pageSwitches[switchId];
                    const event = $gameMap._events[Number(eventId)];
                    
                    if (event && config.selfSwitch) {
                        const key = [$gameMap._mapId, Number(eventId), config.selfSwitch];
                        $gameSelfSwitches.setValue(key, true);
                        console.log(`[Switch #${switchId} ON] Event ${eventId} self switch ${config.selfSwitch} turned on`);
                        
                        event.refresh();
                    }
                }
            }
        }
    };
    
    const _Game_SelfSwitches_setValue = Game_SelfSwitches.prototype.setValue;
    Game_SelfSwitches.prototype.setValue = function(key, value) {
        _Game_SelfSwitches_setValue.call(this, key, value);
        
        if ($gameMap && $gameMap._dynamicEventIds && Array.isArray(key)) {
            const eventId = key[1];
            if ($gameMap._dynamicEventIds.includes(eventId)) {
                const event = $gameMap._events[eventId];
                if (event) {
                    event.refresh();
                }
            }
        }
    };
    
    const _Scene_Map_onMapLoaded = Scene_Map.prototype.onMapLoaded;
    Scene_Map.prototype.onMapLoaded = function() {
        _Scene_Map_onMapLoaded.call(this);
        spawning.clear();
    };
    
    PluginManager.registerCommand("RandomEventSpawn", "resetAllEvents", function(args) {
        console.log('[Full Reset] Start');
        
        $gameSelfSwitches._data = {};
        console.log('[Full Reset] All self switches deleted');
        
        if ($gameSystem._dynamicEventsByMap) {
            $gameSystem._dynamicEventsByMap = {};
            console.log('[Full Reset] Dynamic event records deleted');
        }
        
        if ($gameMap._dynamicEventIds && $gameMap._dynamicEventIds.length > 0) {
            for (let i = $gameMap._dynamicEventIds.length - 1; i >= 0; i--) {
                const eventId = $gameMap._dynamicEventIds[i];
                delete $dataMap.events[eventId];
                delete $gameMap._events[eventId];
            }
            $gameMap._dynamicEventIds = [];
            $gameMap._dynamicEventData = {};
            $gameMap._dynamicEventPageSwitch = {};
            console.log('[Full Reset] Current map dynamic events removed');
        }
        
        console.log('[Full Reset] Complete!');
    });
})();