//=============================================================================
// RPG Maker MZ - Text Picture (Cached Version)
//=============================================================================

/*:
 * @target MZ
 * @plugindesc Displays text as a picture with bitmap caching.
 * @author Yoji Ojima (Modified with caching)
 *
 * @help TextPicture.js
 *
 * This plugin provides a command to show text as a picture.
 * Modified version with bitmap caching for better performance.
 *
 * Use it in the following procedure.
 *   1. Call the plugin command "Set Text Picture".
 *   2. Execute "Show Picture" without specifying an image.
 *
 * @command set
 * @text Set Text Picture
 * @desc Sets text to display as a picture.
 *       After this, execute "Show Picture" without specifying an image.
 *
 * @arg text
 * @type multiline_string
 * @text Text
 * @desc Text to display as a picture.
 *       Control characters are allowed.
 */

/*:ja
 * @target MZ
 * @plugindesc テキストをピクチャとして表示します（キャッシュ版）。
 * @author Yoji Ojima (Modified with caching)
 *
 * @help TextPicture.js
 *
 * このプラグインは、テキストをピクチャとして表示するコマンドを提供します。
 * ビットマップキャッシュによりパフォーマンスが向上しています。
 *
 * 次の手順で使用してください。
 *   1. プラグインコマンド「テキストピクチャの設定」を呼び出します。
 *   2. 画像を指定せずに「ピクチャの表示」を実行します。
 *
 * @command set
 * @text テキストピクチャの設定
 * @desc ピクチャとして表示するテキストを設定します。
 *       この後、画像を指定せずに「ピクチャの表示」を実行してください。
 *
 * @arg text
 * @type multiline_string
 * @text テキスト
 * @desc ピクチャとして表示するテキストです。
 *       制御文字が使用可能です。
 */

(() => {
    const pluginName = "TextPicture";
    let textPictureText = "";
    
    // ビットマップキャッシュ
    const bitmapCache = new Map();
    const MAX_CACHE_SIZE = 50;

    PluginManager.registerCommand(pluginName, "set", args => {
        textPictureText = String(args.text);
    });

    const _Game_Picture_show = Game_Picture.prototype.show;
    Game_Picture.prototype.show = function() {
        _Game_Picture_show.apply(this, arguments);
        if (this._name === "" && textPictureText) {
            this.mzkp_text = textPictureText;
            this.mzkp_textChanged = true;
            textPictureText = "";
        }
    };

    const _Sprite_Picture_destroy = Sprite_Picture.prototype.destroy;
    Sprite_Picture.prototype.destroy = function() {
        // キャッシュ版なので個別の destroy は不要
        _Sprite_Picture_destroy.apply(this, arguments);
    };

    const _Sprite_Picture_updateBitmap = Sprite_Picture.prototype.updateBitmap;
    Sprite_Picture.prototype.updateBitmap = function() {
        _Sprite_Picture_updateBitmap.apply(this, arguments);
        if (this.visible && this._pictureName === "") {
            const picture = this.picture();
            if (!picture) {
                this.mzkp_text = "";
                return;
            }
            
            const text = picture.mzkp_text || "";
            const textChanged = picture.mzkp_textChanged;
            
            // テキストが同じなら何もしない
            if (this.mzkp_text === text && !textChanged) {
                return;
            }
            
            this.mzkp_text = text;
            this.bitmap = getCachedBitmap(text);
            picture.mzkp_textChanged = false;
        } else {
            this.mzkp_text = "";
        }
    };

    function getCachedBitmap(text) {
        // 制御文字を評価
        const evaluatedText = text.replace(/\\V\[(\d+)\]/gi, (_, n) => {
            return $gameVariables.value(parseInt(n));
        });
        
        // キャッシュにあれば返す
        if (bitmapCache.has(evaluatedText)) {
            return bitmapCache.get(evaluatedText);
        }
        
        // 新規作成
        const bitmap = createTextPictureBitmap(text);
        
        // キャッシュに追加（上限管理）
        if (bitmapCache.size >= MAX_CACHE_SIZE) {
            const firstKey = bitmapCache.keys().next().value;
            const oldBitmap = bitmapCache.get(firstKey);
            if (oldBitmap && oldBitmap.mzkp_isTextPicture) {
                oldBitmap.destroy();
            }
            bitmapCache.delete(firstKey);
        }
        
        bitmapCache.set(evaluatedText, bitmap);
        return bitmap;
    }

    function createTextPictureBitmap(text) {
        const tempWindow = new Window_Base(new Rectangle());
        const size = tempWindow.textSizeEx(text);
        tempWindow.padding = 0;
        tempWindow.move(0, 0, size.width, size.height);
        tempWindow.createContents();
        tempWindow.drawTextEx(text, 0, 0, 0);
        const bitmap = tempWindow.contents;
        tempWindow.contents = null;
        tempWindow.destroy();
        bitmap.mzkp_isTextPicture = true;
        return bitmap;
    }

    function destroyTextPictureBitmap(bitmap) {
        if (bitmap && bitmap.mzkp_isTextPicture) {
            bitmap.destroy();
        }
    }
})();