/*:
 * @target MZ
 * @plugindesc Quick item use + return/options unified menu (optimized)
 * @author WD
 *
 * @param itemIdVar
 * @text Item ID Variable
 * @type variable
 * @default 81
 * @desc Variable to store the used item ID
 *
 * @param hotkey
 * @text Hotkey
 * @type select
 * @option None
 * @value none
 * @option Tab Key
 * @value tab
 * @option Shift Key
 * @value shift
 * @option Q Key
 * @value pageup
 * @option E Key
 * @value pagedown
 * @default tab
 * @desc Hotkey to open unified menu
 *
 * @param returnMapId
 * @text Return Map ID
 * @type number
 * @default 1
 * @desc Map ID to return to
 *
 * @param returnX
 * @text Return X Coordinate
 * @type number
 * @default 10
 * @desc X coordinate for return
 *
 * @param returnY
 * @text Return Y Coordinate
 * @type number
 * @default 10
 * @desc Y coordinate for return
 *
 * @param returnDirection
 * @text Return Direction
 * @type select
 * @option Down
 * @value 2
 * @option Left
 * @value 4
 * @option Right
 * @value 6
 * @option Up
 * @value 8
 * @default 2
 * @desc Direction character faces after return
 *
 * @param returnCommonEvent
 * @text Return Common Event
 * @type common_event
 * @default 0
 * @desc Common event to run on return (0 = direct teleport)
 *
 * @command openUnifiedMenu
 * @text Open Unified Menu
 * @desc Open item/return/options unified menu
 *
 * @help
 * ============================================================================
 * Unified Menu
 * - Press Tab key to open "Item Use/Return/Options" menu
 * - Select item use to display item list
 * - Select return to teleport to set location
 * - Select options to open game settings
 * 
 * Item Settings
 * - Item note: <commonEvent:23> (link to common event)
 * - Item note: <oneTime> (close window after single use)
 * 
 * Return Location Settings
 * - Set map ID, X, Y coordinates, and direction in plugin parameters
 * 
 * Return Common Event Settings
 * - If set, runs the common event when return is selected
 * - Configure choices, initialization, and transfers in the common event
 * - Set to 0 for direct return (default behavior)
 * ============================================================================
 */

(() => {
    'use strict';
    
    const pluginName = "WD_ItemUse";
    const params = PluginManager.parameters(pluginName);
    const itemIdVar = Number(params['itemIdVar'] || 81);
    const hotkey = String(params['hotkey'] || 'tab');
    const returnMapId = Number(params['returnMapId'] || 1);
    const returnX = Number(params['returnX'] || 10);
    const returnY = Number(params['returnY'] || 10);
    const returnDirection = Number(params['returnDirection'] || 2);
    const returnCommonEvent = Number(params['returnCommonEvent'] || 0);
    
    if (!Input.keyMapper[9]) Input.keyMapper[9] = 'tab';
    if (!Input.keyMapper[16]) Input.keyMapper[16] = 'shift';
    
    const HOTKEY_MAP = {
        tab: 'tab',
        shift: 'shift',
        pageup: 'pageup',
        pagedown: 'pagedown'
    };
    
    class Scene_UnifiedMenu extends Scene_MenuBase {
        create() {
            super.create();
            this.createCategoryWindow();
        }
        
        createCategoryWindow() {
            const ww = 500, wh = 220;
            const rect = new Rectangle(
                (Graphics.boxWidth - ww) / 2,
                (Graphics.boxHeight - wh) / 2,
                ww, wh
            );
            this._categoryWindow = new Window_UnifiedCategory(rect);
            this._categoryWindow.setHandler('item', this.onItemCategory.bind(this));
            this._categoryWindow.setHandler('return', this.onReturnCategory.bind(this));
            this._categoryWindow.setHandler('options', this.onOptionsCategory.bind(this));
            this._categoryWindow.setHandler('cancel', this.popScene.bind(this));
            this.addWindow(this._categoryWindow);
        }
        
        createBackground() {
            super.createBackground();
            this.setBackgroundOpacity(192);
        }
        
        onItemCategory() {
            this._categoryWindow.hide();
            this._categoryWindow.deactivate();
            this.createItemWindow();
        }
        
        onReturnCategory() {
            SoundManager.playUseSkill();
            if (returnCommonEvent > 0) {
                $gameTemp.reserveCommonEvent(returnCommonEvent);
                SceneManager.goto(Scene_Map);
            } else {
                $gamePlayer.reserveTransfer(returnMapId, returnX, returnY, returnDirection, 0);
                SceneManager.goto(Scene_Map);
            }
        }
        
        onOptionsCategory() {
            SceneManager.push(Scene_Options);
        }
        
        createItemWindow() {
            if (this._itemWindow) return;
            
            const ww = 500, wh = 450;
            const rect = new Rectangle(
                (Graphics.boxWidth - ww) / 2,
                (Graphics.boxHeight - wh) / 2,
                ww, wh
            );
            this._itemWindow = new Window_UnifiedItemList(rect);
            this._itemWindow.setHandler('ok', this.onItemOk.bind(this));
            this._itemWindow.setHandler('cancel', this.onItemCancel.bind(this));
            this.addWindow(this._itemWindow);
            this._itemWindow.activate();
        }
        
        onItemCancel() {
            this._itemWindow.hide();
            this._itemWindow.deactivate();
            this._categoryWindow.show();
            this._categoryWindow.activate();
        }
        
        onItemOk() {
            const item = this._itemWindow.item();
            if (!item) {
                this._itemWindow.activate();
                return;
            }
            
            $gameVariables.setValue(itemIdVar, item.id);
            
            const ceId = Number(item.meta.commonEvent || 0);
            if (ceId > 0) $gameTemp.reserveCommonEvent(ceId);
            
            $gameParty.consumeItem(item);
            SoundManager.playUseItem();
            this._itemWindow.refresh();
            
            if (item.meta.oneTime) {
                this.popScene();
                return;
            }
            
            if (this._itemWindow.maxItems() > 0) {
                this._itemWindow.activate();
            } else {
                this.popScene();
            }
        }
        
        update() {
            super.update();
            if (this.isActive() && this.isHotkeyTriggered()) {
                if (this._itemWindow && this._itemWindow.active) {
                    this.onItemCancel();
                } else if (this._categoryWindow && this._categoryWindow.active) {
                    this.popScene();
                }
            }
        }
        
        isHotkeyTriggered() {
            return hotkey !== 'none' && Input.isTriggered(HOTKEY_MAP[hotkey]);
        }
    }
    
    class Window_UnifiedCategory extends Window_Command {
        initialize(rect) {
            super.initialize(rect);
            this.selectLast();
        }
        
        makeCommandList() {
            this.addCommand("\u30A2\u30A4\u30C6\u30E0\u4F7F\u7528", 'item');
            this.addCommand("\u5E30\u9084", 'return');
            this.addCommand("\u30AA\u30D7\u30B7\u30E7\u30F3", 'options');
        }
        
        maxCols() {
            return 1;
        }
        
        drawItem(index) {
            const rect = this.itemLineRect(index);
            const align = 'center';
            this.resetTextColor();
            this.changePaintOpacity(this.isCommandEnabled(index));
            this.drawText(this.commandName(index), rect.x, rect.y, rect.width, align);
        }
        
        selectLast() {
            this.select(0);
        }
    }
    
    class Window_UnifiedItemList extends Window_ItemList {
        initialize(rect) {
            super.initialize(rect);
            this.refresh();
            this.select(0);
        }
        
        maxCols() {
            return 1;
        }
        
        includes(item) {
            if (!item || !DataManager.isItem(item)) return false;
            if (item.itypeId !== 1) return false;
            if (item.occasion === 3) return false;
            if (item.occasion === 1 && !$gameParty.inBattle()) return false;
            if (item.occasion === 2 && $gameParty.inBattle()) return false;
            return true;
        }
        
        isEnabled(item) {
            return $gameParty.canUse(item);
        }
        
        makeItemList() {
            this._data = $gameParty.items().filter(item => this.includes(item));
        }
        
        drawItem(index) {
            const item = this.itemAt(index);
            if (!item) return;
            
            const rect = this.itemLineRect(index);
            const numW = 60;
            this.changePaintOpacity(this.isEnabled(item));
            this.drawItemName(item, rect.x, rect.y, rect.width - numW);
            this.drawText(':', rect.x, rect.y, rect.width - 32, 'right');
            this.drawText($gameParty.numItems(item), rect.x, rect.y, rect.width - 8, 'right');
            this.changePaintOpacity(true);
        }
    }
    
    PluginManager.registerCommand(pluginName, "openUnifiedMenu", () => {
        SceneManager.push(Scene_UnifiedMenu);
    });
    
    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.call(this);
        if (hotkey !== 'none' && !SceneManager.isSceneChanging() && this.isActive()) {
            if (this.canOpenUnifiedMenu() && Input.isTriggered(HOTKEY_MAP[hotkey])) {
                SceneManager.push(Scene_UnifiedMenu);
            }
        }
    };
    
    Scene_Map.prototype.canOpenUnifiedMenu = function() {
        return !$gameMessage.isBusy();
    };
})();